/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 
#include "mdpApp.h"
#include <stdlib.h>
#include <stdarg.h>
#include <time.h>
 
//For experimental IPsec hack
#ifdef NETSEC
#include <net/security.h>
void *netsec_request = NULL;
int netsec_requestlen = 0;
#endif // NETSEC

// Session flags
enum 
{
    MDP_CLIENT =      0x01,
    MDP_SERVER =      0x02,
    UNICAST_NACKS =   0x04,
    MULTICAST_ACKS =  0x08,
    EMCON =           0x10,
    POSITIVE_ACK =    0x20
};
    
extern MdpError ParseAckListFile(MdpSessionHandle sessionHandle, const char *path);

MdpApp::MdpApp()
    : session_handle(0), log_file(NULL),
      session_port(0), session_tx_port(0), session_ttl(8), 
      session_tos(0), session_loopback(false),
      session_tx_rate(64000), session_flags(0),
#ifdef PROTO_DEBUG
      tx_packet_drop_rate(0.0), rx_packet_drop_rate(0.0),
#endif  // PROTO_DEBUG
      status_reporting(true), reuse_ports(false), archive_mode(false), 
      rx_cache_count_min(8), rx_cache_count_max(16), 
      rx_cache_size_max(8*1024*1024), 
      rx_buffer_size(MDP_DEFAULT_BUFFER_SIZE), stream_integrity(true),
      tx_cleanup(false), flush_tx_files(false), tx_repeat_count(0), 
      tx_object_interval(0.0), tx_repeat_interval(1.0), flow_control(false), 
      tx_rate_min((float)0.0), tx_rate_max((float)0.0),
      initial_grtt((float)MDP_DEFAULT_GRTT_ESTIMATE),
      segment_size(1024), ndata(64), nparity(32), auto_parity(0),
      tx_cache_count_min(MDP_DEFAULT_TX_CACHE_COUNT_MIN), 
      tx_cache_count_max(MDP_DEFAULT_TX_CACHE_COUNT_MAX), 
      tx_cache_size_max(MDP_DEFAULT_TX_CACHE_SIZE_MAX)    
{
    strcpy(session_name, "MDPv2 Session");
    session_address[0] = '\0';
    interface_address[0] = '\0';
    archive_dir[0] = '\0';
    ack_list_file[0] = '\0';
    
    post_processor[0] = '\0';
    memset(processor_opt, 0, 32*sizeof(char*));   
    
    // Init base_object_id to random value by default
    struct timeval theTime;
    GetSystemTime(&theTime);
    srand(theTime.tv_usec);  // seed random number generator
    base_object_id  =  rand() * (0xffffffff/RAND_MAX); 
}     

MdpApp::~MdpApp()
{
    for(int i = 0; i < 32; i++)
    {
        if (processor_opt[i]) 
        {
            delete [] processor_opt[i];
            processor_opt[i] = NULL;
        }
    }
    if (log_file) CloseLogFile();
}

void MdpApp::Alert(const char *format, ...)
{
    fprintf(stderr, "mdp: ");
    va_list args;
    va_start(args, format);
    vfprintf(stderr, format, args);
    va_end(args);
    fprintf(stderr, "\n");     
}  // end MdpApp::Alert() 

// Call this _before_ processing command line arguments
bool MdpApp::Init(MdpTimerInstallCallback*  timerInstaller,
                  MdpSocketInstallCallback* socketInstaller)
{       
	mdp_instance = MdpInit();
	if (MDP_NULL_INSTANCE == mdp_instance)
    {
        Alert("Error initializing MDPv2 library!");
        return false;
    }	
    MdpInstanceSetUserData(mdp_instance, (const void*)this);
    MdpSetNotifyCallback(mdp_instance, MdpApp::MdpNotifyFunc); 
    MdpSetTimerInstallCallback(mdp_instance, timerInstaller);
    MdpSetSocketInstallCallback(mdp_instance, socketInstaller);
    return true;
}  // end MdpApp::Init()

// Call this _after_ processing command line arguments
bool MdpApp::Start()
{
    // Check for validity of launch options
    if (!session_flags) 
    {
        Alert("Error! Must pick \"client\" mode (-D <cache/archive directory>),\n"
              "       \"server mode\" (i.e. supply file(s) to transmit), or both.\n"
              "       Use \"-h\" option for usage help.");
        return false;
    };
    
    if ('\0' == session_address[0])
    {
        Alert("Error! Must specify session address/port.\n"
			   "      (\"-A\" <address/port>)");
        return false;
    };
       
    // Create instance of MDP session
    MdpError err;
    session_handle = MdpNewSession(mdp_instance, 
                                   session_address, session_port, 
                                   session_tx_port, session_ttl, &err);      
    if (MDP_NULL_SESSION == session_handle)
    {
       Alert("Error creating new session!");
       return false;
    }
    
    // General MdpSession initialization
    MdpSessionSetTxRate(session_handle, session_tx_rate);
    MdpSessionSetServerGrttEstimate(session_handle, initial_grtt);
    MdpSessionSetStatusReporting(session_handle, status_reporting);
    if ('\0' != interface_address[0])
    {
        if (MDP_ERROR_NONE != 
                MdpSessionSetMulticastInterfaceAddress(session_handle, interface_address))
        {
            Alert("Error setting interface address to:\n"
                  "\"%-.32s\"", interface_address);
            return false;   
        }
    }
    MdpSessionSetTOS(session_handle, session_tos);
    MdpSessionSetMulticastLoopback(session_handle, session_loopback);
    MdpSessionSetPortReuse(session_handle, reuse_ports);
            
    // Debug options
#ifdef PROTO_DEBUG
    MdpSessionSetRecvDropRate(session_handle, rx_packet_drop_rate);
    MdpSessionSetSendDropRate(session_handle, tx_packet_drop_rate);
#endif // PROTO_DEBUG
    
    
    // Client initialization and startup
    if (session_flags & MDP_CLIENT) 
    {
        rx_file_cache.SetCacheSize(rx_cache_count_min, rx_cache_count_max, rx_cache_size_max);
        MdpSessionSetArchiveMode(session_handle, archive_mode);  
        MdpSessionSetArchiveDirectory(session_handle, archive_dir);
        if (session_flags & EMCON) MdpSessionSetClientEmcon(session_handle, true);
        if (session_flags & UNICAST_NACKS) 
        {
            MdpSessionSetClientUnicastNacks(session_handle, true);
            fprintf(stderr, "UNICASTING NACKS ...\n");
        }
        if (session_flags & MULTICAST_ACKS) MdpSessionSetClientMulticastAcks(session_handle, true);
        if (session_flags & POSITIVE_ACK) MdpSessionSetClientAcking(session_handle, true); 
        MdpSessionSetClientStreamIntegrity(session_handle, stream_integrity);
        
        // Open client
        err = MdpSessionOpenClient(session_handle, rx_buffer_size);
        if (MDP_ERROR_NONE != err)
        {
            Alert("Error opening client! (error = %d)\n", err);
            return false;
        }      
    }
    
    
    // Server initialization and startup 
    if (session_flags & MDP_SERVER)
    {
        // Establish reference for updated file transmission
        // (Note: All files will be transmitted on the first pass)
        tx_file_list.InitUpdateTime(time(NULL));
        // Uncomment the following line to cause _no_ file to tx'd first pass 
        // (i.e. only files changed _after_ server startup get transmitted)
        // tx_file_list.ResetIterator();
    
        if (session_flags & EMCON) MdpSessionSetServerEmcon(session_handle, true);
        
        // This list of clients will be requested to ACK (unless the clients
        // launch without the "-K" option and thus their reports will get
        // them removed from the positive ack list)
        if ('\0' != ack_list_file[0])
        {
            MdpError err = ParseAckListFile(session_handle, ack_list_file);
            if (MDP_ERROR_NONE != err)
            {
                Alert("Error parsing positive ACK list file");
                return false;
            }
        }
        MdpSessionSetBaseObjectTransportId(session_handle, base_object_id);
        MdpSessionSetCongestionControl(session_handle, flow_control);
        MdpSessionSetTxRateBounds(session_handle, (double)tx_rate_min, (double)tx_rate_max);
        MdpSessionSetTxCacheDepth(session_handle, tx_cache_count_min, tx_cache_count_max, tx_cache_size_max);
        MdpSessionSetAutoParity(session_handle, auto_parity);
        
        // Open server
        err = MdpSessionOpenServer(session_handle, segment_size, ndata, nparity);
        if (MDP_ERROR_NONE != err)
        {
            Alert("Error opening server! (error = %d\n", err);
            if (session_flags & MDP_CLIENT) MdpSessionCloseClient(session_handle);
            return false;
        }
        
        char path[PATH_MAX], name[PATH_MAX];
        // For -W option, flush file list first
        if (flush_tx_files)
        {
           while (tx_file_list.GetNextFile(path, name));
        }
        // Queue first file for transmission
        
        if (tx_file_list.GetNextFile(path, name))
        {
            MdpObjectHandle obj = MdpSessionQueueTxFile(session_handle, path, name, &err);
            if (MDP_NULL_OBJECT == obj)
            {
                Alert("Error queueing tx file! (error = %d)\n", err);
                MdpSessionCloseServer(session_handle);
                if (session_flags & MDP_CLIENT) MdpSessionCloseClient(session_handle);
                return false;
            }  
        }
        else
        {
            if (tx_file_list.UpdatesOnly() && tx_repeat_count) 
            {
                // Poll for future file updates/additions
                StartTxIntervalTimer(tx_repeat_interval);  
                SetStatusText("Server: No current tx files, monitoring \n"
                              "        directory/file list for updates ...");
            }
            else
            {
                SetStatusText("Server: Warning! No files to transmit.");  
            }          
        }         
    }  
    return true;
}  // end MdpApp:Start()


void MdpApp::OnExit()
{
    if (log_file) CloseLogFile();
	StopTxIntervalTimer();
	MdpDeleteSession(mdp_instance, session_handle);
	session_handle = NULL;
	tx_file_list.Destroy();
	rx_file_cache.Destroy();
	MdpDestroy(mdp_instance);
	mdp_instance = NULL;
}  // end MdpApp::OnExit();

// This function is called as a callback by the MDPv2 protocol engine
bool MdpApp::MdpNotifyFunc(MdpNotifyCode        notifyCode,
                           MdpInstanceHandle    instanceHandle,
                           MdpSessionHandle     sessionHandle,
                           MdpNodeHandle        nodeHandle,
                           MdpObjectHandle      objectHandle,
                           MdpError             errorCode)
{
    MdpApp* app = (MdpApp*)MdpInstanceGetUserData(instanceHandle);
	return app->OnNotify(notifyCode, sessionHandle, nodeHandle,
                         objectHandle, errorCode);
}  // end MdpApp::MdpNotifyFunc()

bool MdpApp::OnNotify(MdpNotifyCode     notifyCode,
                      MdpSessionHandle  /*sessionHandle*/,
                      MdpNodeHandle     /*nodeHandle*/,
                      MdpObjectHandle   objectHandle,
                      MdpError          errorCode)
{
	static char text[PATH_MAX], name[PATH_MAX];
    unsigned long recvd, size, progress;
    unsigned short nameLen = PATH_MAX;
    
    switch(notifyCode)
    {
        case MDP_NOTIFY_ERROR:
            Alert("Notify error code = %d\n", errorCode);
            break;
            
        case MDP_NOTIFY_TX_OBJECT_START:
            if (MdpObjectGetInfo(objectHandle, name, &nameLen))
            {            
                sprintf(text, "Tx file: \"%.64s\"\n(Beginning transmission ...)\n",
                                name);
                SetStatusText(text);
            }
            else
            {
                SetStatusText("Error getting tx file name!");
            }
            break;
            
        case MDP_NOTIFY_TX_OBJECT_FIRST_PASS:
            if (MdpObjectGetInfo(objectHandle, name, &nameLen))
            {            
                sprintf(text, "Tx file: \"%.64s\"\n(First pass transmission complete)\n",
                                name);
                SetStatusText(text);
            }
            else
            {
                SetStatusText("Error getting tx file name!");
            }
            break;
            
        case MDP_NOTIFY_TX_OBJECT_ACK_COMPLETE:
            if (MdpObjectGetInfo(objectHandle, name, &nameLen))
            {            
                sprintf(text, "Tx file: \"%.64s\"\n(Acknowledgement complete)\n",
                                name);
                SetStatusText(text);
            }
            else
            {
                SetStatusText("Error getting tx file name!");
            }
            break;
        
        case MDP_NOTIFY_TX_OBJECT_FINISHED:
            if (MdpObjectGetInfo(objectHandle, name, &nameLen))
            {            
                sprintf(text, "Tx file: \"%.64s\"\n(Server finished with file)\n", name);
                SetStatusText(text);
                if (MDP_OBJECT_FILE == MdpObjectGetType(objectHandle))
                {
                    if (tx_cleanup)
                    {
                        MdpObjectGetFileName(objectHandle, name, PATH_MAX);
                        unlink(name);
                    }
                    if (log_file)
                    {
                        time_t theTime = time(NULL);
                        fprintf(log_file, "TX FILE FINISHED : \"%s\" : %s",
                                          name, ctime(&theTime));
                        fflush(log_file);
                    }
                }
            }
            else
            {
                SetStatusText("Error getting tx file name!");
            }
            break;
            
        case MDP_NOTIFY_TX_QUEUE_EMPTY:
            // If there's not time between objects, 
            // queue next object immediately
            // This prevents FLUSH messages from being 
            // sent between objects
            if (tx_object_interval > 0.0)
            {
                StartTxIntervalTimer(tx_object_interval);
            }
            else
            {
                StopTxIntervalTimer();
                MdpApp::OnTxIntervalTimeout();
            }
            break;
            
        case MDP_NOTIFY_RX_OBJECT_INFO:
            if (MdpObjectGetInfo(objectHandle, name, &nameLen))
            {         
                if (log_file)
                {
                    time_t theTime = time(NULL);
                    fprintf(log_file, "RX FILE INFO : \"%s\" : %s",
                                      name, ctime(&theTime));
                    fflush(log_file);
                }
            }
            break;
            
        case MDP_NOTIFY_RX_OBJECT_START:
            if (MdpObjectGetInfo(objectHandle, name, &nameLen))
            {            
                sprintf(text, "\"%.64s\"", name);
                SetStatusText(text);
            }
            else
            {
                SetStatusText("New recv file starting ...");
            }
            SetProgressMeter((unsigned int) 0);
            break;
            
        case MDP_NOTIFY_RX_OBJECT_UPDATE:
            recvd = MdpObjectGetRecvBytes(objectHandle);
            size = MdpObjectGetSize(objectHandle);
            if (size)
                progress = (100 * recvd)/size;
            else
                progress = 100;
            if (MdpObjectGetInfo(objectHandle, name, &nameLen))
            {            
                sprintf(text, "\"%.64s\" (%lu/%lu)", name, recvd, size);
                SetStatusText(text);
            }
            else
            {
                SetStatusText("New recv file update ...");
            }
            SetProgressMeter((int) progress);
            break;
        
        case MDP_NOTIFY_RX_OBJECT_COMPLETE:
            if (MdpObjectGetInfo(objectHandle, name, &nameLen))
            {            
                sprintf(text, "Rx file: \"%.64s\"\n         (Complete)", name);
                SetStatusText(text);
            }
            SetProgressMeter(100);
            if (MDP_OBJECT_FILE == MdpObjectGetType(objectHandle))
            {
                MdpObjectGetFileName(objectHandle, name, PATH_MAX);
                if (HasPostProcessor()) PostProcess(name);
                if (!archive_mode)
                {
					if (!rx_file_cache.CacheFile(name))
                        Alert("Error cacheing receive file!\n");
                }
                if (log_file)
                {
                    time_t theTime = time(NULL);
                    fprintf(log_file, "RX FILE COMPLETE : \"%s\" : %s",
                        name, ctime(&theTime));
                    fflush(log_file);
                }
            }
            break;
            
        case MDP_NOTIFY_OBJECT_DELETE:
            break;
            
        case MDP_NOTIFY_LOCAL_SERVER_CLOSED:
            fprintf(stderr, "mdp: NOTIFY_SERVER_CLOSED ...\n");
            break;
        
        default:
            fprintf(stderr, "mdp: Notify ??? (%d) ...\n", notifyCode);
            break;           
    }
    return true;
}  // end MdpApp::OnNotify()


void MdpApp::OnTxIntervalTimeout()
{
    char path[PATH_MAX], name[PATH_MAX], text[256];
    if (tx_file_list.GetNextFile(path, name))
    {
        MdpError err;
        MdpObjectHandle obj = 
            MdpSessionQueueTxFile(session_handle, path, name, &err);
        if (MDP_NULL_OBJECT == obj)
        {
            sprintf(text, "Error %d queueing \"%.64s%.64s\" \nfor transmission.",
                        err, path, name); 
            SetStatusText(text);                     
        }  
    }
    else
    {
		if (tx_repeat_count)
        {
            if (tx_repeat_count > 0) tx_repeat_count--;
            tx_file_list.ResetIterator();
            
            // Install timeout for (tx_repeat_interval - tx_object_interval)
            double delay;
            if (tx_repeat_count)
            {
                if (tx_repeat_interval > tx_object_interval)
                    delay = tx_repeat_interval - tx_object_interval;
                else
                    delay = 0.0;
            }
            else if (tx_file_list.UpdatesOnly())
            {
                delay = tx_repeat_interval;
            }
            else
            {
                delay = 0.0;  // this one should never happen
            }            
            StartTxIntervalTimer(delay);
        }
        else
        {
            sprintf(text, "Server transmission complete");
            SetStatusText(text);
        }
    }   
} // end MdpApp:OnTxIntervalTimeout()

void MdpApp::SetPostProcessor(const char* theCmd)
{
    if (!theCmd || !strcmp(theCmd, "none") || ('\0' == theCmd[0]))// No post processor
    {
        post_processor[0] = '\0';
        SetProcessorOpt(0, NULL);
        return;
    }
    strncpy(post_processor, theCmd, PATH_MAX);
    char *tk = strtok(post_processor, "\t \n\r");
    int i = 0;
    while (tk)
    {
        if (i != 0) SetProcessorOpt(i-1, tk);
        tk = strtok(NULL, "\t \n\r");
        if (++i >= 32)
        {
            Alert("PostProcessor has too many command line options!\n");
            break;
        }
    }  
    if (i) 
    {
        SetProcessorOpt(i-1, NULL);  
    }
    else
    {
        post_processor[0] = '\0';
        SetProcessorOpt(0, NULL);
    }
}  // end MdpApp::SetPostProcessor()

void MdpApp::SetProcessorOpt(int i, char *opt)
{
    ASSERT((i>=0) && (i<32));
    if (processor_opt[i]) delete[] processor_opt[i];
    if (opt)
    {
        if (!(processor_opt[i] = new char[strlen(opt)+1]))
        {
            perror("tkMdp: calloc(post processor option) error:");
            processor_opt[i] = (char *) NULL;
            return;
        }
        strcpy(processor_opt[i], opt);
    }
    else
    {
        processor_opt[i] = NULL;
    }   
}  // end MdpApp::SetProcessorOpt()

bool MdpApp::OpenLogFile(const char* thePath)
{
    if (!(log_file = fopen(thePath, "w+")))
        return false;
    else
        return true;
}  // MdpApp::OpenLogFile()

void MdpApp::CloseLogFile()
{
    if (log_file) fclose(log_file);
    log_file = NULL;
}  // end MdpApp::CloseLogFIle()

void MdpApp::SetStatusText(const char* /*theText*/)
{
    
}

void MdpApp::SetProgressMeter(unsigned int /*percent*/)
{
    
}

void MdpApp::PostProcess(const char* /*thePath*/)
{
    
}

// (TBD) Replace this with a more sophisticated command list
// (i.e. "words/strings" instead of single letter flags)

const char* MDPAPP_CMD_LINE_FLAGS = 
    "aA:b:BcC:d:D:e:E:fF:g:G:i:I:Jk:K:l:L:mMn:No:p:P:Q:r:R:s:S:t:TuUWwX:y:z:Z:";

// Process MdpApp command line options and arguments
bool MdpApp::ProcessCommand(char opt, const char* optarg)
{
	char* ptr;
    switch(opt) 
    {
        case 'S':
#ifdef NETSEC
            if (net_security_strtorequest(optarg, &netsec_request,
                                          &netsec_requestlen))
            {
                Alert("net_security_strtorequest('%s', ...) failed.",
                        optarg);
                return false;
            }
#else
            Alert("'-S' option not supported in this build!");
            return false;
#endif  // NETSEC
            break;            
        case 'C':
            strncpy(session_name, optarg, MDP_SESSION_NAME_MAX);
            break;
        case 'A':
            ptr = strchr(optarg, '/');
            if (!ptr)
            {
                Alert("Invalid destination address/port: \"%s\"", optarg);
                return false;
            }
            *ptr++ = '\0';
            strncpy(session_address, optarg, MDP_NODE_NAME_MAX);
            session_port = atoi(ptr);
            break;
        case 'J':
            reuse_ports = true;
            break;
        case 't':
            session_ttl = atoi(optarg);
            break;
        case 'F':
            strncpy(interface_address, optarg, 32);
            break;
        case 'r':
            if (1 != sscanf(optarg, "%lu", &session_tx_rate))
            {
                Alert("Invalid 'txRate' specified!");
                return false;
            }
            break;   
        case 'M':
            session_flags |= EMCON;
            break;

        case 'L':
            if (!OpenLogFile(optarg))
            {
                perror("Error opening log file");
                return false;   
            }
            break;
        case 'B':
            session_loopback = true;
            break;
        case 'Q':
        {
            int tempint;
            if (1 != sscanf(optarg, "%i", &tempint))
            {
                Alert("Invalid '-Q <tos>' specification!");
                return false;   
            }                
            session_tos = (unsigned char) tempint;
            break;
        }

#ifdef PROTO_DEBUG
        case 'd':
            SetDebugLevel(atoi(optarg));
            break;

        case 'l':
            if (!OpenDebugLog(optarg))
            {
                Alert("Error opening debug log file!");
                return false;
            }
            break;

        case 'T':
            MdpSetMessageTrace(true);  // Turn on tracing regardless of debug level
            break;

        case 'e':
            sscanf(optarg, "%f", &rx_packet_drop_rate);
            if (rx_packet_drop_rate < 0.0)
            {
                Alert("Invalid rx_packet_drop_rate specified!");
                return false;
            }
            break;
        case 'E':
            sscanf(optarg, "%f", &tx_packet_drop_rate);
            if (tx_packet_drop_rate < 0.0)
            {
                Alert("Invalid tx_packet_drop_rate specified!");
                return false;
            }
            break;
#endif // PROTO_DEBUG           
        
        case 'N':
            status_reporting = false;
            break;
  
    ///////////////////////////////////////////////////////////////////      
    // Server options 
        case 's':
            if (1 != sscanf(optarg, "%hu", &segment_size))
            {
                Alert("Invalid segment_size!");
                return false;
            }
            break;
        case 'b':
            ndata = (unsigned char) atoi(optarg);
            break;
        case 'p':
            nparity = (unsigned char) atoi(optarg);
            break;
        case 'n':
            auto_parity = (unsigned char) atoi(optarg);
            break;
        case 'f':
            flow_control = true;
            break;  
        case 'G':
            if (2 != sscanf(optarg, "%f:%f", &tx_rate_min, &tx_rate_max))
            {
                Alert("Error parsing tx cache specification!");
                return false;
            }
            flow_control = true;
            break;           
        case 'g':
            if (1 != sscanf(optarg, "%f", &initial_grtt))
            {
                Alert("Invalid 'initialGrtt' specified!");
                return false;
            }
            break;   
        case 'o':
            base_object_id = atoi(optarg);
            break;
        case 'i':
            sscanf(optarg, "%lf", &tx_object_interval);
            if (tx_object_interval < 0)
            {
                Alert("Invalid tx_object_interval specified!");
                return false;
            }
            break;
        case 'R':
            tx_repeat_count = atoi(optarg);
            break;
        case 'U':
			tx_file_list.SetUpdatesOnly(true);  // _should_ also set session_repeats!! 
            break;
        case 'W':  // Don't send existing files 
			tx_file_list.SetUpdatesOnly(true); 
            flush_tx_files = true;  
            break;
        
        case 'I':
            sscanf(optarg, "%lf", &tx_repeat_interval);
            if (tx_repeat_interval < 0)
            {
                Alert("Invalid tx_repeat_interval specified!");
                return false;
            }
            break;
        case 'c':
            tx_cleanup = true;
            break;
        case 'k':
            strncpy(ack_list_file, optarg, PATH_MAX);
            break;
        case 'Z':
            if (3 != sscanf(optarg, "%lu:%lu:%lu", &tx_cache_count_min, 
                             &tx_cache_count_max, &tx_cache_size_max))
            {
                Alert("Error parsing tx cache specification!");
                return false;
            }
            break;
        // case ZERO means no option flag, therefore it's the tx file list
        case 0:
            session_flags |= MDP_SERVER;
            if((NULL == tx_file_list.NewFileItem(optarg)))
            {
               Alert("Error adding file/directory to tx list!");
               return false;
            }
			break;
            
    ///////////////////////////////////////////////////////////////////      
    // Client options 
        case 'D':
			session_flags |= MDP_CLIENT;
            strncpy(archive_dir, optarg, PATH_MAX);
            // Test that the session_archive_dir is writable
            if (!MdpFileIsWritable(archive_dir))
            {
                Alert("Can't write to archive directory: \"%s\"", archive_dir);
                return false;
            }

            break;
        case 'a':
            archive_mode = true;
            break;
        case 'K':
            session_flags |= POSITIVE_ACK;
            break;
        case 'u':
            session_flags |= UNICAST_NACKS;
            break;
        case 'm':
            session_flags |= MULTICAST_ACKS;
            break;
        case 'X':
            SetPostProcessor(optarg);
            break;
        case 'y':
            if (1 != sscanf(optarg, "%lu", &rx_buffer_size))
            {
                Alert("Error parsing recv buffer specification!");
                return false;
            } 
            break;
        case 'z':
            if (3 != sscanf(optarg, "%lu:%lu:%lu", &rx_cache_count_min, 
                             &rx_cache_count_max, &rx_cache_size_max))
            {
                Alert("Error parsing recv cache specification!");
                return false;
            }
            break;
        
        case 'w':
            // Disable stream integrity
            stream_integrity = false;
            break;

        case 'h':
        default:
            Usage();
            return false;
            break;
    }  // end switch(opt)
    return true;
}  // end MdpApp::ProcessCommand

void MdpApp::Usage() 
{
   printf("tkMdp -A <address>/<port> [-C <sessionName>][-t <ttl>][-Q <tos>]\n"
          "       [-M][-N][-F <interfaceAddress>][-B][-r <txRate> (bits/sec)]\n"
#ifdef PROTO_DEBUG
          "       [-d <debugLevel>][-l <debugLogFile>][-T]\n"
          "       [-e <percentRecvLoss> (0-100%%)][-E <percentSendLoss> (0-100%%)]\n"
#endif // PROTO_DEBUG
          "   (client-mode:)   -D <cache/archive directory> [-a][-u][-m][-K]\n"
          "                    [-z <rxMinCount:rxMaxCount:rxMinSize>]\n"
          "                    [-X <\"post processor cmd and options\">]\n"
          "   (server-mode:)   [-U][-c][-f][-G <minRate:maxRate>][-s <segmentSize> (bytes)]\n"
          "                    [-b <blockSize>][-p <numParity>][-n <autoParity>]\n"
          "                    [-g <initialGrtt>]\n"
          "                    [-o <baseObjectId>][-i <objectInterval> (sec)]\n"
          "                    [-R <repeats>][-I <repeatInterval> (sec)] \n"
          "                    [-Z <txMinCount:txMaxCount:txMinSize>]\n"
          "                    [-L <logFile>]\n"
          "                    [-k <ackListFile>] <tx file/dir names>\n"
          "   (help:)          [-h]\n\n");
}  // end usage()

