/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 
#ifndef _MDP_BIT_MASK
#define _MDP_BIT_MASK

#include <string.h>   // for memset()
#include "debug.h"

// The "ParseMask" routines return the number of bits set in the
// mask and fill the "locs" array with the indices of the set bits
        
// This can only handle masks for up to 256-member vectors
int MaskParse(const char *mask, int len, unsigned char *locs);

// This can handle very large bit masks 
unsigned long MaskParseLong(const char *mask, unsigned long len,
                            unsigned long *locs, unsigned long locs_len);

inline void MaskSet(char *mask, unsigned long index)
{
    mask[(index >> 3)] |= (0x80 >> (index & 0x07));
}

inline void MaskUnset(char *mask, unsigned long index)
{
    mask[(index >> 3)] &= ~(0x80 >> (index & 0x07));
}

void MaskSetBits(char *mask, unsigned long index, unsigned long count);
void MaskUnsetBits(char *mask, unsigned long index, unsigned long count);


unsigned long MaskFirstSet(const char* mask, unsigned long mask_len);
unsigned long MaskNextSet(unsigned long index, const char *mask, 
                          unsigned long mask_len);
unsigned long MaskLastSet(const char *mask, unsigned long mask_len);


inline bool MaskTest(const char *mask, unsigned long index)
{
    return (0 != (mask[(index >> 3)] & (0x80 >> (index & 0x07))));
}

// Compute mask1 = mask1 + mask2 (mask1 | mask2)
void MaskAdd(char* mask1, const char* mask2, unsigned long maskLen);
// Compute mask1 = mask1 - mask2 (mask1 & ~mask2)
void MaskSubtract(char* mask1, const char* mask2, unsigned long maskLen);

// Set mask1 = mask2
inline void MaskCopy(char* mask1, const char* mask2, unsigned long maskLen)
{
    memcpy(mask1, mask2, maskLen);   
}

// This class is a useful smarter bit mask (i.e it tracks where
// the lowest set bit location is (Note everything is set on reset)
class MdpBitMask
{
    friend class MdpBitMaskIterator;
    friend class MdpSlidingMask;     // dynamic BitMask variation
    
    // Members
    protected:  
        unsigned long   num_bits;
        unsigned long   first_set;  // index of lowest set bit
        unsigned long   mask_len;
        unsigned char*  mask;
        
    // Methods
    public:
        MdpBitMask();
        ~MdpBitMask();
        bool Init(unsigned long numBits);
        bool IsInited() {return (NULL != mask);}
        void Destroy();
        
        void Reset()  // set to all one's
        {
            memset(mask, 0xff, (mask_len-1));
            mask[mask_len-1] = 0x00ff << ((8 - (num_bits & 0x07)) & 0x07);
            first_set = 0;
        }
        void Clear()  // set to all zero's
        {
            memset(mask, 0, mask_len);
            first_set = num_bits;
        };
        unsigned long NumBits() {return num_bits;}
        const char* Mask() {return (char*)mask;}
        unsigned long MaskLen() {return mask_len;}
        unsigned long FirstSet() {return first_set;}
        unsigned long NextSet(unsigned long index);
        unsigned long NextUnset(unsigned long index);
        void Set(unsigned long index);
        void Unset(unsigned long index);
        void SetBits(unsigned long baseIndex, unsigned long count);
        void UnsetBits(unsigned long baseIndex, unsigned long count);
        bool Test(unsigned long index)
        {
            ASSERT(index < num_bits);
			return((0 != (mask[(index >> 3)] & (0x80 >> (index & 0x07)))));
        }
        bool IsSet() {return (first_set < num_bits);}
        bool IsClear() {return (first_set >= num_bits);}
        
        void Copy(MdpBitMask &theMask)
        {
            ASSERT(mask_len == theMask.mask_len);
            memcpy(mask, theMask.mask, mask_len);
            first_set = theMask.first_set;
        }        
        void Add(MdpBitMask *theMask);
        void AddRawMask(const char *theMask, unsigned long theLen);
        void AddRawOffsetMask(const char *theMask, unsigned long theLen,
                              unsigned long theOffset);
        void Display();
        
        void XCopy(MdpBitMask &theMask);
        void And(MdpBitMask &theMask);
};

// This class could be re-implemented using a circular buffer
// strategy to optimize performance further (no memmove()
class MdpSlidingMask
{
    public:
    // Methods
        MdpSlidingMask();
        bool Init(unsigned long numBits)
        {
            offset = 0;
            // Make sure numBits is byte multiple
            return bit_mask.Init(((numBits+7) >> 3) << 3);
        }
        bool IsInited() {return bit_mask.IsInited();}
        void Clear()
        {
            offset = 0;
            bit_mask.Clear();
        }
        void Destroy()
        {
            bit_mask.Destroy();
            offset = 0;
        }
        void SetOffset(unsigned long theOffset)
        {
            offset = theOffset & (~((unsigned long)0x07));
        }
        bool IsSet() {return bit_mask.IsSet();}
        bool InCurrentRange(unsigned long index);
        bool CanFreeSet(unsigned long index);
        unsigned long FirstSet() {return offset + bit_mask.FirstSet();}
        bool Set(unsigned long index);
        bool Test(unsigned long index)
        {
            return (InCurrentRange(index) ? bit_mask.Test(index - offset) : false);
        }
        void Unset(unsigned long index)
        {
            // Make sure it's in range
            if(InCurrentRange(index))
            {
                bit_mask.Unset(index - offset);
                Slide();
            }
        }
        bool FirstSet(unsigned long *index)
        {
            if (bit_mask.IsSet())
            {
                ASSERT(index);
                *index = offset + bit_mask.FirstSet();
                return true;
            }
            else
            {
                return false;
            }
        }
        bool NextSet(unsigned long *index);
        unsigned long NumBits() {return bit_mask.NumBits();}
        void Copy(MdpSlidingMask &theMask)
        {
            offset = theMask.offset;
            bit_mask.Copy(theMask.bit_mask);
        }
        
#ifdef PROTO_DEBUG
        void Display();
#endif // PROTO_DEBUG
                  
    private:
    // Members
        unsigned long offset;  
        MdpBitMask    bit_mask; 
    // Methods
        void Slide();
};

#endif // _MDP_BIT_MASK
