#!/usr/bin/tclsh

# Random number generator for dither taken in part from Welch
proc randomInit { seed } {
    global rand
    set rand(ia) 9301;# Multiplier
    set rand(ic) 49297; #Constant
    set rand(im) 233280; # Divisor
    set rand(seed) $seed; #last result
}
proc random {} {
    global rand
    set rand(seed) [expr (($rand(seed)*$rand(ia) + $rand(ic)) % $rand(im))]
    return [expr $rand(seed)/double($rand(im))]
}
proc drand { range } {
    return [expr ([random]*$range*2)-($range)]
}

proc bw2gp {title dither inFileName outFileName outputType} {
    
    # Init our random number generator   
    randomInit [pid]
        
    # First, open input file and find the different flow types
    # (Each flow type will have at least one "mean" entry)
    set inFile [open $inFileName "r"]
    set flowTypeList []
    while {[eof $inFile]==0} {
        set line [gets $inFile]
        if {[string first "mean" $line] == 0} {
    	    set temp [lindex [split $line " "] 1]
    	    set temp [lindex [split $temp ":"] 0]
    	    if {[lsearch $flowTypeList $temp] == -1} {
    	         lappend flowTypeList $temp
	    }
        }
    }
    
    # Open output file and write gnuplot header based on flow types discovered
    set outFile [open $outFileName "w"]
    puts $outFile "set title '$title'"
    if {$outputType == "PS"} {
        puts $outFile "set terminal postscript"
        puts $outFile "set output \"$outFileName.ps\""
    } elseif {$outputType == "GIF"} {
	puts $outFile "set terminal gif"
	puts $outFile "set output \"$outFileName.gif\""
    } elseif {$outputType == "X11"} {
	puts $outFile "set terminal x11"	
	puts $outFile "set output"
    } else {
	error "Invalid outputType <X11 | GIF | PS>"
    }
    puts -nonewline $outFile  "plot "
    set first 1
    foreach flowType $flowTypeList {
        if {$first} {
            puts $outFile "\\"
            set first 0
        } else {
            puts $outFile ", \\"
        }
        puts $outFile "'-' t '$flowType' with points, \\"
        puts -nonewline $outFile "'-' t '$flowType mean' with lines"
    }
    puts $outFile "\n"
        
    # Write data points and keep "mean" data for each flow
    foreach flowType $flowTypeList {
        seek $inFile 0 start
        while {[eof $inFile]==0} {
            set line [gets $inFile]
            if {0 == [string first "test" $line]} {
                set testId [lindex [split $line " "] 1]
            } elseif {0 == [string first "flow" $line]} {
                set type [lindex [split [lindex [split $line " "] 1] ":"] 0]
                if {$type == $flowType} {
                    set xvalue [expr $testId + [drand $dither]]
                    set yvalue [lindex [split $line " "] 2]
                    puts $outFile  "$xvalue, $yvalue"
                }
    	    } elseif {0 == [string first "mean" $line]} {
                set type [lindex [split [lindex [split $line " "] 1] ":"] 0]
                if {$type == $flowType} {
                    set xvalue $testId
                    set yvalue [lindex [split $line " "] 2]
                    lappend meanList "$xvalue, $yvalue"
                }
            }
        }
        puts $outFile "e"
        
        # Write data for mean values for this flow type
        foreach entry $meanList {
            puts $outFile $entry
        }
        puts $outFile "e"
        set meanList []
    }
    close $inFile
    close $outFile
}

# This bit of code let's the script be used directly with
# command line arguments or be "sourced" with no args from 
# another tcl script for access to its procs.

if {5 == $argc } {
    set title [lindex $argv 0]
    set dither [lindex $argv 1]
    set inFileName [lindex $argv 2]
    set outFileName [lindex $argv 3]
    set outputType [lindex $argv 4]
    bw2gp $title $dither $inFileName $outFileName $outputType
} elseif {4 == $argc } {
    set title [lindex $argv 0]
    set dither [lindex $argv 1]
    set inFileName [lindex $argv 2]
    set outFileName [lindex $argv 3]
    set outputType "X11"
    bw2gp $title $dither $inFileName $outFileName $outputType
} else { 
    puts $argc
    error "Usage: bw2gp.tcl <graphTitle> <dither> <inFile> <outFile> <X11 | PS | GIF>"
}







