
# This routine uses rateplot to plot the observed data
# rate on a link from node <src> to node <dst>
# from an ns trace file (<traceFile>).  <window> is
# the width of ratplot's averaging window
# (<plotType> can by "X11", "GIF", or "PS" (POSTSCRIPT))

proc RatePlot {traceFile fileName src dst window plotType} {

if {"X11" == $plotType} {
    set plotArg ""
    set outFile "X11 Display"
} elseif {"GIF" == $plotType} {
    set plotArg "gif"
    set outFile "$fileName.rate.gif"
} elseif {"PS" == $plotType} {
    set plotArg "post"
    set outFile "$fileName.rate.ps"
} else {
    puts "mdpTools:RatePlot: Error! Invalid <plotType>."
    exit
}
puts "Plotting link($src,$dst) data rate to $outFile"
# Run "rateplot" ...
append plotCmd "rateplot $plotArg window $window auto 4 flow X link $src,$dst "
append plotCmd "trace $traceFile $fileName.rate"
catch {eval exec $plotCmd}
# Process/Display with gnuplot (also deletes file created by rateplot)
set plotCmd "load '$fileName.rate'; !rm -f $fileName.rate;\n"
catch {exec echo $plotCmd | gnuplot -persist &}

}

proc LossPlot {traceFile fileName src dst window plotType} {

if {"X11" == $plotType} {
    set plotArg ""
    set outFile "X11 Display"
} elseif {"GIF" == $plotType} {
    set plotArg "gif"
    set outFile "$fileName.loss.gif"
} elseif {"PS" == $plotType} {
    set plotArg "post"
    set outFile "$fileName.loss.ps"
} else {
    puts "mdpTools:LossPlot: Error! Invalid <plotType>."
    exit
}

puts "Plotting link($src,$dst) loss fraction to $outFile"
# Run "lossplot" ...
append plotCmd "lossplot $plotArg window $window auto 4 flow X link $src,$dst "
append plotCmd "trace $traceFile $fileName.loss"
catch {eval exec $plotCmd}
# Process/Display with gnuplot (also deletes file created by lossplot)
set plotCmd "load '$fileName.loss'; !rm -f $fileName.rate;"
catch {exec echo $plotCmd | gnuplot -persist &}

}


# This routine uses gnuplot to plot the Mdp receiver
# LossIntervalTracking from a MDP Log file
# (<plotType> can by "X11", "GIF", or "PS" (POSTSCRIPT))

proc MdpLossIntervalPlot {logFile fileName plotType} {

# Create an appropriate Gnuplot terminal type
if {"X11" == $plotType} {
    set termType "X11"
    set outFile "X11 Display"
} elseif {"GIF" == $plotType} {
    set termType "gif"
    set outFile "$fileName.li.gif"
} elseif {"PS" == $plotType} {
    set termType "postscript color solid"
    set outFile "$fileName.li.ps"
} else {
    puts "mdpTools:LossIntervalPlot: Error! Invalid <plotType>."
    exit
}

# Build a set of commands for Gnuplot
set plotCmd    "set title '\"$logFile\" Loss Interval Tracking'\n"
append plotCmd "set term $termType \n"
append plotCmd "set output '$outFile' \n"
append plotCmd "set data style lines\n"
append plotCmd "set xlabel 'Time (sec)'\n"
append plotCmd "set ylabel 'Interval (pkts)'\n"
append plotCmd "plot \\\n"
append plotCmd "'$logFile' using 1:2 \\\n"
append plotCmd "'LossIntervalTracking> time>%lf current>%lf mdp>%lf' t 'Current', \\\n"
append plotCmd "'$logFile' using 1:3 \\\n"
append plotCmd "'LossIntervalTracking> time>%lf current>%lf mdp>%lf' t 'Mdp', \\\n"
append plotCmd "'$logFile' using 1:4 \\\n"
append plotCmd "'LossIntervalTracking> time>%lf current>%lf mdp>%lf tfrc>%lf' t 'Tfrc'\n"

# Now execute plot command
puts "Plotting MdpLossIntervalTracking to $outFile"
exec echo $plotCmd | gnuplot -persist &

}


# This routine uses gnuplot to plot the Mdp sender
# congestion control metrics from a MDP Log file
# (<plotType> can by "X11", "GIF", or "PS" (POSTSCRIPT))
proc MdpCCMetricPlot {logFile fileName plotType} {

# Create an appropriate Gnuplot terminal type
if {"X11" == $plotType} {
    set termType "X11"
    set outFile "X11 Display"
} elseif {"GIF" == $plotType} {
    set termType "gif"
    set outFile "$fileName.cc.gif"
} elseif {"PS" == $plotType} {
    set termType "postscript color solid"
    set outFile "$fileName.cc.ps"
} else {
    puts "mdpTools:LossIntervalPlot: Error! Invalid <plotType>."
    exit
}

# Build a set of commands for Gnuplot
set plotCmd    "set title '\"$logFile\" MDP Congestion Control Metrics'\n"
append plotCmd "set term $termType \n"
append plotCmd "set output '$outFile' \n"
append plotCmd "set data style lines\n"
append plotCmd "set xlabel 'Time (sec)'\n"
append plotCmd "set ylabel 'Rate (kbps)'\n"
append plotCmd "set y2label 'Rtt Measurements (sec)'\n"
append plotCmd "set y2tics\n"
append plotCmd "plot \\\n"
append plotCmd "'$logFile' using 1:5 \\\n"
append plotCmd "'RateControlStatus> time>%lf loss>%lf rtt>%lf tzero>%lf goal_rate>%lf' t 'Goal Rate', \\\n"
append plotCmd "'$logFile' using 1:6 \\\n"
append plotCmd "'RateControlStatus> time>%lf loss>%lf rtt>%lf tzero>%lf goal_rate>%lf tx_rate>%lf' t 'Tx Rate', \\\n"
append plotCmd "'$logFile' using 1:3 \\\n"
append plotCmd "'RateControlStatus> time>%lf loss>%lf rtt>%lf tzero>%lf goal_rate>%lf' axes x1y2 t 'RTT', \\\n"
append plotCmd "'$logFile' using 1:4 \\\n"
append plotCmd "'RateControlStatus> time>%lf loss>%lf rtt>%lf tzero>%lf goal_rate>%lf tx_rate>%lf' axes x1y2 t 'T0', \\\n"
append plotCmd "'$logFile' using 1:(\$2 * 100) \\\n"
append plotCmd "'RateControlStatus> time>%lf loss>%lf rtt>%lf tzero>%lf goal_rate>%lf' axes x1y2 t 'Loss Estimate', \\\n"
append plotCmd "'$logFile' using 1:7 \\\n"
append plotCmd "'RateControlStatus> time>%lf loss>%lf rtt>%lf tzero>%lf goal_rate>%lf tx_rate>%lf frate>%lf' t 'F-Rate'\n"

# Now execute plot command
puts "Plotting Mdp-CC Metrics to $outFile"
exec echo $plotCmd | gnuplot -persist &

}
