/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
  
#include "udpSocket.h"
#include "mdpMessage.h"        

class MdpTracer : public UdpSocketOwner
{
    public:
        MdpTracer();
        bool Open(NetworkAddress* theAddr);
        void Run();
        
    private:
        UdpSocket socket;
        char rx_buffer[MSG_SIZE_MAX];
        bool HandleRecvPacket(char* buffer, int len, NetworkAddress* src);
};

MdpTracer::MdpTracer()
{
    // Init session socket
    //socket.SetInstaller(mgr->socket_installer, mgr->socket_user_data);
	socket.SetRecvPacketHandler((UdpSocketOwner *) this, 
            (UdpSocketHandlerFunc) &MdpTracer::HandleRecvPacket,
            rx_buffer, MSG_SIZE_MAX);
    
}

bool MdpTracer::Open(NetworkAddress* theAddr)
{
    if(UDP_SOCKET_NO_ERROR != socket.Open(theAddr->GetPort()))
    {
        fprintf(stderr, "mdpTrace: error opening socket!\n");
        return false;
    }
    if (theAddr->IsMulticast())
    {
        if(UDP_SOCKET_NO_ERROR != socket.JoinGroup(theAddr, 64, false))
	    {
	        fprintf(stderr, "Error joining multicast group!");
	        socket.Close();
	        return false;
	    }
    }
    return true;
}

void MdpTracer::Run()
{
    while(1)
    {
        socket.OnReceive(0);
    }
}
        
static void MessageTrace(bool snd, unsigned long node_id, MdpMessage *msg, int len, const NetworkAddress *src);

static void usage() 
{
    fprintf(stderr, "Usage: mdpTrace -A <address>/<port>\n");
}  // end usage()

        
int main(int argc, char *argv[])
{
    MdpTracer tracer;
    NetworkAddress addr;
    
    char *session_address = NULL;
    int session_port = 0;
    
    extern char *optarg;
    int c;
    while ((c = getopt(argc, argv, "A:")) != EOF) 
    {
        char *ptr;
        switch(c) 
        {
            case 'A':
                ptr = strchr(optarg, '/');
                if (!ptr)
                {
                    fprintf(stderr, "mdp: Invalid address/port: %s\n\n",
                                     optarg);
                    usage();
                    exit(-1);
                }
                *ptr++ = '\0';
                session_address = optarg;
                session_port = atoi(ptr); 
                break;
                  
            default:
                fprintf(stderr, "mdpTrace: Bad command line option!\n");
                usage();
                exit(1);        
        }
    }
    
    if (!session_address)
    {
        fprintf(stderr, "mdpTrace: Must specify session address/port!\n");
        usage();
        exit(1);
    }
    
    addr.type = IPv4;
    addr.LookupHostAddress(session_address);
    addr.SetPort(session_port);
        
    if(!tracer.Open(&addr))
    {
        exit(-1);
    }
    
    tracer.Run();
    
}  // end main()

bool MdpTracer::HandleRecvPacket(char* buffer, int len, NetworkAddress* src)
{
    MdpMessage msg;
    
    if(!msg.Unpack(buffer, len))
    {
        fprintf(stdout, "mdp: Recv'd bad message from \"%s\"\n", src->HostAddressString());
        return true;
    }
    
    if (MDP_REPORT != msg.type) 
        MessageTrace(false, msg.sender, &msg, len, src);
    return true;
}  // end MdpTracer::HandleRecvPacket()

static const char cmd_flavor[5][13] = 
{
    "MDP_NULL_CMD",
    "MDP_FLUSH   ",
    "MDP_SQUELCH ",
    "MDP_ACK_REQ ",
    "MDP_GRTT_REQ"
};
    
static const char dst[2][4] = 
{
    "src",
    "dst"
};
    
static void DataTrace(unsigned long  object_id,
			  unsigned short segment_size, 
		      unsigned char  ndata, 
			  unsigned long  offset,
		      unsigned char  parity_id);
    
static void MessageTrace(bool snd, unsigned long node_id, MdpMessage *msg, int len, const NetworkAddress *src)
{   
    struct timeval rxTime;
	
    ::GetSystemTime(&rxTime);
    
    if (snd) snd = 1;
    
    struct tm *rx_time = gmtime((time_t *)&rxTime.tv_sec);
    DMSG(0, "mdp_trace>%02d:%02d:%02d.%06lu ",
                    rx_time->tm_hour, 
                    rx_time->tm_min,
                    rx_time->tm_sec,
                    rxTime.tv_usec); 	
	DMSG(0, "node>%5lu ", node_id);
	
    struct in_addr server_addr;
    switch(msg->type)
    {
        case MDP_REPORT:
            DMSG(0, "MDP_REPORT       %s>%-16s len>%04d\n", 
                    dst[snd], src->HostAddressString(), len);
            break;
            
        case MDP_INFO:
            DMSG(0, "MDP_INFO         %s>%-16s len>%04d obj>%04lu\n", 
                dst[snd], src->HostAddressString(), len, msg->info.object_id);
            break;
            
        case MDP_DATA:
            DMSG(0, "MDP_DATA         %s>%-16s len>%04d ", dst[snd], 
				    src->HostAddressString(), len);
		    DataTrace(msg->data.object_id, (msg->data.flags & MDP_DATA_FLAG_RUNT ) ?
					   msg->data.segment_size : msg->data.len, 
					  msg->data.ndata, msg->data.offset, 0);
            break;
             
        case MDP_PARITY:
            DMSG(0, "MDP_PARITY       %s>%-16s len>%04d ", 
                dst[snd], src->HostAddressString(), len);
			DataTrace(msg->parity.object_id, msg->parity.len, msg->parity.ndata, 
					  msg->parity.offset, msg->parity.id);
            break;
            
        case MDP_CMD:
            DMSG(0, "%-16s %s>%-16s len>%04d", 
                cmd_flavor[msg->cmd.flavor], dst[snd], src->HostAddressString(), 
                len);
            switch(msg->cmd.flavor)
            {
                case MDP_CMD_SQUELCH:
                    DMSG(0, "obj>%04lu\n", msg->cmd.squelch.object_id);
                    break;
                case MDP_CMD_ACK_REQ:
                    DMSG(0, "obj>%04lu\n", msg->cmd.ack_req.object_id);
                    break;
                default:
                    DMSG(0, "\n");
                    break;
            }
            break;
            
        case MDP_NACK:
            server_addr.s_addr = htonl(msg->nack.server_id);
            DMSG(0, "MDP_NACK         %s>%-16s len>%04d \n", 
                    dst[snd], src->HostAddressString(), len);
            //DMSG(0, "server>%-16s\n", inet_ntoa(server_addr));
            break;
            
        case MDP_ACK:
            server_addr.s_addr = htonl(msg->ack.server_id);
            DMSG(0, "MDP_ACK          %s>%-16s len>%04d obj>0x%08x \n", 
                    dst[snd], src->HostAddressString(), len, msg->ack.object_id);
            //DMSG(0, "server>%-16s\n", inet_ntoa(server_addr));
            break;
            
        default:
            DMSG(0, "MDP_INVALID_MSG %s>%s len>%04d\n", 
                    dst[snd], src->HostAddressString(), len);
    }  // end switch(msg->type)   
}  // end MessageTrace()

static void DataTrace(unsigned long  object_id, 
		    		  unsigned short segment_size, 
		    		  unsigned char  ndata, 
					  unsigned long  offset,
		    		  unsigned char  parity_id)
{
	unsigned long block_size = ndata * segment_size;
	unsigned long block_id = offset / block_size;
	unsigned int vector_id;
	if (parity_id)
		vector_id = parity_id;
	else
		vector_id = (offset - (block_id * block_size)) / segment_size;
	DMSG(0, "obj>%04lu blk>%04lu vec>%03d\n", object_id, block_id, vector_id);
}

