/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 

#include "protocolTimer.h"
#include "debug.h"

#include <math.h>     // for fabs() & isnan()
#include <stdio.h>

ProtocolTimer::ProtocolTimer()
    : timeout(-1.0), repeat_count(0), 
      interval(0), repeats(0), 
      list(NULL), prev(NULL), next(NULL) 
{
}  // end ProtocolTimer::ProtocolTimer()

ProtocolTimer::~ProtocolTimer()
{ 
    if (IsActive()) Deactivate(); 
}

void ProtocolTimer::Reset()
{
    if (IsActive())
    {
        ProtocolTimerMgr *mgr = list->mgr;
	    mgr->RemoveTimer(this);
	    mgr->InstallTimer(this);
    }
}  // end ProtocolTimer::Reset()

double ProtocolTimer::TimeRemaining()
{
    return (list ? list->DeltaTime(timeout, list->GetCurrentTime()) : -1.0);   
}


void ProtocolTimer::Deactivate()
{
    ASSERT(list);
    list->mgr->RemoveTimer(this);
}  // end ProtocolTimer::Deactivate()




ProtocolTimerList::ProtocolTimerList()
    : head(NULL), tail(NULL),  mgr(NULL)
{

}  // end ProtocolTimerList()


void ProtocolTimerList::InsertTimer(ProtocolTimer *theTimer)
{
    ASSERT(!theTimer->list);
    ProtocolTimer *nextTimer = head;	
    theTimer->list = this;
    while(nextTimer)
    {
	    if (theTimer->timeout <= nextTimer->timeout)
	    {
		    if((theTimer->prev = nextTimer->prev))
		        theTimer->prev->next = theTimer;
		    else
		        head = theTimer;
		    nextTimer->prev = theTimer;
		    theTimer->next = nextTimer;
		    return;
	    }
	    else
	    {
		    nextTimer = nextTimer->next;
	    }
    }   
    if ((theTimer->prev = tail))
	    tail->next = theTimer;
    else
	    head = theTimer;    
    theTimer->next = NULL;
    tail = theTimer;
}  // end InsertTimer()


// Remove timer from linked list and mark as inactive (list == NULL)
void ProtocolTimerList::RemoveTimer(ProtocolTimer *theTimer)
{
    if (theTimer->prev)
	    theTimer->prev->next = theTimer->next;
    else
	    head = theTimer->next;
    if (theTimer->next)
	    theTimer->next->prev = theTimer->prev;
    else
	    tail = theTimer->prev;
    theTimer->list = NULL;
}  // end RemoveTimer()


double ProtocolTimerList::DoTimers()
{
    ProtocolTimer *nextTimer = head;
    double current_time = GetCurrentTime();
    while (nextTimer)
    {
	    if (nextTimer->timeout <= current_time)
	    {
			if(nextTimer->DoTimeout())
	        {
                mgr->RemoveTimer(nextTimer);
                int repeatCount = nextTimer->repeat_count;
                if (repeatCount) 
                {
				    mgr->InstallTimer(nextTimer);
                    if (repeatCount > 0) repeatCount--;
                    nextTimer->repeat_count = repeatCount;
                }
	        }
	        nextTimer = head;
	    }
	    else
	    {
	        nextTimer = NULL;
	    }
    } 
    return TimeRemaining();
}  // end ProtocolTimerList::DoTimers() 


double ProtocolTimerList::TimeRemaining()
{
    if (head)
    {
	    double nextTime = head->timeout - GetCurrentTime();
	    if (nextTime < 0.0)
	        return 0.0;	
	    else
	        return nextTime;
    }
    else
    {
	    return -1.0;
    }
}  // end ProtocolTimerList::TimeRemaining()


void ProtocolPrecisionTimerList::InsertTimer(ProtocolTimer *theTimer)
{
    ASSERT(!theTimer->list);
    ProtocolTimer *nextTimer = head;
    theTimer->list = this;
	double t_out = theTimer->timeout;
    while(nextTimer)
    {
	    if (DeltaTime(t_out, nextTimer->timeout) <= (double)0.0)
	    {
	        if((theTimer->prev = nextTimer->prev))
		        theTimer->prev->next = theTimer;
	        else
		        head = theTimer;
	        nextTimer->prev = theTimer;
	        theTimer->next = nextTimer;
	        return;
	    }
	    else
	    {
	        nextTimer = nextTimer->next;
	    }
    }
    if ((theTimer->prev = tail))
	    tail->next = theTimer;
    else
	    head = theTimer;
    theTimer->next = NULL;
    tail = theTimer;
}  // end ProtocolPrecisionTimerList::InsertTimer()


double ProtocolPrecisionTimerList::DoTimers()
{
	ProtocolTimer* nextTimer = head;
    double current_time = GetCurrentTime();
	
    while (nextTimer)
    {
		//if (DeltaTime(nextTimer->timeout, current_time) <= (double)0.0)
	    // (The code should read as the above comment, but floating point
		//  comparisons don't always seem to work quite as expected)
		// This works within a microsecond of accuracy
		if (DeltaTime(nextTimer->timeout, current_time) < (double)1.0e-06)
        {
			if(nextTimer->DoTimeout())
	        {
                int repeatCount = nextTimer->repeat_count;
                if (repeatCount) 
                {
                    mgr->ReinstallTimer(nextTimer, current_time);
		            if (repeatCount > 0) repeatCount--;
                    nextTimer->repeat_count = repeatCount;
                }
                else
                {
                    mgr->RemoveTimer(nextTimer);
                }
	        }
	        nextTimer = head;
	    } 
	    else
	    {
			nextTimer = NULL;
	    }
    }
	return TimeRemaining();
}  // end ProtocolPrecisionTimerList::DoTimers() 


// Properly calculates t1 - t2
double ProtocolPrecisionTimerList::DeltaTime(double t1, double t2)
{
	double delta = t1 - t2;    
    if (delta < -PRECISION_HALF_TIME)
        return delta + PRECISION_TIME_MODULUS;
    else if (delta < PRECISION_HALF_TIME)
        return delta;
    else
        return delta - PRECISION_TIME_MODULUS;
}  // end ProtocolPrecisionTimerList::DeltaTime()

// Properly calculates t1 + t2
double ProtocolPrecisionTimerList::TotalTime(double t1, double t2)
{
    double total = t1 + t2; 
    if (total > PRECISION_TIME_MAX)
	    return (total - PRECISION_TIME_MODULUS);
    else
	    return total;
}  // end ProtocolPrecisionTimerList::TotalTime()

double ProtocolPrecisionTimerList::TimeRemaining()
{
    if (head)
    {
		double curtime = GetCurrentTime();
		double nextTime = DeltaTime(head->timeout, curtime);
        if (nextTime < (double) 0.0)
	        return (double) 0.0;
	    else
	        return nextTime;
    }
    else
    {
		return -1.0;
    }
}  // end ProtocolPrecisionTimerList::TimeRemaining()



ProtocolTimerMgr::ProtocolTimerMgr()
    : installer_func(NULL), installer_data(NULL),
      user_data(NULL), timer_timeout(-1.0)
{
	precision_list.SetMgr(this);
    long_list.SetMgr(this);
	long_base_timer.Init(1.0, -1, (ProtocolTimerOwner *)this, 
						 (ProtocolTimeoutFunc)&ProtocolTimerMgr::DoLongTimers); 
    long_base_timer.repeat_count = -1;
}

bool ProtocolTimerMgr::DoLongTimers()
{
    return (long_list.DoTimers() >= 0);
};

ProtocolTimerMgr::~ProtocolTimerMgr()
{
    Halt();
}  // end ProtocolTimerMgr::~ProtocolTimerMgr()

void ProtocolTimerMgr::Update()
{
    double next_timeout = NextTimeout();
    if (next_timeout != timer_timeout)
    {
        if (next_timeout < 0)
        {
            // Remove pending timeout   
            installer_func(PROTOCOL_TIMER_REMOVE, 0.0, this, installer_data);
			timer_timeout = -1;
		}
        else if (timer_timeout < 0.0)
        {
            // Install new timeout (none currently installed)
            if (installer_func(PROTOCOL_TIMER_INSTALL, TimeRemaining(), 
                               this, installer_data))
            {
			    timer_timeout = next_timeout;
            }
			else
            {
                timer_timeout = -1;
                DMSG(0, "ProtocolTimerMgr::Update() INSTALL error!\n");
            }
        }
        else
        {
            // Modify existing timeout (replace installed timer) 
            if (installer_func(PROTOCOL_TIMER_MODIFY, TimeRemaining(), 
                               this, installer_data))
            {
			    timer_timeout = next_timeout;
            }
			else
            {
                timer_timeout = -1;
                DMSG(0, "ProtocolTimerMgr::Update() MODIFY error!\n");
            }  
        }
    }
}  // end ProtocolTimerMgr::Update() 

void ProtocolTimerMgr::InstallTimer(ProtocolTimer *theTimer)
{
    if (theTimer->interval > PRECISION_TIME_THRESHOLD)
    {
        theTimer->timeout = long_list.GetCurrentTime() + 
						    theTimer->interval;
	    long_list.InsertTimer(theTimer);
	    if (!long_base_timer.IsActive())
		{
			long_base_timer.timeout = 
				precision_list.TotalTime(precision_list.GetCurrentTime(), 
				          long_base_timer.interval);
	        precision_list.InsertTimer(&long_base_timer);
		}
    }
    else
    {
		double curtime = precision_list.GetCurrentTime();
		theTimer->timeout = 
			precision_list.TotalTime(theTimer->interval, curtime);
	    precision_list.InsertTimer(theTimer);
    }
    theTimer->repeat_count = theTimer->repeats;
	Update();
}  // end ProtocolTimerMgr::InstallTimer()
	
void ProtocolTimerMgr::ReinstallTimer(ProtocolTimer *theTimer,
                                      double   now)
{
	ASSERT(theTimer->list);
    theTimer->list->RemoveTimer(theTimer);
    if (theTimer->interval <= PRECISION_TIME_THRESHOLD)
    { 
        theTimer->timeout = 
		    precision_list.TotalTime(theTimer->timeout,
								     theTimer->interval);
        // *** If a "precision timer" falls more than 1 second behind
        // don't keep forcing it to try and catch up
        if (precision_list.DeltaTime(theTimer->timeout, now) < (double) -1.0)
             theTimer->timeout = now;

	    precision_list.InsertTimer(theTimer);
        theTimer->repeat_count = theTimer->repeats;
        Update();
    }
    else
    {
        InstallTimer(theTimer);
    }
}  // end ProtocolTimerMgr::ReinstallTimer()

void ProtocolTimerMgr::RemoveTimer(ProtocolTimer *theTimer)
{
    if (theTimer->list)
    {
        if (&long_list == theTimer->list)
        {
            long_list.RemoveTimer(theTimer);
            if (!long_list.head) 
	            precision_list.RemoveTimer(&long_base_timer);
        }
        else
        {
            precision_list.RemoveTimer(theTimer);
        }
        Update();
    }
}  // end ProtocolTimerMgr::RemoveTimer()

void ProtocolTimerMgr::Halt()
{
	if (timer_timeout >= 0.0)
	{
		installer_func(PROTOCOL_TIMER_REMOVE, 0.0, this, installer_data);
        timer_timeout = -1.0;
	}
}  // end ProtocolTimerMgr::HaltTimers()


