#ifndef _UDP_SOCKET
#define _UDP_SOCKET

#include "networkAddress.h"
#include "debug.h"

#ifdef UNIX
typedef int SocketHandle;
const int INVALID_SOCKET = -1;
#endif // UNIX
#ifdef WIN32
typedef SOCKET SocketHandle;
#endif // WIN32

// The "UdpSocketOwner" class exists only for the derivation of 
// other classes. Therefore, no constructor is defined.
class UdpSocketOwner {};
typedef bool (UdpSocketOwner::*UdpSocketRecvHandler)(class UdpSocket* theSocket);

// For socket read notification installation
enum UdpSocketCmd{UDP_SOCKET_INSTALL, UDP_SOCKET_REMOVE};
typedef bool (UdpSocketInstallFunc)(UdpSocketCmd        cmd,
                                    class UdpSocket*    theSocket,
                                    const void*         installData);


// Possible errors returned by some UdpSocket calls
typedef enum UdpSocketError
{
    UDP_SOCKET_ERROR_NONE = 0, 
    UDP_SOCKET_ERROR,         // generic error
    UDP_SOCKET_ERROR_OPEN,    // socket() error
    UDP_SOCKET_ERROR_BIND,    // bind() error
    UDP_SOCKET_ERROR_CONNECT, // connect() error
    UDP_SOCKET_ERROR_NAME,    // getsockname() error
    UDP_SOCKET_ERROR_SEND,    // sendto() error
    UDP_SOCKET_ERROR_RECV,    // recvfrom() error        
    UDP_SOCKET_ERROR_MCAST,   // join group error
    UDP_SOCKET_ERROR_TTL,     // mcast ttl error
    UDP_SOCKET_ERROR_IFACE,   // mcast iface error
    UDP_SOCKET_ERROR_TOS,     // ip tos error
    UDP_SOCKET_ERROR_LOOP,    // loopback error
    UDP_SOCKET_ERROR_REUSE,   // port/addr reuse error
    UDP_SOCKET_ERROR_SEC      // ipsec error
} UdpSocketError;

class UdpSocket
{
    public:
    // Construction
	    UdpSocket();
	    ~UdpSocket();
        
    // Initialization
        void Init(UdpSocketOwner*       theOwner, 
                  UdpSocketRecvHandler  recvHandler,
                  UdpSocketInstallFunc* installFunc,
                  void*                 installData)
        {
            owner = theOwner;
            recv_handler = recvHandler;
            install_func = installFunc;
            install_data = installData;
        }
        
        UdpSocketOwner* Owner() {return owner;}
        UdpSocketRecvHandler RecvHandler() {return recv_handler;}
    
        void SetUserData(const void* userData) {user_data = userData;}
        const void* GetUserData() {return user_data;}
        
        // Control
	    UdpSocketError Open(unsigned short thePort = 0, NetworkAddressType destType = IPv4);
	    void Close();
		bool Connect(const NetworkAddress* theAddress);
		void Disconnect();
	    UdpSocketError JoinGroup(const NetworkAddress* theAddress, 
                                 unsigned char ttl);
	    void LeaveGroup(const NetworkAddress *theAddress);
	    UdpSocketError SetTTL(unsigned char ttl);
        UdpSocketError SetLoopback(bool loopback);
        UdpSocketError SetTOS(unsigned char tos);
        UdpSocketError SetReuse(bool state);
        UdpSocketError SetMulticastInterface(unsigned long interfaceAddress);
        
	    UdpSocketError SendTo(const NetworkAddress* dest, char* buffer, unsigned int len);
	    UdpSocketError RecvFrom(char* buffer, unsigned int* buflen, NetworkAddress* src);
        
        // Status
        bool IsOpen() {return (bool)(handle != INVALID_SOCKET);}        
		bool IsConnected() {return connected;}
        unsigned short Port() {return port;}
        SocketHandle Handle() {return handle;}
	
        void OnReceive()
        {
            ASSERT(owner && recv_handler);
            (owner->*recv_handler)(this);
        }
#ifdef HAVE_IPV6
        static bool UdpSocket::IsIPv6Capable();
#endif            
    private:
    // Members
	    SocketHandle            handle;
	    int                     domain;  // PF_INET or PF_INET6
	    unsigned short          port;
	    bool		            connected;
	    UdpSocketOwner*         owner;
	    UdpSocketRecvHandler    recv_handler;
        UdpSocketInstallFunc*   install_func;
        const void*             install_data;
        const void*             user_data;
	    
};  // end class UdpSocket


#endif // _UDP_SOCKET

