/*
 * Bart's version of su2
 *
 * 7/91
 * Based on source from HP 1984/1985.
 * Copyright notice and revision history at bottom.
 *
 * Pseudo su program.  This program will check your username
 * in a system file to see if you are authorized to become root.
 * If it is found, your password will be asked for, and a root
 * shell exec'd.
 *
 * the following compile options are provided:
 *
 *  SAFE -- if non zero, go into "paranoid" mode, and remove rw access
 *	from sulog and super-users files.
 *
 *  FULLPATH -- if defined, requires that su2 be run by it's full path
 *	name (i.e. argv[0] starts with a '/').  This is provided for
 *	Trojan Horse protection.
 *
 *  DEBUG -- if defined, adds the "-D#" option for debugging.
 *
 *  SUPERDEBUG -- if defined, does an immediate set[ug]id to the
 *	e[ug]id's (should be root).  This allows for successful
 *	core dumps.  This should NOT be a security hazard.
 *
 *  NOVFORK -- use fork() instead of vfork().
 *
 *  CHANGEHOME -- if defined, $HOME will be redefined to the directory
 *	listed in the passwd file.
 *
 *  CHANGESHELL -- if defined, $SHELL will be redefined to the shell
 *	entry listed in the passwd file.
 *
 *  CHANGEUSER -- if defined, $USER will be redefined to the user
 *	entry listed in the passwd file.
 *
 *  NOSU2RC -- if defined, does not allow users to create their own
 *	equivalent to the "super-users" file.  The file must be setuid'ed
 *	to either the to user, or to root.  If the file is setuid'ed
 *	to the to user, it can be modified after su2'ing.  By making
 *	the file suid'ed to root, it can be protected.  The file is
 *	.su2rc in the to user's home directory.
 *
 *  INITGROUPS -- if set, does an initgroups(3C) call.  Done as well
 *	for BSD4.2.
 *
 *  IOCTL -- if set, uses ioctl(2) to get the terminal atributes
 *	for restoring after returning from shell
 *
 *  NOCHOWNTTY -- if defined, do not change owner of tty unless
 *	the "-s" flag is used.  On 4.2bsd systems, the chown()
 *	of tty seems to send SIGHUP to other processes.
 *
 *  NOCHMODTTY -- if defined, do not change mode of tty.
 *
 *  NOUSERCHECK -- if defined, the super-users and .su2rc files will
 *	not be checked.  Everyone will be granted access.  This is
 *	a major security hole, but I know that someone will want it.
 *
 *  NOPASSWDCHECK -- if defined, a user's password will not be checked.
 *	This is also a major security hole, but I am sure someone will
 *	want this one too.  NOPASSWDCHECK and NOUSERCHECK are independent
 *	of one another.
 *
 *  3rd February 1995
 *  Modifications by Matthew Wojcik, woj@ccs.neu.edu
 *  
 *  BROKENCUSERID -- if defined, use mycuserid() instead of the broken
 *      POSIX.1 version.  If your cuserid() function is broken, it will
 *      return the name of the _owner_ of an suid program, instead of
 *      the real uid of the executor of the same.
 *
 *      (Remember that P1003.1a dropped cuserid() from the standard;
 *       however, some POSIX.1-compliant systems implement cuserid()
 *       as it existed in the original standard.  OSF/1, as of 3.0,
 *       is an example of such a system.)
 *  
 *  OSF -- OSF/1's setpwent() returns an int; most return void.
 *
 *  18 July 1995
 *  Modifications by Brian Dowling, bdowling@ccs.neu.edu
 *
 *  USE_SHADOW	-- define if you use shadow passwords.  Note that this
 *                 has only been tested under Solaris if it works elsewhere,
 *                 that's great.
 *
 *  Removed "register" from two variables where it is used in 
 *  pointer arithmetic.
 *
 */

#ifdef BSD42
# define BSD
#endif

#ifdef DEBUG
#define Debug(level, fmt, arg)	if (debug >= level) (void) fprintf(stderr, fmt, arg)
#else DEBUG
#define Debug(level, fmt, string)
#endif DEBUG

#ifndef	SAFE
#define SAFE	0
#endif SAFE

#ifndef SUPERUSERS
#define SUPERUSERS "/etc/super-users"
#endif SUPERUSERS

char   *SystemNameFile = SUPERUSERS;
#ifndef NOSU2RC
char   *UsersNameFile = ".su2rc";
#endif NOSU2RC
char   *SULog = "/usr/adm/sulog";

char   *UtmpFile = "/etc/utmp";

#ifndef PATH
# ifdef BSD
#  define PATH "PATH=/usr/ucb:/bin:/usr/bin:"
# else BSD
#  define PATH "PATH=/bin:/usr/bin:"
# endif BSD
#endif PATH

#ifndef ROOTPATH
# ifdef BSD
#  define ROOTPATH "PATH=/usr/ucb:/bin:/etc:/usr/bin"
# else BSD
#  define ROOTPATH "PATH=/bin:/etc:/usr/bin"
# endif BSD
#endif ROOTPATH

#ifndef DEFAULTSHELL
# ifdef BSD
#  define DEFAULTSHELL "/bin/csh"
# else BSD
#  define DEFAULTSHELL "/bin/sh"
# endif BSD
#endif DEFAULTSHELL

char   *UserPath = PATH;
char   *RootPath = ROOTPATH;
char   *DefaultShell = DEFAULTSHELL;

#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <fcntl.h>
#include <stdio.h>
#include <time.h>
#include <utmp.h>
#include <signal.h>
#ifdef IOCTL
#include <sys/ioctl.h>
#ifdef BSD
#include <sgtty.h>
#else BSD
#include <termio.h>
#endif BSD
#endif IOCTL

#ifdef USE_SHADOW
#include <shadow.h>
struct spwd *spw;
#endif

enum ChangeType_ {
    ToRoot,
    FromRoot,
    NoChange,
};

enum NameCheck_ {
    NameFound,
    PlusNameFound,
    StarFound,
    NotFound,
};

char   *malloc ();

#ifdef OSF
int    setpwent ();
#else
void   setpwent ();
#endif

#ifndef sgi
struct passwd  *getpwuid ();
struct passwd  *getpwnam ();
void    endpwent ();
int     chmod ();
#endif

struct passwd  *pw;

char   *ttyname ();
char   *TTY;			/* last comp of tty path i.e. tty01  */
char   *FullTTY;		/* full path of tty name             */
int     ValidTTY = -1;		/* != -1 associated with a tty       */
int     Uid;			/* current user id                   */
int     Gid;			/* current group id                  */
char   *Program;		/* program name for error messages   */
char   *Empty = "";		/* empty string                      */

#ifdef BSD
# define strchr index
# define strrchr rindex
#endif BSD

#ifdef NOVFORK
# define vfork fork
#endif NOVFORK

char   *strchr ();
char   *strrchr ();
char   *strcpy ();
char   *strncpy ();

char   *getenv ();
char   *cuserid ();
#if defined(BROKENCUSERID)
char   *mycuserid ();
#endif
char   *crypt ();
char   *getpass ();

enum NameCheck_ NameCheck ();
void CleanPath ();
void ChangePS1 ();
void AddSuLog ();
int     ModifyUtmp ();
void Usage ();

char   *NewEnvironment[] = {
    0,				/* NewHomeDir                        */
    0,				/* NewPath                           */
    0,				/* NewLogname                        */
    0,				/* NewShell                          */
    0,				/* NewTerm                           */
    0,
};

#define NewHomeDir	NewEnvironment[0]
#define NewPath		NewEnvironment[1]
#define NewLogname	NewEnvironment[2]
#define NewShell	NewEnvironment[3]
#define NewTerm		NewEnvironment[4]

#ifdef DEBUG
int     debug;
#endif DEBUG

main (argc, argv, envp)
int     argc;
char  **argv;
char  **envp;
{
    char  **newenvp;		/* new envp array                    */
    char   *newargv[4];		/* new argv array                    */
    enum ChangeType_ ChangeType;/* type of switch                    */

    struct stat st;		/* Status return structure           */

    char   *MyPasswd;		/* my replied password               */
    char   *MyCryptPasswd;	/* my passwd, encrypted              */
    char   *Shell;		/* shell for arg[0] of exec          */

    FILE * f;			/* file to scan for Authorized names */
    char    Buffer[BUFSIZ];	/* Buffer to read names into         */
    int     AskPasswd;		/* true means do not ask for passwd  */
    int     Authorized;		/* true means not in super-users     */
    int     pid;		/* return pid of wait                */
    int     Status;		/* return Status of wait             */
#ifndef NOCHMODTTY
    int     TTYmode;		/* TTYmode for later restoring       */
#endif NOCHMODTTY
    int     TTYuid;		/* TTYuid for later restoring        */
    int     TTYgid;		/* TTYgid for later restoring        */
    int     TTYStatError = 1;	/* stat of FullTTY                   */
#ifdef IOCTL
#ifdef BSD
    struct sgttyb   TTYtermio;	/* TTY termio for later restoring    */
#else BSD
    struct termio   TTYtermio;	/* TTY termio for later restoring    */
#endif BSD
    int     TTYIoctlError;	/* TTY ioctl error return            */
#endif IOCTL
    int     Flag_c;		/* "sh -c" flag                      */
    int     Flag_d;		/* set SHELL to DefaultShell         */
    int     Flag_r;		/* reset utmp entry and exit         */
    int     Flag_s;		/* change /etc/utmp entry            */
    int     Flag_u;		/* from user specified               */
    int     Flag_x;		/* exec shell without forking        */
    int	    Flag_h;		/* don't change home directory	     */

    char   *NewUserName;	/* name of user to change to         */
    char   *OldUtmpName;	/* current UTMP entry                */
    char   *FromUserName;	/*
				 * name of user switching.  comes from
				 * either argv or cuserid ().
				 */
#ifndef NOSU2RC
    char   *su2rcFile;		/* users .su2rc full file name       */
#endif NOSU2RC
    char   *cString;		/* file for -c option                */

    int     FullLogin;		/* if ==1 change environment & -sh   */
    char  **eptr;		/* env pointer */
    char  **neptr;		/* new env pointer */
    char   *ShellPointer;	/* ptr to SHELL= value */

    int     NiceInc;		/* value to nice to */
    register int    i;		/* g.p. index */
    register char  *c;		/* g.p. char pointer */

    char  **EnvPath;		/* pointer to "PATH=" string of env  */
    char  **EnvRootPath;	/* pointer to "SUPATH=" string of env */
    char  **EnvUserPath;	/* pointer to "UPATH=" string of env */
    char  **EnvPS1;		/* ptr to PS1= string of env */

    /*
     * initialize variables
     */

    Program = argv[0];
    Flag_c = Flag_d = Flag_r = Flag_s = Flag_u = Flag_x = Flag_h = 0;
    EnvPath = EnvRootPath = EnvUserPath = EnvPS1 = (char **) 0;
    FromUserName = (char *) 0;	/* default from name                 */
    NewUserName = "root";	/* default NewUserName               */
    FullLogin = 0;		/* do not do full login/change env.  */
    cString = (char *) 0;	/* default -c string                 */
    AskPasswd = 0;		/* default is do not ask for passwd  */
    Authorized = 0;		/* default is not authorized         */
    NiceInc = 0;		/* default nice increment            */
#ifdef DEBUG
    debug = 0;
#endif DEBUG

    /*
     * get current user/group ids
     */

    Uid = (int) getuid ();
    Gid = (int) getgid ();

#ifdef SUPERDEBUG
    /*
     * this will allow for core dumps to produce valid core files
     */

    setgid (getegid ());
    setuid (geteuid ());
#endif SUPERDEBUG

    /*
     * lets check file descriptors 0, 1, and 2 (in that order) for a
     * valid tty
     */

    for (i = 0; i < 3; i++)
	if (isatty (i)) {
	    ValidTTY = i;
	    break;
	}
    Debug (1, "-> ValidTTY=%d\n", ValidTTY);

    /*
     * get name of TTY for everyone who needs it
     */

    if (ValidTTY != -1) {
	FullTTY = ttyname (ValidTTY);
	if ((FullTTY == (char *) 0) || (*FullTTY == (char) 0))
	    (void) fprintf (stderr, "%s: ttyname(%d) failed\n", argv[0],
		    ValidTTY);
    }

    if (FullTTY == (char *) 0)
	FullTTY = "/dev/TTY??";

    TTY = strrchr (FullTTY, '/') + 1;

    Debug (1, "-> FullTTY=\"%s\"\n", FullTTY);
    Debug (1, "-> TTY=\"%s\"\n", TTY);

#ifdef FULLPATH
    if (*Program != '/') {
	fprintf (stderr, "%s: may only be run with full path name\n", Program);
	(void) exit (1);
    }
#endif FULLPATH

    /*
     * parse off arguments
     */

    for (i = 1; i < argc; i++) {
	c = argv[i];
	Debug (1, "-> Parsing \"%s\"\n", c);
	if (*c++ != '-') {
	    NewUserName = argv[i];
	    if (*NewUserName == '\0')
		(void) Usage ();
	    Debug (1, "-> NewUserName=\"%s\"\n", NewUserName);
	    continue;
	}

	/*
	 * skip past '-' and parse off flags
	 */

	if (*c == '\0') {

	    /*
	     * change environment and run
	     * shell as "-sh"
	     */

	    (void) FullLogin++;
	    continue;
	}

	while (*c) {
	    Debug (1, "-> switching on '%c'\n", *c);
	    switch (*c++) {

		case 'D':
#ifdef DEBUG
		    if (*c) {
			debug = atoi (c);
			c = Empty;
		    }
		    if (debug <= 0)
			debug = 1;
		    Debug (1, "-> debug level set to %d\n", debug);
#else  DEBUG
		    if (*c)
			c = Empty;
		    fprintf( stderr, "warning: not compiled for debugging\n" );
#endif DEBUG
		    break;

		case 'u':

		    /*
		     * username to check passwd and access
		     * against.  disables the +username
		     * option.
		     */

		    if (!*c) {
			if (++i >= argc)
			    (void) Usage ();
			c = argv[i];
		    }

		    (void) Flag_u++;
		    FromUserName = c;
		    c = Empty;
		    Debug (1, "-> FromUserName=\"%s\"\n", FromUserName);
		    break;

		case 'c':

		    /*
		     * execute command string as in sh -c "..."
		     */

		    if (!*c) {
			if (++i >= argc)
			    (void) Usage ();
			c = argv[i];
		    }

		    (void) Flag_c++;
		    cString = c;
		    c = Empty;
		    Debug (1, "-> cString=\"%s\"\n", cString);
		    break;

		case 'd':

		    /*
		     * set SHELL to DefaultShell
		     */

		    (void) Flag_d++;
		    Debug (1, "-> Flag_d set\n", 0);
		    break;

		case 'r':

		    /*
		     * reset /etc/utmp entry and terminate
		     */

		    (void) Flag_r++;
		    Debug (1, "-> Flag_r set\n", 0);
		    break;

		case 'x':

		    /*
		     * exec shell rather than forking one
		     */

		    (void) Flag_x++;
		    Debug (1, "-> Flag_x set\n", 0);
		    break;

		case 's':

		    /*
		     * change /etc/utmp entry before forking
		     */

		    (void) Flag_s++;
		    Debug (1, "-> Flag_s set\n", 0);
		    break;

		case 'h':

		    /*
		     * don't set HOME
		     */

		    (void) Flag_h++;
		    Debug (1, "-> Flag_h set\n", 0);
		    break;

		case '-':
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':

		    /*
		     * nice value to change to.
		     * change now to help if the system is
		     * bogged down.
		     */

		    if (sscanf (c, "%d", &NiceInc) != 1)
			(void) Usage ();

		    /*
		     * execute nice immediately
		     */

		    if (nice (NiceInc) == -1)
			(void) perror ("nice");
		    c = Empty;
		    Debug (1, "-> NiceInc=%d\n", NiceInc);
		    break;

		default:
		    (void) Usage ();
		    break;
	    }
	}
    }

#ifndef sgi
    setpwent ();
#endif

    /*
     * check to see if we are only supposed to
     * reset /etc/utmp
     */

    if (Flag_r) {

	/*
	 * in this case, get the default name from the current
	 * ruid if needed.
	 */

	if (FromUserName == NULL) {
	    pw = getpwuid (Uid);
	    if (pw == NULL) {
		(void) fprintf (stderr,
			"%s: unable to get username\n", Program);
		(void) exit (1);
	    }
	    FromUserName = pw -> pw_name;
	    Debug (1, "-> FromUserName=\"%s\"\n", FromUserName);
	}

	if (ModifyUtmp (FromUserName))
	    (void) fprintf (stderr, "update of %s entry failed\n", UtmpFile);
	else
	    (void) printf ("%s entry fixed\n", UtmpFile);

	if (ValidTTY != -1) {
	    if (chown (FullTTY, Uid, Gid))
		(void) perror (FullTTY);
	    if (stat (FullTTY, &st)) {
		(void) perror (FullTTY);
	    } else {
		if (chmod (FullTTY, (int) ((st.st_mode & 07777) | 022)))
		    (void) perror (FullTTY);
	    }
	}
	(void) printf ("%s ownership and mode fixed\n", FullTTY);

	(void) exit (0);
    }

    /*
     * get OldUtmpName from cuserid ();
     */
#ifdef BROKENCUSERID
    OldUtmpName = mycuserid ((char *) 0);
#else
    OldUtmpName = cuserid ((char *) 0);
#endif
    Debug (1, "-> OldUtmpName from cuserid = \"%s\"\n", OldUtmpName);
    if (OldUtmpName == NULL) {
	(void) fprintf (stderr, "%s: unable to get username\n", Program);
	(void) exit (1);
    }

    /*
     * assign it as the default (if needed) current name
     */

    if (FromUserName == NULL) {
	FromUserName = malloc ((unsigned) strlen (OldUtmpName) + 1);
	(void) strcpy (FromUserName, OldUtmpName);
    }

    /*
     * check and see if FromUserName is in /etc/passwd
     */

    if ((pw = getpwnam (FromUserName)) == NULL) {
	(void) fprintf (stderr, "%s: unable to find user: %s in /etc/passwd\n",
		Program, FromUserName);
	(void) exit (1);
    }

    /*
     * get crypt passwd
     */

    
#ifdef USE_SHADOW
    if ((spw = getspnam (FromUserName)) == NULL) {
	(void) fprintf (stderr, "%s: unable to find user: %s in /etc/shadow\n",
		Program, FromUserName);
	(void) exit (1);
    }
    MyCryptPasswd = malloc ((unsigned) strlen (spw -> sp_pwdp) + 1);
    (void) strcpy (MyCryptPasswd, spw -> sp_pwdp);
#else
    MyCryptPasswd = malloc ((unsigned) strlen (pw -> pw_passwd) + 1);
    (void) strcpy (MyCryptPasswd, pw -> pw_passwd);
#endif

    /*
     * can we find new name?
     */

    if ((pw = getpwnam (NewUserName)) == NULL) {
	(void) fprintf (stderr,
		"%s: unable to find user: %s in /etc/passwd\n",
		Program, NewUserName);
	(void) exit (1);
    }


#ifndef NOUSERCHECK

    /*
     * let's open SystemNameFile
     */

    if ((f = fopen (SystemNameFile, "r")) == NULL) {
	(void) perror (SystemNameFile);
	(void) exit (1);
    }

    switch (NameCheck (f, FromUserName)) {
	case NameFound:
	    (void) Authorized++;
	    (void) AskPasswd++;
	    break;

	case PlusNameFound:
	    (void) Authorized++;
	    break;

	case NotFound:
	case StarFound:
	    break;
    }

    (void) fclose (f);

    /*
     * remove write access from name file for security
     */

    if (stat (SystemNameFile, &st) ||
	    chmod (SystemNameFile, (int) ((st.st_mode & 07777) &
		    (SAFE ? ~0666 : ~0222))))
	(void) perror (SystemNameFile);

#ifndef NOSU2RC
    if (!(Authorized && AskPasswd) && *pw -> pw_dir) {

	/*
	 * check that users .su2rc file
	 */

	su2rcFile = (char *) malloc ((unsigned) strlen (pw -> pw_dir) +
		(unsigned) strlen (UsersNameFile) + 2);
	(void) sprintf (su2rcFile, "%s/%s", pw -> pw_dir, UsersNameFile);

	if ((!stat (su2rcFile, &st)) && (st.st_mode & 04000) &&
		((st.st_uid == pw -> pw_uid) || (st.st_uid == 0)) &&
		(st.st_nlink == 1)) {

	    /*
	     * open su2rcFile
	     */

	    if (f = fopen (su2rcFile, "r"))
		switch (NameCheck (f, FromUserName)) {

		    case NameFound:
			/*
			 * if they are Authorized by the master
			 * super-users file for access without
			 * requiring a passwd, do not set the AskPasswd
			 * flag.
			 */

			if (!Authorized)
			    (void) AskPasswd++;
			(void) Authorized++;
			break;

		    case PlusNameFound:
			(void) Authorized++;
			break;

		    case StarFound:
			/*
			 * a "*" entry was found
			 */

			(void) Authorized++;
			break;

		    case NotFound:
			break;
		}

	    (void) fclose (f);
	}
    }
#endif NOSU2RC
#endif NOUSERCHECK

#ifndef NOPASSWDCHECK
    if ((AskPasswd || !(Authorized || AskPasswd) || Flag_u) &&
	    (Uid != 0)) {
	(void) sprintf (Buffer, "Password:");

	/*
	 * get this users password
	 */

	MyPasswd = getpass (Buffer);

	/*
	 * encrypt and check it
	 */

	if (MyPasswd[0] == '\0' ||
		strcmp (crypt (MyPasswd, MyCryptPasswd), MyCryptPasswd)) {
	    (void) AddSuLog (FromUserName, NewUserName, "-");
	    (void) fprintf (stderr, "Password incorrect\n");
	    (void) exit (1);
	}
    }

#endif NOPASSWDCHECK

#ifndef NOUSERCHECK
    if (!Authorized) {
	(void) AddSuLog (FromUserName, NewUserName, "-");
	(void) fprintf (stderr,
		"%s: User %s is not authorized for this system.\n",
		Program, FromUserName);
	(void) fprintf (stderr,
		"%s: For authorization, see your system administrator.\n",
		Program);
	(void) exit (1);
    }
#endif NOUSERCHECK

    /*
     * close passwd file
     */

#ifndef sgi
    (void) endpwent ();
#endif

    /*
     * check and see if we should set /etc/utmp entry
     */

    (void) AddSuLog (FromUserName, NewUserName, "+");
    if (Flag_s)
	if (ModifyUtmp (NewUserName)) {
	    (void) fprintf (stderr, "update of %s entry failed\n", UtmpFile);
	    (void) exit (1);
	}
    if (Flag_s && Flag_x)
	(void) printf ("%s: please do \"%s -f\" after exiting new shell\n",
		Program, Program);

    /*
     * what type of transfer is this
     */

    ChangeType = NoChange;

    if (Uid == 0) {
	if (pw -> pw_uid != 0)
	    ChangeType = FromRoot;
    } else {
	if (pw -> pw_uid == 0)
	    ChangeType = ToRoot;
    }

    /*
     * if from or to uid is not root and the nice value was
     * negative, un-nice the process
     */

    if ((NiceInc < 0) && (pw -> pw_uid != 0) && (Uid != 0))
	(void) nice (-NiceInc);

    /*
     * does user have a default shell
     */

    if (*pw -> pw_shell == '\0')
	pw -> pw_shell = DefaultShell;

    if (c = strrchr (pw -> pw_shell, '/'))
	(void) c++;
    else
	c = pw -> pw_shell;

    /*
     * make sure default shell is not something like uucico
     */

    if (strcmp (c, "sh") && strcmp (c + 1, "sh")) {
	pw -> pw_shell = DefaultShell;
	if (c = strrchr (pw -> pw_shell, '/'))
	    (void) c++;
	else
	    c = pw -> pw_shell;
    }
    Debug (1, "-> pw -> pw_shell=\"%s\"\n", pw -> pw_shell);

    /*
     * build arg[0] of shell.  add leading '-' if we want to
     * do a full login.
     */

    Shell = malloc ((unsigned) strlen (c) + 2);
    (void) sprintf (Shell, FullLogin ? "-%s" : "%s", c);
    Debug (1, "-> Shell=\"%s\"\n", Shell);

    if (FullLogin) {

	/*
	 * we will exec the newusers shell
	 */

	ShellPointer = pw -> pw_shell;

	/*
	 * set up $HOME, $PATH, $LOGNAME
	 */

	NewHomeDir = malloc ((unsigned) strlen (pw -> pw_dir) + 6);
	(void) sprintf (NewHomeDir, "HOME=%s", pw -> pw_dir);

	NewPath = (pw -> pw_uid == 0) ? RootPath : UserPath;

#ifdef BSD
	NewLogname = malloc ((unsigned) strlen (pw -> pw_name) + 6);
	(void) sprintf (NewLogname, "USER=%s", pw -> pw_name);
#else
	NewLogname = malloc ((unsigned) strlen (pw -> pw_name) + 9);
	(void) sprintf (NewLogname, "LOGNAME=%s", pw -> pw_name);
#endif BSD

	NewShell = malloc ((unsigned) strlen (pw -> pw_shell) + 7);
	(void) sprintf (NewShell, "SHELL=%s", pw -> pw_shell);

#ifdef BSD
	if ((NewTerm = getenv ("TERM")) != NULL)
	    NewTerm = (char *) ((int) NewTerm - 5);/* back up to "TERM=" */
#endif BSD

	envp = newenvp = NewEnvironment;
    } else {

	/*
	 * modify old env variables
	 */

	/*
	 * find out size of current environment and allocate a new bigger
	 * one.
	 */

	for (i = 0, eptr = envp; *eptr; i++, eptr++);
	newenvp = (char **) malloc ((unsigned) ((i + 20) * sizeof (char *)));

	for (eptr = envp, neptr = newenvp; *eptr; eptr++, neptr++) {
	    *neptr = *eptr;
	    Debug (1, "-> *neptr=%s\n", *neptr);

#ifdef CHANGEHOME
	    if (!Flag_h && strncmp (*neptr, "HOME=", 5) == 0) {
		*neptr = malloc ((unsigned) strlen (pw -> pw_dir) + 6);
		(void) sprintf (*neptr, "HOME=%s", pw -> pw_dir);
	    }
#endif CHANGEHOME
#ifdef CHANGEUSER
	    if (strncmp (*neptr, "USER=", 5) == 0) {
		*neptr = malloc ((unsigned) strlen (pw -> pw_name) + 6);
		(void) sprintf (*neptr, "USER=%s", pw -> pw_name);
	    }
#endif CHANGEUSER

	    /*
	     * change shell
	     */
	    if (strncmp (*neptr, "SHELL=", 6) == 0) {
		if (Flag_d) {
		    *neptr = malloc ((unsigned) strlen (DefaultShell) + 7);
		    (void) sprintf (*neptr, "SHELL=%s", DefaultShell);
		}
#ifdef CHANGESHELL
 		else {
		    *neptr = malloc ((unsigned) strlen (pw -> pw_shell) + 7);
		    (void) sprintf (*neptr, "SHELL=%s", pw -> pw_shell);
		}
#endif CHANGESHELL
		ShellPointer = *neptr + 6;
		if (Shell = strrchr (ShellPointer, '/'))
		    (void) Shell++;
		else
		    Shell = ShellPointer;

		Debug (1, "-> new shell=\"%s\"\n", *neptr);
		continue;
	    }

	    if (!strncmp (*neptr, "SUPATH=", 7)) {
		EnvRootPath = neptr;
		Debug (1, "-> *EnvRootPath=\"%s\"\n", *EnvRootPath);
		continue;
	    }

	    if (!strncmp (*neptr, "UPATH=", 6)) {
		EnvUserPath = neptr;
		Debug (1, "-> *EnvUserPath=\"%s\"\n", *EnvUserPath);
		continue;
	    }

	    if (!strncmp (*neptr, "PATH=", 5)) {
		EnvPath = neptr;
		Debug (1, "-> *EnvPath=\"%s\"\n", *EnvPath);
		continue;
	    }

	    /*
	     * change prompt if necessary
	     */

	    if (strncmp (*neptr, "PS1=", 4) == 0) {
		EnvPS1 = neptr;
		Debug (1, "-> *EnvPS1=\"%s\"\n", *EnvPS1);
		continue;
	    }
	}			/* for */

	/*
	 * fix up last character of PS1 if it makes sense to
	 */

	if (EnvPS1) {
	    Debug (1, "-> changing PS1\n", 0);
	    ChangePS1 (*EnvPS1, ChangeType, Shell);
	    Debug (1, "-> PS1 changed\n", 0);
	    Debug (1, "-> new PS1=\"%s\"\n", *EnvPS1);
	}

	/*
	 * fix up "PATH=..." as follows:  If change is from non-root to root,
	 * take path from "SUPATH=..." (if found).  If change is from root to
	 * non-root, take path from "UPATH=..." (if found).
	 * If change is from non-root to root, clean up the path.
	 */

	if (ChangeType == ToRoot) {

	    /*
	     * if no "UPATH=...", stuff current "PATH=..." there
	     */

	    if (!EnvUserPath) {
		Debug (1, "-> assigning $PATH->$UPATH\n", 0);
		EnvUserPath = neptr++;
		*EnvUserPath = malloc ((unsigned) strlen (*EnvPath) + 2);
		(void) sprintf (*EnvUserPath, "UPATH=%s", (*EnvPath) + 5);
		Debug (1, "-> new *EnvUserPath=%s\n", *EnvUserPath);
	    }

	    /*
	     * if "SUPATH=...", stuff it into "PATH=..."
	     */

	    if (EnvRootPath) {
		Debug (1, "-> assigning $SUPATH->$PATH\n", 0);
		*EnvPath = malloc ((unsigned) strlen (*EnvRootPath) - 1);
		(void) sprintf (*EnvPath, "PATH=%s", (*EnvRootPath) + 7);
		Debug (1, "-> new *EnvPath=%s\n", *EnvPath);
	    }

	    /*
	     * strip path of all entries that do not begin
	     * in a / if we are becoming super-user
	     */

	    Debug (1, "-> stripping *EnvPath=%s\n", *EnvPath);
	    Debug (2, "-> &EnvPath=%lx\n", (long) EnvPath);
	    Debug (2, "-> &*EnvPath=%lx\n", (long) * EnvPath);

	    CleanPath ((*EnvPath) + 5);

	    Debug (2, "-> *EnvPath stripped\n", 0);
	    Debug (2, "-> &EnvPath=%lx\n", (long) EnvPath);
	    Debug (2, "-> &*EnvPath=%lx\n", (long) * EnvPath);
	    Debug (1, "-> stripped *EnvPath=%s\n", *EnvPath);
	}
	if ((ChangeType == FromRoot) && EnvUserPath) {

	    /*
	     * stuff "UPATH=..." into "PATH=..."
	     */

	    Debug (1, "-> assigning $UPATH->$PATH\n", 0);
	    *EnvPath = malloc ((unsigned) strlen (*EnvUserPath) + 0);
	    (void) sprintf (*EnvPath, "PATH=%s", *EnvUserPath + 6);
	    Debug (1, "-> new *EnvPath=%s\n", *EnvPath);
	}

	/*
	 * terminate new environment with a null string and assign
	 * it to envp
	 */

	Debug (1, "-> terminating new environ\n", 0);
	*neptr = (char *) 0;
	Debug (2, "-> envp = newenvp\n", 0);
	envp = newenvp;
    }				/* else */

    /*
     * if a valid tty, do some things to it
     */

    Debug (1, "-> changing tty mode/owner\n", 0);
    if (ValidTTY != -1) {

	/*
	 * get the terminal's characteristics for later restoring
	 */

#ifdef IOCTL
	Debug (1, "-> saving tty ioctl\n", 0);
#ifdef BSD
	if (TTYIoctlError = ioctl (ValidTTY, TIOCGETP, &TTYtermio))
#else BSD
	if (TTYIoctlError = ioctl (ValidTTY, TCGETA, &TTYtermio))
#endif BSD
	    (void) perror ("ioctl");
#endif IOCTL

	/*
	 * get the terminal's current mode, uid, gid (and maybe change
	 * them
	 */

	if (TTYStatError = stat (FullTTY, &st)) {
	    (void) perror (FullTTY);
	} else {

#ifndef NOCHMODTTY

	    /*
	     * only remove TTY rw access if changing to root
	     */

	    TTYmode = st.st_mode & 07777;

	    if (ChangeType == ToRoot)
		if (chmod (FullTTY, TTYmode & ~066))
		    (void) perror (FullTTY);
#endif NOCHMODTTY

	    TTYuid = st.st_uid;
	    TTYgid = st.st_gid;

#ifdef NOCHOWNTTY

	    /*
	     * only change ownership if changing utmp
	     */

	    if (Flag_s)
		if (chown (FullTTY, pw -> pw_uid, pw -> pw_gid))
		    (void) perror (FullTTY);
#else NOCHOWNTTY
	    if (chown (FullTTY, pw -> pw_uid, pw -> pw_gid))
		(void) perror (FullTTY);
#endif NOCHOWNTTY
	}
    }

    /*
     * unless we had a -x option we will fork a shell
     */

    Debug (1, "-> forking new shell\n", 0);
    if (Flag_x || ((pid = vfork ()) == 0)) {

	/*
	 * change group id
	 *
	 * this is done before changing userid incase the
	 * uid is not 0
	 */

	if (setgid (pw -> pw_gid)) {
	    (void) perror ("setgid");
	    (void) exit (1);
	}

#if defined(BSD42) || defined(INITGROUPS)
	if (initgroups (pw -> pw_name, pw -> pw_gid)) {
	    (void) perror ("initgroups");
	    (void) exit (1);
	}
#endif defined(BSD42) || defined(INITGROUPS)

	/*
	 * change user id
	 */

	if (setuid (pw -> pw_uid)) {
	    (void) perror ("setgid");
	    (void) exit (1);
	}

	Debug (1, "-> exec program=\"%s\"\n", ShellPointer);
	Debug (1, "-> exec argv[0]=\"%s\"\n", Shell);
#ifdef DEBUG
	Debug (1, "-> exec envp=\n", 0);
	{
	    char  **eptr;
	    for (eptr = envp; *eptr; eptr++)
		Debug (1, "-> 	\"%s\"\n", *eptr);
	}
#endif DEBUG

	/*
	 * Try to cd to the home directory, if doing a full login.
	 * If it fails, then run in the current directory.
	 */

	if (FullLogin)
	    if (chdir (NewHomeDir + 5) < 0)
		(void) perror (NewHomeDir + 5);

	/*
	 * exec shell
	 */

	newargv[0] = Shell;
	if (Flag_c) {
	    newargv[1] = "-c";
	    Debug (1, "-> newargv[1]=\"%s\"\n", newargv[1]);
	    newargv[2] = cString;
	    Debug (1, "-> newargv[2]=\"%s\"\n", newargv[2]);
	    newargv[3] = (char *) 0;
	} else {
	    newargv[1] = (char *) 0;
	}

	(void) execve (ShellPointer, newargv, newenvp);

	(void) perror (ShellPointer);
	(void) exit (1);

    } else {

	/*
	 * wait for shell to exit
	 */

	(void) signal (SIGINT, SIG_IGN);
	(void) signal (SIGQUIT, SIG_IGN);
	while (((i = wait (&Status)) != pid) && (i != -1));
	(void) signal (SIGINT, SIG_DFL);
	(void) signal (SIGQUIT, SIG_DFL);

	/*
	 * reset utmp if necessary
	 */

	if (Flag_s)
	    if (ModifyUtmp (OldUtmpName)) {
		(void) fprintf (stderr, "update of %s entry failed\n",
			UtmpFile);
		(void) Status++;
	    }

#ifdef IOCTL

	/*
	 * reset TTY characteristics
	 */

	if ((ValidTTY != -1) && !TTYIoctlError) {
	    Debug (1, "-> restoring tty ioctl\n", 0);
#ifdef BSD
	    if (ioctl (ValidTTY, TIOCSETP, &TTYtermio))
		(void) perror ("ioctl");
#else BSD
	    if (ioctl (ValidTTY, TCSETAF, &TTYtermio))
		(void) perror ("ioctl");
#endif BSD
	}
#endif IOCTL

	/*
	 * reset TTY mode, uid, and gid
	 */

	if ((ValidTTY != -1) && !TTYStatError) {
#ifndef NOCHMODTTY
	    if (ChangeType == ToRoot)
		if (chmod (FullTTY, TTYmode))
		    (void) perror (FullTTY);
#endif NOCHMODTTY
#ifdef NOCHOWNTTY
	    if (Flag_s)
		if (chown (FullTTY, TTYuid, TTYgid))
		    (void) perror (FullTTY);
#else NOCHOWNTTY
	    if (chown (FullTTY, TTYuid, TTYgid))
		(void) perror (FullTTY);
#endif NOCHOWNTTY
	}

	(void) exit (Status != 0);
    }
}


#if defined(BSD) || defined(hpux) || defined(BROKENCUSERID)

/**********
 *
 * char *cuserid ()
 * get name of current userid
 *
 * This function written by Marion Hakanson (hakanson@orstcs)
 *	at Oregon State University, 09/23/85.
 *
 * Parameters:
 *	none (ignored, anyway).
 *
 * Returns:
 *	NULL if not found.
 *	Pointer to null-terminated name, otherwise.
 *
 * Side Effects:
 *	Copies name into an internal static buffer.
 */

#define MAXNAME sizeof(((struct utmp *)nptr)->ut_name)

#ifdef BROKENCUSERID
char *mycuserid ()
#else
char
*cuserid ()
#endif
{
    char   *getlogin ();
    char   *nptr;
    struct passwd  *pwptr;
    static char name[MAXNAME + 1];

    if ((nptr = getlogin ()) == NULL) {/* name from utmp file */
#ifndef sgi
      setpwent ();
#endif
	pwptr = getpwuid (Uid);
#ifndef sgi
	(void) endpwent ();
#endif
	if (pwptr == NULL)
	    return (NULL);
	nptr = pwptr -> pw_name;/* name from passwd file */
    }
    strncpy (name, nptr, MAXNAME);
    return (name);
}

#endif BSD || hpux

/**********
 * enum NameCheck_ NameCheck (f, FromUserName)
 *
 * check FromUserName in file f
 *
 * Parameters:
 *	FILE *f -- file to search for username
 *	char *FromUserName -- name of user to search for
 *
 * Returns:
 *	NameNotFound if not found
 *	NameFound if found
 *	PlusNameFound if found and preceeded by a '+'
 *	StarFound if file contains the wildcard '*'
 *
 * Side Effects:
 *	File f is read.
 */

enum NameCheck_
NameCheck (f, FromUserName)
FILE * f;
register char  *FromUserName;
{
    char   Buffer[BUFSIZ];
    register int    Starred = 0;/* non zero == star found */

    /*
     * read through file.  check and see if name is in list
     */

    while (fgets (Buffer, 80, f)) {

	/*
	 * remove trailing nl
	 */

	Buffer[strlen (Buffer) - 1] = '\0';

	/*
	 * is this the user?
	 */

	if (!strcmp (Buffer, FromUserName))
	    return (NameFound);

	/*
	 * first char '+'?  if so we do not need to AskPasswd for
	 * a password unless they specified the "-u name" option.
	 */

	if ((*Buffer == '+') && !strcmp (Buffer + 1, FromUserName))
	    return (PlusNameFound);

	/*
	 * is the entry a "*"?
	 */

	if ((*Buffer == '*') && (*(Buffer + 1) == '\0'))
	    Starred++;
    }
    return (Starred ? StarFound : NotFound);
}


/**********
 * int ModifyUtmp (NewUserName)
 *
 * Change entry for this TTY in /etc/utmp to NewUserName.
 *
 * Parameters:
 *	char *NewUserName -- username to change entry to.
 *
 * Returns:
 *	0 -- success.
 *	!0 -- failure.
 *
 * Side Effects:
 *	/etc/utmp (or whatever the file name is) is modified.
 */

int
ModifyUtmp (NewUserName)
register char  *NewUserName;
{
    register int    fd;		/* /etc/utmp file */
    register int    i;		/* index */
#ifdef hpux
    struct utmp    Utmp;    /* hpux cc complains about taking address */
#else
    struct utmp    Utmp;  /* for /etc/utmp manipulation */
#endif


    if ((fd = open (UtmpFile, O_RDWR)) < 0) {
	(void) perror (UtmpFile);
	return (1);
    }

    /*
     * find our entry in UTMP
     */

    while (read (fd, (char *) & Utmp, sizeof (Utmp)) == sizeof (Utmp))
	if (!strncmp (TTY, Utmp.ut_line, sizeof (Utmp.ut_line)) &&
		*Utmp.ut_name)
	    goto UtmpMatch;

    (void) printf ("No entry in %s\n", UtmpFile);
    (void) close (fd);
    return (1);

UtmpMatch:

    /*
     * backup to beginning of this utmp entry
     */

    (void) lseek (fd, (long) - sizeof (Utmp), 1);

    /*
     * fill utmp.ut_name with nulls
     */

    for (i = 0; i < 8; i++)
	Utmp.ut_name[i] = '\0';

    /*
     * copy name to Utmp.ut_name
     */

    (void) strncpy (Utmp.ut_name, NewUserName, sizeof (Utmp.ut_name));

    (void) write (fd, (char *) & Utmp, sizeof (Utmp));
    (void) close (fd);
    return (0);
}


/**********
 * void AddSuLog (FromName, ToName, ChangeType)
 *
 * add switch from user "FromName" to user "ToName" to sulog.
 * ChangeType is "+" for success, "-" for failure.
 *
 * Parameters:
 *	char *FromName -- from name (for logging).
 *	char *ToName -- to name (for logging).
 *	char *ChangeType -- +/- (for logging).
 *
 * Returns:
 *	nothing.
 *
 * Side Effects:
 *	A change record is appended to /usr/lib/sulog.
 */

void
AddSuLog (FromName, ToName, ChangeType)
char   *FromName;		/* user name  */
char   *ToName;
char   *ChangeType;		/* '+' or '-' */
{
  /* This is a redeclaration on most systems:
   * long    time ();
   */
    struct tm  *localtime ();

    register    FILE * f;
    struct stat st;
    long    timenow;
    struct tm  *now;

    if ((f = fopen (SULog, "a")) == NULL) {
	(void) perror (SULog);
	(void) exit (1);
    }

    time (&timenow);
    now = localtime (&timenow);

    if (fprintf (f, "SU2 %.2d/%.2d %.2d:%.2d %1.1s %s %s-%s\n",
		now -> tm_mon + 1, now -> tm_mday, now -> tm_hour,
		now -> tm_min, ChangeType, TTY, FromName, ToName) < 0) {
	(void) perror (SULog);
	(void) exit (2);
    }

    (void) fclose (f);

    /*
     * take away write access FromName SULog
     */

    if (stat (SULog, &st) || chmod (SULog, (int) (st.st_mode & 07777) &
	    (SAFE ? ~0666 : ~0222)))
	(void) perror (SULog);

    return;
}


/**********
 * void CleanPath (Path);
 *
 * remove any directories from the path that are
 *    - null (leading/trailing colon or double colon)
 *    - not anchored to root (no leading /)
 *
 * the returned path is the original path with any such
 * directories stripped
 *
 * Parameters:
 *	char *Path -- $PATH to clean
 *
 * Returns:
 *	none.
 *
 * Side Effects:
 *	Unanchored paths will be stripped off of Path.
 */

void
CleanPath (Path)
register char  *Path;
{
    register char  *StrippedPath;
    register char  *PathHead;

    StrippedPath = PathHead = Path;

    while (*Path) {

	/*
	 * remove multiple ':'s
	 */

	while (*Path && (*Path == ':')) {
	    Debug (2, "-> skip colon: *Path=%c\n", *Path);
	    (void) Path++;
	}

	/*
	 * is the first character of this
	 * directory a '/'????
	 */

	if (*Path == '/') {

	    /*
	     * copy directory intact;
	     */

	    while (*Path && (*Path != ':')) {
		Debug (3, "-> copy directory: *Path=%c\n", *Path);
		*StrippedPath++ = *Path++;
	    }

	    if (*Path == ':') {
		Debug (3, "-> copy colon: *Path=%c\n", *Path);
		*StrippedPath++ = *Path++;
	    }
	}
	else {

	    /*
	     * zip past directory
	     */

	    while (*Path && (*Path != ':')) {
		Debug (3, "-> skip directory: *Path=%c\n", *Path);
		(void) Path++;
	    }

	    if (*Path == ':') {
		Debug (3, "-> copy colon: *Path=%c\n", *Path);
		(void) Path++;
	    }
	}

    }

    /*
     * remove all trailing ':'s
     */

    while ((StrippedPath > PathHead) && (StrippedPath[-1] == ':')) {
	Debug (2, "-> drop colon: StrippedPath[-1]=%c\n", StrippedPath[-1]);
	StrippedPath--;
    }

    /*
     * null terminate the path
     */

    *StrippedPath = '\0';
    return;
}


/**********
 * void ChangePS1(PS1, ChangeType, Shell);
 *
 * Change last character of prompt to reflect the new user.  If the
 * switch is from non-root to root, make it a "#".  If the switch is
 * from root to non-root, make it a '%' (for csh), '$' for everything
 * else.
 *
 * Parameters:
 *	char *PS1 -- pointer to PS1 environment variable
 *	enum ChangeType_ ChangeType -- type of change
 *	char *Shell -- pointer to last component of shell (i.e. "sh")
 *
 * Returns:
 *	none.
 *
 * Side effects:
 *	PS1 will be changed as indicated above.
 */

void
ChangePS1 (PS1, ChangeType, Shell)
register char  *PS1;
enum ChangeType_ ChangeType;
char   *Shell;
{
    /*
     * find last character
     */

    while (*PS1)
	(void) PS1++;

    /*
     * back up to last character
     */

    (void) PS1--;

    /*
     * back up past spaces
     */

    while ((*PS1 == ' ') || (*PS1 == '\t'))
	(void) PS1--;

    if (ChangeType == ToRoot)
	*PS1 = '#';
    if (ChangeType == FromRoot)
	*PS1 = strcmp (Shell, "csh") ? '$' : '%';
    return;
}


/**********
 * void Usage ()
 *
 * print usage message.
 *
 * Parameters:
 *	none.
 *
 * Returns:
 *	none.
 *
 * Side Effects:
 *	Usage message is printed, and program exits with a return value
 *	of 1.
 */

void
Usage ()
{
    (void) fprintf (stderr,
	    "usage: %s [-] [-xsfd] [-ufromname] [-niceval] [toname] [-c string]\n",
	    Program);
    (void) exit (1);
}

/*
 * su2.c
 *
 * Copyright Hewlett-Packard Company, 1985.  Permission is granted for
 * unlimited modification, use, and distribution, except that this software
 * may not be sold for profit directly nor as part of any software package.
 * This software is made available with no warranty of any kind, express
 * or implied.
 *
 * Writen by:
 *	Dave Serisky (hp-pcd!daves)
 *	Handheld Computer and Calculator Operation
 *	Hewlett-Packard
 *	1000 NE Circle BLVD.
 *	Corvallis, OR   97330
 *	(503) 757-2000
 *
 ******	Revision history ***********************************************

	851029	Added the -d option
		Fixed -c bug.  Worked for "-cstring" but not for
		"-c string".
	851016	Added IOCTL and INITGROUPS compile options.
	851007	Included the following mods from hakanson@orstcs:
		Added CHANGEHOME compile option.
		Added BSD compile option,
		Added NOCHOWNTTY and NOCHMODTTY compile options.
		For BSD option:
 			Includes cuserid() code.
			Alternate default PATH.
			Added definition of TERM in NewEnvironment.
			Does initgroups().
		Now chdir() to home directory for FullLogin;
		Added conditional use of vfork() instead of fork().
		Fixed bug in environment-passing for FullLogin option.
		Fixed bug where tty ownership got reset on exit only
		if ChangeType == ToRoot.
	850828	cleaned up some of the code, and added additional compile
		options.
	850725	added "UPATH" and "SUPATH" environment variables.  Fixed
		.su2rc handler so that a leading "+" in the .su2rc file
		would not be blocked by its absence in the super-users
		file.  Moved the path stripper and change prompt code
		into a function to use registers on the character pointers.
	850326	added the ability for users to install their own .su2rc
		file.  must be suid to that user.
	850322	fixed chmod so that it sets the correct bits
	841226	changes gid before uid.  if new uid == 0: makes tty mode 600;
		removes from path all directories not beginning with /.
	841127	added -r -s options
	841018	added "-" option, cleaned up program somewhat
	840926	full rewrite
	840906  if prompt declared, set last non-blank character to #
	840830  Identifies tty properly now, notes failure as well as success.
	840727	Original issue.

***********************************************************************/
