/* tn5250 -- an implentation of the 5250 telnet protocol.
 * Copyright (C) 1997 Michael Madore
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include <assert.h>
#include <sys/time.h>
#include <curses.h>
#include <string.h>
#include <ctype.h>
#include "utility.h"
#include "displaybuf.h"
#include "terminal.h"
#include "cursesterm.h"

/* Mapping of 5250 colors to curses colors */
#define A_5250_GREEN    (COLOR_PAIR(COLOR_GREEN))
#define A_5250_WHITE    (COLOR_PAIR(COLOR_WHITE) | A_BOLD)
#define A_5250_RED      (COLOR_PAIR(COLOR_RED))
#define A_5250_TURQ     (COLOR_PAIR(COLOR_CYAN))
#define A_5250_YELLOW   (COLOR_PAIR(COLOR_YELLOW) | A_BOLD)
#define A_5250_PINK     (COLOR_PAIR(COLOR_MAGENTA))
#define A_5250_BLUE     (COLOR_PAIR(COLOR_CYAN) | A_BOLD)

static int attribute_map[] = { A_5250_GREEN,
                   A_5250_GREEN | A_REVERSE,
                   A_5250_WHITE,
                   A_5250_WHITE | A_REVERSE,
                   A_5250_GREEN | A_UNDERLINE,
                   A_5250_GREEN | A_UNDERLINE | A_REVERSE,
                   A_5250_WHITE | A_UNDERLINE,
                   0x00,
                   A_5250_RED,
                   A_5250_RED | A_REVERSE,
                   A_5250_RED | A_BLINK,
                   A_5250_RED | A_BLINK | A_REVERSE,
                   A_5250_RED | A_UNDERLINE,
                   A_5250_RED | A_UNDERLINE | A_REVERSE,
                   A_5250_RED | A_UNDERLINE | A_BLINK,
                   0x00,
                   A_5250_TURQ | A_VERTICAL,
                   A_5250_TURQ | A_VERTICAL | A_REVERSE,
                   A_5250_YELLOW | A_VERTICAL,
                   A_5250_YELLOW | A_VERTICAL | A_REVERSE,
                   A_5250_TURQ | A_UNDERLINE | A_VERTICAL,
                   A_5250_TURQ | A_UNDERLINE | A_REVERSE | A_VERTICAL,
                   A_5250_YELLOW | A_UNDERLINE | A_VERTICAL,
                   0x00,
                   A_5250_PINK,
                   A_5250_PINK | A_REVERSE,
                   A_5250_BLUE,
                   A_5250_BLUE | A_REVERSE,
                   A_5250_PINK | A_UNDERLINE,
                   A_5250_PINK | A_UNDERLINE | A_REVERSE,
                   A_5250_BLUE | A_UNDERLINE,
                   0x00};

void CursesTerminal::init ()
{
   initscr ();  
   raw ();
   keypad (stdscr, TRUE);
   nodelay (stdscr, TRUE);
   noecho ();
   if (has_colors ()) {
      start_color ();
      init_pair(COLOR_BLACK, COLOR_BLACK, COLOR_BLACK);
      init_pair(COLOR_GREEN, COLOR_GREEN, COLOR_BLACK);
      init_pair(COLOR_RED, COLOR_RED, COLOR_BLACK);
      init_pair(COLOR_CYAN, COLOR_CYAN, COLOR_BLACK);
      init_pair(COLOR_WHITE, COLOR_WHITE, COLOR_BLACK);
      init_pair(COLOR_MAGENTA, COLOR_MAGENTA, COLOR_BLACK);
      init_pair(COLOR_BLUE, COLOR_BLUE, COLOR_BLACK);
      init_pair(COLOR_YELLOW, COLOR_YELLOW, COLOR_BLACK);
   }
   conn_fd = -1;
   quit_flag = false;
   underscores = false;
}

void CursesTerminal::term ()
{
   endwin ();
}

void CursesTerminal::connection_fd (int fd)
{
   conn_fd = fd;
}

int CursesTerminal::width () const
{
   int y, x;
   getmaxyx (stdscr, y, x);
   return x+1;
}

int CursesTerminal::height () const
{
   int y, x;
   getmaxyx (stdscr, y, x);
   return y+1;
}

int CursesTerminal::flags () const
{
   int f = 0;
   if (has_colors ())
      f |= HAS_COLOR;
   return f;
}

void CursesTerminal::update (const DisplayBuffer& buffer)
{
   int my, mx;
   int curs_attr;
   unsigned char a = 0x20, c;

   attrset (A_NORMAL);
   getmaxyx (stdscr, my, mx);
   for (int y = 0; y < buffer.height (); y++) {
      if (y > my)
	 break;

      move (y, 0);
      for (int x = 0; x < buffer.width (); x++) {

 	 c = buffer[y][x]; 
 	 if ((c & 0xe0) == 0x20) {  /* ATTRIBUTE */
 	    a = (c & 0xff);
 	    addch (attribute_map[0] | ' ');
 	 } else {		    /* DATA */
 	    curs_attr = attribute_map[a - 0x20];
 	    if (curs_attr == 0x00) { /* NONDISPLAY */
 	       addch (attribute_map[0] | ' ');
 	    } else {
	       c = ebcdic2ascii (c);
	       if (((c & 0x60) == 0) || c == 0x7f) { /* UNPRINTABLE */
		  c = ' ';
		  curs_attr ^= A_REVERSE;
	       }
 	       if ((curs_attr & A_VERTICAL) != 0) {
 		  curs_attr |= A_UNDERLINE;
 		  curs_attr &= ~A_VERTICAL;
 	       }
	       // This is a kludge since vga hardware doesn't support under-
	       // lining characters.  It's pretty ugly.
	       if (underscores) {
		  if ((curs_attr & A_UNDERLINE) != 0) {
		     curs_attr &= ~A_UNDERLINE;
		     if (c == ' ')
			c = '_';
		  }
	       }
 	       addch (c|curs_attr);
 	    }
 	 } /* if ((c & 0xe0) ... */
       } /* for (int x ... */
   } /* for (int y ... */

   move (buffer.cursor_y (), buffer.cursor_x ());
   refresh ();
}

void CursesTerminal::update_indicators (const DisplayBuffer& buffer)
{
   int inds = buffer.indicators ();
   char ind_buf [80];
   
   memset (ind_buf, ' ', sizeof (ind_buf));
   memcpy (ind_buf, "5250", 4);
   if ((inds & DisplayBuffer::IND_MESSAGE_WAITING) != 0) {
      memcpy (ind_buf + 23, "MW", 2);
   }
   if ((inds & DisplayBuffer::IND_INHIBIT) != 0) {
      memcpy (ind_buf + 9, "X II", 4);
   } else if ((inds & DisplayBuffer::IND_X_CLOCK) != 0) {
      memcpy (ind_buf + 9, "X CLOCK", 7);
   } else if ((inds & DisplayBuffer::IND_X_SYSTEM) != 0) {
      memcpy (ind_buf + 9, "X SYSTEM", 8);
   }
   if ((inds & DisplayBuffer::IND_INSERT) != 0) {
      memcpy (ind_buf + 30, "IM", 2);
   }

   attrset (COLOR_PAIR (COLOR_WHITE));
   mvaddnstr (buffer.height (), 0, ind_buf, 80);
   move (buffer.cursor_y (), buffer.cursor_x ());
   attrset (A_NORMAL);
   refresh ();
}

int CursesTerminal::waitevent ()
{
   fd_set fdr;
   int result = 0;

   if (quit_flag)
      return EVENT_QUIT; 

   FD_ZERO (&fdr);
   FD_SET (0, &fdr);
   FD_SET (conn_fd, &fdr);
   select (conn_fd+1, &fdr, NULL, NULL, NULL);

   if (FD_ISSET (0, &fdr))
      result |= EVENT_KEY;

   if (FD_ISSET (conn_fd, &fdr))
      result |= EVENT_DATA;

   return result;
}

int CursesTerminal::getkey ()
{
   int key = getch ();

   while (1) {
      switch (key) {

      case 0x0d:
      case 0x0a:
	 return K_ENTER; 

      case 0x1b:
	 if ((key = get_esc_key (true)) != ERR)
	    return key;
	 break;

      case K_CTRL('A'): return K_ATTENTION;
      case K_CTRL('B'): return K_ROLLDN;
      case K_CTRL('C'): return K_SYSREQ;
      case K_CTRL('D'): return K_ROLLUP;
      case K_CTRL('E'): return K_ERASE;
      case K_CTRL('F'): return K_ROLLUP;
      case K_CTRL('K'): return K_FIELDEXIT;
      case K_CTRL('L'): return K_REFRESH;
      case K_CTRL('O'): return K_HOME;
      case K_CTRL('P'): return K_PRINT;
      case K_CTRL('R'): return K_RESET; /* Error Reset */
      case K_CTRL('T'): return K_TESTREQ;
      case K_CTRL('U'): return K_ROLLDN;
      case K_CTRL('X'): return K_FIELDEXIT;

      case K_CTRL('Q'):
         quit_flag = true;
      	 return ERR;

      case K_CTRL('G'): /* C-g <function-key-shortcut> */
         if ((key = get_esc_key (false)) != ERR)
	    return key;
	 break;

      case ERR:
         return -1;

      case 127:
	 return K_DELETE;

      case KEY_A1:
	 return K_HOME;

      case KEY_A3:
	 return K_ROLLDN;

      case KEY_C1:
	 return KEY_END;

      case KEY_C3:
	 return K_ROLLUP;

      case KEY_ENTER:
	 return K_FIELDEXIT;

      default:
	 return key;
      }
   }
}

int CursesTerminal::get_esc_key (bool is_esc)
{
   int y, x, key, display_key;
   fd_set fdr;

   getyx (stdscr, y, x);
   attrset (COLOR_PAIR (COLOR_WHITE));
   if (is_esc)
      mvaddstr (24, 60, "Esc ");
   else
      mvaddstr (24, 60, "C-g ");
   move (y, x);
   refresh ();

   FD_ZERO (&fdr);
   FD_SET (0, &fdr);
   select (1, &fdr, NULL, NULL, NULL);
   key = getch ();

   if (isalpha (key))
      key = toupper (key);

   display_key = key;
   switch (key) { 

   /* Function keys */
   case '1': key = K_F1; break;
   case '2': key = K_F2; break;
   case '3': key = K_F3; break;
   case '4': key = K_F4; break;
   case '5': key = K_F5; break;
   case '6': key = K_F6; break;
   case '7': key = K_F7; break;
   case '8': key = K_F8; break;
   case '9': key = K_F9; break;
   case '0': key = K_F10; break;
   case '-': key = K_F11; break;
   case '=': key = K_F12; break;
   case '!': key = K_F13; break;
   case '@': key = K_F14; break;
   case '#': key = K_F15; break;
   case '$': key = K_F16; break;
   case '%': key = K_F17; break;
   case '^': key = K_F18; break;
   case '&': key = K_F19; break;
   case '*': key = K_F20; break;
   case '(': key = K_F21; break;
   case ')': key = K_F22; break;
   case '_': key = K_F23; break;
   case '+': key = K_F24; break;

   /* AS/400 strangeness */
   case 'A': key = K_ATTENTION; break;
   case 'C': key = K_CLEAR; break;
   case 'D': key = K_DUPLICATE; break;
   case 'H': key = K_HELP; break;
   case 'I': key = K_INSERT; break;
   case 'L': key = K_REFRESH; break;
   case 'M': key = K_FIELDMINUS; break;
   case 'P': key = K_PRINT; break;
   case 'R': key = K_RESET; break;
   case 'S': key = K_SYSREQ; break;
   case 'T': key = K_TOGGLE; break;
   case 'X': key = K_FIELDEXIT; break;

   case 127: key = K_INSERT; break;	/* ESC DEL */
   case KEY_DC: key = K_INSERT; break;	/* ESC DEL, also */
   case K_CTRL('J'): key = K_NEWLINE; break;

   case 'Q':
      quit_flag = true; 
      key = ERR;
      break;

   default:
      beep ();
      key = ERR;
      break;
   }

   if (key == ERR)
      mvaddstr (24, 64, "???");
   else
      mvaddch (24, 64, display_key);
   move (y, x);
   refresh ();
   return key;
}

bool CursesTerminal::use_underscores (bool v)
{
   bool oldval = underscores;
   underscores = v;
   return oldval;
}
