/*
 * Copyright (C) 1997 Michael Madore
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#ifdef WIN32
#include <windows.h>
#include <winsock.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <sys/types.h>
#ifndef WIN32
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <netdb.h>
#include <unistd.h>
#endif
#include "buffer.h"
#include "stream5250.h"
#include "utility.h"

#ifdef WIN32
/* Windows' braindead socketing */
#define CLOSESOCKET(x)		closesocket ((x))
#define IOCTLSOCKET(x,y,z)	ioctlsocket ((x),(y),(z))
#define LAST_ERROR		(WSAGetLastError ())
#define ERR_INTR		WSAEINTR
#define ERR_AGAIN 		WSAEWOULDBLOCK
#define WAS_ERROR_RET(r)	((r) == SOCKET_ERROR)
#define WAS_INVAL_SOCK(r)	((r) == INVALID_SOCKET)
#else
/* Real, UNIX socketing */
#define CLOSESOCKET(x)		close ((x))
#define IOCTLSOCKET(x,y,z)	ioctl ((x),(y),(z))
#define LAST_ERROR		(errno)
#define ERR_INTR		EINTR
#define ERR_AGAIN		EAGAIN
#define WAS_ERROR_RET(r)	((r) < 0)
#define WAS_INVAL_SOCK(r)	((r) < 0)
#endif

Stream5250::Stream5250 ()
{
	packets = NULL;
	current_packet = NULL;
	packet_count = 0;
	termtype = NULL;
}

Stream5250::~Stream5250 ()
{
	Packet *iter, *next;

	// Delete any packets that might exist.
	if ((iter = packets)) {
		do {
			next = iter->next;
			delete iter;
			iter = next;
		} while (iter != packets);
	}
	if (termtype != NULL)
		delete[] termtype;
}

void Stream5250::SetTerminalType (const char *str)
{
	termtype = new char[strlen (str)+1];
	strcpy (termtype, str);
}

int Stream5250::Connect(char * address, char * port, char * sessionname,
                        char * transform)
/*
    Connects to the server.
*/
{
   struct servent *pserv_servent;
   struct sockaddr_in serv_addr;
   u_long ioctlarg = 1;
   int r;

   strcpy(devicename, sessionname);
   strcpy(transformname, transform);

   memset ((char *) &serv_addr, 0, sizeof(serv_addr));
   serv_addr.sin_family = AF_INET;

   serv_addr.sin_addr.s_addr = inet_addr (address);
   if (serv_addr.sin_addr.s_addr == INADDR_NONE) {
      struct hostent *pent = gethostbyname (address);
      if(pent != NULL)
	 serv_addr.sin_addr.s_addr = *((u_long*)(pent->h_addr));
   }

   if (serv_addr.sin_addr.s_addr == INADDR_NONE)
      return LAST_ERROR;

   serv_addr.sin_port = htons((u_short)atoi(port));
   if (serv_addr.sin_port == 0) {
      struct servent *pent = getservbyname (port, NULL);
      if (pent != NULL)
	 serv_addr.sin_port = pent->s_port;
   } 

   if (serv_addr.sin_port == 0)
      return LAST_ERROR;

   sockfd = socket (PF_INET, SOCK_STREAM, 0);
   if (WAS_INVAL_SOCK (sockfd)) {
      return LAST_ERROR;
   }

   //printf("Connecting to %s\n", inet_ntoa(serv_addr.sin_addr));

   r = connect (sockfd, (struct sockaddr *) &serv_addr, sizeof (serv_addr));
   if (WAS_ERROR_RET (r)) {
      return LAST_ERROR;
   }      

   // Set socket to non-blocking mode.
   IOCTLSOCKET (sockfd, FIONBIO, &ioctlarg);

   state = STATE_DATA;
   return 0;
}

void Stream5250::Disconnect()
/*
   Disconnects from the server.
*/
{
   CLOSESOCKET (sockfd);
}

int Stream5250::getnext()
/*
   Gets the next byte from the socket.

   Returns:
      -1    No data on socket.
      -2    We've been disconnected.
      >= 0  The next byte on the socket.
*/
{
   unsigned char curchar;
   int rc;
   fd_set fdr;
   struct timeval tv;

   FD_ZERO (&fdr);
   FD_SET (sockfd, &fdr);
   tv.tv_sec = 0;
   tv.tv_usec = 0;
   select (sockfd+1, &fdr, NULL, NULL, &tv);
   if (!FD_ISSET (sockfd, &fdr))
   	return -1;	/* No data on socket. */
   
   rc = recv (sockfd, (char *) &curchar, 1, 0);
   if (WAS_ERROR_RET (rc)) {
   	if (LAST_ERROR != ERR_AGAIN && LAST_ERROR != ERR_INTR) {
		printf("Error reading from socket: %s\n", strerror(LAST_ERROR));
		return -2;
	} else
		return -1;
   }

   /* If the socket was readable, but there is no data, that means that we
      have been disconnected. */
   if (rc == 0)
   	return -2;

   return (int)curchar;
}

void Stream5250::do_verb (unsigned char verb, unsigned char what)
{
   unsigned char reply[3];
   int ret;

   reply[0] = IAC;
   reply[2] = what;
   switch (verb) {
   case DO:
      switch (what) {
      case TERMINAL_TYPE:
      case END_OF_RECORD:
      case TRANSMIT_BINARY:
      case NEW_ENVIRON:
	 reply[1] = WILL;
	 break;

      default:
	 reply[1] = WONT;
	 break;
      }
      break;

   case DONT:
      break;

   case WILL:
      switch (what) {
      case TERMINAL_TYPE:
      case END_OF_RECORD:
      case TRANSMIT_BINARY:
      case NEW_ENVIRON:
	 reply[1] = DO;
	 break;

      case TIMING_MARK:
         LOG (("do_verb: IAC WILL TIMING_MARK received.\n"));
      default:
	 reply[1] = DONT;
	 break;
      }
      break;

   case WONT:
      break;
   }

   /* FIXME: We have to keep track of states here, but... */

   ret = send (sockfd, (char *) reply, 3, 0);
   if (WAS_ERROR_RET (ret)) {
      printf("Error writing to socket: %s\n", strerror(LAST_ERROR));
      exit(5);
   } 
}

void Stream5250::sb (unsigned char *sb_buf, int sb_len)
{
   Buffer out_buff;
   int ret;

   if (sb_len <= 0)
      return;

   if (sb_buf[0] == TERMINAL_TYPE)
   {
      if (sb_buf[1] != SEND)
         return;

      out_buff.length (4);
      out_buff[0] = IAC;
      out_buff[1] = SB;
      out_buff[2] = TERMINAL_TYPE;
      out_buff[3] = IS; 

      out_buff.append ((unsigned char*)termtype, strlen (termtype));
      out_buff.append (IAC);
      out_buff.append (SE);
      ret = send (sockfd, (char *)out_buff.get_data (), out_buff.length(), 0);
      if (WAS_ERROR_RET (ret)) {
         printf("Error writing to socket: %s\n", strerror(LAST_ERROR));
         exit(5);
      }
      status = status | TERMINAL;
   }
   else if(sb_buf[0] == NEW_ENVIRON)
   {
      out_buff.length(28);
      out_buff[0] = IAC;
      out_buff[1] = SB;
      out_buff[2] = NEW_ENVIRON;
      out_buff[3] = IS;
      if(strcmp(devicename,"") != 0) {
         out_buff.append(VAR);
         out_buff.append((unsigned char *)"DEVNAME",7);
         out_buff.append(VALUE);
         out_buff.append((unsigned char *)devicename,sizeof(devicename));
      }

      if(!strcmp(termtype,"IBM-3812-1"))
      {
         out_buff.append(VAR);
         out_buff.append((unsigned char *)"IBMFONT",7);
         out_buff.append(VALUE);
         out_buff.append((unsigned char *)"12",2);

         if(strcmp(transformname,"")) {
            out_buff.append(VAR);
            out_buff.append((unsigned char *)"IBMTRANSFORM",11);
            out_buff.append(VALUE);
            out_buff.append((unsigned char *)"1", 1);

            out_buff.append(VAR);
            out_buff.append((unsigned char *)"IBMMFRTYPMDL",12);
            out_buff.append(VALUE);
            out_buff.append((unsigned char *)transformname, 10);
         }
         else {
            out_buff.append(VAR);
            out_buff.append((unsigned char *)"IBMTRANSFORM",11);
            out_buff.append(VALUE);
            out_buff.append((unsigned char *)"0", 1);
         }         
      }

      out_buff.append(IAC);
      out_buff.append(SE);
      ret = send (sockfd, (char *)out_buff.get_data (), out_buff.length(), 0);
      if (WAS_ERROR_RET (ret)) {
         printf("Error writing to socket: %s\n", strerror(LAST_ERROR));
         exit(5);
      }
   }
   else
   {
      return;
   }
}

void Stream5250::badstate(int location)
/*
   We received something we weren't expecting from the server.
*/
{
   printf("Could not negotiate session. -> %d\n", location);
   CLOSESOCKET (sockfd);
}

int Stream5250::GetByte()
/*
   Gets the next byte from the 5250 data stream.  This method transparently
   parses telnet escape sequences (with the exception of EOR, which it
   passes along).


   Returns:
   	-1 		= No data waiting on socket.
	-2 		= Disconnected.
	-END_OF_RECORD	= Telnet EOR escape encountered.
	>= 0		= Next byte on stream.
*/
{
   int temp;
   unsigned char verb;
   Buffer sb_buf;

   do {
      if (state == STATE_NO_DATA)
	 state = STATE_DATA;

      temp = getnext ();
      if (temp < 0)
         return temp;

      switch (state) {
      case STATE_DATA:
	 if (temp == IAC)
	    state = STATE_HAVE_IAC;
	 break;

      case STATE_HAVE_IAC:
         switch (temp) {
	 case IAC:
	    state = STATE_DATA;
	    break;

	 case DO:
	 case DONT:
	 case WILL:
	 case WONT:
	    verb = temp;
	    state = STATE_HAVE_VERB;
	    break;

	 case SB:
	    state = STATE_HAVE_SB;
	    sb_buf.length (0);
	    break;

	 case EOR:
	    state = STATE_DATA;
	    return -END_OF_RECORD;

	 default:
	    LOG (("GetByte: unknown escape 0x%02x in telnet stream.\n", temp));
	    state = STATE_NO_DATA; /* Hopefully a good recovery. */
	 }
	 break;

      case STATE_HAVE_VERB:
	 do_verb (verb, temp);
	 state = STATE_NO_DATA;
	 break;

      case STATE_HAVE_SB:
	 if (temp == IAC)
	    state = STATE_HAVE_SB_IAC;
	 else
	    sb_buf.append ((unsigned char)temp);
	 break;

      case STATE_HAVE_SB_IAC:
	 switch (temp) {
	 case IAC:
	    sb_buf.append (IAC);
	    break;

	 case SE:
	    sb (sb_buf.get_data (), sb_buf.length ());
	    sb_buf.length (0);
	    state = STATE_NO_DATA;
	    break;

	 default: /* Should never happen -- server error */
	    LOG (("GetByte: huh? Got IAC SB 0x%02X.\n", temp));
	    state = STATE_HAVE_SB;
	    break;
	 }
	 break;

      default:
         LOG (("GetByte: huh? Invalid state %d.\n", state));
	 ASSERT (0);
         break; /* Avoid compiler warning. */
      }
   } while (state != STATE_DATA);

   return (int)temp;
}

void Stream5250::Write(unsigned char * data, int size)
/*
   Writes size bytes of data (pointed to by *data) to the 5250 data stream.
   This is also a temporary method to aid in the conversion process.  
*/
{
	int r;
	int last_error = 0;
	fd_set fdw;

	/* There was an easier way to do this, but I can't remember.  This
	   just makes sure that non blocking writes that don't have enough
	   buffer space get completed anyway. */
	do {
		FD_ZERO (&fdw);
		FD_SET (sockfd, &fdw);
		r = select (sockfd+1, NULL, &fdw, NULL, NULL);
		if (WAS_ERROR_RET (r)) {
			last_error = LAST_ERROR;
			switch (last_error) {
			case ERR_INTR:
			case ERR_AGAIN:
			   r = 0;
			   continue;

			default:
			   perror ("select");
			   exit (5);
			}
		}
	
		if (FD_ISSET (sockfd, &fdw)) {
			r = send (sockfd, (char *)data, size, 0);
			if (WAS_ERROR_RET (r)) {
			        last_error = LAST_ERROR;
				if (last_error != ERR_AGAIN) {
				   perror ("Error writing to socket");
				   exit (5);
				}
			}

			if (r > 0) {
				data += r;
				size -= r;
			}
		}
	} while (size && (r >= 0 || last_error == ERR_AGAIN));
}

void Stream5250::SendPacket(int length, int flowtype, unsigned char flags, 
                            unsigned char opcode, unsigned char * data)
/*
   Any IAC characters are doubled before the data is sent.
*/
{
   unsigned char temp[10];
   Buffer out_buff;


   length = length + 10;
  
   out_buff.length (10);
   /* Fixed length portion of header */
   out_buff[0] = length >> 8;	/* Header length */
   out_buff[1] = length & 0xff;
   out_buff[2] = 0x12;		/* Record type = General data stream (GDS) */
   out_buff[3] = 0xA0;
   out_buff[4] = flowtype >> 8;			/* Reserved */
   out_buff[5] = flowtype & 0xff;			
   /* Variable length portion of header */
   out_buff[6] = 4;			/* Variable portion length = 4 */
   out_buff[7] = flags;		/* Flags */
   out_buff[8] = 0;
   out_buff[9] = opcode;		/* Opcode */
   out_buff.append (data, length-10);

   out_buff = TelnetEscape (out_buff);

   out_buff.append (IAC);
   out_buff.append (EOR);

#ifndef NDEBUG
   LOG (("SendPacket: length = %d\nSendPacket: data follows.",
   	out_buff.length ()));
   for (int n = 0; n < out_buff.length (); n++) {
   	if ((n % 16) == 0) {
		LOG (("\nSendPacket: data: "));
	}
	LOG (("%02X ", out_buff[n]));
   }
   LOG (("\n"));
#endif

   Write (out_buff.get_data (), out_buff.length ());
}
  
void Stream5250::GetPacket(Record * p)
{
	Buffer b;
	int length;
	int loop;
        int offset;

	assert (p != NULL);
	assert (packet_count >= 1);

	b = packets->data;
	assert (b.length() >= 10);

	p->Init ();
	length = (b[0] << 8) | b[1];
	p->length (length);

        p->SetFlowType(b[4], b[5]);
	p->SetFlags(b[7]);
	p->SetOpcode(b[9]);

        offset = 6+b[6];

        p->SetHeaderLength(offset);

	LOG (("GetPacket: length = 0x%X\n", length));

	for(loop = 0; loop < length-offset; loop++)
		(*p)[loop] = b[loop+offset];

	p->Dump(); 

	// Remove packet from queue.
	if (packets->next == packets) {
		delete packets;
		packets = NULL;
		packet_count = 0;
	} else {
		Packet *old_packet = packets;
		packets = packets->next;
		old_packet->prev->next = old_packet->next;
		old_packet->next->prev = old_packet->prev;
		delete old_packet;
		packet_count --;
	}
}

void Stream5250::QueryReply()
{
   unsigned char temp[61];

   temp[0]   = 0x00;	// Cursor Row/Column (set to zero) 
   temp[1]   = 0x00;
   temp[2]   = 0x88;	// Inbound Write Structured Field Aid 
   temp[3]   = 0x00;	// Length of Query Reply 
   temp[4]   = 0x3A;	
   temp[5]   = 0xD9;	// Command class 
   temp[6]   = 0x70;	// Command type - Query 
   temp[7]   = 0x80;	// Flag byte 
   temp[8]   = 0x06;	// Controller hardware class 
   temp[9]   = 0x00;	
   temp[10]  = 0x01;	// Controller code level (Version 1 Release 1.0 
   temp[11]  = 0x01;    
   temp[12]  = 0x00;
   temp[13]  = 0x00;    // Reserved (set to zero) 
   temp[14]  = 0x00;
   temp[15]  = 0x00;
   temp[16]  = 0x00;
   temp[17]  = 0x00;
   temp[18]  = 0x00;
   temp[19]  = 0x00;
   temp[20]  = 0x00;
   temp[21]  = 0x00;
   temp[22]  = 0x00;
   temp[23]  = 0x00;
   temp[24]  = 0x00;
   temp[25]  = 0x00;
   temp[26]  = 0x00;
   temp[27]  = 0x00;
   temp[28]  = 0x00;
   temp[29]  = 0x01;	// 5250 Display or 5250 emulation 
   temp[30]  = ascii2ebcdic('5');
   temp[31]  = ascii2ebcdic('2');
   temp[32]  = ascii2ebcdic('5');
   temp[33]  = ascii2ebcdic('1');
   temp[34]  = ascii2ebcdic('0');
   temp[35]  = ascii2ebcdic('1');
   temp[36]  = ascii2ebcdic('1');
   temp[37]  = 0x02;	// Keyboard ID (standard keyboard) 
   temp[38]  = 0x00;	// Extended keyboard ID 
   temp[39]  = 0x00;	// Reserved 
   temp[40]  = 0x00;	// Display serial number 
   temp[41]  = 0x61;	
   temp[42]  = 0x50;
   temp[43]  = 0x00;
   temp[44]  = 0x01;	// Maximum number of input fields (256) 
   temp[45]  = 0x00;
   temp[46]  = 0x00;	// Reserved (set to zero) 
   temp[47]  = 0x00;
   temp[48]  = 0x00;
   temp[49]  = 0x01;	// Controller/Display Capability 
   temp[50]  = 0x10;
   temp[51]  = 0x00;	// Reserved 
   temp[52]  = 0x00;
   temp[53]  = 0x00;
   temp[54]  = 0x00;	// Reserved (set to zero) 
   temp[55]  = 0x00;
   temp[56]  = 0x00;
   temp[57]  = 0x00;
   temp[58]  = 0x00;
   temp[59]  = 0x00;
   temp[60]  = 0x00;

   SendPacket(61, Record::DISPLAY, Record::H_NONE, Record::NO_OP,
   	(unsigned char *)&temp[0]);

}

void Stream5250::PrintComplete()
{

   SendPacket(0, Record::CLIENTO, Record::H_NONE, Record::PRINT_COMPLETE,
     NULL );
}

SOCKET_TYPE Stream5250::GetSocketHandle()
{
   return(sockfd);
}

bool Stream5250::HandleReceive ()
/*
	This is called when there is data waiting on the socket
	(from MainLoop ()).
*/
{
	int c;

	/* -1 = no more data, -2 = we've been disconnected */
	while ((c = GetByte ()) != -1 && c != -2) {

		if (c == -END_OF_RECORD && current_packet != NULL) {
			/* End of current packet. */
			current_packet = NULL;
			packet_count ++;
			continue;
		}

		if (current_packet == NULL) {
			/* Start of new packet. */
			current_packet = new Packet;
			if (packets == NULL) {
				packets = current_packet;
				current_packet->next = current_packet;
				current_packet->prev = current_packet;
			} else {
				current_packet->next = packets;
				current_packet->prev = packets->prev;
				current_packet->next->prev = current_packet;
				current_packet->prev->next = current_packet;
			}
		}

		current_packet->data.append ((unsigned char)c);
	}

	return (c != -2);
}

int Stream5250::GetPacketCount ()
{
	return packet_count;
}

Buffer Stream5250::TelnetEscape (Buffer in)
{
	Buffer out;
	int n;

	for (n = 0; n < in.length (); n++) {
		out.append (in[n]);
		if (in[n] == IAC)
			out.append (IAC);
	}
	return out;
}
