/*
 * Copyright (C) 1997 Michael Madore
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include <stdlib.h>
#include <getopt.h> 
#include <errno.h>

#include "buffer.h"
#include "stream5250.h"
#include "utility.h"
#include "displaybuf.h"
#include "formattable.h"
#include "codes5250.h"
#include "terminal.h"
#include "cursesterm.h"
#include "session.h"
#include "printsession.h"

extern char *version_string;

char* remotehost;
char* transmapname = "en";
char* sessionname = "";
char port[50] = "23";
bool underscores = false;
bool printsession = false;
int hosttransform = 0;
char* transformname = "";
char* outputcommand = "";

PrintSession *printsess;
Session *sess;
Stream5250 *strm;
Terminal *term;

static void syntax (void);
static int parse_options (int argc, char *argv[]);

int main(int argc, char * argv[])
{
   int cn_result;


   if (parse_options (argc, argv)<0) {
      syntax ();
      exit (1);
   }

   settransmap(transmapname);


   if(printsession) {
      printsess = new PrintSession;
   }
   else {
      term = new CursesTerminal;
      term->init ();
      ((CursesTerminal*)term)->use_underscores (underscores);
      sess = new Session; 
      sess->terminal (term);
   }

   strm = new Stream5250;
   cn_result = strm->Connect(remotehost, port, sessionname, transformname);
   if (cn_result != 0)
      goto bomb_out;
   
   if(printsession) {

      printsess->connection_fd (strm->GetSocketHandle ());
      printsess->stream (strm);
      printf("-%s-\n", outputcommand);
      printsess->OutputCommand (outputcommand);
      printsess->MainLoop ();
   }
   else {
      term->connection_fd (strm->GetSocketHandle ());
      sess->stream (strm);
      sess->MainLoop ();
   }

bomb_out:
   strm->Disconnect ();
   if(!printsession) {
      term->term();
      delete sess;
   }
   if(cn_result != 0)
      printf("Could not start session: %s\n", strerror(cn_result));
#ifndef NDEBUG
   log_close ();
#endif
   exit (0);
}

static int parse_options (int argc, char *argv[])
{
   int arg;
   while ((arg = getopt(argc, argv, "m:s:t:T:P:uVp")) != EOF) {
      switch (arg) {
      case 'm':
	 transmapname = optarg;
	 break;

      case 'P':
         outputcommand = optarg;
         break;

      case 's':
	 sessionname = optarg;
	 break;

      case 'u':
	 underscores = true;
	 break;

#ifndef NDEBUG
      case 't':
	 log_open (optarg);
	 break;
#endif
      case 'T':
         hosttransform = true;
         transformname = optarg;
         break;
      case 'V':  
         printf("tn5250 version %s\n\n", version_string);
	 exit (0);
	 break;

      case 'p':
         printsession = true;
         break;

      case '?':
      case ':':
	 return -1;
      }
   }

   if (optind >= argc) {
      return -1;
   }

   remotehost = argv[optind++];

   if (optind < argc)
      strcpy(port, argv[optind]);

   return 0;
}

static void syntax ()
{
   printf("Usage:  tn5250 [options] <host> [port]\n"
      "Options:\n"
      "\t-m map      specify translation map\n"
      "\t-s name     specify session name\n"
#ifndef NDEBUG
      "\t-t file     specify trace file\n"
#endif
      "\t-u          use underscores instead of underline attribute\n"
      "\t-V          display version\n");
}
