/****************************************************************************/
/*                                                                          */
/*                         TCPA Key Handling Routines                       */
/*                                                                          */
/* This file is copyright 2003 IBM. See "License" for details               */
/****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netinet/in.h>
#include <tcpa.h>
#include <buildbuff.h>
#include <keys.h>
#include <oiaposap.h>
#include <hmac.h>
#include <openssl/rsa.h>
#include <openssl/bn.h>
#include <openssl/rand.h>

/****************************************************************************/
/*                                                                          */
/* Read the TCPA Endorsement public key                                     */
/*                                                                          */
/****************************************************************************/
uint32_t TPM_ReadPubek(pubkeydata * k, FILE * log)
{
    unsigned char read_pubek_fmt[] = "00 c1 T 00 00 00 7c %";
    unsigned char tcpadata[TCPA_MAX_BUFF_SIZE];
    unsigned char nonce[TCPA_HASH_SIZE];
    uint32_t ret;

    /* check input argument */
    if (k == NULL)
        return -1;
    /* generate random nonce */
    ret = RAND_bytes(nonce, TCPA_NONCE_SIZE);
    if (ret != 1)
        return 1;
    /* copy Read PubKey request template to buffer */
    ret = buildbuff(read_pubek_fmt, tcpadata, TCPA_HASH_SIZE, nonce);
    if (ret <= 0)
        return 1;
    ret = TPM_Transmit(tcpadata, log, "TPM_ReadPubek");
    if (ret)
        return ret;
    PubKeyExtract(tcpadata + TCPA_DATA_OFFSET, k, 0);
    return 0;
}

/****************************************************************************/
/*                                                                          */
/* Create and Wrap a Key                                                    */
/*                                                                          */
/* The arguments are...                                                     */
/*                                                                          */
/* keyhandle is the TCPA_KEY_HANDLE of the parent key of the new key        */
/*           0x40000000 for the SRK                                         */
/* keyauth   is the authorization data (password) for the parent key        */
/* newauth   is the authorization data (password) for the new key           */
/* migauth   is the authorization data (password) for migration of the new  */
/*           key, or NULL if the new key is not migratable                  */
/*           all authorization values must be 20 bytes long                 */
/* keyparms  is a pointer to a keydata structure with parms set for the new */
/*           key                                                            */
/* key       is a pointer to a keydata structure returned filled in         */
/*           with the public key data for the new key                       */
/*                                                                          */
/****************************************************************************/
uint32_t TPM_CreateWrapKey(uint32_t keyhandle,
                           unsigned char *keyauth,
                           unsigned char *newauth,
                           unsigned char *migauth,
                           keydata *keyparms, keydata *key, FILE *log)
{
    unsigned char create_key_fmt[] = "00 c2 T l l % % % l % o %";
    unsigned char tcpadata[TCPA_MAX_BUFF_SIZE];
    unsigned char kparmbuf[TCPA_MAX_BUFF_SIZE];
    unsigned char encauth1[TCPA_HASH_SIZE];
    unsigned char encauth2[TCPA_HASH_SIZE];
    unsigned char xorwork[TCPA_HASH_SIZE * 2];
    unsigned char xorhash[TCPA_HASH_SIZE];
    unsigned char nonceodd[TCPA_NONCE_SIZE];
    unsigned char pubauth[TCPA_HASH_SIZE];
    osapsess sess;
    uint32_t ret;
    int i;
    unsigned char c;
    uint32_t ordinal;
    uint32_t keyhndl;
    uint16_t keytype;
    int kparmbufsize;

    /* check input arguments */
    if (keyauth == NULL || newauth == NULL || keyparms == NULL
        || key == NULL)
        return -1;
    if (keyhandle == 0x40000000)        /* SRK */
        keytype = 0x0004;
    else
        keytype = 0x0001;
    /* generate odd nonce */
    RAND_bytes(nonceodd, TCPA_NONCE_SIZE);
    /* Open OSAP Session */
    ret = TPM_OSAP(&sess, keyauth, keytype, keyhandle, log);
    if (ret)
        return ret;
    /* calculate encrypted authorization value for new key */
    memcpy(xorwork, sess.ssecret, TCPA_HASH_SIZE);
    memcpy(xorwork + TCPA_HASH_SIZE, sess.enonce, TCPA_HASH_SIZE);
    sha1(xorwork, TCPA_HASH_SIZE * 2, xorhash);
    for (i = 0; i < TCPA_HASH_SIZE; ++i)
        encauth1[i] = xorhash[i] ^ newauth[i];
    /* calculate encrypted authorization value for migration of new key */
    if (migauth != NULL) {
        memcpy(xorwork, sess.ssecret, TCPA_HASH_SIZE);
        memcpy(xorwork + TCPA_HASH_SIZE, nonceodd, TCPA_HASH_SIZE);
        sha1(xorwork, TCPA_HASH_SIZE * 2, xorhash);
        for (i = 0; i < TCPA_HASH_SIZE; ++i)
            encauth2[i] = xorhash[i] ^ migauth[i];
    } else
        memset(encauth2, 0, TCPA_HASH_SIZE);
    /* move Network byte order data to variables for hmac calculation */
    ordinal = htonl(0x1F);
    keyhndl = htonl(keyhandle);
    c = 0;
    /* convert keyparm structure to buffer */
    kparmbufsize = BuildKey(kparmbuf, keyparms);
    if (kparmbufsize < 0) {
        TPM_Terminate_Handle(sess.handle, log);
        return 1;
    }
    /* calculate authorization HMAC value */
    ret = authhmac(pubauth, sess.ssecret, TCPA_HASH_SIZE, sess.enonce,
                   nonceodd, c, 4, &ordinal, TCPA_HASH_SIZE,
                   encauth1, TCPA_HASH_SIZE, encauth2, kparmbufsize,
                   kparmbuf, 0, 0);
    if (ret < 0) {
        TPM_Terminate_Handle(sess.handle, log);
        return 1;
    }
    /* build the request buffer */
    ret = buildbuff(create_key_fmt, tcpadata,
                    ordinal,
                    keyhndl,
                    TCPA_HASH_SIZE, encauth1,
                    TCPA_HASH_SIZE, encauth2,
                    kparmbufsize, kparmbuf,
                    sess.handle,
                    TCPA_NONCE_SIZE, nonceodd, c, TCPA_HASH_SIZE, pubauth);
    if (ret <= 0) {
        TPM_Terminate_Handle(sess.handle, log);
        return -1;
    }
    /* transmit the request buffer to the TPM device and read the reply */
    ret = TPM_Transmit(tcpadata, log, "CreateWrapKey");
    if (ret != 0) {
        TPM_Terminate_Handle(sess.handle, log);
        return ret;
    }
    kparmbufsize = KeySize(tcpadata + TCPA_DATA_OFFSET);
    ret = checkhmac1(tcpadata, ordinal, nonceodd, sess.ssecret,
                     TCPA_HASH_SIZE, kparmbufsize, TCPA_DATA_OFFSET, 0, 0);
    if (ret != 0) {
        TPM_Terminate_Handle(sess.handle, log);
        return -1;
    }
    TPM_Terminate_Handle(sess.handle, log);
    /* convert the returned key to a structure */
    KeyExtract(tcpadata + TCPA_DATA_OFFSET, key);
    return 0;
}

/****************************************************************************/
/*                                                                          */
/* Load a new Key into the TPM                                              */
/*                                                                          */
/* The arguments are...                                                     */
/*                                                                          */
/* keyhandle is the TCPA_KEY_HANDLE of parent key for the new key           */
/*           0x40000000 for the SRK                                         */
/* keyauth   is the authorization data (password) for the parent key        */
/* keyparms  is a pointer to a keydata structure with all data  for the new */
/*           key                                                            */
/* newhandle is a pointer to a 32bit word which will receive the handle     */
/*           of the new key                                                 */
/*                                                                          */
/****************************************************************************/
uint32_t TPM_LoadKey(uint32_t keyhandle, unsigned char *keyauth,
                     keydata *keyparms, uint32_t *newhandle, FILE *log)
{
    unsigned char load_key_fmt[] = "00 c2 T l l % l % o %";
    unsigned char tcpadata[TCPA_MAX_BUFF_SIZE];
    unsigned char kparmbuf[TCPA_MAX_BUFF_SIZE];
    unsigned char nonceodd[TCPA_NONCE_SIZE];
    unsigned char evennonce[TCPA_NONCE_SIZE];
    unsigned char pubauth[TCPA_HASH_SIZE];
    uint32_t ret;
    unsigned char c;
    uint32_t ordinal;
    uint32_t keyhndl;
    uint32_t authhandle;
    int kparmbufsize;

    /* check input arguments */
    if (keyauth == NULL || keyparms == NULL || newhandle == NULL)
        return 1;
    /* generate odd nonce */
    RAND_bytes(nonceodd, TCPA_NONCE_SIZE);
    /* Open OIAP Session */
    ret = TPM_OIAP(&authhandle, evennonce, log);
    if (ret != 0)
        return ret;
    /* move Network byte order data to variables for hmac calculation */
    ordinal = htonl(0x20);
    keyhndl = htonl(keyhandle);
    c = 0;
    /* convert keyparm structure to buffer */
    kparmbufsize = BuildKey(kparmbuf, keyparms);
    if (kparmbufsize < 0) {
        TPM_Terminate_Handle(authhandle, log);
        return 1;
    }
    /* calculate authorization HMAC value */
    ret = authhmac(pubauth, keyauth, TCPA_HASH_SIZE, evennonce, nonceodd,
                   c, 4, &ordinal, kparmbufsize, kparmbuf, 0, 0);
    if (ret < 0) {
        TPM_Terminate_Handle(authhandle, log);
        return 1;
    }
    /* build the request buffer */
    ret = buildbuff(load_key_fmt, tcpadata,
                    ordinal,
                    keyhndl,
                    kparmbufsize, kparmbuf,
                    htonl(authhandle),
                    TCPA_NONCE_SIZE, nonceodd, c, TCPA_HASH_SIZE, pubauth);
    if (ret <= 0) {
        TPM_Terminate_Handle(authhandle, log);
        return 1;
    }
    /* transmit the request buffer to the TPM device and read the reply */
    ret = TPM_Transmit(tcpadata, log, "LoadKey");
    if (ret != 0) {
        TPM_Terminate_Handle(authhandle, log);
        return ret;
    }
    TPM_Terminate_Handle(authhandle, log);
    ret = checkhmac1(tcpadata, ordinal, nonceodd, keyauth,
                     TCPA_HASH_SIZE, 4, TCPA_DATA_OFFSET, 0, 0);
    if (ret != 0)
        return 1;
    *newhandle = ntohl(*(uint32_t *) (tcpadata + TCPA_DATA_OFFSET));
    return 0;
}

/****************************************************************************/
/*                                                                          */
/* Evict (delete) a  Key from the TPM                                       */
/*                                                                          */
/* The arguments are...                                                     */
/*                                                                          */
/* keyhandle is the TCPA_KEY_HANDLE of the key to be evicted                */
/*                                                                          */
/****************************************************************************/
uint32_t TPM_EvictKey(uint32_t keyhandle, FILE * log)
{
    unsigned char evict_key_fmt[] = "00 c1 T 00 00 00 22 L";
    unsigned char tcpadata[TCPA_MAX_BUFF_SIZE];
    uint32_t ret;

    ret = buildbuff(evict_key_fmt, tcpadata, keyhandle);
    ret = TPM_Transmit(tcpadata, log, "TPM_EvictKey");
    return ret;
}

/****************************************************************************/
/*                                                                          */
/* Create a TCPA_KEY buffer from a keydata structure                        */
/*                                                                          */
/****************************************************************************/
int BuildKey(unsigned char *buffer, keydata * k)
{
    unsigned char build_key_fmt[] = "% S L o L S S L L L L @ @ @";
    int ret;

    ret = buildbuff(build_key_fmt, buffer,
                    4, k->version,
                    k->keyusage,
                    k->keyflags,
                    k->authdatausage,
                    k->pub.algorithm,
                    k->pub.encscheme,
                    k->pub.sigscheme,
                    12,
                    k->pub.keybitlen,
                    k->pub.numprimes,
                    0,
                    k->pub.pcrinfolen, k->pub.pcrinfo,
                    k->pub.keylength, k->pub.modulus,
                    k->privkeylen, k->encprivkey);
    return ret;
}


/****************************************************************************/
/*                                                                          */
/* Walk down the TCPA_Key Structure extracting information                  */
/*                                                                          */
/****************************************************************************/
int KeyExtract(unsigned char *keybuff, keydata * k)
{
    int offset;
    int pubkeylen;

    /* fill in  keydata structure */
    offset = 0;
    memcpy(k->version, keybuff + offset, sizeof(k->version));
    offset += 4;
    k->keyusage = ntohs(*(uint16_t *) (keybuff + offset));
    offset += 2;
    k->keyflags = ntohl(*(uint32_t *) (keybuff + offset));
    offset += 4;
    k->authdatausage = keybuff[offset];
    offset += 1;
    pubkeylen = PubKeyExtract(keybuff + offset, &(k->pub), 1);
    offset += pubkeylen;
    k->privkeylen = ntohl(*(uint32_t *) (keybuff + offset));
    offset += 4;
    if (k->privkeylen > 0 && k->privkeylen <= 1024)
        memcpy(k->encprivkey, keybuff + offset, k->privkeylen);
    offset += k->privkeylen;
    return offset;
}

/****************************************************************************/
/*                                                                          */
/* Walk down the TCPA_PUBKey Structure extracting information               */
/*                                                                          */
/****************************************************************************/
int PubKeyExtract(unsigned char *keybuff, pubkeydata * k, int pcrpresent)
{
    uint32_t parmsize;
    uint32_t pcrisize;
    int offset;

    offset = 0;
    k->algorithm = ntohl(*(uint32_t *) (keybuff + offset));
    offset += 4;
    k->encscheme = ntohs(*(uint16_t *) (keybuff + offset));
    offset += 2;
    k->sigscheme = ntohs(*(uint16_t *) (keybuff + offset));
    offset += 2;
    parmsize = ntohl(*(uint32_t *) (keybuff + offset));
    offset += 4;
    if (k->algorithm == 0x00000001 && parmsize > 0) {   /* RSA */
        k->keybitlen = ntohl(*(uint32_t *) (keybuff + offset));
        offset += 4;
        k->numprimes = ntohl(*(uint32_t *) (keybuff + offset));
        offset += 4;
        k->expsize = ntohl(*(uint32_t *) (keybuff + offset));
        offset += 4;
    } else {
        offset += parmsize;
    }
    if (k->expsize != 0)
        offset += k->expsize;
    else {
        k->exponent[0] = 0x01;
        k->exponent[1] = 0x00;
        k->exponent[2] = 0x01;
        k->expsize = 3;
    }
    if (pcrpresent) {
        pcrisize = ntohl(*(uint32_t *) (keybuff + offset));
        offset += 4;
        if (pcrisize > 0 && pcrisize <= 256)
            memcpy(k->pcrinfo, keybuff + offset, pcrisize);
        offset += pcrisize;
        k->pcrinfolen = pcrisize;
    }
    k->keylength = ntohl(*(uint32_t *) (keybuff + offset));
    offset += 4;
    if (k->keylength > 0 && k->keylength <= 256)
        memcpy(k->modulus, keybuff + offset, k->keylength);
    offset += k->keylength;
    return offset;
}

/****************************************************************************/
/*                                                                          */
/* Get the size of a TCPA_KEY                                               */
/*                                                                          */
/****************************************************************************/
int KeySize(unsigned char *keybuff)
{
    int offset;
    int privkeylen;

    offset = 0 + 4 + 2 + 4 + 1;
    offset += PubKeySize(keybuff + offset, 1);
    privkeylen = ntohl(*(uint32_t *) (keybuff + offset));
    offset += 4 + privkeylen;
    return offset;
}

/****************************************************************************/
/*                                                                          */
/* Get the size of a TCPA_PUBKEY                                            */
/*                                                                          */
/****************************************************************************/
int PubKeySize(unsigned char *keybuff, int pcrpresent)
{
    uint32_t parmsize;
    uint32_t pcrisize;
    uint32_t keylength;

    int offset;

    offset = 0 + 4 + 2 + 2;
    parmsize = ntohl(*(uint32_t *) (keybuff + offset));
    offset += 4;
    offset += parmsize;
    if (pcrpresent) {
        pcrisize = ntohl(*(uint32_t *) (keybuff + offset));
        offset += 4;
        offset += pcrisize;
    }
    keylength = ntohl(*(uint32_t *) (keybuff + offset));
    offset += 4;
    offset += keylength;
    return offset;
}

/****************************************************************************/
/*                                                                          */
/* Convert a TCPA public key to an OpenSSL RSA public key                   */
/*                                                                          */
/****************************************************************************/
RSA *convpubkey(pubkeydata * k)
{
    RSA *rsa;
    BIGNUM *mod;
    BIGNUM *exp;

    /* create the necessary structures */
    rsa = RSA_new();
    mod = BN_new();
    exp = BN_new();
    if (rsa == NULL || mod == NULL || exp == NULL)
        return NULL;
    /* convert the raw public key values to BIGNUMS */
    BN_bin2bn(k->modulus, k->keylength, mod);
    BN_bin2bn(k->exponent, k->expsize, exp);
    /* set up the RSA public key structure */
    rsa->n = mod;
    rsa->e = exp;
    return rsa;
}
