/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net
Copyright (C) 2003, Dave Hodges - hodgesda@rose-hulman.edu

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  Checks to see if all UBS processes are alive and well, and sends out email notification in case of failure.
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "ubs.h"

// global variable table
ubs_table MONITOR;

int mailalert(int errnum, char *errdaemon)
{
 FILE *mail;
 char buf[FILEBUF];

 snprintf(buf, FILEBUF, "%s %s %s", ubs_table_data(&MONITOR, "mailprog"), "-s 'UBS Error Message'", ubs_table_data(&MONITOR, "email"));
 mail = popen(buf, "w");
 if(!mail) {
  log_error_msg(LOG_ERROR, "Unable to open pipe to mail program '%s'", ubs_table_data(&MONITOR, "mailprog"));
  return FAIL;
 }

 fprintf(mail, " UBS Error #%d on the %s Daemon.\n Please refer to your system for more information.\n", errnum, errdaemon);

 if(pclose(mail)) {
  fprintf(stderr, "Mail failed!\n");
  exit(1);
 }
 return (0);
}

// maybe later, eh Dave? =)
int pagealert()
{
 /*
 printf("./page -p 8125310004");
 */
 return (0);
}

int main(int argc, char *argv[])
{
 int status, sleeptime;
 DIR *dp;
 struct dirent *ds;

 ubs_table_init(&GLOBAL);
 ubs_table_init(&MONITOR);
 ubs_init(argv[0]);

 if(read_config(DEF_CONFIG, "global", &GLOBAL)) {
  console_error("Can't read global settings in configuration file", FAIL);
 }
 if(read_config(DEF_CONFIG, "monitor", &MONITOR)) {
  console_error("Can't read monitor settings in configuration file", FAIL);
 }

 // see if we have all the variables we need
 if(!ubs_table_exists(&MONITOR, "mailprog")) {
  console_error("No mail program specified", FAIL);
 }
 if(!ubs_table_exists(&MONITOR, "email")) {
  console_error("No emergency email address specified", FAIL);
 }

 // default sleep time -> 15 seconds
 if(!ubs_table_exists(&MONITOR, "sleeptime")) {
  sleeptime = 15;
 }
 else {
  sleeptime = atoi(ubs_table_data(&MONITOR, "sleeptime"));
  // some value that is borked
  if(!sleeptime) {
   sleeptime = 15;
  }
 }

 if(check_running() == FAIL) {
  console_error("Daemon already seems to be running", FAIL);
 }

#ifndef WIN32
 log_error_msg(LOG_EMERG, "Process started as uid/gid %d/%d", geteuid(), getgid());
#else
 log_error_msg(LOG_EMERG, "Process started");
#endif

#ifndef WIN32
 // fork into the background
 if(fork()) {
  // parent exits
  exit(0);
 }
#endif

 log_error_msg(LOG_DEBUG, "Becoming the process leader");
 setsid();
 log_error_msg(LOG_DEBUG, "Changing root directory to PREFIX");
 chdir(ubs_table_data(&GLOBAL, "prefix"));
 log_error_msg(LOG_DEBUG, "Setting umask to 000");
 umask(0);

 record_pid(getpid());
 record_status("OK");

 while(1) {
  // sleep at the top of the loop to avoid potential false alarms when
  // starting the ubs up.
  sleep(sleeptime);

  if((dp = opendir("bin")) == NULL) {
   console_error("Unable to open bin/ directory", NO_DIR);
  }

  while((ds = readdir(dp)) != NULL) {
   if(!strncmp(ds->d_name, "ubs-", 4)) {
    status = ping_pid(ds->d_name);
    log_error_msg(LOG_DEBUG, "%s status: %d", ds->d_name, status);
    if(status != OK) {
     mailalert(status, ds->d_name);
    }
   }
  }
 }

 return OK;
}
