/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  Deals with song/artist info inside of a music file.  In the case of mp3, this means ID3 version 1 tags.  For ogg vorbis, it means just grabbing this information out of the comments.
****/

#ifndef WIN32
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#ifdef HAVE_LIBVORBIS
#include <vorbis/vorbisfile.h>
#include <vorbis/vorbisenc.h>
#include <vorbis/codec.h>
#endif

#include "ubs.h"

/*+
  Determines the type of file (mp3 or ogg are supported at the moment), and calls the appropriate function to grab the song information

  int get_media_tag Returns value of get_mp3_tag()/get_ogg_tag() on success, FAIL if an unknown extension is given, FAIL if libvorbis isn't around and an ogg was given anyways

  char *sname The filename of the music file

  struct media_tag *song_tag The structure that will be filled in with the relevant information from the song's ID3 or vorbis tag
+*/
int get_media_tag(char *sname, struct media_tag *song_tag) {
 char ext[4];

 memset(ext, 0x0, 4);
 strncpy(ext, sname + strlen(sname) - 3, 3);
 if(!strcasecmp(ext, "ogg")) {
#ifdef HAVE_LIBVORBIS
  return get_ogg_tag(sname, song_tag);
#else
  log_error_msg(LOG_DEBUG, "Ogg Vorbis file found, but no libraries");
  return FAIL;
#endif
 }
 else if(!strcasecmp(ext, "mp3")) {
  return get_mp3_tag(sname, song_tag);
 }
 else {
  log_error_msg(LOG_DEBUG, "Unknown file type given for extension '%s' in file '%s'", ext, sname);
  return FAIL;
 }

 return OK;
}

/*+
  Gets the song information from an ogg vorbis tag.  This is done via the ogg vorbis libraries.

  int get_ogg_tag Returns OK on success, NO_FILE if the file can't be opened

  char *sname The song filename to scan

  struct media_tag *song_tag The structure to write data to
+*/
int get_ogg_tag(char *sname, struct media_tag *song_tag)
{
#ifdef HAVE_LIBVORBIS
 FILE *fp;
 char *p, key[STRBUF], value[STRBUF];
 int i;
 OggVorbis_File vf;
 struct vorbis_comment *tag;

 if((fp = fopen(sname, "r")) == NULL) {
  log_error_msg(LOG_ERROR, "Can't open file '%s' to read ogg tag", sname);
  return NO_FILE;
 }

 if(ov_open(fp, &vf, NULL, 0) < 0) {
  log_error_msg(LOG_ERROR, "Can't create ogg filehandle");
  fclose(fp);
  return NO_FILE;
 }

 memset(key, 0x0, STRBUF);
 memset(value, 0x0, STRBUF);
 tag = ov_comment(&vf, -1);
 strncpy(song_tag->artist, "Unknown", STRBUF);
 strncpy(song_tag->title, "Unknown", STRBUF);
 strncpy(song_tag->album, "Unknown", STRBUF);

 for(i = 0; i < tag->comments; i++) {
  if((p = strchr(tag->user_comments[i], '=')) != NULL) {
   strncpy(key, tag->user_comments[i], p - tag->user_comments[i]);
   strncpy(value, p + 1, STRBUF);
  }

  if(!strncasecmp(key, "artist", strlen("artist"))) {
   memset(song_tag->artist, 0x0, STRBUF);
   strncpy(song_tag->artist, value, STRBUF);
  }
  else if(!strncasecmp(key, "title", strlen("title"))) {
   memset(song_tag->title, 0x0, STRBUF);
   strncpy(song_tag->title, value, STRBUF);
  }
  else if(!strncasecmp(key, "album", strlen("album"))) {
   memset(song_tag->album, 0x0, STRBUF);
   strncpy(song_tag->album, value, STRBUF);
  }
  else {
   // ignore
  }
 }
 ov_clear(&vf);

#endif
 return OK;
}

/*+
  Gets information from an ID3 version 1 tag, which is the "old" mp3 tagging format.  This is done by just opening up the file and reading the last 128 bytes.

  int get_mp3_tag Returns OK on success, NO_FILE if the file can't be opened

  char *sname The song filename to scan

  struct media_tag *song_tag The structure to fill
+*/
int get_mp3_tag(char *sname, struct media_tag *song_tag) {
 FILE *fp;
 char tag[129];

 memset(tag, 0x0, 129);
 memset(song_tag->title, 0x0, STRBUF);
 memset(song_tag->artist, 0x0, STRBUF);
 memset(song_tag->album, 0x0, STRBUF);

 if((fp = fopen(sname, "r")) == NULL) {
  log_error_msg(LOG_ERROR, "Can't open file '%s' to read mp3 tag", sname);
  return NO_FILE;
 }

 fseek(fp, -128, SEEK_END);
 fgets(tag, 128, fp);
 fclose(fp);

 if(!strncmp(tag, "TAG", 3)) {
  strncpy(song_tag->title, tag + 3, 30);
  strncpy(song_tag->artist, tag + 33, 30);
  strncpy(song_tag->album, tag + 63, 30);
 }
 else {
  log_error_msg(LOG_ERROR, "mp3 file '%s' has no id3 tag", sname);

  strncpy(song_tag->artist, "Unknown", STRBUF);
  strncpy(song_tag->title, "Unknown", STRBUF);
  strncpy(song_tag->album, "Unknown", STRBUF);
 }
 return OK;
}

/*+
  Sees if any part of any tag in file2 is present in file1

  int check_against_tags Returns YES if a match is found, FAIL if either get_media_tag() call is bad, or NO if no match is found

  char *file1 File to check against

  char *file2 File to check
+*/
int check_against_tags(char *file1, char *file2) {
 struct media_tag fs1, fs2;

 if(get_media_tag(file1, &fs1) || get_media_tag(file2, &fs2)) {
  return FAIL;
 }

 if(!strcasecmp(fs1.title, fs2.title)) {
  log_error_msg(LOG_DEBUG, "%s vs. %s: Title match found", file1, file2);
  return YES;
 }
 else {
  log_error_msg(LOG_DEBUG, "%s vs. %s: No title match clear", file1, file2);
 }

 if(ubs_strcasestr(fs1.artist, fs2.artist)) {
  log_error_msg(LOG_DEBUG, "%s vs. %s: Artist match found", file1, file2);
  return YES;
 }
 else {
  log_error_msg(LOG_DEBUG, "%s vs. %s: No artist match found", file1, file2);
 }

 return NO;
}
