/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  This file is responsible for monitoring the queue and adding new music files to the end of the queue.  Also handles parsing various types of playlists.
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>

#include "ubs-sched.h"

/*+
  This function monitors the queue and adds new files to it when needed.  Basically, whenever the number of files in the queue is less than three, it will find new files from the current show and add them to the queue until the queue is at three files.

  int enqueue Returns OK on success, FAIL if the call to get_song() fails

  int cur_show The index of the current show in the ALLSHOWS array
+*/
int enqueue(int cur_show) {
 int i, j, num_lines = 0;
 char sbuf[FILEBUF];

 // if there are less than 3 songs in the queue, then queue up some more.
 // This algorithm will repeat AT MOST 30 times, which should be a gods
 // plenty.
 num_lines = queue_length();
 if(num_lines < 3) {
  for(i = num_lines; i < 3; i++) {
   for(j = 0; j < 30; j++) {
    if(get_song(sbuf) != OK || sbuf[0] == '\0') {
     log_error_msg(LOG_DEBUG, "get_song() failed, sbuf: %s", sbuf);
     return FAIL;
    }

    log_error_msg(LOG_DEBUG, "Checking song '%s' against songlog", sbuf);
    if(check_against_log(sbuf, ubs_table_data(&GLOBAL, "songlog"), atoi(ubs_table_data(&SCHED, "searchback"))) == NO) {
     log_error_msg(LOG_DEBUG, "Passed. Checking against queue");
     if(check_against_queue(sbuf) == NO) {
      log_error_msg(LOG_DEBUG, "Passed. Checking against current song");
      if(check_against_cur(sbuf) == NO) {
       log_error_msg(LOG_DEBUG, "Passed. Queueing song '%s'", sbuf);
       queue_push(sbuf);
       return OK;
      }
     }
    }
   }
   log_error_msg(LOG_DEBUG, "All checks failed; queueing '%s' anyways", sbuf);
   queue_push(sbuf);
  }
 }

 return OK;
}

/*+
  Parses a m3u playlist to the filecache, stripping all the junk that makes it unusable with the internal UBS format

  int parse_m3u_playlist Returns num_files on success, NO_FILE if the playlist can't be read in

  char *playlist Filename to read in
+*/
int parse_m3u_playlist(char *playlist)
{
 char *cp, buf[FILEBUF], full_path[FILEBUF], *workdir;
 FILE *fp_list;
 int num_files = 0;

 // Open the file pointer streams
 if((fp_list = fopen(playlist, "r")) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to open m3u playlist");
  return NO_FILE;
 }

 // Code to tokenize the directory for the path
 workdir = strdup(playlist);
 cp = strrchr(workdir, '/');
 if(cp) {
  *cp = '\0';
 }

 // Tokenize the strings for new lines and remove # lines
 while((fgets(buf, FILEBUF, fp_list)) != NULL) {
  chomp_line(buf);

  if(buf[0] != '#') {
   // relative path
   if(buf[0] != '/') {
    memset(full_path, 0x0, FILEBUF);
    snprintf(full_path, FILEBUF, "%s/%s", workdir, buf);
    queue_push(full_path);
   }
   // absolute path
   else {
    queue_push(buf);
   }
   num_files++;
  }
 }

 fclose(fp_list);

 return num_files;
}

// not ready for prime time
int parse_itunes_playlist(char *playlist)
{
 /*
 char buf[FILEBUF], key[STRBUF], *b1, *b2, *b3, *b4;
 FILE *fp_list;
 int i, num_files = 0;

 if((fp_list = fopen(playlist, "r")) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to open iTunes playlist");
  return NO_FILE;
 }

 while((fgets(buf, FILEBUF, fp_list)) != NULL) {
  chomp_line(buf);
  memset(key, 0x0, STRBUF);
  if((b1 = strchr(buf, '<')) && (b2 = strchr(buf, '>'))) {
   strncpy(key, b1 + 1, b2 - b1 - 1);
   if(!strcasecmp(key, "location")) {
    if((b3 = strchr(b2, '<')) && (b4 = strchr(b2, '>'
 fclose(fp_list);
    exit(0);

 return 0;
 */
}
