/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  This file contains functions called by the UBS shell for displaying logs and queues.
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "ubs-main.h"

/*+
  This function displays the end part of a logfile, with the number of lines being an arbitrary integer.  This is very similar to the "tail" command in unix.

  int get_log Returns OK on success, NO_FILE if the logfile cannot be opened, NO_MEM if memory allocation fails

  char *log The name of the logfile to open

  int num_lines Number of lines to display

  int mode The output mode
+*/
int get_log(char *log, int num_lines, int mode) {
 FILE *fp;
 char ch, buf[BIGBUF];
 int i = 0, j = 0;
 struct log_entry *entry;

 if((entry = (struct log_entry *)malloc(sizeof(struct log_entry))) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to allocate memory for entry");
  return NO_MEM;
 }

 if((fp = fopen(log, "r")) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to open logfile '%s'", log);
  return NO_FILE;
 }
 fseek(fp, 0, SEEK_END);
 while(i <= num_lines) {
  fseek(fp, -j, SEEK_END);
  if(ftell(fp) == SEEK_SET) {
   log_error_msg(LOG_DEBUG, "Could not seek to file end, breaking");
   break;
  }
  ch = fgetc(fp);
  if(ch == '\n') {
   i++;
  }
  j++;
 }
 for(i = 0; i < num_lines; i++) {
  bzero(buf, BIGBUF);
  bzero(entry, sizeof(struct log_entry));
  if(fgets(buf, BIGBUF, fp) != NULL) {
   read_log(entry, buf);
   print_log(entry, mode);
  }
 }
 fclose(fp);
 free(entry);

 return OK;
}

/*+
  Prints the contents of the queue to standard output

  int show_queue Returns OK on success, NO_FILE if the queue cannot be opened

  int format Output format
+*/
int show_queue(int format) {
 FILE *fp;
 char buf[FILEBUF];

 if((fp = fopen(ubs_table_data(&GLOBAL, "queue"), "r")) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to open queue");
  return NO_FILE;
 }
 while(fgets(buf, FILEBUF, fp) != NULL) {
  chomp_line(buf);
  if(format == ENGLISH || format == CSV) {
   printf("%s\n", buf);
  }
  else if(format == HTML) {
   printf("<tr><td>%s</td></tr>\n", buf);
  }
 }
 fclose(fp);

 return OK;
}
