/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  This module does the most work of any part of the UBS.  It is responsible for knowing what time it is, what show we are in, the attributes of the said show, and scheduling music from this show to the queue when the queue runs out of music.
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "ubs.h"
#include "ubs-sched.h"

/*+
  The main function

  int argc Should be empty; this program takes no arguments

  char *argv[] Should be empty; this program takes no arguments
+*/
int main(int argc, char *argv[])
{
 int cur_day, cur_slot, cur_show, tmp_show;

 // initialize the global table and such
 ubs_table_init(&GLOBAL);
 ubs_table_init(&SCHED);

 ubs_init(argv[0]);

 // initialize values specific to the scheduler daemon
 ubs_table_add(&SCHED, "miscdir", DEF_MISCDIR);
 ubs_table_add(&SCHED, "numshows", DEF_NUMSHOWS);
 ubs_table_add(&SCHED, "searchback", DEF_SEARCHBACK);

 if(read_config(DEF_CONFIG, "global", &GLOBAL)) {
  console_error("Can't read global settings in configuration file", FAIL);
 }
 // schedule specific context
 if(read_config(DEF_CONFIG, "sched", &SCHED)) {
  console_error("Can't read scheduler settings in configuration file", FAIL);
 }

 if(atoi(ubs_table_data(&SCHED, "numshows")) == 0) {
  console_error("Directive 'sched.numshows' undefined", FAIL);
 }
 if(read_sched(DEF_CONFIG)) {
  console_error("Can't parse schedule from configuration file", FAIL);
 }

 if(check_running() == FAIL) {
  console_error("Daemon already seems to be running", FAIL);
 }

#ifndef WIN32
 log_error_msg(LOG_EMERG, "Process started as uid/gid %d/%d", geteuid(), getegid());
#else
 log_error_msg(LOG_EMERG, "Process started");
#endif

 // fork into the background
 if(fork()) {
  exit(0);
 }
 log_error_msg(LOG_DEBUG, "Becoming the process leader");
 setsid();
 log_error_msg(LOG_DEBUG, "Changing working directory to PREFIX");
 chdir(ubs_table_data(&GLOBAL, "prefix"));
 log_error_msg(LOG_DEBUG, "Setting umask to 000");
 umask(0);

 record_pid(getpid());
 record_status("OK");

 log_error_msg(LOG_DEBUG, "Preparing to initialize the WEEK array");
 init_week();
 log_error_msg(LOG_DEBUG, "Preparing to initialize the queue");
 queue_init();

 gettime();
 cur_day = CUR_TIME->tm_wday;
 cur_slot = (CUR_TIME->tm_hour * 2) + (int)(CUR_TIME->tm_min / 30);
 tmp_show = find_show(cur_day, cur_slot);

 if(tmp_show < 0) {
  log_error_msg(LOG_DEBUG, "Scheduling from the miscdir");
  make_cache(ubs_table_data(&SCHED, "miscdir"));
 }
 else {
  // play banner event if necessary
  if(ALLSHOWS[tmp_show].banner[0] != '\0') {
   log_error_msg(LOG_DEBUG, "Preparing to push banner event '%s' for show", ALLSHOWS[tmp_show].banner);
   queue_push(ALLSHOWS[tmp_show].banner);
  }

  if(ALLSHOWS[tmp_show].type == LIST_RANDOM) {
   log_error_msg(LOG_DEBUG, "Making cache for a random show");
   make_cache(ALLSHOWS[tmp_show].dir);
  }
  else if(ALLSHOWS[tmp_show].type == LIST_M3U) {
   log_error_msg(LOG_DEBUG, "Making cache for a m3u playlist");
   parse_m3u_playlist(ALLSHOWS[tmp_show].dir);
  }
  else if(ALLSHOWS[tmp_show].type == LIST_ITUNES) {
   log_error_msg(LOG_DEBUG, "Making cache for a iTunes playlist");
   parse_itunes_playlist(ALLSHOWS[tmp_show].dir);
  }
  else {
   log_error_msg(LOG_DEBUG, "Unknown playlist type");
  }
 }

 // this is the primary control loop.  Hopefully it shall never die
 while(1) {
  gettime();
  cur_day = CUR_TIME->tm_wday;
  cur_slot = (CUR_TIME->tm_hour * 2) + (int)(CUR_TIME->tm_min / 30);
  cur_show = find_show(cur_day, cur_slot);
  // new show is being played
  if(tmp_show != cur_show) {
   log_error_msg(LOG_DEBUG, "New show coming on; clearing everything out");
   queue_init();
   tmp_show = cur_show;
   if(cur_show < 0) {
    log_error_msg(LOG_DEBUG, "Scheduling from the miscdir");
    make_cache(ubs_table_data(&SCHED, "miscdir"));
   }
   else {
    // play banner event if necessary
    if(ALLSHOWS[tmp_show].banner[0] != '\0') {
     log_error_msg(LOG_DEBUG, "Preparing to push banner event '%s' for show", ALLSHOWS[tmp_show].banner);
     queue_push(ALLSHOWS[tmp_show].banner);
    }

    if(ALLSHOWS[cur_show].type == LIST_RANDOM) {
     log_error_msg(LOG_DEBUG, "Making cache for a random show");
     make_cache(ALLSHOWS[tmp_show].dir);
    }
    else if(ALLSHOWS[cur_show].type == LIST_M3U) {
     log_error_msg(LOG_DEBUG, "Making cache for a m3u playlist");
     parse_m3u_playlist(ALLSHOWS[tmp_show].dir);
    }
    else if(ALLSHOWS[tmp_show].type == LIST_ITUNES) {
     log_error_msg(LOG_DEBUG, "Making cache for a iTunes playlist");
     parse_itunes_playlist(ALLSHOWS[tmp_show].dir);
    }
    else {
     log_error_msg(LOG_DEBUG, "Unknown playlist type");
     return FAIL;
    }
   }
  }
  log_error_msg(LOG_DEBUG, "Perparing to enqueue for show");
  enqueue(cur_show);
  sleep(10);
  record_status("OK");
 }

 return OK;
}
