//---------------------------------------------------------------------------
// Copyright (C) 2000 Dallas Semiconductor Corporation, All Rights Reserved.
// 
// Permission is hereby granted, free of charge, to any person obtaining a 
// copy of this software and associated documentation files (the "Software"), 
// to deal in the Software without restriction, including without limitation 
// the rights to use, copy, modify, merge, publish, distribute, sublicense, 
// and/or sell copies of the Software, and to permit persons to whom the 
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included 
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
// MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
// IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES 
// OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
// OTHER DEALINGS IN THE SOFTWARE.
// 
// Except as contained in this notice, the name of Dallas Semiconductor 
// shall not be used except as stated in the Dallas Semiconductor 
// Branding Policy. 
//---------------------------------------------------------------------------
//
//  BUILDTAG.C   - source file containing the Object-Based Messaging
//                 build functions.
//
//  Version: 2.00
//  Revisions:
//

#include "msginc.h"

// exportable functions
// Build TAG functions 
int AddTAGGroup(ushort, TAGType *);
int EndTAGGroup(TAGType *);
int AddTAGData(ushort, void *, int, TAGType *);

//--------------------------------------------------------------------------
// Build TAG functions
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
//  Add a data object to the TAGessage.
//
//  Returns:    =TRUE  object added
//              =FALSE object not added
//
int AddTAGData(ushort OD, void *data, int datalen, TAGType *TAG)
{
   short i,rt=FALSE,templen,l;
   uchar *ucharP;
   ulong *ulongP;

   // get a temporary length of the message
   templen = TAG->msglen;

   // first add the data descriprot
   TAG->msg[templen++] = (uchar)(OD & 0xFF);
   TAG->msg[templen++] = (uchar)((OD >> 8) & 0xFF);

   // case depending on Data Object Descriptor
   switch (OD & DOD_MASK)
   {
      case DOD_NBO_Int:  // NBO decimal integer
         // get a ulong pointer to the data 
         ulongP = data;
         // loop to find out how many characters I need to send this int
         for (l = 3; l >= 0; l--)
            if ((*ulongP >> (8 * l)) & 0xFF)
               break;
         // zero exception
         if (l <= 0)
            l = 1;
         // add the length
         TAG->msg[templen++] = (uchar)(l & 0xFF);

         // check if enough room in message buffer 
         if (TAG->msglen < (MSGMAX - l + 1))
         {
            for (i = l; i >= 0; i--)
               TAG->msg[templen++] = (uchar)((*ulongP >> (8 * i)) & 0xFF);
            rt = TRUE;
         }
         break;
      case DOD_Integer:  // decimal numeric value, leading zero supressed
         // get a ulong pointer to the data 
         ulongP = data;
         // loop to find out how many characters I need to send this int
         for (l = 3; l >= 0; l--)
            if ((*ulongP >> (8 * l)) & 0xFF)
               break;
         // zero exception
         l++;
         if (l <= 0)
            l = 1;
         // add the length
         TAG->msg[templen++] = (uchar)(l & 0xFF);

         // check if enough room in message buffer 
         if (TAG->msglen < (MSGMAX - l + 1))
         {
            for (i = 0; i < l; i++)
            {
              TAG->msg[templen++] = (uchar)(*ulongP & 0xFF);
              *ulongP >>= 8;
            }
            rt = TRUE;
         }
         break;
      case DOD_TimeDate:  // Time / Date
         // assume the same as a fixed ulong
         // add the length
         TAG->msg[templen++] = 4;
         // get a ulong pointer to the data 
         ulongP = data;
         // check if enough room in message buffer 
         if (TAG->msglen < (MSGMAX - 4))
         {
            for (i = 0; i < 4; i++)
               TAG->msg[templen++] = (uchar)((*ulongP >> (8 * i)) & 0xFF);
            rt = TRUE;
         }
         break;
         
      case DOD_DotDec:   // Dotted-decimal notation (as in IP addresses) NBO
      case DOD_ColHex:  // Colon delimited Hex (as in eithernet address) NBO
      case DOD_DSROM:  // Dallas Serial ROM number uncompressed
      case DOD_DSROM_Comp:   // Dallas Serial ROM number compressed (nocrc)
      case DOD_String:  // printable ASCII string
      case DOD_Boolean:  // flags
      case DOD_Unknown: // Unknown format 
      case DOD_Password:  // password (not displayed or masked by IBFAR * characters
      case DOD_ByteArray:  // Byte array of hex digits
      default:
         // add the length
         TAG->msg[templen++] = (uchar)(datalen & 0xFF);
         // get a uchar pointer to the data 
         ucharP = data;
         // check if enough room in message buffer 
         if (TAG->msglen < (MSGMAX - datalen))
         {
            for (i = 0; i < datalen; i++)
               TAG->msg[templen++] = *(ucharP + i);
            rt = TRUE;
         }
         break;
   }
   
   // if successful then keep changes to length of message
   if (rt == TRUE)
      TAG->msglen = templen;

   return rt;
}


//--------------------------------------------------------------------------
//  Add a group to the TAG 
//
//  Returns:    TRUE  : group added
//              FALSE : not enough room 
//
int AddTAGGroup(ushort OD, TAGType *TAG)
{
   // check if there is room for a group 
   if (TAG->msglen < (MSGMAX - 3 - TAG->depth)) 
   {
      // adding a group so increase the depth
      TAG->depth++;
      // add the group
      TAG->msg[TAG->msglen++] = (uchar)(OD & 0xFF);
      TAG->msg[TAG->msglen++] = (uchar)((OD >> 8) & 0xFF);
      return TRUE;
   }
   else
      return FALSE;
}


//--------------------------------------------------------------------------
//  End a group by adding a 0xFF and decreasing the depth
//
//  Returns:    TRUE  : group ended
//              FALSE : not enough room or no group to end
//
int EndTAGGroup(TAGType *TAG)
{
   // check if there is room for end group 
   if ((TAG->msglen < (MSGMAX - 1)) && (TAG->depth > 0))
   {
      // ending a group so decrease the depth
      TAG->depth--;
      // add the group
      TAG->msg[TAG->msglen++] = 0xFF;
      return TRUE;
   }
   else
      return FALSE;
}






