#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>

#include "opts.h"
#include "stats.h"
#include "version.h"


// Take a relative path as argument and return the absolute path by using the current working directory. The return string is malloc'ed with this function.
char *make_absolute(char *relpath) {
	// Already an absolute path
	if (*relpath == '/') return relpath;

	char cwd[PATHLEN_MAX];
	if (!getcwd(cwd, PATHLEN_MAX)) {
		perror("Unable to get current working directory");
		return NULL;
	}

	size_t cwdlen = strlen(cwd);
	if (!cwdlen) {
		fprintf(stderr, "Zero-sized length of CWD!\n");
		return NULL;
	}

	int abslen = cwdlen + strlen(relpath) + 2;
	if (abslen > PATHLEN_MAX) {
		fprintf(stderr, "Absolute path too long!\n");
		return NULL;
	}

	char *abspath = malloc(abslen);
	sprintf(abspath, "%s/%s", cwd, relpath);

	return abspath;
}

void print_help(const char *progname) {
	fprintf (stderr,
	"unionfs-fuse version "VERSION"\n"
	"by Radek Podgorny <radek@podgorny.cz>\n"
	"\n"
	"Usage: %s [options] root[:root...] mountpoint\n"
	"The first argument is a colon separated list of directories to merge\n"
	"\n"
	"general options:\n"
	"    -o opt,[opt...]        mount options\n"
	"    -h   --help            print help\n"
	"    -V   --version         print version\n"
	"\n"
	"UnionFS options:\n"
	"    -o stats               show statistics in the file 'stats' under the mountpoint\n"
	"\n",
	progname);
}

int unionfs_opt_proc(void *data, const char *arg, int key, struct fuse_args *outargs) {
	(void)data;

	switch (key) {
		case FUSE_OPT_KEY_NONOPT:
			if (!nroots) {
				// We don't free the buf as parts of it may go to roots
				char *buf = strdup(arg);
				char **ptr = (char **)&buf;
				char *root;
				while ((root = strsep(ptr, ROOT_SEP)) != NULL) {
					if (strlen(root) == 0) continue;

					roots = realloc(roots, (nroots+1) * sizeof(root));
					roots[nroots++] = make_absolute(root);
				}
				return 0;
			}
			return 1;
		case KEY_STATS:
			stats_enabled = 1;
			return 0;
		case KEY_HELP:
			print_help(outargs->argv[0]);
			fuse_opt_add_arg(outargs, "-ho");
			doexit = 1;
			return 0;
		case KEY_VERSION:
			printf("unionfs-fuse version: "VERSION"\n");
			doexit = 1;
			return 1;
		default:
			return 1;
	}
}
