/* vifm
 * Copyright (C) 2001 Ken Steen.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include<ncurses.h>
#include<ctype.h> /* isspace() */
#include<string.h> /* strchr() */

#include "bookmarks.h"
#include "commands.h"
#include "config.h"
#include "filelist.h"
#include "fileops.h"
#include "filetype.h"
#include "keys.h"
#include "status.h"
#include "ui.h"
#include "utils.h"

typedef struct menu_info 
{
	int top;
	int current;
	int len;
	int pos;
	int win_rows;
	int type;
}menu_info;



enum {
	BOOKMARK,
	COMMAND,
	FILETYPE,
	HISTORY,
	VIFM
};


static void draw_menu(FileView *view, menu_info *m);

void
show_error_msg(char *title, char *message)
{
	int x, y;
	int key;
	int done = 0;
	char *p = NULL;

	if((p = strrchr(message, '\n')) != NULL)
	{
		*p = '\0';
	}

	curr_stats.popup_showing = 1;
	curr_stats.freeze = 1;
	wattroff(curr_view->win, COLOR_PAIR(CURR_LINE_COLOR) | A_BOLD);
	mvwaddstr(curr_view->win, curr_view->curr_line, 0, "  ");
	curs_set(0);
	werase(error_win);
	box(error_win, 0, 0);


	getmaxyx(error_win, y, x);
	if(strlen(message) > x -2)
		message[x -2] = '\0';

	mvwaddstr(error_win, 0, (x - strlen(title))/2, title);
	mvwaddstr(error_win, 2, (x - strlen(message))/2, message);

	mvwaddstr(error_win, y -2, (x - 25)/2, "Press Return to continue");

	while(!done)
	{
		key = wgetch(error_win);
		if(key == 13 || key == 3) /* ascii Return  ascii Ctrl-c */
			done = 1;
	}

	curr_stats.popup_showing = 0;
	curr_stats.freeze = 0;

	werase(error_win);
	wrefresh(error_win);

	wattron(curr_view->win, COLOR_PAIR(CURR_LINE_COLOR) | A_BOLD);
	mvwaddstr(curr_view->win, curr_view->curr_line, 0, "=>");
	
	touchwin(stdscr);
	update_all_windows();

	if(curr_stats.need_redraw)
		redraw_window();

}
void
reset_popup_menu(void)
{
	werase(popup_win);
	
	update_all_windows();
	curr_stats.popup_showing = 0;
	if(curr_stats.need_redraw)
		redraw_window();
	moveto_list_pos(curr_view, curr_view->list_pos);
}

void
setup_menu(FileView *view)
{
	scrollok(popup_win, FALSE);
	curr_stats.popup_showing = 1;
	wattroff(view->win, COLOR_PAIR(CURR_LINE_COLOR) | A_BOLD);
	mvwaddstr(view->win, view->curr_line, 0, "  ");
	curs_set(0);
	werase(popup_win);
}

static void
moveto_menu_pos(FileView *view, int pos,  menu_info *m)
{
	int redraw = 0;

	mvwaddstr(popup_win, m->current, 1, "  ");

	if(pos < 1)
		pos = 0;

	if(pos > m->len -1)
		pos = m->len -1;

	if((m->top <=  pos) && (pos <= (m->top + m->win_rows +1)))
	{
		m->current = pos - m->top +1;
	}
	if((pos >= (m->top + m->win_rows -2)))
	{
		while(pos >= (m->top + m->win_rows -2))
			m->top++;

		m->current = m->win_rows -2;
		redraw = 1;
	}
	else if(pos  < m->top)
	{
		while(pos  < m->top)
			m->top--;
		m->current = 1;
		redraw = 1;
	}
	if(redraw)
		draw_menu(view, m);

	wattron(popup_win, COLOR_PAIR(CURR_LINE_COLOR) | A_BOLD);
	mvwaddstr(popup_win, m->current, 1, "=>");
	wattroff(popup_win, COLOR_PAIR(CURR_LINE_COLOR) | A_BOLD);
	m->pos = pos;
}

static void
menu_key_cb(FileView *view, menu_info *m)
{
	int done = 0;
	int abort = 0;

	wattron(popup_win, COLOR_PAIR(CURR_LINE_COLOR) | A_BOLD);
	mvwaddstr(popup_win, 1, 1, "=>");
	wattroff(popup_win, COLOR_PAIR(CURR_LINE_COLOR) | A_BOLD);
	keypad(popup_win, TRUE);

	wrefresh(popup_win);
	
	while(!done)
	{
		int key = wgetch(popup_win);

		switch(key)
		{
			case 'q':
				{
					done = 1;
					abort = 1;
				}
				break;
			case 'd':
				{
					key = wgetch(popup_win);

					if(key != 'd')
						break;

					if(m->type == COMMAND)
					{
						remove_command(command_list[m->pos].name);
						m->len = cfg.command_num;
						draw_menu(view, m);
						if(m->pos -1 >= 0)
							moveto_menu_pos(view, m->pos -1, m);
						else
							moveto_menu_pos(view, 0, m);
					}
					else if(m->type == BOOKMARK)
					{
						remove_bookmark(bookmarks[m->pos].mark);
						m->len = cfg.bookmarks_pos;
						draw_menu(view, m);
						if(m->pos -1 >= 0)
							moveto_menu_pos(view, m->pos -1, m);
						else
							moveto_menu_pos(view, 0, m);
					}
				}
				break;
			case KEY_DOWN:
			case 'j':
				{
					m->pos++;
					moveto_menu_pos(view, m->pos, m);
				}
				break;
			case KEY_UP:
			case 'k':
				{
					m->pos--;
					moveto_menu_pos(view, m->pos, m);

					wrefresh(popup_win);
				}
				break;
			case 3: /* ascii Ctrl C */
			case 27: /* ascii Escape */
				done = 1;
				abort = 1;
				break;
			case 'l':
			case 13: /* ascii Return */
						done = 1;
				break;
			default:
				break;
		}
	} /* end of while(!done) */

	if(abort)
	{
		reset_popup_menu();
		return;
	}

	switch(m->type)
	{
		case BOOKMARK:
			{
				reset_popup_menu();
				move_to_bookmark(view, bookmarks[m->pos].mark);
				return;
			}
			break;
		case COMMAND:
		{
			reset_popup_menu();
			execute_command(view, command_list[m->pos].name);
			return;
		}
			break;
		case HISTORY:
			{
				reset_popup_menu();
				change_directory(view, view->history[m->pos].dir);
				load_dir_list(view, 1);
				moveto_list_pos(view, find_file_pos_in_list(view, 
							view->history[m->pos].file));
				return;
			}
			break;
		case VIFM:
			status_bar_message("Vifm menu is not implemented yet.");
			break;
		case FILETYPE:
			{
				char *filename = get_current_file_name(view);
				char *prog_str = get_all_programs_for_file(filename);
				char command[NAME_MAX];
				char *ptr = NULL;

				reset_popup_menu();

				if((ptr = strchr(prog_str, ',')) == NULL)
				{
					if(strchr(prog_str, '%'))
					{
						char *expanded_command = expand_macros(view, prog_str, NULL);
						shellout(expanded_command, 0);
						my_free(expanded_command);
						return;
					}
					else
					{
						snprintf(command, sizeof(command), "%s %s", prog_str, filename);
						shellout(command, 0);
						return;
					}
				}
				else
				{
					char *prog_copy = strdup(prog_str);
					char *free_this = prog_copy;
					char *ptr1 = NULL;
					int x = 1;

					while((ptr = ptr1 = strchr(prog_copy, ',')) != NULL)
					{
						*ptr = '\0';
						ptr1++;

						if(x == m->pos +1)
						{
							if(strchr(prog_str, '%'))
							{
								char *expanded_command = expand_macros(view, prog_copy, NULL);
								shellout(expanded_command, 0);
								my_free(expanded_command);
								free(free_this);
								return;
							}
							else
							{
								snprintf(command, sizeof(command), "%s %s", 
										prog_copy, filename);
								shellout(command, 0);
								free(free_this);
								return;
							}
						}
						prog_copy = ptr1;
						x++;
					}
						if(strchr(prog_str, '%'))
						{
							char *expanded_command = expand_macros(view, prog_copy, NULL);
							shellout(expanded_command, 0);
							my_free(expanded_command);
							free(free_this);
							return;
						}
						else
						{
							snprintf(command, sizeof(command), "%s %s", prog_copy, filename);
							shellout(command, 0);
							free(free_this);
							return;
						}
				}
			}
			break;
		default:
			break;
	}

	wattroff(popup_win, A_BOLD);
	reset_popup_menu();
}

static void
draw_menu(FileView *view,  menu_info *m)
{

	int i;
	int x, y;

	getmaxyx(popup_win, y, x);
	werase(popup_win);

	box(popup_win, 0, 0);
	if(m->win_rows >= m->len)
	{
		m->top = 0;
	}

	x = m->top;

	switch(m->type)
	{
		case BOOKMARK:
			{
				char buf[PATH_MAX];
				mvwaddstr(popup_win, 0, 2, " Mark ");
				mvwaddstr(popup_win, 0, 10, " File ");
				for(i = 1; x < m->len; i++)
				{
					if(!strcmp(bookmarks[x].directory, "/"))
						snprintf(buf, sizeof(buf), "/%s", bookmarks[x].file);
					else if(!strcmp(bookmarks[x].file, "../"))
						snprintf(buf, sizeof(buf),  bookmarks[x].directory);
					else
						snprintf(buf, sizeof(buf), "%s/%s", bookmarks[x].directory,
								bookmarks[x].file);

					mvwaddch(popup_win, i, 4, bookmarks[x].mark);
					mvwaddstr(popup_win, i, 8, buf);
					x++;
					if(i +3 > y)
						break;
				}
			}
			break;
		case COMMAND:
			{
				mvwaddstr(popup_win, 0, 3, " Command ");
				mvwaddstr(popup_win, 0, 16, " Action ");
				for(i = 1; x < m->len; i++)
				{
					mvwaddstr(popup_win, i, 4, command_list[x].name);
					mvwaddstr(popup_win, i, 14, command_list[x].action);

					/* This will show the expanded command instead of the macros
					char *expanded = expand_macros(view, command_list[x].action, NULL);
					mvwaddstr(popup_win, i, 14, expanded);
					free(expanded);
					*/
					x++;
					if(i +3 > y)
						break;
				}
			}
			break;
		case HISTORY:
			{
				char his_buf[PATH_MAX];
				mvwaddstr(popup_win, 0, 3, " History ");
				for(i = 1; x < m->len; i++)
				{
					if(strlen(view->history[x].dir) < 1)
						break;

					/* Change the current dir to reflect the current file.*/ 
					if(!strcmp(view->history[x].dir, view->curr_dir))
						snprintf(view->history[x].file, sizeof(view->history[x].file),
									view->dir_entry[view->list_pos].name);

					if(!strcmp(view->history[x].dir, "/"))
						snprintf(his_buf, sizeof(his_buf), "/%s", view->history[x].file);
					else
						snprintf(his_buf, sizeof(his_buf), "%s/%s", view->history[x].dir,
								view->history[x].file);

					mvwaddstr(popup_win, i, 4, his_buf);
					x++;
					if(i +3 > y)
						break;
				}
			}
			break;
		case VIFM:
			{
			}
			break;
		case FILETYPE:
			{
				char *filename = get_current_file_name(view);
				char *prog_str = get_all_programs_for_file(filename);
				char file_buf[PATH_MAX];
				char *ptr = NULL;

				snprintf(file_buf, sizeof(file_buf), "  Select program to open %s  ",
						filename);

				mvwaddstr(popup_win, 0, 4, file_buf);

				if((ptr = strchr(prog_str, ',')) == NULL)
				{
					m->len = 1;
					mvwaddstr(popup_win, 1, 4, prog_str);
					break;
				}
				else
				{
					char *prog_copy = strdup(prog_str);
					char *free_this = prog_copy;
					char *ptr1 = NULL;
					int x = 1;

					while((ptr = ptr1 = strchr(prog_copy, ',')) != NULL)
					{
						*ptr = '\0';
						ptr1++;
						mvwaddstr(popup_win, x, 4, prog_copy);
						prog_copy = ptr1;
						x++;
					}

					mvwaddstr(popup_win, x, 4, prog_copy);
					m->len = x;
					free(free_this);

					break;
				}
			}
			break;
		default:
			break;
	}

	wrefresh(popup_win);

}


void
show_commands_menu(FileView *view)
{

	menu_info m;
	int x;
	m.top = 0;
	m.current = 1;
	m.len = cfg.command_num;
	m.pos = 0;
	m.type = COMMAND;

	getmaxyx(popup_win, m.win_rows, x);
	qsort(command_list, cfg.command_num, sizeof(command_t),
			sort_this);
	setup_menu(view);

	draw_menu(view, &m);
	moveto_menu_pos(view, 0, &m);
	menu_key_cb(view, &m);
}

void
show_bookmarks_menu(FileView *view)
{
	menu_info m;
	int x;
	m.top = 0;
	m.current = 1;
	m.len = cfg.bookmarks_pos;
	m.pos = 0;
	m.type = BOOKMARK;

	getmaxyx(popup_win, m.win_rows, x);
	qsort(bookmarks, cfg.bookmarks_pos, sizeof(bookmarks_t),
			sort_bookmarks);

	setup_menu(view);
	draw_menu(view, &m);
	moveto_menu_pos(view, 0, &m);
	menu_key_cb(view, &m);
}

void
show_history_menu(FileView *view)
{
	menu_info m;
	int x;
	m.top = 0;
	m.current = 1;
	m.len = view->history_num;
	m.pos = 0;
	m.type = HISTORY;

	getmaxyx(popup_win, m.win_rows, x);

	setup_menu(view);
	draw_menu(view, &m);
	moveto_menu_pos(view, 0, &m);
	menu_key_cb(view, &m);
}

void
show_vifm_menu(FileView *view)
{
	/*
	menu_info m;
	int x;
	m.top = 0;
	m.current = 1;
	m.len = NULL;
	m.pos = 0;
	m.type = VIFM;

	getmaxyx(popup_win, m.win_rows, x);

	setup_menu(view);
	draw_menu(view, &m);
	moveto_menu_pos(view, 0, &m);
	menu_key_cb(view, &m);
	*/
}


void
show_filetypes_menu(FileView *view)
{
	char *filename = get_current_file_name(view);

	if((get_all_programs_for_file(filename)) == NULL)
	{
		show_error_msg("  Filetype is not set.  ", 
				"No programs set for this filetype.");
		return;
	}
	else
	{
		int x;
		menu_info m;
		m.top = 0;
		m.current = 1;
		m.len = cfg.filetypes_num;
		m.pos = 0;
		m.type = FILETYPE;

		getmaxyx(popup_win, m.win_rows, x);

		setup_menu(view);
		draw_menu(view, &m);
		moveto_menu_pos(view, 0, &m);
		menu_key_cb(view, &m);
	}
}





