/*
 * Symbol.c -- Implementation of Scheme Symbols
 *
 * (C) m.b (Matthias Blume); Mar 1992, HUB; Jan 1993 PU/CS
 *         Humboldt-University of Berlin
 *         Princeton University, Dept. of Computer Science
 *
 * $Id: Symbol.c,v 2.15 1994/11/12 22:17:04 blume Exp $
 */

# include "rcs.h"
RCSID ("$Id: Symbol.c,v 2.15 1994/11/12 22:17:04 blume Exp $")

# include <stdio.h>
# include <string.h>
# include <ctype.h>

# include "storext.h"
# include "Symbol.h"
# include "Primitive.h"
# include "identifier.h"
# include "type.h"
# include "except.h"

# define SYMTAB_HASH_SIZE 41

static ScmSymbol *symbol_table [SYMTAB_HASH_SIZE];
static ScmSymbol *weak_symbol_table [SYMTAB_HASH_SIZE];

static MEM_cnt measure (void *vsym)
{
  return MEM_UNITS (sizeof (ScmSymbol) + ((ScmSymbol *)vsym)->length - 1);
}

static void iterator (void *vsym, MEM_visitor proc, void *cd)
{
  ScmSymbol *sym = (ScmSymbol *) vsym;

  if (sym->strong)
    (*proc) ((void *)&sym->hashlink, cd);
  (*proc) ((void *)&sym->value, cd);
}

static void dumper (void *vsym, FILE *file)
{
  ScmSymbol *sym = (ScmSymbol *) vsym;
  unsigned i;

  MEM_dump_ul (sym->primno_succ, file);
  MEM_dump_ul (sym->length, file);
  for (i = 0; i < sym->length; i++)
    putc (sym->array[i], file);
}

static void *excavator (FILE *file)
{
  ScmSymbol *sym;
  unsigned i, c;
  unsigned primno_succ, length;

  primno_succ = MEM_restore_ul (file);
  length = MEM_restore_ul (file);
  SCM_VNEW (sym, Symbol, length, char);
  sym->primno_succ = primno_succ;
  sym->length = length;
  sym->strong = 1;
  for (i = 0; i < length; i++)
    if ((c = getc (file)) == EOF)
      fatal ("bad dump file format (Symbol)");
    else
      sym->array[i] = c;
  return sym;
}

static void display (void *vsym, putc_proc pp, void *cd)
{
  ScmSymbol *sym = vsym;
  unsigned i;

  for (i = 0; i < sym->length; i++)
    (* pp) (sym->array[i], cd);
}

static void write_this (void *vsym, putc_proc pp, void *cd)
{
  ScmSymbol *sym = vsym;
  unsigned i;
  int c;

  for (i = 0; i < sym->length; i++) {
    c = (unsigned char) sym->array[i];
    if (isprint (c))
      (* pp) (c, cd);
    else {
      char buf[16];
      sprintf (buf, "\\%03o", (unsigned char)c);
      putc_string (buf, pp, cd);
    }
  }
}

static void symtab_iterator (void *vsymtab, MEM_visitor proc, void *cd)
{
  int i;
  ScmSymbol **symtab = vsymtab;

  for (i = 0; i < SYMTAB_HASH_SIZE; i++)
    (* proc) ((void *)&symtab[i], cd);
}

static int hash_key (const char *string, unsigned length)
/* string is not a C-string (0-terminated), so we need its length */
{
  int sum;

  sum = 0;
  while (length--)
    sum += (unsigned char) *string++;
  return sum % SYMTAB_HASH_SIZE;
}

void *ScmMakeSymbol (const char *name, unsigned length)
/* name is not necessarily a C-string (0-terminated), so we need its length */
{
  int key = hash_key (name, length);
  ScmSymbol *l = symbol_table [key];

  while (l != NULL) {
    if (length == l->length && memcmp (name, l->array, length) == 0)
      break;
    l = l->hashlink;
  }

  if (l == NULL) {
    SCM_VNEW (l, Symbol, length, char);
    l->hashlink = symbol_table [key];
    symbol_table [key] = l;
    l->value = NULL;
    l->strong = 1;
    l->primno_succ = 0;
    l->length = length;
    memcpy (l->array, name, length);
  }
  return l;
}

void ScmInitSymtab (void)
{
  int i;
  unsigned long seq_num;
  ScmPrimitive *prim;
  ScmSymbol *sym;

  /* Make an empty hash table */
  for (i = 0; i < SYMTAB_HASH_SIZE; i++)
    symbol_table [i] = weak_symbol_table [i] = NULL;

  /* register symbol_table at storage module */
  MEM_root (symbol_table, symtab_iterator);

  for (seq_num = 0; seq_num < ScmPrimitive_num; seq_num++) {
    prim = GetScmPrimitive (seq_num);
    sym = ScmMakeSymbol (prim->name, strlen (prim->name));
    sym->value = prim;
    sym->primno_succ = seq_num + 1;
  }
}

static void before_gc (void)
{
  int i;
  ScmSymbol *strong, *weak, *sym, *l;

  for (i = 0; i < SYMTAB_HASH_SIZE; i++) {
    l = symbol_table [i];
    strong = NULL;
    weak = NULL;

    while (l != NULL) {
      sym = l;
      l = l->hashlink;
      if (sym->value == NULL && sym->primno_succ == 0)
	sym->hashlink = weak, weak = sym, sym->strong = 0;
      else
	sym->hashlink = strong, strong = sym;
    }

    symbol_table [i] = strong;
    weak_symbol_table [i] = weak;
  }
}

static void after_gc (void)
{
  int i;
  ScmSymbol *sym, *n, *strong;

  for (i = 0; i < SYMTAB_HASH_SIZE; i++) {
    sym = weak_symbol_table [i];
    strong = symbol_table [i];

    while (sym != NULL)
      if ((n = MEM_new_location_of (sym)) != sym) {
	n->strong = 1;
	sym = n->hashlink;
	n->hashlink = strong;
	strong = n;
      } else
	sym = sym->hashlink;

    symbol_table [i] = strong;
  }
}

MEM_VECTOR (Symbol,
	    0, measure,
	    iterator, dumper, excavator, MEM_NULL_revisor,
	    MEM_NULL_task, before_gc, after_gc,
	    EXT (SCM_NO_NUMBER,
		 cannot_cvt_real, display, write_this, NULL_eq, NULL_eq));
