(define allow-compiler-inspection #f)
(define ## '())

(define toplevel '())
(define eval '())
(define load '())
(define show-opt '())
(define show-expand '())
(define statistics '())

(define-type driver-sig
  (signature (toplevel rts eval load build-compiler show-expand show-opt
	      compile-files
	      detailed-compile-statistics gen-boot-loader)
    (constant toplevel any)
    (constant rts any)
    (constant eval any)
    (constant load any)
    (constant build-compiler any)
    (constant compile-files any)
    (constant show-expand any)
    (constant show-opt any)
    (constant detailed-compile-statistics any)
    (constant gen-boot-loader any)))

(define (link-driver tabfile)

  (let* ((aux-mod (aux-functor))
	 (names-mod (names-functor))
	 (show-mod (show-functor))
	 (error-mod (error-functor names-mod show-mod))
	 (lgi-mod (lgi-functor))
	 (sa-mod (sa-functor lgi-mod error-mod))
	 (scc-mod (scc-functor))
	 (varset-mod (varset-functor lgi-mod set-functor))
	 (hack-mod (hack-functor lgi-mod))
	 (foldint-mod (foldint-functor error-mod lgi-mod aux-mod))
	 (optutil-mod (optutil-functor error-mod))
	 (lvar-mod (lvar-functor optutil-mod varset-mod error-mod))
	 (fv-mod (fv-functor varset-mod optutil-mod error-mod))
	 (eff-mod (eff-functor aux-mod lgi-mod optutil-mod error-mod))
	 (primv-mod (primv-functor tabfile aux-mod))
	 (high-opt-mod
	  (opt-functor 2
		       lgi-mod error-mod varset-mod aux-mod foldint-mod
		       optutil-mod hack-mod fv-mod eff-mod))
	 (med-opt-mod
	  (opt-functor 1
		       lgi-mod error-mod varset-mod aux-mod foldint-mod
		       optutil-mod hack-mod fv-mod eff-mod))
	 (low-opt-mod
	  (opt-functor 0
		       lgi-mod error-mod varset-mod aux-mod foldint-mod
		       optutil-mod hack-mod fv-mod eff-mod))
	 (cg-mod (cg-functor error-mod lgi-mod varset-mod aux-mod
			     primv-mod hack-mod))
	 (clos-mod (clos-functor
		    lgi-mod aux-mod varset-mod fv-mod optutil-mod))
	 (ltrc-mod (ltrc-functor varset-mod fv-mod aux-mod optutil-mod
				 eff-mod lgi-mod scc-mod error-mod))
	 (soft-mod (soft-functor))
	 (integ-mod (integ-functor lgi-mod optutil-mod soft-mod))
	 (high-opt-be-mod (be-functor sa-mod high-opt-mod integ-mod ltrc-mod
				      clos-mod lvar-mod cg-mod))
	 (low-opt-be-mod (be-functor sa-mod low-opt-mod integ-mod ltrc-mod
				     clos-mod lvar-mod cg-mod))
	 (tfgen-mod (tfgen-functor low-opt-be-mod))
	 (pmac-mod (pmac-functor names-mod error-mod))
	 (envs-mod
	  (envs-functor lgi-mod pmac-mod primv-mod aux-mod names-mod soft-mod))
	 (macro-mod (macro-functor lgi-mod error-mod aux-mod tfgen-mod
				   names-mod envs-mod))
	 (boot-mod (boot-functor macro-mod high-opt-be-mod soft-mod))
	 (inspect-mod (inspect-functor show-mod primv-mod))
	 (rts-mod (rts-functor)))

    (define driver-mod
      (module driver-sig (allow-compiler-inspection driver-sig link-driver ##)

	(use raw-system-sig)
	(use macro-mod macro-sig)
	(use sa-mod sa-sig)
	(use high-opt-mod opt-sig h-opt)
	(use med-opt-mod opt-sig m-opt)
	(use cg-mod cg-sig)
	(use error-mod error-sig)
	(use hack-mod hack-sig)
	(use lvar-mod lvar-sig)
	(use clos-mod clos-sig)
	(use ltrc-mod ltrc-sig)
	(use inspect-mod inspect-sig)
	(use soft-mod soft-sig soft)
	(use boot-mod boot-sig)
	(use rts-mod rts-sig)
	(use integ-mod integ-sig)

	(define pretty-print write)	; cheap substitute...

	(define (show-opt exp)
	  (pretty-print
	   (hack (livevar (closures
			   (decompose-letrecs
			    (integ
			     ((h-opt opt) (sa (expand exp))))))))))

	(define (show-expand exp) (pretty-print (hack (expand exp))))

	(define (scm->asm)
	  (with-error-handler
	      (lambda (m c)
		(with-output-to-port
		    (standard-port 2)
		  (lambda ()
		    (display "!* ")
		    (display m)
		    (newline)
		    (display "!* error during compilation")
		    (newline)
		    (quit 1))))
	    (lambda ()
	      (with-interrupt-handler
		  (lambda ()
		    (with-output-to-port
			(standard-port 2)
		      (lambda ()
			(display "!* compilation interrupted")
			(newline)
			(quit 2))))
		(lambda ()
		  (do ((x (read) (read)))
		      ((eof-object? x) 'done)
		    (let ((code
			   (cg
			    (livevar
			     (closures
			      (decompose-letrecs
			       (integ
				((h-opt opt)
				 (sa
				  (expand x))))))))))
		      (write `(run-asm ',code))
		      (newline))))))))

	(define (compile-files argl)
	  (for-each (lambda (arg)
		      (display ";; File: ") (display arg) (newline) (newline)
		      ((soft with-input-from-file) arg scm->asm)
		      (display ";; End of File: ")
		      (display arg) (newline) (newline))
	    argl))

	(define (build-compiler f)
	  (cond ((dump f)
		 =>
		 (lambda (argl)
		   (compile-files argl)
		   (quit)))
		(else f)))
			   
	(define (driver exp k ek opt)

	  (define phases
	    (list (list expand            "macro expansion       ")
		  (list sa                "assignment elimination")
		  (list opt               "optimization          ")
		  (list integ             "rts factorization     ")
		  (list decompose-letrecs "letrec decomposition  ")
		  (list closures          "closure analysis      ")
		  (list livevar           "liveness analysis     ")
		  (list cg                "code generation       ")))

	  (define start-t (clock))
	  (define start-g (gc-clock))
	  (define compile-t #f)
	  (define compile-g #f)

	  (define (compile exp)
	    (let loop
		((x exp)
		 (p phases)
		 (t start-t)
		 (g start-g))
	      (if (pair? p)
		  (let* ((y ((caar p) x))
			 (nt (clock))
			 (ng (gc-clock)))
		    (set-cdr! (cdar p) (list (- nt t) (- ng g)))
		    (loop y (cdr p) nt ng))
		  x)))

	  (let* ((p (call-with-current-continuation
		     (lambda (c)
		       (with-error-handler
			   (lambda (m ec)
			     (c (cons ek (cons m ec))))
			 (lambda ()
			   (let* ((asm (compile exp))
				  (ign (set! compile-t (clock)))
				  (ign (set! compile-g (gc-clock))))
			     (cons k (run-asm asm))))))))
		 (end-t (clock))
		 (end-g (gc-clock))
		 (compile-t (or compile-t end-t))
		 (compile-g (or compile-g end-g)))
	    ((car p) (cdr p)
		     (- compile-t start-t)
		     (- compile-g start-g)
		     (- end-t compile-t)
		     (- end-g compile-g)
		     phases)))

	(define (eval exp)
	  (driver exp
		  (lambda (res . ign) res)
		  (lambda (res . ign)
		    (reset-error-type!)
		    (re-raise-error
		     (string-append "(EVAL) " (car res))
		     (cdr res)))
		  (m-opt opt)))

	(define (dtime t)
	  (if (negative? t)
	      (display "?.???")
	      (call-with-values
		  (lambda () (divide t 1000))
		(lambda (s ms)
		  (let* ((ms-str (number->string ms))
			 (ms-pad (make-string
				  (- 3 (string-length ms-str)) '#\0)))
		    (display s)
		    (display ".")
		    (display ms-pad)
		    (display ms-str))))))

	(define (dtimes t g)
	  (dtime t)
	  (if (not (zero? g))
	      (begin
		(display "(")
		(dtime g)
		(display ")"))))

	(define (statistics ct cg et eg)

	  (define (time m t g)
	    (display " ") (display m) (display ": ")
	    (dtimes t g))

	  (display ";")
	  (time "compile" ct cg)
	  (time "execute" et eg)
	  (time "TOTAL" (+ ct et) (+ cg eg))
	  (newline))

	(define (statistics-phase-by-phase phases)

	  (define (one-phase p)
	    (display (cadr p))
	    (display ": ")
	    (let ((t (cddr p)))
	      (if (not (pair? t))
		  (display "[didn't run]")
		  (dtimes (car t) (cadr t))))
	    (newline))

	  (for-each one-phase phases))

	(define recent-phases #f)

	(define (detailed-compile-statistics)
	  (if (not recent-phases)
	      (error "no statistics available yet")
	      (statistics-phase-by-phase recent-phases)))

	(define (toplevel argl)

	  (define resume #f)

	  (define (input)
	    (display "D> " (standard-port 2))
	    (flush (standard-port 2))
	    (set! resume #f)
	    (let ((r (read)))
	      (if (and resume (eof-object? r))
		  ''interrupt
		  r)))

	  (if (list? argl)
	      (with-error-handler
		  (lambda (m ec)
		    (with-output-to-port
			(standard-port 2)
		      (lambda ()
			(display m)
			(newline)
			(display "!* error during command line processing")
			(newline)
			(quit 1))))
		(lambda ()
		  (with-interrupt-handler
		      (lambda ()
			(with-output-to-port
			    (standard-port 2)
			  (lambda ()
			    (display "!* command line processing interrupted")
			    (newline)
			    (quit 1))))
		    (lambda ()
		      (do ((l argl (cdr l)))
			  ((not (pair? l)) 'done)
			(load (car l)))

		      (display
		       "Welcome to VSCM [now featuring a module system]")
		      (newline))))))

	  (call-with-current-continuation
	   (lambda (exit)
	     (define (loop)
	       (call-with-current-continuation
		(lambda (continue)
		  (with-error-handler
		      (lambda (m ec)
			(with-output-to-port
			    (standard-port 2)
			  (lambda ()
			    (display "!* Error in top-level loop: ")
			    (display m)
			    (newline)))
			(continue #f))
		    (lambda ()
		      (with-interrupt-handler
			  (lambda ()
			    (with-output-to-port
				(standard-port 2)
			      (lambda ()
				(display
				 "!* Interrupt: type (##) to resume execution")
				(newline)))
			    (if
			     (call-with-current-continuation
			      (lambda (c)
				(set! ## (lambda () (c #f)))
				#t))
			     (continue #f)
			     (set! resume #t)))
			(lambda ()
			  (do ((x (input) (input)))
			      ((eof-object? x) (newline) (exit 'done))
			    (reset-error-type!)
			    (driver
			     x
			     (lambda (r ct cg et eg ph)
			       (statistics ct cg et eg)
			       (display "; result: ")
			       (newline)
			       (write r)
			       (newline)
			       (set! ## r)
			       (set! recent-phases ph))
			     (lambda (r ct cg et eg ph)
			       (display (car r))
			       (newline)
			       (statistics ct cg et eg)
			       (set! recent-phases ph)
			       (display 
				(case error-type
				  ((runtime)
				   (set! ##
				    (lambda () (inspector (cdr r) 1) #f))
				   "!* Error: type (##) to run inspector")
				  ((syntax) "!* Syntax error")
				  ((semantic) "!* Semantic error")
				  (else "!!* Compiler bug")))
			       (newline)
			       (if allow-compiler-inspection
				   (set! ##
				    (lambda () (inspector (cdr r) 1) #f))))
			     (m-opt opt)))))))))
	       (loop))
	     (loop)))
	  (quit 0))

	(define rts (make-rts toplevel (soft name->value)))

	(define (load fname . stats)

	  (define (lstat accu)
	    (display "; +++ LOAD statistics:")
	    (newline)
	    (apply statistics accu)
	    (display "; --- END of LOAD statistics")
	    (newline))

	  (define details? (and (pair? stats) (eq? (car stats) 'details)))
	  (define summary? (and (pair? stats) (car stats)))

	  (let ((p (open-input-file fname)))
	    (let loop
		((accu '(0 0 0 0)))
	      (let ((x (read p)))
		(if (eof-object? x)
		    (begin (close-port p) (if summary? (lstat accu)) '#f)
		    (driver
		     x
		     (lambda (r ct cg et eg ph)
		       (if details? (statistics-phase-by-phase ph))
		       (loop (map + (list ct cg et eg) accu)))
		     (lambda (r ct cg et eg ph)
		       (close-port p)
		       (if details? (statistics-phase-by-phase ph))
		       (if summary?
			   (begin
			     (display "!* error during load")
			     (newline)
			     (lstat accu)))
		       (re-raise-error (string-append "(LOAD) " (car r))
				       (cdr r)))
		     (h-opt opt)))))))))
    driver-mod))

(define (build-repl f)

  (define (discard)
    (set! aux-functor #f)
    (set! be-functor #f)
    (set! boot-functor #f)
    (set! cg-functor #f)
    (set! clos-functor #f)
    (set! link-driver #f)
    (set! build-repl #f)
    (set! build-compiler #f)
    (set! gen-boot-loader #f)
    (set! eff-functor #f)
    (set! envs-functor #f)
    (set! error-functor #f)
    (set! foldint-functor #f)
    (set! fv-functor #f)
    (set! hack-functor #f)
    (set! inspect-functor #f)
    (set! integ-functor #f)
    (set! lgi-functor #f)
    (set! ltrc-functor #f)
    (set! lvar-functor #f)
    (set! macro-functor #f)
    (set! names-functor #f)
    (set! opt-functor #f)
    (set! optutil-functor #f)
    (set! pmac-functor #f)
    (set! primv-functor #f)
    (set! rts-functor #f)
    (set! sa-functor #f)
    (set! scc-functor #f)
    (set! set-functor #f)
    (set! show-functor #f)
    (set! soft-functor #f)
    (set! tfgen-functor #f)
    (set! varset-functor #f))

  (let* ((tabfile (or (getenv "PRIMVTAB")
		      (error "PRIMVTAB?")))
	 (driver-mod (link-driver tabfile)))

    (discard)

    (module (any) (f toplevel eval load show-opt show-expand statistics)

      (use system-sig)
      (use driver-mod driver-sig driver)

      (set! toplevel (driver toplevel))
      (set! eval (driver eval))
      (set! load (driver load))
      (set! show-opt (driver show-opt))
      (set! show-expand (driver show-expand))
      (set! statistics (driver detailed-compile-statistics))

      (set-system-main-loop! (driver rts))

      (cond ((dump f)
	     =>
	     (lambda (argl)
	       ((driver toplevel) argl)
	       (quit)))
	    (else (quit))))))

(define (build-compiler f)

  (let* ((tabfile (or (getenv "PRIMVTAB")
		      (error "PRIMVTAB?")))
	 (driver-mod (link-driver tabfile)))

    (module (any) (f)

      (use scheme-sig)
      (use driver-mod driver-sig)

      (build-compiler f)
      (quit))))

(define (gen-boot-loader scm-file asm-file tabfile)

  (let ((driver-mod (link-driver tabfile)))

    (module (any) (scm-file asm-file)

      (use scheme-sig)
      (use driver-mod driver-sig)

      (gen-boot-loader scm-file asm-file)
      (quit))))
