/*
 * list-prim.c -- Implementation of Scheme's primitive list procedures
 *
 * (C) m.b (Matthias Blume); May 1992, HUB; Jan 1993 PU/CS
 *         Humboldt-University of Berlin
 *         Princeton University, Dept. of Computer Science
 *
 * $Id: list-prim.c,v 2.10 1994/11/12 22:21:19 blume Exp $
 */

# include "rcs.h"
RCSID ("$Id: list-prim.c,v 2.10 1994/11/12 22:21:19 blume Exp $")

# include "Cont.h"
# include "Code.h"
# include "Vector.h"
# include "Boolean.h"
# include "Cons.h"
# include "Numeric.h"
# include "storext.h"
# include "type.h"
# include "except.h"

# include "builtins.tab"

unsigned ScmPrimitiveAppend (unsigned argcnt)
{
  unsigned long len, run, nxt;
  unsigned i;
  ScmCons *cons;
  void *l, *e;

  if (argcnt == 0) {
    Push (&ScmNil);
    return 0;
  }
  if (argcnt == 1)
    return 0;

  len = 0;
  for (i = 0; i < argcnt - 1; i++)
    len += ScmListLength (POS (ScmCC->top - i - 1));
  if (len == 0) {
    while (argcnt-- > 1)
      (void) POP ();
    return 0;
  } else {
    SCM_ALLOC (cons, len * sizeof (ScmCons));
    run = 0;
    e = POS (ScmCC->top - argcnt);
    for (i = 0; i < argcnt - 1; i++) {
      l = POP ();
      while (ScmTypeOf (l) == ScmType (Cons)) {
	nxt = run + 1;
	cons [run]._ = ScmType (Cons);
	cons [run].car = ((ScmCons *) l)->car;
	l = ((ScmCons *) l)->cdr;
	cons [run].cdr = cons + nxt;
	run = nxt;
      }
    }
    cons [len - 1].cdr = e;
    SET_TOP (cons);
  }
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitivePairP (unsigned argcnt)
{
  void *tmp = POP ();
  Push (ScmTypeOf (tmp) == ScmType (Cons)
	  ? &ScmTrue
	  : &ScmFalse);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveCons (unsigned argcnt)
{
  ScmCons *cons;

  SCM_NEW (cons, Cons);
  cons->car = POP ();
  cons->cdr = PEEK ();
  SET_TOP (cons);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveCar (unsigned argcnt)
{
  void *vcons = PEEK ();

  if (ScmTypeOf (vcons) != ScmType (Cons))
    badarg ("car", vcons);
  SET_TOP (((ScmCons *) vcons)->car);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveCdr (unsigned argcnt)
{
  void *vcons = PEEK ();

  if (ScmTypeOf (vcons) != ScmType (Cons))
    badarg ("cdr", vcons);
  SET_TOP (((ScmCons *) vcons)->cdr);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveSetCar (unsigned argcnt)
{
  void *vcons;
  void *item;
  vcons = POP ();
  item = PEEK ();
  if (ScmTypeOf (vcons) != ScmType (Cons))
    badarg ("set-car!", vcons);
  ((ScmCons *) vcons)->car = item;
  SET_TOP (vcons);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveSetCdr (unsigned argcnt)
{
  void *vcons;
  void *item;
  vcons = POP ();
  item = PEEK ();
  if (ScmTypeOf (vcons) != ScmType (Cons))
    badarg ("set-cdr!", vcons);
  ((ScmCons *) vcons)->cdr = item;
  SET_TOP (vcons);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveNullP (unsigned argcnt)
{
  void *tmp = PEEK ();
  SET_TOP (tmp == &ScmNil ? &ScmTrue : &ScmFalse);
  return 0;
}

unsigned ScmPrimitiveList (unsigned argcnt)
{
  ScmCons *cons;
  unsigned i, j;

  if (argcnt == 0)
    Push (&ScmNil);
  else {
    SCM_ALLOC (cons, argcnt * sizeof (ScmCons));
    for (i = 0; i < argcnt; i = j) {
      j = i + 1;
      cons [i]._ = ScmType (Cons);
      cons [i].car = POP ();
      cons [i].cdr = cons + j;
    }
    cons [argcnt - 1].cdr = &ScmNil;
    PUSH (cons);		/* safe, since we've made at least one POP */
  }
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveReverse (unsigned argcnt)
{
  void *tmp;
  tmp = ScmReverseList (PEEK ());
  SET_TOP (tmp);
  return 0;
}

static
void *list_tail (void *list, unsigned long k)
{
  while (k > 0)
    if (ScmTypeOf (list) == ScmType (Cons)) {
      --k;
      list = ((ScmCons *) list)->cdr;
    } else {
      list = &ScmFalse;
      break;
    }
  return list;
}

/*ARGSUSED*/
unsigned ScmPrimitiveListTail (unsigned argcnt)
{
  void *list;
  unsigned long k;

  list = POP ();
  k = ScmNumberToULong (PEEK (), "list-tail");
  list = list_tail (list, k);
  SET_TOP (list);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveListRef (unsigned argcnt)
{
  void *list;
  unsigned long k;

  list = POP ();
  k = ScmNumberToULong (PEEK (), "list-ref");
  list = list_tail (list, k);
  if (ScmTypeOf (list) == ScmType (Cons))
    list = ((ScmCons *) list)->car;
  else
    list = &ScmFalse;
  SET_TOP (list);
  return 0;
}

static
unsigned search (int (* pred) (void *, void *), int like_member)
{
  void *obj;
  void *list;
  void *res;

  obj = POP ();
  list = PEEK ();
  res = &ScmFalse;
  while (ScmTypeOf (list) == ScmType (Cons))
    if ((* pred) (obj, ((ScmCons *) list)->car)) {
      res = like_member ? list : ((ScmCons *) list)->car;
      break;
    } else
      list = ((ScmCons *) list)->cdr;
  SET_TOP (res);
  return 0;
}

static
int eq_pred (void *x, void *y)
{
  return x == y;
}

static
void ensure_ass_pair (void *x)
{
  if (ScmTypeOf (x) != ScmType (Cons))
    error ("impromper a-list to assq/assv/assoc");
}

static
int assq_pred (void *x, void *y)
{
  ensure_ass_pair (y);
  return x == ((ScmCons *) y)->car;
}

static
int assv_pred (void *x, void *y)
{
  ensure_ass_pair (y);
  return eqv_object (x, ((ScmCons *) y)->car);
}

static
int assoc_pred (void *x, void *y)
{
  ensure_ass_pair (y);
  return equal_object (x, ((ScmCons *) y)->car);
}

/*ARGSUSED*/
unsigned ScmPrimitiveMemq (unsigned argcnt)
{
  return search (eq_pred, 1);
}

/*ARGSUSED*/
unsigned ScmPrimitiveMemv (unsigned argcnt)
{
  return search (eqv_object, 1);
}

/*ARGSUSED*/
unsigned ScmPrimitiveMember (unsigned argcnt)
{
  return search (equal_object, 1);
}

/*ARGSUSED*/
unsigned ScmPrimitiveAssq (unsigned argcnt)
{
  return search (assq_pred, 0);
}

/*ARGSUSED*/
unsigned ScmPrimitiveAssv (unsigned argcnt)
{
  return search (assv_pred, 0);
}

/*ARGSUSED*/
unsigned ScmPrimitiveAssoc (unsigned argcnt)
{
  return search (assoc_pred, 0);
}

/*ARGSUSED*/
unsigned ScmPrimitiveLength (unsigned argcnt)
{
  void *tmp = PEEK ();
  tmp = ScmLongToNumber (ScmListLength (tmp));
  SET_TOP (tmp);
  return 0;
}

/*
 * Credit for the following linear-time version of the list? predicate
 * to Henry Cejtin (NEC)...
 */

/*ARGSUSED*/
unsigned ScmPrimitiveListP (unsigned argcnt)
{
  ScmCons	*slow, *fast;

  slow = fast = PEEK ();
  for (;;) {
    if (ScmTypeOf (fast) != ScmType (Cons))
      break;
    fast = fast->cdr;
    if (ScmTypeOf (fast) != ScmType (Cons))
      break;
    fast = fast->cdr;
    slow = slow->cdr;
    if (fast == slow) {
      SET_TOP (&ScmFalse);
      return 0;
    }
  }
  if ((void *)fast == (void *)&ScmNil)
    SET_TOP (&ScmTrue);
  else
    SET_TOP (&ScmFalse);
  return 0;
}
