/*
 * Copyright(c) 1992 Bell Communications Research, Inc. (Bellcore)
 * Copyright(c) 1995-99 Andrew Lister
 * Copyright  1999, 2000, 2001, 2002, 2003, 2004 by the LessTif Developers.
 *
 *                        All rights reserved
 * Permission to use, copy, modify and distribute this material for
 * any purpose and without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies, and that the name of Bellcore not be used in advertising
 * or publicity pertaining to this material without the specific,
 * prior written permission of an authorized representative of
 * Bellcore.
 *
 * BELLCORE MAKES NO REPRESENTATIONS AND EXTENDS NO WARRANTIES, EX-
 * PRESS OR IMPLIED, WITH RESPECT TO THE SOFTWARE, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR ANY PARTICULAR PURPOSE, AND THE WARRANTY AGAINST IN-
 * FRINGEMENT OF PATENTS OR OTHER INTELLECTUAL PROPERTY RIGHTS.  THE
 * SOFTWARE IS PROVIDED "AS IS", AND IN NO EVENT SHALL BELLCORE OR
 * ANY OF ITS AFFILIATES BE LIABLE FOR ANY DAMAGES, INCLUDING ANY
 * LOST PROFITS OR OTHER INCIDENTAL OR CONSEQUENTIAL DAMAGES RELAT-
 * ING TO THE SOFTWARE.
 *
 * MatrixWidget Author: Andrew Wason, Bellcore, aw@bae.bellcore.com
 *
 * $Id: Create.c,v 1.58 2004/12/07 23:55:55 tobiasoed Exp $
 */

/*
 * Create.c created by Andrew Lister (28 Jan, 1996)
 */

#ifdef HAVE_CONFIG_H
#include <XbaeConfig.h>
#endif

#include <stdio.h>
#include <stdlib.h>

#include <Xm/Xm.h>
#include <Xm/ScrollBar.h>

#include <Xbae/MatrixP.h>
#include <Xbae/Macros.h>
#include <Xbae/Utils.h>
#include <Xbae/Actions.h>
#include <Xbae/Create.h>

#include <XbaeDebug.h>

#if (XmVERSION < 2)
#define	XmUNSPECIFIED_PIXEL	((Pixel) (~0))
#endif

static Pixmap createInsensitivePixmap(XbaeMatrixWidget mw);

void xbaeCopyBackground(Widget widget, int offset, XrmValue * value)
{
        value->addr = (XtPointer) & (widget->core.background_pixel);
}


void xbaeCopyForeground(Widget widget, int offset, XrmValue * value)
{
        value->addr = (XtPointer) & (((XmManagerWidget) widget)->manager.foreground);
}

void xbaeCopyDoubleClick(Widget widget, int offset, XrmValue * value)
{
        static int interval;

        interval = XtGetMultiClickTime(XtDisplay(widget));
        value->addr = (XtPointer) & interval;
}

void xbaeCopyRowShadowTypes(XbaeMatrixWidget mw)
{
        unsigned char *copy = NULL;
        int i;
        Boolean bad = False;

        xbaeObjectLock((Widget) mw);

        if (mw->matrix.rows) {
                copy = (unsigned char *) XtMalloc(mw->matrix.rows * sizeof(unsigned char));

                for (i = 0; i < mw->matrix.rows; i++) {
                        if (!bad && mw->matrix.row_shadow_types[i] == BAD_SHADOW) {
                                bad = True;
                                XtAppWarningMsg(XtWidgetToApplicationContext((Widget) mw),
                                                "xbaeCopyRowShadowTypes", "tooShort", "XbaeMatrix",
                                                "XbaeMatrix: rowShadowTypes array is too short",
                                                NULL, 0);
                                copy[i] = 0;
                        } else if (bad) {
                                copy[i] = 0;
                        } else {
                                copy[i] = mw->matrix.row_shadow_types[i];
                        }
                }
        }
        mw->matrix.row_shadow_types = copy;
        xbaeObjectUnlock((Widget) mw);
}

void xbaeCopyColumnShadowTypes(XbaeMatrixWidget mw)
{
        unsigned char *copy = NULL;
        int i;
        Boolean bad = False;

        xbaeObjectLock((Widget) mw);

        if (mw->matrix.columns) {
                copy = (unsigned char *) XtMalloc(mw->matrix.columns * sizeof(unsigned char));

                for (i = 0; i < mw->matrix.columns; i++) {
                        if (!bad && mw->matrix.column_shadow_types[i] == BAD_SHADOW) {
                                bad = True;
                                XtAppWarningMsg(XtWidgetToApplicationContext((Widget) mw),
                                                "xbaeCopyColumnShadowTypes", "tooShort", "XbaeMatrix",
                                                "XbaeMatrix: columnShadowTypes array is too short",
                                                NULL, 0);
                                copy[i] = 0;
                        } else if (bad) {
                                copy[i] = 0;
                        } else {
                                copy[i] = mw->matrix.column_shadow_types[i];
                        }
                }
        }
        mw->matrix.column_shadow_types = copy;
        xbaeObjectUnlock((Widget) mw);
}

void xbaeCopyRowUserData(XbaeMatrixWidget mw)
{
        XtPointer *copy = NULL;
        int i;

        xbaeObjectLock((Widget) mw);

        if (mw->matrix.rows) {
                copy = (XtPointer *) XtMalloc(mw->matrix.rows * sizeof(XtPointer));

                for (i = 0; i < mw->matrix.rows; i++)
                        copy[i] = mw->matrix.row_user_data[i];
        }
        mw->matrix.row_user_data = copy;

        xbaeObjectUnlock((Widget) mw);
}

void xbaeCopyColumnUserData(XbaeMatrixWidget mw)
{
        XtPointer *copy = NULL;
        int i;

        xbaeObjectLock((Widget) mw);

        if (mw->matrix.columns) {
                copy = (XtPointer *) XtMalloc(mw->matrix.columns * sizeof(XtPointer));

                for (i = 0; i < mw->matrix.columns; i++)
                        copy[i] = mw->matrix.column_user_data[i];
        }
        mw->matrix.column_user_data = copy;

        xbaeObjectUnlock((Widget) mw);
}

void xbaeCopyRowLabels(XbaeMatrixWidget mw)
{
        String *copy = NULL;
        int i;
        Boolean bad = False;

        xbaeObjectLock((Widget) mw);

        if (mw->matrix.rows && mw->matrix.row_labels) {
                copy = (String *) XtMalloc(mw->matrix.rows * sizeof(String));

                for (i = 0; i < mw->matrix.rows; i++) {
                        if (!bad && mw->matrix.row_labels[i] == &xbaeBadString) {
                                bad = True;
                                XtAppWarningMsg(XtWidgetToApplicationContext((Widget) mw),
                                                "copyRowLabels", "tooShort", "XbaeMatrix",
                                                "XbaeMatrix: Row labels array is too short",
                                                NULL, 0);
                                copy[i] = NULL;
                        } else if (bad) {
                                copy[i] = NULL;
                        } else {
                                copy[i] = (mw->matrix.row_labels[i] == NULL) ? NULL : XtNewString(mw->matrix.row_labels[i]);
                        }
                }
        }
        mw->matrix.row_labels = copy;

        xbaeObjectUnlock((Widget) mw);
}

void xbaeCopyColumnLabels(XbaeMatrixWidget mw)
{
        String *copy = NULL;
        XmString *xmcopy = NULL;
        int i;
        Boolean bad = False;

        xbaeObjectLock((Widget) mw);

        if (mw->matrix.columns && mw->matrix.column_labels) {
                copy = (String *) XtMalloc(mw->matrix.columns * sizeof(String));

                mw->matrix.column_label_lines =
                    (ColumnLabelLines) XtMalloc(mw->matrix.columns * sizeof(ColumnLabelLinesRec));

                for (i = 0; i < mw->matrix.columns; i++) {
                        if (!bad && mw->matrix.column_labels[i] == &xbaeBadString) {
                                bad = True;
                                XtAppWarningMsg(XtWidgetToApplicationContext((Widget) mw),
                                                "copyColumnLabels", "tooShort", "XbaeMatrix",
                                                "XbaeMatrix: Column labels array is too short",
                                                NULL, 0);
                                copy[i] = NULL;
                        } else if (bad) {
                                copy[i] = NULL;
                        } else {
                                copy[i] = (mw->matrix.column_labels[i] == NULL) ? NULL : XtNewString(mw->matrix.column_labels[i]);
                        }
                        xbaeParseColumnLabel(copy[i], &mw->matrix.column_label_lines[i]);
                }
                
                /*
                 * Determine max number of lines in column labels
                 */
                mw->matrix.column_label_maxlines = xbaeCalculateLabelMaxLines(mw->matrix.column_label_lines,
                                                                              mw->matrix.columns);
        }
        mw->matrix.column_labels = copy;

        /* Straight through for xmcolumn_label */
        if (mw->matrix.columns && mw->matrix.xmcolumn_labels) {
                xmcopy = (XmString *) XtMalloc(mw->matrix.columns * sizeof(XmString));
                for (i = 0; i < mw->matrix.columns; i++) {
                        xmcopy[i] = mw->matrix.xmcolumn_labels[i] == NULL ? NULL : XmStringCopy(mw->matrix.xmcolumn_labels[i]);
                }
        }
        mw->matrix.xmcolumn_labels = xmcopy;
        
        xbaeObjectUnlock((Widget) mw);
}

void xbaeCopyColumnWidths(XbaeMatrixWidget mw)
{
        short *copy = NULL;
        int i;
        Boolean bad = False;

        xbaeObjectLock((Widget) mw);

        if (mw->matrix.columns) {
                copy = (short *) XtMalloc(mw->matrix.columns * sizeof(short));

                for (i = 0; i < mw->matrix.columns; i++) {
                        if (!bad && mw->matrix.column_widths[i] == BAD_WIDTH) {
                                bad = True;
                                XtAppWarningMsg(XtWidgetToApplicationContext((Widget) mw),
                                                "copyColumnWidths", "tooShort", "XbaeMatrix",
                                                "XbaeMatrix: Column widths array is too short",
                                                NULL, 0);
                                copy[i] = 1;
                        } else if (bad)
                                copy[i] = 1;
                        else
                                copy[i] = mw->matrix.column_widths[i];
                }
        }
        mw->matrix.column_widths = copy;

        xbaeObjectUnlock((Widget) mw);
}

void xbaeCopyRowHeights(XbaeMatrixWidget mw)
{
        short *copy = NULL;
        int i;
        Boolean bad = False;

        xbaeObjectLock((Widget) mw);

        if (mw->matrix.rows) {
                copy = (short *) XtMalloc(mw->matrix.rows * sizeof(short));

                for (i = 0; i < mw->matrix.rows; i++) {
                        if (!bad && mw->matrix.row_heights[i] == BAD_HEIGHT) {
                                bad = True;
                                XtAppWarningMsg(XtWidgetToApplicationContext((Widget) mw),
                                                "copyRowHeights", "tooShort", "XbaeMatrix",
                                                "XbaeMatrix: Row heights array is too short", NULL,
                                                0);
                                copy[i] = 1;
                        } else if (bad)
                                copy[i] = 1;
                        else
                                copy[i] = mw->matrix.row_heights[i];
                }
        }
        DEBUGOUT(_XbaeDebug
                 (__FILE__, (Widget) mw, "xbaeCopyRowHeights %p (old %p)\n", copy,
                  mw->matrix.row_heights));
        mw->matrix.row_heights = copy;

        xbaeObjectUnlock((Widget) mw);
}

void xbaeCopyColumnMaxLengths(XbaeMatrixWidget mw)
{
        int *copy = NULL;
        int i;
        Boolean bad = False;

        xbaeObjectLock((Widget) mw);

        if (mw->matrix.columns) {
                copy = (int *) XtMalloc(mw->matrix.columns * sizeof(int));

                for (i = 0; i < mw->matrix.columns; i++) {
                        if (!bad && mw->matrix.column_max_lengths[i] == BAD_MAXLENGTH) {
                                bad = True;
                                XtAppWarningMsg(XtWidgetToApplicationContext((Widget) mw),
                                                "copyColumnMaxLengths", "tooShort", "XbaeMatrix",
                                                "XbaeMatrix: Column max lengths array is too short",
                                                NULL, 0);
                                copy[i] = 0;
                        } else if (bad)
                                copy[i] = 0;
                        else
                                copy[i] = mw->matrix.column_max_lengths[i];
                }
        }
        mw->matrix.column_max_lengths = copy;

        xbaeObjectUnlock((Widget) mw);
}

void xbaeCopyColumnAlignments(XbaeMatrixWidget mw)
{
        unsigned char *copy = NULL;
        int i;
        Boolean bad = False;

        xbaeObjectLock((Widget) mw);

        if (mw->matrix.columns) {
                copy = (unsigned char *) XtMalloc(mw->matrix.columns * sizeof(unsigned char));

                for (i = 0; i < mw->matrix.columns; i++) {
                        if (!bad && mw->matrix.column_alignments[i] == BAD_ALIGNMENT) {
                                bad = True;
                                XtAppWarningMsg(XtWidgetToApplicationContext((Widget) mw),
                                                "copyColumnAlignments", "tooShort", "XbaeMatrix",
                                                "XbaeMatrix: Column alignments array is too short",
                                                NULL, 0);
                                copy[i] = XmALIGNMENT_BEGINNING;
                        } else if (bad)
                                copy[i] = XmALIGNMENT_BEGINNING;
                        else
                                copy[i] = mw->matrix.column_alignments[i];
                }
        }
        mw->matrix.column_alignments = copy;

        xbaeObjectUnlock((Widget) mw);
}

void xbaeCopyColumnButtonLabels(XbaeMatrixWidget mw)
{
        Boolean *copy = NULL;
        int i;

        xbaeObjectLock((Widget) mw);

        if (mw->matrix.columns) {
                copy = (Boolean *) XtMalloc(mw->matrix.columns * sizeof(Boolean));

                for (i = 0; i < mw->matrix.columns; i++) {
                        copy[i] = mw->matrix.column_button_labels[i];
                }
        }
        mw->matrix.column_button_labels = copy;

        xbaeObjectUnlock((Widget) mw);
}

void xbaeCopyShowColumnArrows(XbaeMatrixWidget mw)
{
        Boolean *copy = NULL;
        int i;

        xbaeObjectLock((Widget) mw);

        if (mw->matrix.columns) {
                copy = (Boolean *) XtMalloc(mw->matrix.columns * sizeof(Boolean));

                for (i = 0; i < mw->matrix.columns; i++) {
                        copy[i] = mw->matrix.show_column_arrows[i];
                }
        }
        mw->matrix.show_column_arrows = copy;

        xbaeObjectUnlock((Widget) mw);
}


void xbaeCopyColumnFontBold(XbaeMatrixWidget mw)
{
        Boolean *copy = NULL;
        int i;

        xbaeObjectLock((Widget) mw);

        if (mw->matrix.columns) {
                copy = (Boolean *) XtMalloc(mw->matrix.columns * sizeof(Boolean));

                for (i = 0; i < mw->matrix.columns; i++) {
                        copy[i] = mw->matrix.column_font_bold[i];
                }
        }
        mw->matrix.column_font_bold = copy;

        xbaeObjectUnlock((Widget) mw);
}

void xbaeCopyRowButtonLabels(XbaeMatrixWidget mw)
{
        Boolean *copy = NULL;
        int i;

        xbaeObjectLock((Widget) mw);

        if (mw->matrix.rows) {
                copy = (Boolean *) XtMalloc(mw->matrix.rows * sizeof(Boolean));

                for (i = 0; i < mw->matrix.rows; i++) {
                        copy[i] = mw->matrix.row_button_labels[i];
                }
        }
        mw->matrix.row_button_labels = copy;

        xbaeObjectUnlock((Widget) mw);
}

void xbaeCopyColumnLabelAlignments(XbaeMatrixWidget mw)
{
        unsigned char *copy = NULL;
        int i;
        Boolean bad = False;

        xbaeObjectLock((Widget) mw);

        if (mw->matrix.columns) {
                copy = (unsigned char *) XtMalloc(mw->matrix.columns * sizeof(unsigned char));

                for (i = 0; i < mw->matrix.columns; i++) {
                        if (!bad && mw->matrix.column_label_alignments[i] == BAD_ALIGNMENT) {
                                bad = True;
                                XtAppWarningMsg(XtWidgetToApplicationContext((Widget) mw),
                                                "copyColumnLabelAlignments", "tooShort", "XbaeMatrix",
                                                "XbaeMatrix: Column label alignments array is too short",
                                                NULL, 0);
                                copy[i] = XmALIGNMENT_BEGINNING;
                        } else if (bad)
                                copy[i] = XmALIGNMENT_BEGINNING;
                        else
                                copy[i] = mw->matrix.column_label_alignments[i];
                }
        }
        mw->matrix.column_label_alignments = copy;

        xbaeObjectUnlock((Widget) mw);
}

/*
 * A cache for pixmaps which remembers the screen a pixmap belongs to
 * (see bug #591306) and that permits clearing by screen when that
 * screen gets closed.
 */
static struct pcache {
        Pixmap pixmap;
        Screen *scr;

} *stipple_cache = NULL;
static int ncache = 0;          /* Allocated size of the array. */
                                /* Empty places have NULL screen */

static Pixmap PixmapFromCache(Screen * scr)
{
        int i;

        for (i = 0; i < ncache; i++)
                if (scr == stipple_cache[i].scr)
                        return stipple_cache[i].pixmap;
        return (Pixmap) 0;
}

static void AddPixmapToCache(Screen * scr, Pixmap p)
{
        int i, old;

        for (i = 0; i < ncache; i++)
                if (stipple_cache[i].scr == 0) {
                        stipple_cache[i].scr = scr;
                        stipple_cache[i].pixmap = p;
                        return;
                }

        /* Allocate more */
        if (ncache) {
                old = ncache;
                ncache *= 2;
                stipple_cache =
                    (struct pcache *) XtRealloc((char *) stipple_cache,
                                                ncache * sizeof(struct pcache));
                for (i = old; i < ncache; i++)
                        stipple_cache[i].scr = NULL;
                stipple_cache[old].scr = scr;
                stipple_cache[old].pixmap = p;
        } else {
                ncache = 16;    /* Some silly initial value */
                stipple_cache = (struct pcache *) XtCalloc(ncache, sizeof(struct pcache));
                stipple_cache[0].scr = scr;
                stipple_cache[0].pixmap = p;
        }
}

/*
 * Remove the pixmaps with this screen from the cache
 */
static void RemovePixmapsFromScreen(Screen * scr)
{
        int i;
        for (i = 0; i < ncache; i++)
                if (stipple_cache[i].scr == scr) {
                        XFreePixmap(DisplayOfScreen(stipple_cache[i].scr), stipple_cache[i].pixmap);
                        stipple_cache[i].pixmap = (Pixmap) 0;
                        stipple_cache[i].scr = NULL;
                }
}

/*
 * Create a pixmap to be used for drawing the matrix contents when
 * XmNsensitive is set to False
 */
static Pixmap createInsensitivePixmap(XbaeMatrixWidget mw)
{
        static char stippleBits[] = { 0x01, 0x02 };
        Display *dpy = XtDisplay(mw);
        Screen *scr = XtScreen(mw);
        Pixmap p;

        xbaeObjectLock((Widget) mw);

        p = PixmapFromCache(XtScreen((Widget) mw));
        if (p) {
                xbaeObjectUnlock((Widget) mw);
                return p;
        }

        p = XCreatePixmapFromBitmapData(dpy, RootWindowOfScreen(scr), stippleBits, 2, 2, 0, 1, 1);
        AddPixmapToCache(scr, p);
        xbaeObjectUnlock((Widget) mw);
        return p;
}

void xbaeCreateDrawGC(XbaeMatrixWidget mw)
{
        XGCValues values;
        unsigned long mask = GCForeground | GCStipple;

        xbaeObjectLock((Widget) mw);

        /*
         * GC for drawing cells. We create it instead of using a cached one,
         * since the foreground may change frequently.
         */
        values.foreground = mw->manager.foreground;
        values.stipple = createInsensitivePixmap(mw);
        /* Font id isn't used for fontsets */
        if (mw->matrix.font_struct) {
                mask |= GCFont;
                values.font = mw->matrix.fid;
        }

        DEBUGOUT(_XbaeDebug(__FILE__, (Widget) mw,
                            "xbaeCreateDrawGC(dpy %p win %p fg %d font %p stip %p)\n",
                            XtDisplay(mw), GC_PARENT_WINDOW(mw),
                            values.foreground, values.font, values.stipple));

        mw->matrix.draw_gc = XCreateGC(XtDisplay(mw), GC_PARENT_WINDOW(mw), mask, &values);

        xbaeObjectUnlock((Widget) mw);
}

void xbaeCreatePixmapGC(XbaeMatrixWidget mw)
{
        XGCValues values;
        unsigned long mask = GCForeground | GCGraphicsExposures | GCStipple;

        xbaeObjectLock((Widget) mw);

        values.foreground = mw->manager.foreground;
        values.graphics_exposures = False;
        values.stipple = createInsensitivePixmap(mw);

        mw->matrix.pixmap_gc = XCreateGC(XtDisplay(mw), GC_PARENT_WINDOW(mw), mask, &values);

        xbaeObjectUnlock((Widget) mw);
}

void xbaeCreateLabelGC(XbaeMatrixWidget mw)
{
        XGCValues values;
        unsigned long mask = GCForeground | GCStipple;

        xbaeObjectLock((Widget) mw);

        /*
         * GC for drawing labels
         */
        values.foreground = mw->manager.foreground;
        values.stipple = createInsensitivePixmap(mw);
        /* Font id isn't used for fontsets */
        if (mw->matrix.label_font_struct) {
                mask |= GCFont;
                values.font = mw->matrix.label_fid;
        }
        mw->matrix.label_gc = XCreateGC(XtDisplay(mw), GC_PARENT_WINDOW(mw), mask, &values);

        xbaeObjectUnlock((Widget) mw);
}

void xbaeGetGridLineGC(XbaeMatrixWidget mw)
{
        XGCValues values;
        XtGCMask mask = GCForeground | GCBackground;

        xbaeObjectLock((Widget) mw);

        /*
         * GC for drawing grid lines
         */
        values.foreground = mw->matrix.grid_line_color;
        values.background = mw->manager.foreground;

        /* Release the GC before getting another one */
        if (mw->matrix.grid_line_gc)
                XtReleaseGC((Widget) mw, mw->matrix.grid_line_gc);

        mw->matrix.grid_line_gc = XtGetGC((Widget) mw, mask, &values);

        xbaeObjectUnlock((Widget) mw);
}

void xbaeGetResizeTopShadowGC(XbaeMatrixWidget mw)
{
        XGCValues values;
        XtGCMask mask = GCForeground | GCBackground | GCFunction;

        xbaeObjectLock((Widget) mw);

        /*
         * GC for drawing the top shadow when resizing
         */
        values.foreground = mw->manager.top_shadow_color;
        values.background = mw->manager.foreground;
        values.function = GXxor;

        if (mw->manager.top_shadow_pixmap != XmUNSPECIFIED_PIXMAP) {
                mask |= GCFillStyle | GCTile;
                values.fill_style = FillTiled;
                values.tile = mw->manager.top_shadow_pixmap;
        }

        /* Release the GC before getting another one */
        if (mw->matrix.resize_top_shadow_gc)
                XtReleaseGC((Widget) mw, mw->matrix.resize_top_shadow_gc);

        mw->matrix.resize_top_shadow_gc = XtGetGC((Widget) mw, mask, &values);

        xbaeObjectUnlock((Widget) mw);
}

void xbaeGetResizeBottomShadowGC(XbaeMatrixWidget mw)
{
        XGCValues values;
        XtGCMask mask = GCForeground | GCBackground | GCFunction;

        xbaeObjectLock((Widget) mw);

        /*
         * GC for drawing the bottom shadow when resizing
         */
        values.foreground = mw->manager.bottom_shadow_color;
        values.background = mw->manager.foreground;
        values.function = GXxor;

        if (mw->manager.bottom_shadow_pixmap != XmUNSPECIFIED_PIXMAP) {
                mask |= GCFillStyle | GCTile;
                values.fill_style = FillTiled;
                values.tile = mw->manager.bottom_shadow_pixmap;
        }

        /* Release the GC before getting another one */
        if (mw->matrix.resize_bottom_shadow_gc)
                XtReleaseGC((Widget) mw, mw->matrix.resize_bottom_shadow_gc);

        mw->matrix.resize_bottom_shadow_gc = XtGetGC((Widget) mw, mask, &values);

        xbaeObjectUnlock((Widget) mw);
}

static short xbaeGetFontWidth(font_struct)
XFontStruct *font_struct;
{
        short width;
        unsigned long fp;
        unsigned char char_idx;

        /*
         *  From the XmText man page: If the em-space value is
         *  available, it is used. If not, the width of the  numeral  "0"
         *  is used. If this is not available, the maximum width is used.
         */
        if (XGetFontProperty(font_struct, XA_QUAD_WIDTH, &fp) && fp != 0) {
                width = (short) fp;
        } else {
                if (font_struct->min_char_or_byte2 <= '0' &&
                    font_struct->max_char_or_byte2 >= '0' && font_struct->per_char) {
                        char_idx = '0' - font_struct->min_char_or_byte2;
                        width = font_struct->per_char[char_idx].width;
                } else {
                        width = font_struct->max_bounds.width;
                }
        }

        /* last safety check */
        if (width <= 0) {
                width = 1;
        }

        return width;
}

void xbaeNewFont(XbaeMatrixWidget mw, int frominit)
{
        XmFontContext context;
        XmFontListEntry font_list_entry;
        XmFontType type;
        XtPointer fontp;

        xbaeObjectLock((Widget) mw);

        /*
         * Make a private copy of the FontList
         */
        if (!frominit)
                mw->matrix.font_list = XmFontListCopy(mw->matrix.font_list);

        /*
         * Get XmFontListEntry from FontList
         */
        if (!XmFontListInitFontContext(&context, mw->matrix.font_list))
                XtAppErrorMsg(XtWidgetToApplicationContext((Widget) mw),
                              "newFont", "badFont", "XbaeMatrix",
                              "XbaeMatrix: XmFontListInitFontContext failed, bad fontList",
                              NULL, 0);

        if ((font_list_entry = XmFontListNextEntry(context)) == NULL)
                XtAppErrorMsg(XtWidgetToApplicationContext((Widget) mw),
                              "newFont", "badFont", "XbaeMatrix",
                              "XbaeMatrix: XmFontListNextEntry failed, no next fontList", NULL, 0);

        fontp = XmFontListEntryGetFont(font_list_entry, &type);

        if (type == XmFONT_IS_FONTSET) {
                XFontSetExtents *extents;

                mw->matrix.font_set = (XFontSet) fontp;
                mw->matrix.font_struct = (XFontStruct *) NULL;

                extents = XExtentsOfFontSet(mw->matrix.font_set);
                mw->matrix.font_width = extents->max_logical_extent.width;
                mw->matrix.font_height = extents->max_logical_extent.height;
                mw->matrix.font_y = extents->max_logical_extent.y;
                mw->matrix.fid = 0;     /* not used for fontsets */
        } else {
                XFontStruct *font_struct = (XFontStruct *) fontp;

                mw->matrix.font_set = (XFontSet) NULL;
                mw->matrix.font_struct = font_struct;

                mw->matrix.font_width = xbaeGetFontWidth(font_struct);
                mw->matrix.font_height = (font_struct->max_bounds.descent +
                                          font_struct->max_bounds.ascent);
                mw->matrix.font_y = -font_struct->max_bounds.ascent;
                mw->matrix.fid = font_struct->fid;
        }

        XmFontListFreeFontContext(context);

        xbaeObjectUnlock((Widget) mw);
}

void xbaeNewLabelFont(XbaeMatrixWidget mw)
{
        XmFontContext context;
        XmFontListEntry font_list_entry;
        XmFontType type;
        XtPointer fontp;

        xbaeObjectLock((Widget) mw);

        /*
         * Make a private copy of the FontList
         */
        mw->matrix.label_font_list = XmFontListCopy(mw->matrix.label_font_list);

        /*
         * Get XmFontListEntry from FontList
         */
        if (!XmFontListInitFontContext(&context, mw->matrix.label_font_list))
                XtAppErrorMsg(XtWidgetToApplicationContext((Widget) mw), "newFont", "badLabelFont",
                              "XbaeMatrix",
                              "XbaeMatrix: XmFontListInitFontContext failed, bad labelFontList",
                              NULL, 0);

        if ((font_list_entry = XmFontListNextEntry(context)) == NULL)
                XtAppErrorMsg(XtWidgetToApplicationContext((Widget) mw), "newFont", "badLabelFont",
                              "XbaeMatrix",
                              "XbaeMatrix: XmFontListNextEntry failed, no next fontList", NULL, 0);

        fontp = XmFontListEntryGetFont(font_list_entry, &type);

        if (type == XmFONT_IS_FONTSET) {
                XFontSetExtents *extents;

                mw->matrix.label_font_set = (XFontSet) fontp;
                mw->matrix.label_font_struct = (XFontStruct *) NULL;

                extents = XExtentsOfFontSet(mw->matrix.label_font_set);
                mw->matrix.label_font_width = extents->max_logical_extent.width;
                mw->matrix.label_font_height = extents->max_logical_extent.height;
                mw->matrix.label_font_y = extents->max_logical_extent.y;
                mw->matrix.label_fid = 0;       /* not used for fontsets */
        } else {
                XFontStruct *font_struct = (XFontStruct *) fontp;

                mw->matrix.label_font_set = (XFontSet) NULL;
                mw->matrix.label_font_struct = font_struct;

                mw->matrix.label_font_width = xbaeGetFontWidth(font_struct);
                mw->matrix.label_font_height = (font_struct->max_bounds.descent
                                                + font_struct->max_bounds.ascent);
                mw->matrix.label_font_y = -font_struct->max_bounds.ascent;
                mw->matrix.label_fid = font_struct->fid;
        }

        XmFontListFreeFontContext(context);

        xbaeObjectUnlock((Widget) mw);
}



void xbaeFreeRowLabels(XbaeMatrixWidget mw)
{
        int i;

        if (mw->matrix.row_labels) {
                xbaeObjectLock((Widget) mw);

                for (i = 0; i < mw->matrix.rows; i++) {
                        if (mw->matrix.row_labels[i]) {
                                XtFree((char *) mw->matrix.row_labels[i]);
                        }
                }

                XtFree((char *) mw->matrix.row_labels);
                mw->matrix.row_labels = NULL;

                xbaeObjectUnlock((Widget) mw);
        }
}

void xbaeFreeColumnLabels(XbaeMatrixWidget mw)
{
        int i;

        xbaeObjectLock((Widget) mw);
        if (mw->matrix.column_labels) {

                for (i = 0; i < mw->matrix.columns; i++) {
                        if (mw->matrix.column_labels[i]) {
                                XtFree((char *) mw->matrix.column_labels[i]);
                                XtFree((char *) mw->matrix.column_label_lines[i].lengths);
                        }
                }

                XtFree((char *) mw->matrix.column_label_lines);
                XtFree((char *) mw->matrix.column_labels);
                mw->matrix.column_labels = NULL;
                mw->matrix.column_label_lines = NULL;
        }
        
        if (mw->matrix.xmcolumn_labels) {
                for (i = 0; i < mw->matrix.columns; i++) {
                        if (mw->matrix.xmcolumn_labels[i]) {
                                XmStringFree(mw->matrix.xmcolumn_labels[i]);
                        }
                }
                mw->matrix.xmcolumn_labels = NULL;
        }
        xbaeObjectUnlock((Widget) mw);
}

/*
 * Make sure to know when our display connection dies.
 */
static void DisplayDied(Widget w, XtPointer client, XtPointer call)
{
        XtDestroyHookDataRec *p = (XtDestroyHookDataRec *) call;

        if (p == NULL || p->type != XtHdestroy)
                return;

        if (XtIsSubclass(p->widget, xmPrimitiveWidgetClass))
                RemovePixmapsFromScreen(XtScreen(p->widget));
}

void xbaeRegisterDisplay(Widget w)
{
        Display *d = XtDisplay(w);
        XtAddCallback(XtHooksOfDisplay(d), XtNdestroyHook, DisplayDied, NULL);
}

/*
 * ARCAD SYSTEMHAUS
 */
void xbaeFill_WithEmptyValues_PerCell(XbaeMatrixWidget mw, struct _XbaeMatrixPerCellRec *p)
{
	p->shadow_type = 0; /* 0 means to use matrix.cell_shadow_type */
	p->highlighted = HighlightNone;
	p->selected = False;
	p->underlined = False;
	p->user_data = NULL;
	p->background = XmUNSPECIFIED_PIXEL;
	p->color = XmUNSPECIFIED_PIXEL;
	p->widget = NULL;
	p->font_list = NULL;
	p->pixmap = XmUNSPECIFIED_PIXMAP;
	p->mask = XmUNSPECIFIED_PIXMAP;
	p->CellValue = NULL;
}

void xbaeCreatePerCell(XbaeMatrixWidget mw)
{
        /*
         * Create with empty values
         */
        struct _XbaeMatrixPerCellRec **copy = NULL;
        int i, j;

	DEBUGOUT(_XbaeDebug(__FILE__, (Widget)mw, "xbaeCreatePerCell(%d, %d)\n",
				mw->matrix.rows, mw->matrix.columns));
        xbaeObjectLock((Widget) mw);

        /*
         * Malloc an array of row pointers
         */
        if (mw->matrix.rows && mw->matrix.columns) {
                copy =
                    (struct _XbaeMatrixPerCellRec **) XtCalloc((Cardinal) mw->matrix.rows,
                                                               sizeof(struct _XbaeMatrixPerCellRec
                                                                      *));
                for (i = 0; i < mw->matrix.rows; i++) {
                        copy[i] =
                            (struct _XbaeMatrixPerCellRec *) XtCalloc((Cardinal) mw->matrix.columns,
                                                                      sizeof(struct
                                                                             _XbaeMatrixPerCellRec));
                        for (j = 0; j < mw->matrix.columns; j++)
                                xbaeFill_WithEmptyValues_PerCell(mw, &copy[i][j]);
                }
        }
        mw->matrix.per_cell = copy;
        xbaeObjectUnlock((Widget) mw);
}

static void __FreePixmap(XbaeMatrixWidget mw, Pixmap * p)
{
        if (*p && (*p != XmUNSPECIFIED_PIXMAP)) {
                XFreePixmap(XtDisplay(mw), *p);
                *p = XmUNSPECIFIED_PIXMAP;
        }
}

static void __FreeString(String * p)
{
        if (*p != NULL) {
                XtFree(*p);
                *p = NULL;
        }
}


void xbaeFreePerCellEntity(XbaeMatrixWidget mw, int row, int column)
{
	DEBUGOUT(_XbaeDebug(__FILE__, (Widget)mw, "xbaeFreePerCellEntity(%d,%d)\n", row, column));
        /*
         * CellValue: must be freed, alloc in xbae!
         */
        __FreeString(&mw->matrix.per_cell[row][column].CellValue);

        /*
         * user_data: must not be freed, user-defined
         */

        /*
         * Pixmap: free it, alloc in xbae!
         */
        __FreePixmap(mw, &mw->matrix.per_cell[row][column].pixmap);

        /*
         * mask: free it, alloc in xbae!
         */
        __FreePixmap(mw, &mw->matrix.per_cell[row][column].mask);

}


void xbaeFreePerCellRow(XbaeMatrixWidget mw, int row)
{
        int j = 0;
        if (mw->matrix.per_cell[row]) {
                for (j = 0; j < mw->matrix.columns; j++)
                        xbaeFreePerCellEntity(mw, row, j);
                XtFree((char *) mw->matrix.per_cell[row]);
                mw->matrix.per_cell[row] = NULL;
        }
}

void xbaeFreePerCell(XbaeMatrixWidget mw)
{
        int i = 0;

        /*
         * Free each row of XtPointer pointers
         */
        if (mw->matrix.per_cell) {

                xbaeObjectLock((Widget) mw);

                for (i = 0; i < mw->matrix.rows; i++)
                        xbaeFreePerCellRow(mw, i);

                XtFree((char *) mw->matrix.per_cell);
                mw->matrix.per_cell = NULL;

                xbaeObjectUnlock((Widget) mw);
        }

}
