/* xdaliclock - a melting digital clock
 * Copyright (c) 1991-2009 Jamie Zawinski <jwz@jwz.org>
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *
 * This is a stripped-down version of OSX/digital.c for converting the font
 * pixmaps to a JavaScript representation of `struct raw_number'.
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <sys/time.h>

#include "xdaliclock.h"

typedef unsigned short POS;
typedef unsigned char BOOL;

#ifdef BUILTIN_FONTS

/* static int use_builtin_font; */

struct raw_number {
  const unsigned char *bits;
  POS width, height;
};

#endif /* BUILTIN_FONTS */

#ifdef BUILTIN_FONTS

# include "zeroB2.xbm"
# include "oneB2.xbm"
# include "twoB2.xbm"
# include "threeB2.xbm"
# include "fourB2.xbm"
# include "fiveB2.xbm"
# include "sixB2.xbm"
# include "sevenB2.xbm"
# include "eightB2.xbm"
# include "nineB2.xbm"
# include "colonB2.xbm"
# include "slashB2.xbm"

static struct raw_number numbers_B2 [] = {
  { zeroB2_bits,  zeroB2_width,  zeroB2_height  },
  { oneB2_bits,   oneB2_width,   oneB2_height   },
  { twoB2_bits,   twoB2_width,   twoB2_height   },
  { threeB2_bits, threeB2_width, threeB2_height },
  { fourB2_bits,  fourB2_width,  fourB2_height  },
  { fiveB2_bits,  fiveB2_width,  fiveB2_height  },
  { sixB2_bits,   sixB2_width,   sixB2_height   },
  { sevenB2_bits, sevenB2_width, sevenB2_height },
  { eightB2_bits, eightB2_width, eightB2_height },
  { nineB2_bits,  nineB2_width,  nineB2_height  },
  { colonB2_bits, colonB2_width, colonB2_height },
  { slashB2_bits, slashB2_width, slashB2_height },
  { 0, }
};

# include "zeroC2.xbm"
# include "oneC2.xbm"
# include "twoC2.xbm"
# include "threeC2.xbm"
# include "fourC2.xbm"
# include "fiveC2.xbm"
# include "sixC2.xbm"
# include "sevenC2.xbm"
# include "eightC2.xbm"
# include "nineC2.xbm"
# include "colonC2.xbm"
# include "slashC2.xbm"

static struct raw_number numbers_C2 [] = {
  { zeroC2_bits,  zeroC2_width,  zeroC2_height  },
  { oneC2_bits,   oneC2_width,   oneC2_height   },
  { twoC2_bits,   twoC2_width,   twoC2_height   },
  { threeC2_bits, threeC2_width, threeC2_height },
  { fourC2_bits,  fourC2_width,  fourC2_height  },
  { fiveC2_bits,  fiveC2_width,  fiveC2_height  },
  { sixC2_bits,   sixC2_width,   sixC2_height   },
  { sevenC2_bits, sevenC2_width, sevenC2_height },
  { eightC2_bits, eightC2_width, eightC2_height },
  { nineC2_bits,  nineC2_width,  nineC2_height  },
  { colonC2_bits, colonC2_width, colonC2_height },
  { slashC2_bits, slashC2_width, slashC2_height },
  { 0, }
};

# include "zeroC3.xbm"
# include "oneC3.xbm"
# include "twoC3.xbm"
# include "threeC3.xbm"
# include "fourC3.xbm"
# include "fiveC3.xbm"
# include "sixC3.xbm"
# include "sevenC3.xbm"
# include "eightC3.xbm"
# include "nineC3.xbm"
# include "colonC3.xbm"
# include "slashC3.xbm"

static struct raw_number numbers_C3 [] = {
  { zeroC3_bits,  zeroC3_width,  zeroC3_height  },
  { oneC3_bits,   oneC3_width,   oneC3_height   },
  { twoC3_bits,   twoC3_width,   twoC3_height   },
  { threeC3_bits, threeC3_width, threeC3_height },
  { fourC3_bits,  fourC3_width,  fourC3_height  },
  { fiveC3_bits,  fiveC3_width,  fiveC3_height  },
  { sixC3_bits,   sixC3_width,   sixC3_height   },
  { sevenC3_bits, sevenC3_width, sevenC3_height },
  { eightC3_bits, eightC3_width, eightC3_height },
  { nineC3_bits,  nineC3_width,  nineC3_height  },
  { colonC3_bits, colonC3_width, colonC3_height },
  { slashC3_bits, slashC3_width, slashC3_height },
  { 0, }
};

# include "zeroD3.xbm"
# include "oneD3.xbm"
# include "twoD3.xbm"
# include "threeD3.xbm"
# include "fourD3.xbm"
# include "fiveD3.xbm"
# include "sixD3.xbm"
# include "sevenD3.xbm"
# include "eightD3.xbm"
# include "nineD3.xbm"
# include "colonD3.xbm"
# include "slashD3.xbm"

static struct raw_number numbers_D3 [] = {
  { zeroD3_bits,  zeroD3_width,  zeroD3_height  },
  { oneD3_bits,   oneD3_width,   oneD3_height   },
  { twoD3_bits,   twoD3_width,   twoD3_height   },
  { threeD3_bits, threeD3_width, threeD3_height },
  { fourD3_bits,  fourD3_width,  fourD3_height  },
  { fiveD3_bits,  fiveD3_width,  fiveD3_height  },
  { sixD3_bits,   sixD3_width,   sixD3_height   },
  { sevenD3_bits, sevenD3_width, sevenD3_height },
  { eightD3_bits, eightD3_width, eightD3_height },
  { nineD3_bits,  nineD3_width,  nineD3_height  },
  { colonD3_bits, colonD3_width, colonD3_height },
  { slashD3_bits, slashD3_width, slashD3_height },
  { 0, }
};

# include "zeroD4.xbm"
# include "oneD4.xbm"
# include "twoD4.xbm"
# include "threeD4.xbm"
# include "fourD4.xbm"
# include "fiveD4.xbm"
# include "sixD4.xbm"
# include "sevenD4.xbm"
# include "eightD4.xbm"
# include "nineD4.xbm"
# include "colonD4.xbm"
# include "slashD4.xbm"

static struct raw_number numbers_D4 [] = {
  { zeroD4_bits,  zeroD4_width,  zeroD4_height  },
  { oneD4_bits,   oneD4_width,   oneD4_height   },
  { twoD4_bits,   twoD4_width,   twoD4_height   },
  { threeD4_bits, threeD4_width, threeD4_height },
  { fourD4_bits,  fourD4_width,  fourD4_height  },
  { fiveD4_bits,  fiveD4_width,  fiveD4_height  },
  { sixD4_bits,   sixD4_width,   sixD4_height   },
  { sevenD4_bits, sevenD4_width, sevenD4_height },
  { eightD4_bits, eightD4_width, eightD4_height },
  { nineD4_bits,  nineD4_width,  nineD4_height  },
  { colonD4_bits, colonD4_width, colonD4_height },
  { slashD4_bits, slashD4_width, slashD4_height },
  { 0, }
};

# include "zeroE.xbm"
# include "oneE.xbm"
# include "twoE.xbm"
# include "threeE.xbm"
# include "fourE.xbm"
# include "fiveE.xbm"
# include "sixE.xbm"
# include "sevenE.xbm"
# include "eightE.xbm"
# include "nineE.xbm"
# include "colonE.xbm"
# include "slashE.xbm"

static struct raw_number numbers_E [] = {
  { zeroE_bits,  zeroE_width,  zeroE_height  },
  { oneE_bits,   oneE_width,   oneE_height   },
  { twoE_bits,   twoE_width,   twoE_height   },
  { threeE_bits, threeE_width, threeE_height },
  { fourE_bits,  fourE_width,  fourE_height  },
  { fiveE_bits,  fiveE_width,  fiveE_height  },
  { sixE_bits,   sixE_width,   sixE_height   },
  { sevenE_bits, sevenE_width, sevenE_height },
  { eightE_bits, eightE_width, eightE_height },
  { nineE_bits,  nineE_width,  nineE_height  },
  { colonE_bits, colonE_width, colonE_height },
  { slashE_bits, slashE_width, slashE_height },
  { 0, }
};

#endif /* BUILTIN_FONTS */

#undef countof
#define countof(x) (sizeof((x))/sizeof(*(x)))

/* Number of horizontal segments/line.  Enlarge this if you are trying
   to use a font that is too "curvy" for XDaliClock to cope with.
   This code was sent to me by Dan Wallach <c169-bg@auriga.berkeley.edu>.
   I'm highly opposed to ever using statically-sized arrays, but I don't
   really feel like hacking on this code enough to clean it up.
 */
#ifndef MAX_SEGS_PER_LINE
# define MAX_SEGS_PER_LINE 5
#endif

struct scanline {
  POS left[MAX_SEGS_PER_LINE], right[MAX_SEGS_PER_LINE];
};

struct frame {
  struct scanline scanlines [1]; /* scanlines are contiguous here */
};


/* The runtime settings (some initialized from system prefs, but changable.)
 */
struct render_state {

  int char_width, char_height, colon_width;

  struct frame *base_frames [12];
};


static struct frame *
make_blank_frame (int width, int height)
{
  int size = sizeof (struct frame) + (sizeof (struct scanline) * (height - 1));
  struct frame *frame;
  int x, y;

  frame = (struct frame *) calloc (size, 1);
  for (y = 0; y < height; y++)
    for (x = 0; x < MAX_SEGS_PER_LINE; x++)
      frame->scanlines[y].left [x] = frame->scanlines[y].right [x] = width / 2;
  return frame;
}


static struct frame *
number_to_frame (const unsigned char *bits, int width, int height)
{
  int x, y;
  struct frame *frame;
  POS *left, *right;

  frame = make_blank_frame (width, height);

  for (y = 0; y < height; y++)
    {
      int seg, end;
      x = 0;
# define GETBIT(bits,x,y) \
         (!! ((bits) [((y) * ((width+7) >> 3)) + ((x) >> 3)] \
              & (1 << ((x) & 7))))

      left = frame->scanlines[y].left;
      right = frame->scanlines[y].right;

      for (seg = 0; seg < MAX_SEGS_PER_LINE; seg++)
        left [seg] = right [seg] = width / 2;

      for (seg = 0; seg < MAX_SEGS_PER_LINE; seg++)
        {
          for (; x < width; x++)
            if (GETBIT (bits, x, y)) break;
          if (x == width) break;
          left [seg] = x;
          for (; x < width; x++)
            if (! GETBIT (bits, x, y)) break;
          right [seg] = x;
        }

      for (; x < width; x++)
        if (GETBIT (bits, x, y))
          {
            /* This means the font is too curvy.  Increase MAX_SEGS_PER_LINE
               and recompile. */
            fprintf (stderr, "%s: font is too curvy\n", progname);
            exit (-1);
          }

      /* If there were any segments on this line, then replicate the last
         one out to the end of the line.  If it's blank, leave it alone,
         meaning it will be a 0-pixel-wide line down the middle.
       */
      end = seg;
      if (end > 0)
        for (; seg < MAX_SEGS_PER_LINE; seg++)
          {
            left [seg] = left [end-1];
            right [seg] = right [end-1];
          }

# undef GETBIT
    }

  return frame;
}



static void
init_numbers (dali_config *c, int size)
{
  struct render_state *state = c->render_state;
  int i;
#ifdef BUILTIN_FONTS
  struct raw_number *raw;

  switch (size)
    {
    case 0: raw = numbers_E;  break;
    case 1: raw = numbers_D4; break;
    case 2: raw = numbers_D3; break;
    case 3: raw = numbers_C3; break;
    case 4: raw = numbers_C2; break;
    case 5: raw = numbers_B2; break;
    default: abort(); break;
    }

  state->char_width  = raw[0].width;
  state->char_height = raw[0].height;
  state->colon_width = raw[10].width;

  for (i = 0; i < countof(state->base_frames); i++)
    state->base_frames [i] =
      number_to_frame (raw[i].bits, raw[i].width, raw[i].height);
#endif /* BUILTIN_FONTS */
}


static void
print_numbers (dali_config *c, int font)
{
  int i;
  struct render_state *state = c->render_state;
  int width  = state->char_width;
  int height = state->char_height;
  int cw     = state->colon_width;

  fprintf (stdout, "{\n");
  fprintf (stdout, " \"font_number\"   \t: %d,\n", font);
  fprintf (stdout, " \"char_width\"    \t: %d,\n", width);
  fprintf (stdout, " \"char_height\"   \t: %d,\n", height);
  fprintf (stdout, " \"colon_width\"   \t: %d,\n", cw);
  fprintf (stdout, " \"segments\":\n");
  fprintf (stdout, "      [\n");

  for (i = 0; i < countof(state->base_frames); i++)
    {
      struct frame *frame = state->base_frames[i];
      int x, y;
      fprintf (stdout, "       [\n");
      for (y = 0; y < height; y++)
        {
          int ol = -99999, or = -99999;
          fprintf (stdout, "\t[");
          for (x = 0; x < MAX_SEGS_PER_LINE; x++)
            {
              int left  = frame->scanlines[y].left  [x];
              int right = frame->scanlines[y].right [x];
              if (ol == left && or == right) continue;
              ol = left;
              or = right;
              if (x != 0) fprintf (stdout, ",");
              fprintf (stdout, "[%d,%d]", left, right);
            }
          fprintf (stdout, "],\n");
        }
      fprintf (stdout, "       ],\n");
    }
  fprintf (stdout, "      ]\n");
  fprintf (stdout, "}\n");
}


char *progname;

int
main (int argc, char **argv)
{
  dali_config C, *c = &C;
  int font;

  progname = argv[0];
  if (argc != 2)
    {
      fprintf (stderr, "usage: %s font > out.json\n", progname);
      exit (1);
    }

  c->render_state = (struct render_state *)
    calloc (1, sizeof (struct render_state));

  font = atoi (argv[1]);
  init_numbers (c, font);
  print_numbers (c, font);

  return 0;
}
