/*
 *	FM-7 EMULATOR "XM7"
 *
 *	Copyright (C) 1999,2000 ohD(ytanaka@ipc-tokai.or.jp)
 *	[ 6809CPU(C,Original Source from M.A.M.E.) ]
 */

/*-[ [N ]-----------------------------------------------------------*/

static WORD eaddr;				/* AhX */

/*-[ e[u ]-------------------------------------------------------------*/

#define E   0x80				/* 0x80͎AhX̌vZKvȃR[h */

/* ʏy[W̃TCN\ */
static BYTE cycles1[] =
{
	/*    0   1   2   3   4   5   6   7   8   9   A   B   C   D   E   F */
  /*0*/	  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  3,  6,
  /*1*/	255,255,  2,  2,  0,  0,  5,  9,  3,  2,  3,  2,  2,  2,  8,  6,
  /*2*/	  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
  /*3*/	E+4,E+4,E+4,E+4,  5,  5,  5,  5,  4,  5,  3,  6, 20, 11,  1, 19,
  /*4*/	  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,
  /*5*/	  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,
  /*6*/	E+6,E+6,E+6,E+6,E+6,E+6,E+6,E+6,E+6,E+6,E+6,E+6,E+6,E+6,E+3,E+6,
  /*7*/	  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  4,  7,
  /*8*/	  2,  2,  2,  4,  2,  2,  2,  2,  2,  2,  2,  2,  4,  7,  3,  3,
  /*9*/	  4,  4,  4,  6,  4,  4,  4,  4,  4,  4,  4,  4,  6,  7,  5,  5,
  /*A*/	E+4,E+4,E+4,E+6,E+4,E+4,E+4,E+4,E+4,E+4,E+4,E+4,E+6,E+7,E+5,E+5,
  /*B*/	  5,  5,  5,  7,  5,  5,  5,  5,  5,  5,  5,  5,  7,  8,  6,  6,
  /*C*/	  2,  2,  2,  4,  2,  2,  2,  2,  2,  2,  2,  2,  3,  0,  3,  3,
  /*D*/	  4,  4,  4,  6,  4,  4,  4,  4,  4,  4,  4,  4,  5,  5,  5,  5,
  /*E*/	E+4,E+4,E+4,E+6,E+4,E+4,E+4,E+4,E+4,E+4,E+4,E+4,E+5,E+5,E+5,E+5,
  /*F*/	  5,  5,  5,  7,  5,  5,  5,  5,  5,  5,  5,  5,  6,  6,  6,  6
};

/* $10y[W̃TCN\ */
static BYTE cycles2[] =
{
	/*    0   1   2   3   4   5   6   7   8   9   A   B   C   D   E   F */
  /*0*/	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  /*1*/	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  /*2*/	  0,  5,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
  /*3*/	E+0,E+0,E+0,E+0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 20,
  /*4*/	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  /*5*/	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  /*6*/	E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,
  /*7*/	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  /*8*/	  0,  0,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  4,  0,
  /*9*/	  0,  0,  0,  7,  0,  0,  0,  0,  0,  0,  0,  0,  7,  0,  6,  6,
  /*A*/	E+0,E+0,E+0,E+7,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+7,E+0,E+6,E+6,
  /*B*/	  0,  0,  0,  8,  0,  0,  0,  0,  0,  0,  0,  0,  8,  0,  7,  7,
  /*C*/	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,
  /*D*/	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  6,  6,
  /*E*/	E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+6,E+6,
  /*F*/	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  7,  7
};

static BYTE flags8i[256] = {
0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x0a,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08
};

static BYTE flags8d[256] = {
0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08
};

/*-[ }N` ]-----------------------------------------------------------*/

/* C~fBGCgAhbVO */
#define IMMBYTE(b)		{b = READB(PCREG++);}
#define IMMWORD(w)		{w = READW(PCREG); PCREG += 2;}

/* _CNgAhbVO */
#define DIRECT			{IMMBYTE(eaddr); eaddr |= (DPREG << 8);}
#define DIRBYTE(b)		{DIRECT; b=READB(eaddr);}
#define DIRWORD(w)		{DIRECT; w=READW(eaddr);}

/* GNXehAhbVO */
#define EXTENDED		IMMWORD(eaddr)
#define EXTBYTE(b)		{IMMWORD(eaddr); b = READB(eaddr);}
#define EXTWORD(w)		{IMMWORD(eaddr); w = READW(eaddr);}

/* eBuAhbVO */
#define BRANCH(f)		{IMMBYTE(t); if (f) PCREG += SIGNED(t);}
#define LBRANCH(f)		{IMMWORD(t); if (f) {PCREG += t; CYCLE++;}}
#define NXORV			( (CCREG & 0x08) ^ ((CCREG & 0x02) << 2) )

/* tO */
#define SEC				(CCREG |= 0x01)
#define SEZ				(CCREG |= 0x04)

#define CLR_HNZVC		(CCREG &= 0xd0)
#define CLR_NZV			(CCREG &= 0xf1)
#define CLR_NZVC		(CCREG &= 0xf0)
#define CLR_Z			(CCREG &= 0xfb)
#define CLR_NZC			(CCREG &= 0xf2)
#define CLR_ZC			(CCREG &= 0xfa)

#define SET_Z(a)		if (!a) SEZ
#define SET_Z8(a)		SET_Z((BYTE)a)
#define SET_Z16(a)		SET_Z((WORD)a)
#define SET_N8(a)		CCREG |= ((a & 0x80) >> 4)
#define SET_N16(a)		CCREG |= ((a & 0x8000) >> 12)
#define SET_H(a,b,r)	CCREG |= (((a^b^r) & 0x10) << 1)
#define SET_C8(a)		CCREG |= ((a & 0x100) >> 8)
#define SET_C16(a)		CCREG |= ((a & 0x10000) >> 16)
#define SET_V8(a,b,r)	CCREG |= (((a^b^r^(r >> 1)) & 0x80) >> 6)
#define SET_V16(a,b,r)	CCREG |= (((a^b^r^(r >> 1)) & 0x8000) >> 14)
#define SET_NZ8(a)		{SET_N8(a); SET_Z(a);}
#define SET_NZ16(a)		{SET_N16(a); SET_Z(a);}

#define SET_FLAGS8I(a)	{CCREG |= flags8i[(a) & 0xff];}
#define SET_FLAGS8D(a)	{CCREG |= flags8d[(a) & 0xff];}
#define SET_FLAGS8(a,b,r)	{SET_N8(r); SET_Z8(r); SET_V8(a,b,r); SET_C8(r);}
#define SET_FLAGS16(a,b,r)	{SET_N16(r); SET_Z16(r); SET_V16(a,b,r); SET_C16(r);}

/* g⏕ */
#define SIGNED(b)		((WORD)((b & 0x80) ? (b | 0xff00) :(b)))

/* vbVA|bv */
#define PUSHBYTE(b)		{--SREG; WRITEB(SREG, b);}
#define PUSHWORD(w)		{SREG -= 2; WRITEW(SREG, w);}
#define PULLBYTE(b)		{b = READB(SREG); SREG++;}
#define PULLWORD(w)		{w = READW(SREG); SREG += 2;}
#define PSHUBYTE(b)		{--UREG; WRITEB(UREG, b);}
#define PSHUWORD(w)		{UREG -= 2; WRITEW(UREG, w);}
#define PULUBYTE(b)		{b = READB(UREG); UREG++;}
#define PULUWORD(w)		{w = READW(UREG); UREG += 2;}

/* WX^擾Aݒ */
#define GETREG(val,reg)	switch (reg) { \
							case 0x0: val = DREG; break; \
							case 0x1: val = XREG; break; \
							case 0x2: val = YREG; break; \
							case 0x3: val = UREG; break; \
							case 0x4: val = SREG; break; \
							case 0x5: val = PCREG; break; \
							case 0x8: val = (WORD)(0xff00 | AREG); break; \
							case 0x9: val = (WORD)(0xff00 | BREG); break; \
							case 0xa: val = (WORD)(0xff00 | CCREG); break; \
							case 0xb: val = (WORD)(0xff00 | DPREG); break; \
							default: val = 0xffff;  break; } 
#define SETREG(val,reg)	switch (reg) { \
							case 0x0: DREG = val; break; \
							case 0x1: XREG = val; break; \
							case 0x2: YREG = val; break; \
							case 0x3: UREG = val; break; \
							case 0x4: SREG = val; break; \
							case 0x5: PCREG = val; break; \
							case 0x8: AREG = (BYTE)val; break; \
							case 0x9: BREG = (BYTE)val; break; \
							case 0xa: CCREG = (BYTE)val; break; \
							case 0xb: DPREG = (BYTE)val; break; \
							default: break; }

/*-[ ߃Zbg ]-----------------------------------------------------------*/

/* $00 NEG direct ?**** */
static void neg_di( void )
{
	WORD t, r;

	DIRBYTE(t);
	r = -t;
	CLR_NZVC;
	SET_FLAGS8(0, t, r);
	WRITEB(eaddr, r);
}

/* $02 NGC direct */
static void com_di( void );
static void ngc_di( void )
{
	if (CCREG & 0x01) {
		com_di();
	}
	else {
		neg_di();
	}
}

/* $03 COM direct -**01 */
static void com_di( void )
{
	BYTE t;

	DIRBYTE(t);
	t = ~t;
	CLR_NZV;
	SET_NZ8(t);
	SEC;
	WRITEB(eaddr,t);
}

/* $04 LSR direct -0*-* */
static void lsr_di( void )
{
	BYTE t;

	DIRBYTE(t);
	CLR_NZC;
	CCREG |=(t&0x01);
	t>>=1;
	SET_Z8(t);
	WRITEB(eaddr,t);
}

/* $05 ILLEGAL */

/* $06 ROR direct -**-* */
static void ror_di( void )
{
	BYTE t,r;

	DIRBYTE(t);
	r=(CCREG &0x01)<<7;
	CLR_NZC;
	CCREG |=(t&0x01);
	r |= t>>1;
	SET_NZ8(r);
	WRITEB(eaddr,r);
}

/* $07 ASR direct ?**-* */
static void asr_di( void )
{
	BYTE t;

	DIRBYTE(t);
	CLR_NZC;
	CCREG |=(t&0x01);
	t>>=1;
	t|=((t&0x40)<<1);
	SET_NZ8(t);
	WRITEB(eaddr,t);
}

/* $08 ASL direct ?**** */
static void asl_di( void )
{
	WORD t, r;

	DIRBYTE(t);
	r = t << 1;
	CLR_NZVC;
	SET_FLAGS8(t, t, r);
	WRITEB(eaddr, r);
}

/* $09 ROL direct -**** */
static void rol_di( void )
{
	WORD t, r;

	DIRBYTE(t);
	r = CCREG &0x01;
	r |= t << 1;
	CLR_NZVC;
	SET_FLAGS8(t, t, r);
	WRITEB(eaddr, r);
}

/* $0A DEC direct -***- */
static void dec_di( void )
{
	BYTE t;

	DIRBYTE(t);
	--t;
	CLR_NZV;
	SET_FLAGS8D(t);
	WRITEB(eaddr,t);
}

/* $0B DCC direct -**** */
static void dcc_di( void )
{
	BYTE t;

	DIRBYTE(t);
	--t;
	CLR_NZVC;
	SET_FLAGS8D(t);
	WRITEB(eaddr,t);
	if (t!=0) {
		SEC;
	}
}

/* $OC INC direct -***- */
static void inc_di( void )
{
	BYTE t;

	DIRBYTE(t);
	++t;
	CLR_NZV;
	SET_FLAGS8I(t);
	WRITEB(eaddr,t);
}

/* $OD TST direct -**0- */
static void tst_di( void )
{
	BYTE t;

	DIRBYTE(t);
	CLR_NZV;
	SET_NZ8(t);
}

/* $0E JMP direct ----- */
static void jmp_di( void )
{
	DIRECT;
	PCREG = eaddr;
}

/* $0F CLR direct -0100 */
static void clr_di( void )
{
	DIRECT;
	READB(eaddr);
	WRITEB(eaddr,0);
	CLR_NZVC;
	SEZ;
}

/* $12 NOP inherent ----- */
static void nop( void )
{
}

/* $13 SYNC inherent ----- */
static void sync( void )
{
	/* `FbN */
	if (!(INTR & INTR_SYNC_IN)) {
		INTR |= INTR_SYNC_IN;
		INTR &= ~INTR_SYNC_OUT;
		PCREG--;
		return;
	}

	/* `FbN */
	if (!(INTR & INTR_SYNC_OUT)) {
		PCREG--;
		return;
	}

	/* SYNCI */
	INTR &= ~INTR_SYNC_IN;
	INTR &= ~INTR_SYNC_OUT;
}

/* $14 HALT */

/* $15 HALT */

/* $16 LBRA relative ----- */
static void lbra( void )
{
	IMMWORD(eaddr);
	PCREG += eaddr;
}

/* $17 LBSR relative ----- */
static void lbsr( void )
{
	IMMWORD(eaddr);
	PUSHWORD(PCREG);
	PCREG += eaddr;
}

/* $18 ASLCC inherent ****0 */
static void aslcc( void )
{
	if (CCREG & 0x04) {
		CCREG |= 0x01;
	}

	CCREG += CCREG;
	CCREG &= 0x3e;
}

/* $19 DAA inherent (AREG) -**0* */
static void daa( void )
{
	BYTE msn, lsn;
	WORD t, cf = 0;

	msn=AREG & 0xf0;
	lsn=AREG & 0x0f;
	if ( lsn>0x09 || CCREG &0x20 ) cf |= 0x06;
	if ( msn>0x80 && lsn>0x09 ) cf |= 0x60;
	if ( msn>0x90 || CCREG &0x01 ) cf |= 0x60;
	t = cf + AREG;
	CLR_NZV;
	SET_NZ8((BYTE)t);
	SET_C8(t);
	AREG = t & 0xff;
}

/* $1A ORCC immediate ##### */
static void orcc( void )
{
	BYTE t;

	IMMBYTE(t);
	CCREG |=t;
}

/* $1B ILLEGAL */

/* $1C ANDCC immediate ##### */
static void andcc( void )
{
	BYTE t;

	IMMBYTE(t);
	CCREG &=t;
}

/* $1D SEX inherent -**0- */
static void sex( void )
{
	WORD t;

	t = SIGNED(BREG);
	DREG = t;
	CLR_NZV;
	SET_NZ16(t);
}

/* $1E EXG inherent ----- */
static void exg( void )
{
	WORD t1=0,t2=0;
	BYTE tb;

	IMMBYTE(tb);
	GETREG(t1,tb>>4);
	GETREG(t2,tb&15);
	SETREG(t2,tb>>4);
	SETREG(t1,tb&15);
}

/* $1F TFR inherent ----- */
static void tfr( void )
{
	BYTE tb;
	WORD t=0;

	IMMBYTE(tb);
	GETREG(t,tb>>4);
	SETREG(t,tb&15);
}

/* $20 BRA relative ----- */
static void bra( void )
{
	BYTE t;

	IMMBYTE(t);
	PCREG += SIGNED(t);
}

/* $21 BRN relative ----- */
static void brn( void )
{
	PCREG++;
}

/* $1021 LBRN relative ----- */
static void lbrn( void )
{
	PCREG += 2;
}

/* $22 BHI relative ----- */
static void bhi( void )
{
	BYTE t;

	BRANCH(!(CCREG & 0x05));
}

/* $1022 LBHI relative ----- */
static void lbhi( void )
{
	WORD t;

	LBRANCH(!(CCREG & 0x05));
}

/* $23 BLS relative ----- */
static void bls( void )
{
	BYTE t;

	BRANCH(CCREG & 0x05);
}

/* $1023 LBLS relative ----- */
static void lbls( void )
{
	WORD t;

	LBRANCH(CCREG & 0x05);
}

/* $24 BCC relative ----- */
static void bcc( void )
{
	BYTE t;

	BRANCH(!(CCREG & 0x01));
}

/* $1024 LBCC relative ----- */
static void lbcc( void )
{
	WORD t;

	LBRANCH(!(CCREG & 0x01));
}

/* $25 BCS relative ----- */
static void bcs( void )
{
	BYTE t;

	BRANCH(CCREG & 0x01);
}

/* $1025 LBCS relative ----- */
static void lbcs( void )
{
	WORD t;

	LBRANCH(CCREG & 0x01);
}

/* $26 BNE relative ----- */
static void bne( void )
{
	BYTE t;

	BRANCH(!(CCREG & 0x04));
}

/* $1026 LBNE relative ----- */
static void lbne( void )
{

	WORD t;

	LBRANCH(!(CCREG & 0x04));
}

/* $27 BEQ relative ----- */
static void beq( void )
{
	BYTE t;

	BRANCH(CCREG & 0x04);
}

/* $1027 LBEQ relative ----- */
static void lbeq( void )
{
	WORD t;

	LBRANCH(CCREG & 0x04);
}

/* $28 BVC relative ----- */
static void bvc( void )
{
	BYTE t;

	BRANCH(!(CCREG & 0x02));
}

/* $1028 LBVC relative ----- */
static void lbvc( void )
{
	WORD t;

	LBRANCH(!(CCREG & 0x02));
}

/* $29 BVS relative ----- */
static void bvs( void )
{
	BYTE t;

	BRANCH(CCREG & 0x02);
}

/* $1029 LBVS relative ----- */
static void lbvs( void )
{
	WORD t;

	LBRANCH(CCREG & 0x02);
}

/* $2A BPL relative ----- */
static void bpl( void )
{
	BYTE t;

	BRANCH(!(CCREG & 0x08));
}

/* $102A LBPL relative ----- */
static void lbpl( void )
{
	WORD t;

	LBRANCH(!(CCREG & 0x08));
}

/* $2B BMI relative ----- */
static void bmi( void )
{
	BYTE t;

	BRANCH(CCREG & 0x08);
}

/* $102B LBMI relative ----- */
static void lbmi( void )
{
	WORD t;

	LBRANCH(CCREG & 0x08);
}

/* $2C BGE relative ----- */
static void bge( void )
{
	BYTE t;

	BRANCH(!NXORV);
}

/* $102C LBGE relative ----- */
static void lbge( void )
{
	WORD t;

	LBRANCH(!NXORV);
}

/* $2D BLT relative ----- */
static void blt( void )
{
	BYTE t;

	BRANCH(NXORV);
}

/* $102D LBLT relative ----- */
static void lblt( void )
{
	WORD t;

	LBRANCH(NXORV);
}

/* $2E BGT relative ----- */
static void bgt( void )
{
	BYTE t;

	BRANCH(!(NXORV || (CCREG & 0x04)));
}

/* $102E LBGT relative ----- */
static void lbgt( void )
{
	WORD t;

	LBRANCH(!(NXORV || (CCREG & 0x04)));
}

/* $2F BLE relative ----- */
static void ble( void )
{
	BYTE t;

	BRANCH(NXORV || (CCREG & 0x04));
}

/* $102F LBLE relative ----- */
static void lble( void )
{
	WORD t;

	LBRANCH(NXORV || (CCREG & 0x04));
}

/* $30 LEAX indexed --*-- */
static void leax( void )
{
	XREG = eaddr;
	CLR_Z;
	SET_Z(XREG);
}

/* $31 LEAY indexed --*-- */
static void leay( void )
{
	YREG = eaddr;
	CLR_Z;
	SET_Z(YREG);
}

/* $32 LEAS indexed ----- */
static void leas( void )
{
	SREG = eaddr;
}

/* $33 LEAU indexed ----- */
static void leau( void )
{
	UREG = eaddr;
}

/* $34 PSHS inherent ----- */
static void pshs( void )
{
	BYTE t;

	IMMBYTE(t);
	if(t & 0x80) { PUSHWORD(PCREG); CYCLE+=2; }
	if(t & 0x40) { PUSHWORD(UREG); CYCLE+=2; }
	if(t & 0x20) { PUSHWORD(YREG); CYCLE+=2; }
	if(t & 0x10) { PUSHWORD(XREG); CYCLE+=2; }
	if(t & 0x08) { PUSHBYTE(DPREG); CYCLE++; }
	if(t & 0x04) { PUSHBYTE(BREG); CYCLE++; }
	if(t & 0x02) { PUSHBYTE(AREG); CYCLE++; }
	if(t & 0x01) { PUSHBYTE(CCREG); CYCLE++; }
}

/* $35 PULS inherent ----- */
static void puls( void )
{
	BYTE t;

	IMMBYTE(t);
	if(t & 0x01) { PULLBYTE(CCREG); CYCLE++; }
	if(t & 0x02) { PULLBYTE(AREG); CYCLE++; }
	if(t & 0x04) { PULLBYTE(BREG); CYCLE++; }
	if(t & 0x08) { PULLBYTE(DPREG); CYCLE++; }
	if(t & 0x10) { PULLWORD(XREG); CYCLE+=2; }
	if(t & 0x20) { PULLWORD(YREG); CYCLE+=2; }
	if(t & 0x40) { PULLWORD(UREG); CYCLE+=2; }
	if(t & 0x80) { PULLWORD(PCREG); CYCLE+=2; }
}

/* $36 PSHU inherent ----- */
static void pshu( void )
{
	BYTE t;

	IMMBYTE(t);
	if(t & 0x80) { PSHUWORD(PCREG); CYCLE+=2; }
	if(t & 0x40) { PSHUWORD(SREG); CYCLE+=2; }
	if(t & 0x20) { PSHUWORD(YREG); CYCLE+=2; }
	if(t & 0x10) { PSHUWORD(XREG); CYCLE+=2; }
	if(t & 0x08) { PSHUBYTE(DPREG); CYCLE++; }
	if(t & 0x04) { PSHUBYTE(BREG); CYCLE++; }
	if(t & 0x02) { PSHUBYTE(AREG); CYCLE++; }
	if(t & 0x01) { PSHUBYTE(CCREG); CYCLE++; }
}

/* 37 PULU inherent ----- */
static void pulu( void )
{
	BYTE t;

	IMMBYTE(t);
	if(t & 0x01) { PULUBYTE(CCREG); CYCLE++; }
	if(t & 0x02) { PULUBYTE(AREG); CYCLE++; }
	if(t & 0x04) { PULUBYTE(BREG); CYCLE++; }
	if(t & 0x08) { PULUBYTE(DPREG); CYCLE++; }
	if(t & 0x10) { PULUWORD(XREG); CYCLE+=2; }
	if(t & 0x20) { PULUWORD(YREG); CYCLE+=2; }
	if(t & 0x40) { PULUWORD(SREG); CYCLE+=2; }
	if(t & 0x80) { PULUWORD(PCREG); CYCLE+=2; }
}

/* $38 ILLEGAL */

/* $39 RTS inherent ----- */
static void rts( void )
{
	PULLWORD(PCREG);
}

/* $3A ABX inherent ----- */
static void abx( void )
{
	XREG += BREG;
}

/* $3B RTI inherent ##### */
static void rti( void )
{
	PULLBYTE(CCREG);

	if (CCREG & 0x80) {
		PULLBYTE(AREG);
		PULLBYTE(BREG);
		PULLBYTE(DPREG);
		PULLWORD(XREG);
		PULLWORD(YREG);
		PULLWORD(UREG);
		CYCLE += 9;
	}
	PULLWORD(PCREG);
}

/* $3C CWAI inherent ----1 */
static void cwai( void )
{
	BYTE t;

	/* CWAI`FbN */
	if (!(INTR & INTR_CWAI_IN)) {
		IMMBYTE(t);
		CCREG &= t;
		INTR |= INTR_CWAI_IN;
		INTR &= ~INTR_CWAI_OUT;
		PCREG -= 2;
		return;
	}

	/* CWAII`FbN */
	if (!(INTR & INTR_CWAI_OUT)) {
		PCREG--;
		return;
	}

	/* CWAII */
	INTR &= ~INTR_CWAI_IN;
	INTR &= ~INTR_CWAI_OUT;
	PCREG++;
}

/* $3D MUL inherent --*-@ */
static void mul( void )
{
	WORD t;

	t = AREG * BREG;
	CLR_ZC;
	SET_Z16(t);
	if (t & 0x80) SEC;
	DREG = t;
}

/* $3E RST inherent ----- */
static void rst( void )
{
	cpu_reset();
}

/* $3F SWI (SWI2 SWI3) absolute indirect ----- */
static void swi( void )
{
	CCREG |= 0x80;

	PUSHWORD(PCREG);
	PUSHWORD(UREG);
	PUSHWORD(YREG);
	PUSHWORD(XREG);
	PUSHBYTE(DPREG);
	PUSHBYTE(BREG);
	PUSHBYTE(AREG);
	PUSHBYTE(CCREG);

	/* SWÎ݁AORCC #$50̌ʂ */
	CCREG |= 0x50;

	PCREG = READW(0xfffa);
}

/* $103F SWI2 absolute indirect ----- */
static void swi2( void )
{
	CCREG |=0x80;

	PUSHWORD(PCREG);
	PUSHWORD(UREG);
	PUSHWORD(YREG);
	PUSHWORD(XREG);
	PUSHBYTE(DPREG);
	PUSHBYTE(BREG);
	PUSHBYTE(AREG);
	PUSHBYTE(CCREG);

	PCREG = READW(0xfff4);
}

/* $113F SWI3 absolute indirect ----- */
static void swi3( void )
{
	CCREG |=0x80;

	PUSHWORD(PCREG);
	PUSHWORD(UREG);
	PUSHWORD(YREG);
	PUSHWORD(XREG);
	PUSHBYTE(DPREG);
	PUSHBYTE(BREG);
	PUSHBYTE(AREG);
	PUSHBYTE(CCREG);

	PCREG = READW(0xfff2);
}

/* $40 NEGA inherent ?**** */
static void nega( void )
{
	WORD r;

	r = -AREG;
	CLR_NZVC;
	SET_FLAGS8(0, AREG, r);
	AREG = (BYTE)r;
}

/* $42 NGCA inherent */
static void coma( void );
static void ngca( void )
{
	if (CCREG & 0x01) {
		coma();
	}
	else {
		nega();
	}
}

/* $43 COMA inherent -**01 */
static void coma( void )
{
	AREG = ~AREG;
	CLR_NZV;
	SET_NZ8(AREG);
	SEC;
}

/* $44 LSRA inherent -0*-* */
static void lsra( void )
{
	CLR_NZC;
	CCREG |=(AREG&0x01);
	AREG>>=1;
	SET_Z8(AREG);
}

/* $45 ILLEGAL */

/* $46 RORA inherent -**-* */
static void rora( void )
{
	BYTE r;

	r=(CCREG &0x01) << 7;
	CLR_NZC;
	CCREG |= (AREG&0x01);
	r |= AREG>>1;
	SET_NZ8(r);
	AREG=r;
}

/* $47 ASRA inherent ?**-* */
static void asra( void )
{
	CLR_NZC;
	CCREG |= (AREG&0x01);
	AREG >>= 1;
	AREG |= ((AREG&0x40) << 1);
	SET_NZ8(AREG);
}

/* $48 ASLA inherent ?**** */
static void asla( void )
{
	WORD r;

	r = AREG << 1;
	CLR_NZVC;
	SET_FLAGS8(AREG, AREG, r);
	AREG = (BYTE)r;
}

/* $49 ROLA inherent -**** */
static void rola( void )
{
	WORD t, r;

	t = AREG;
	r = CCREG & 0x01;
	r |= t << 1;
	CLR_NZVC;
	SET_FLAGS8(t, t, r);
	AREG = (BYTE)r;
}

/* $4A DECA inherent -***- */
static void deca( void )
{
	--AREG;
	CLR_NZV;
	SET_FLAGS8D(AREG);
}

/* $4B DCCA inherent -**** */
static void dcca( void )
{
	--AREG;
	CLR_NZVC;
	SET_FLAGS8D(AREG);
	if (AREG!=0) {
		SEC;
	}
}

/* $4C INCA inherent -***- */
static void inca( void )
{
	++AREG;
	CLR_NZV;
	SET_FLAGS8I(AREG);
}

/* $4D TSTA inherent -**0- */
static void tsta( void )
{
	CLR_NZV;
	SET_NZ8(AREG);
}

/* $4E CLCA inherent -010- */
static void clca( void )
{
	AREG=0;
	CLR_NZV;
	SEZ;
}

/* $4F CLRA inherent -0100 */
static void clra( void )
{
	AREG=0;
	CLR_NZVC;
	SEZ;
}

/* $50 NEGB inherent ?**** */
static void negb( void )
{
	WORD r;

	r = -BREG;
	CLR_NZVC;
	SET_FLAGS8(0, BREG, r);
	BREG = (BYTE)r;
}

/* $52 NGCB inherent */
static void comb( void );
static void ngcb( void )
{
	if (CCREG & 0x01) {
		comb();
	}
	else {
		negb();
	}
}

/* $53 COMB inherent -**01 */
static void comb( void )
{
	BREG = ~BREG;
	CLR_NZV; SET_NZ8(BREG); SEC;
}

/* $54 LSRB inherent -0*-* */
static void lsrb( void )
{
	CLR_NZC; CCREG |=(BREG&0x01);
	BREG>>=1; SET_Z8(BREG);
}

/* $55 ILLEGAL */

/* $56 RORB inherent -**-* */
static void rorb( void )
{
	BYTE r;
	r=(CCREG &0x01)<<7;
	CLR_NZC; CCREG |=(BREG&0x01);
	r |= BREG>>1; SET_NZ8(r);
	BREG=r;
}

/* $57 ASRB inherent ?**-* */
static void asrb( void )
{
	CLR_NZC; CCREG |=(BREG&0x01);
	BREG>>=1; BREG|=((BREG&0x40)<<1);
	SET_NZ8(BREG);
}

/* $58 ASLB inherent ?**** */
static void aslb( void )
{
	WORD r;

	r= BREG << 1;
	CLR_NZVC;
	SET_FLAGS8(BREG, BREG, r);
	BREG = (BYTE)r;
}

/* $59 ROLB inherent -**** */
static void rolb( void )
{
	WORD t, r;

	t = BREG;
	r = CCREG & 0x01;
	r |= t << 1;
	CLR_NZVC;
	SET_FLAGS8(t, t, r);
	BREG = (BYTE)r;
}

/* $5A DECB inherent -***- */
static void decb( void )
{
	--BREG;
	CLR_NZV;
	SET_FLAGS8D(BREG);
}

/* $5B DCCB inherent -**** */
static void dccb( void )
{
	--BREG;
	CLR_NZVC;
	SET_FLAGS8D(BREG);
	if (BREG!=0) {
		SEC;
	}
}

/* $5C INCB inherent -***- */
static void incb( void )
{
	++BREG;
	CLR_NZV;
	SET_FLAGS8I(BREG);
}

/* $5D TSTB inherent -**0- */
static void tstb( void )
{
	CLR_NZV;
	SET_NZ8(BREG);
}

/* $5E CLCB inherent -010- */
static void clcb( void )
{
	BREG=0;
	CLR_NZV;
	SEZ;
}

/* $5F CLRB inherent -0100 */
static void clrb( void )
{
	BREG=0;
	CLR_NZVC;
	SEZ;
}

/* $60 NEG indexed ?**** */
static void neg_ix( void )
{
	WORD t, r;
	
	t = READB(eaddr);
	r = -t;
	CLR_NZVC;
	SET_FLAGS8(0, t, r);
	WRITEB(eaddr, r);
}

/* $62 NGC indexed */
static void com_ix( void );
static void ngc_ix( void )
{
	if (CCREG & 0x01) {
		com_ix();
	}
	else {
		neg_ix();
	}
}

/* $63 COM indexed -**01 */
static void com_ix( void )
{
	BYTE t;
	t = ~READB(eaddr);
	CLR_NZV; SET_NZ8(t); SEC;
	WRITEB(eaddr,t);
}

/* $64 LSR indexed -0*-* */
static void lsr_ix( void )
{
	BYTE t;
	t=READB(eaddr); CLR_NZC; CCREG |=(t&0x01);
	t>>=1; SET_Z8(t);
	WRITEB(eaddr,t);
}

/* $65 ILLEGAL */

/* $66 ROR indexed -**-* */
static void ror_ix( void )
{
	BYTE t,r;
	t=READB(eaddr); r=(CCREG &0x01)<<7;
	CLR_NZC; CCREG |=(t&0x01);
	r |= t>>1; SET_NZ8(r);
	WRITEB(eaddr,r);
}

/* $67 ASR indexed ?**-* */
static void asr_ix( void )
{
	BYTE t;
	t=READB(eaddr); CLR_NZC; CCREG |=(t&0x01);
	t>>=1; t|=((t&0x40)<<1);
	SET_NZ8(t);
	WRITEB(eaddr,t);
}

/* $68 ASL indexed ?**** */
static void asl_ix( void )
{
	WORD t, r;

	t = READB(eaddr);
	r = t << 1;
	CLR_NZVC;
	SET_FLAGS8(t, t, r);
	WRITEB(eaddr, r);
}

/* $69 ROL indexed -**** */
static void rol_ix( void )
{
	WORD t, r;

	t = READB(eaddr);
	r = CCREG &0x01;
	r |= t << 1;
	CLR_NZVC;
	SET_FLAGS8(t, t, r);
	WRITEB(eaddr, r);
}

/* $6A DEC indexed -***- */
static void dec_ix( void )
{
	BYTE t;
	t=READB(eaddr)-1;
	CLR_NZV; SET_FLAGS8D(t);
	WRITEB(eaddr,t);
}

/* $6B DCC indexed -**** */
static void dcc_ix( void )
{
	BYTE t;
	t=READB(eaddr)-1;
	CLR_NZVC; SET_FLAGS8D(t);
	WRITEB(eaddr,t);
	if (t!=0) {
		SEC;
	}
}

/* $6C INC indexed -***- */
static void inc_ix( void )
{
	BYTE t;
	t=READB(eaddr)+1;
	CLR_NZV; SET_FLAGS8I(t);
	WRITEB(eaddr,t);
}

/* $6D TST indexed -**0- */
static void tst_ix( void )
{
	BYTE t;

	t = READB(eaddr);
	CLR_NZV;
	SET_NZ8(t);
}

/* $6E JMP indexed ----- */
static void jmp_ix( void )
{
	PCREG = eaddr;
}

/* $6F CLR indexed -0100 */
static void clr_ix( void )
{
	READB(eaddr);
	WRITEB(eaddr,0);
	CLR_NZVC;
	SEZ;
}

/* $70 NEG extended ?**** */
static void neg_ex( void )
{
	WORD t, r;

	EXTBYTE(t);
	r = -t;
	CLR_NZVC;
	SET_FLAGS8(0, t, r);
	WRITEB(eaddr, r);
}

/* $72 NGC extended */
static void com_ex( void );
static void ngc_ex( void )
{
	if (CCREG & 0x01) {
		com_ex();
	}
	else {
		neg_ex();
	}
}

/* $73 COM extended -**01 */
static void com_ex( void )
{
	BYTE t;
	EXTBYTE(t); t = ~t;
	CLR_NZV; SET_NZ8(t); SEC;
	WRITEB(eaddr,t);
}

/* $74 LSR extended -0*-* */
static void lsr_ex( void )
{
	BYTE t;
	EXTBYTE(t); CLR_NZC; CCREG |=(t&0x01);
	t>>=1; SET_Z8(t);
	WRITEB(eaddr,t);
}

/* $75 ILLEGAL */

/* $76 ROR extended -**-* */
static void ror_ex( void )
{
	BYTE t,r;
	EXTBYTE(t); r=(CCREG &0x01)<<7;
	CLR_NZC; CCREG |=(t&0x01);
	r |= t>>1; SET_NZ8(r);
	WRITEB(eaddr,r);
}

/* $77 ASR extended ?**-* */
static void asr_ex( void )
{
	BYTE t;
	EXTBYTE(t); CLR_NZC; CCREG |=(t&0x01);
	t>>=1; t|=((t&0x40)<<1);
	SET_NZ8(t);
	WRITEB(eaddr,t);
}

/* $78 ASL extended ?**** */
static void asl_ex( void )
{
	WORD t, r;

	EXTBYTE(t);
	r = t << 1;
	CLR_NZVC;
	SET_FLAGS8(t, t, r);
	WRITEB(eaddr, r);
}

/* $79 ROL extended -**** */
static void rol_ex( void )
{
	WORD t, r;

	EXTBYTE(t);
	r = CCREG & 0x01;
	r |= t << 1;
	CLR_NZVC;
	SET_FLAGS8(t, t, r);
	WRITEB(eaddr, r);
}

/* $7A DEC extended -***- */
static void dec_ex( void )
{
	BYTE t;
	EXTBYTE(t); --t;
	CLR_NZV; SET_FLAGS8D(t);
	WRITEB(eaddr,t);
}

/* $7B DCC extended -**** */
static void dcc_ex( void )
{
	BYTE t;
	EXTBYTE(t); --t;
	CLR_NZV; SET_FLAGS8D(t);
	WRITEB(eaddr,t);
	if (t!=0) {
		SEC;
	}
}

/* $7C INC extended -***- */
static void inc_ex( void )
{
	BYTE t;
	EXTBYTE(t); ++t;
	CLR_NZV; SET_FLAGS8I(t);
	WRITEB(eaddr,t);
}

/* $7D TST extended -**0- */
static void tst_ex( void )
{
	BYTE t;

	EXTBYTE(t);
	CLR_NZV;
	SET_NZ8(t);
}

/* $7E JMP extended ----- */
static void jmp_ex( void )
{
	EXTENDED;
	PCREG = eaddr;
}

/* $7F CLR extended -0100 */
static void clr_ex( void )
{
	EXTENDED;
	READB(eaddr);
	WRITEB(eaddr,0);
	CLR_NZVC;
	SEZ;
}

/* $80 SUBA immediate ?**** */
static void suba_im( void )
{
	WORD t, r;

	IMMBYTE(t);
	r = AREG - t;
	CLR_NZVC;
	SET_FLAGS8(AREG, t, r);
	AREG = (BYTE)r;
}

/* $81 CMPA immediate ?**** */
static void cmpa_im( void )
{
	WORD t, r;

	IMMBYTE(t);
	r = AREG - t;
	CLR_NZVC;
	SET_FLAGS8(AREG, t, r);
}

/* $82 SBCA immediate ?**** */
static void sbca_im( void )
{
	WORD t, r;

	IMMBYTE(t);
	r = AREG - t - (CCREG &0x01);
	CLR_NZVC;
	SET_FLAGS8(AREG, t, r);
	AREG = (BYTE)r;
}

/* $83 SUBD (CMPD CMPU) immediate -**** */
static void subd_im( void )
{
	DWORD t, r;

	IMMWORD(t); t &= 0xffff;
	r = DREG - t;
	CLR_NZVC;
	SET_FLAGS16(DREG, t, r);
	DREG = (WORD)r;
}

/* $1083 CMPD immediate -**** */
static void cmpd_im( void )
{
	DWORD t, r;

	IMMWORD(t); t &= 0xffff;
	r = DREG - t;
	CLR_NZVC;
	SET_FLAGS16(DREG, t, r);
}

/* $1183 CMPU immediate -**** */
static void cmpu_im( void )
{
	DWORD r,b;

	IMMWORD(b); b &= 0xffff;
	r = UREG-b;
	CLR_NZVC;
	SET_FLAGS16(UREG,b,r);
}

/* $84 ANDA immediate -**0- */
static void anda_im( void )
{
	BYTE t;

	IMMBYTE(t); AREG &= t;
	CLR_NZV; SET_NZ8(AREG);
}

/* $85 BITA immediate -**0- */
static void bita_im( void )
{
	BYTE t,r;
	IMMBYTE(t); r = AREG&t;
	CLR_NZV; SET_NZ8(r);
}

/* $86 LDA immediate -**0- */
static void lda_im( void )
{
	IMMBYTE(AREG);
	CLR_NZV;
	SET_NZ8(AREG);
}

/* $87 FLAG immediate -100- */
static void flag8_im( void )
{
	BYTE b;

	IMMBYTE(b);
	CLR_NZV;
	CCREG |= 0x08;
}

/* $88 EORA immediate -**0- */
static void eora_im( void )
{
	BYTE t;
	IMMBYTE(t); AREG ^= t;
	CLR_NZV; SET_NZ8(AREG);
}

/* $89 ADCA immediate ***** */
static void adca_im( void )
{
	WORD t, r;

	IMMBYTE(t);
	r = AREG + t + (CCREG &0x01);
	CLR_HNZVC;
	SET_FLAGS8(AREG, t, r);
	SET_H(AREG, t, r);
	AREG = (BYTE)r;
}

/* $8A ORA immediate -**0- */
static void ora_im( void )
{
	BYTE t;
	IMMBYTE(t); AREG |= t;
	CLR_NZV; SET_NZ8(AREG);
}

/* $8B ADDA immediate ***** */
static void adda_im( void )
{
	WORD t, r;

	IMMBYTE(t);
	r = AREG + t;
	CLR_HNZVC;
	SET_FLAGS8(AREG, t, r);
	SET_H(AREG, t, r);
	AREG = (BYTE)r;
}

/* $8C CMPX (CMPY CMPS) immediate -**** */
static void cmpx_im( void )
{
	DWORD r,d,b;
	IMMWORD(b); b &= 0xffff;
	d = XREG; r = d-b;
	CLR_NZVC; SET_FLAGS16(d,b,r);
}

/* $108C CMPY immediate -**** */
static void cmpy_im( void )
{
	DWORD r,d,b;
	IMMWORD(b); b &= 0xffff;
	d = YREG; r = d-b;
	CLR_NZVC; SET_FLAGS16(d,b,r);
}

/* $118C CMPS immediate -**** */
static void cmps_im( void )
{
	DWORD r,d,b;

	IMMWORD(b);
	b &= 0xffff;
	d = SREG;
	r = d-b;
	CLR_NZVC;
	SET_FLAGS16(d,b,r);
}

/* $8D BSR ----- */
static void bsr( void )
{
	BYTE t;

	IMMBYTE(t);
	PUSHWORD(PCREG);
	PCREG += SIGNED(t);
}

/* $8E LDX (LDY) immediate -**0- */
static void ldx_im( void )
{
	IMMWORD(XREG);
	CLR_NZV;
	SET_NZ16(XREG);
}

/* $108E LDY immediate -**0- */
static void ldy_im( void )
{
	IMMWORD(YREG);
	CLR_NZV;
	SET_NZ16(YREG);
}

/* $8F FLAG immediate -100- */
static void flag16_im( void )
{
	WORD w;

	IMMWORD(w);
	CLR_NZV;
	CCREG |= 0x08;
}

/* is this a legal instruction? */
/* $108F STY immediate -**0- */
static void sty_im( void )
{
}

/* $90 SUBA direct ?**** */
static void suba_di( void )
{
	WORD t, r;

	DIRBYTE(t);
	r = AREG - t;
	CLR_NZVC;
	SET_FLAGS8(AREG, t, r);
	AREG = (BYTE)r;
}

/* $91 CMPA direct ?**** */
static void cmpa_di( void )
{
	WORD t, r;

	DIRBYTE(t);
	r = AREG - t;
	CLR_NZVC;
	SET_FLAGS8(AREG, t, r);
}

/* $92 SBCA direct ?**** */
static void sbca_di( void )
{
	WORD t, r;

	DIRBYTE(t);
	r = AREG - t - (CCREG &0x01);
	CLR_NZVC;
	SET_FLAGS8(AREG, t, r);
	AREG = (BYTE)r;
}

/* $93 SUBD (CMPD CMPU) direct -**** */
static void subd_di( void )
{
	DWORD b, r;

	DIRWORD(b); b &= 0xffff;
	r = DREG - b;
	CLR_NZVC;
	SET_FLAGS16(DREG, b, r);
	DREG = (WORD)r;
}

/* $1093 CMPD direct -**** */
static void cmpd_di( void )
{
	DWORD b, r;

	DIRWORD(b); b &= 0xffff;
	r = DREG - b;
	CLR_NZVC;
	SET_FLAGS16(DREG, b, r);
}

/* $1193 CMPU direct -**** */
static void cmpu_di( void )
{
	DWORD r,b;
	DIRWORD(b); b &= 0xffff;
	r = UREG-b;
	CLR_NZVC; SET_FLAGS16(UREG,b,r);
}

/* $94 ANDA direct -**0- */
static void anda_di( void )
{
	BYTE t;
	DIRBYTE(t); AREG &= t;
	CLR_NZV; SET_NZ8(AREG);
}

/* $95 BITA direct -**0- */
static void bita_di( void )
{
	BYTE t,r;
	DIRBYTE(t); r = AREG&t;
	CLR_NZV; SET_NZ8(r);
}

/* $96 LDA direct -**0- */
static void lda_di( void )
{
	DIRBYTE(AREG);
	CLR_NZV; SET_NZ8(AREG);
}

/* $97 STA direct -**0- */
static void sta_di( void )
{
	CLR_NZV; SET_NZ8(AREG);
	DIRECT; WRITEB(eaddr,AREG);
}

/* $98 EORA direct -**0- */
static void eora_di( void )
{
	BYTE t;
	DIRBYTE(t); AREG ^= t;
	CLR_NZV; SET_NZ8(AREG);
}

/* $99 ADCA direct ***** */
static void adca_di( void )
{
	WORD t, r;

	DIRBYTE(t);
	r = AREG + t + (CCREG &0x01);
	CLR_HNZVC;
	SET_FLAGS8(AREG, t, r);
	SET_H(AREG, t, r);
	AREG = (BYTE)r;
}

/* $9A ORA direct -**0- */
static void ora_di( void )
{
	BYTE t;
	DIRBYTE(t); AREG |= t;
	CLR_NZV; SET_NZ8(AREG);
}

/* $9B ADDA direct ***** */
static void adda_di( void )
{
	WORD t, r;

	DIRBYTE(t);
	r = AREG + t;
	CLR_HNZVC;
	SET_FLAGS8(AREG, t, r);
	SET_H(AREG, t, r);
	AREG = (BYTE)r;
}

/* $9C CMPX (CMPY CMPS) direct -**** */
static void cmpx_di( void )
{
	DWORD r,d,b;

	DIRWORD(b);
	b &= 0xffff;
	d = XREG;
	r = d - b;
	CLR_NZVC;
	SET_FLAGS16(d,b,r);
}

/* $109C CMPY direct -**** */
static void cmpy_di( void )
{
	DWORD r,d,b;

	DIRWORD(b);
	b &= 0xffff;
	d = YREG;
	r = d - b;
	CLR_NZVC;
	SET_FLAGS16(d,b,r);
}

/* $119C CMPS direct -**** */
static void cmps_di( void )
{
	DWORD r,d,b;

	DIRWORD(b);
	b &= 0xffff;
	d = SREG;
	r = d - b;
	CLR_NZVC;
	SET_FLAGS16(d,b,r);
}

/* $9D JSR direct ----- */
static void jsr_di( void )
{
	DIRECT;
	PUSHWORD(PCREG);
	PCREG = eaddr;
}

/* $9E LDX (LDY) direct -**0- */
static void ldx_di( void )
{
	DIRWORD(XREG);
	CLR_NZV;
	SET_NZ16(XREG);
}

/* $109E LDY direct -**0- */
static void ldy_di( void )
{
	DIRWORD(YREG);
	CLR_NZV;
	SET_NZ16(YREG);
}

/* $9F STX (STY) direct -**0- */
static void stx_di( void )
{
	CLR_NZV;
	SET_NZ16(XREG);
	DIRECT;
	WRITEW(eaddr, XREG);
}

/* $109F STY direct -**0- */
static void sty_di( void )
{
	CLR_NZV;
	SET_NZ16(YREG);
	DIRECT;
	WRITEW(eaddr, YREG);
}

/* $a0 SUBA indexed ?**** */
static void suba_ix( void )
{
	WORD t, r;

	t = READB(eaddr);
	r = AREG - t;
	CLR_NZVC;
	SET_FLAGS8(AREG, t, r);
	AREG = (BYTE)r;
}

/* $a1 CMPA indexed ?**** */
static void cmpa_ix( void )
{
	WORD t, r;

	t = READB(eaddr);
	r = AREG - t;
	CLR_NZVC;
	SET_FLAGS8(AREG, t, r);
}

/* $a2 SBCA indexed ?**** */
static void sbca_ix( void )
{
	WORD t, r;

	t = READB(eaddr);
	r = AREG - t - (CCREG &0x01);
	CLR_NZVC;
	SET_FLAGS8(AREG, t, r);
	AREG = (BYTE)r;
}

/* $a3 SUBD (CMPD CMPU) indexed -**** */
static void subd_ix( void )
{
	DWORD t, r;

	t = READW(eaddr); t &= 0xffff;
	r = DREG - t;
	CLR_NZVC;
	SET_FLAGS16(DREG, t, r);
	DREG = (WORD)r;
}

/* $10a3 CMPD indexed -**** */
static void cmpd_ix( void )
{
	DWORD t, r;

	t = READW(eaddr); t &= 0xffff;
	r = DREG - t;
	CLR_NZVC;
	SET_FLAGS16(DREG, t, r);
}

/* $11a3 CMPU indexed -**** */
static void cmpu_ix( void )
{
	DWORD t, r;

	t = READW(eaddr); t &= 0xffff;
	r = UREG - t;
	CLR_NZVC;
	SET_FLAGS16(UREG, t, r);
}

/* $a4 ANDA indexed -**0- */
static void anda_ix( void )
{
	AREG &= READB(eaddr);
	CLR_NZV; SET_NZ8(AREG);
}

/* $a5 BITA indexed -**0- */
static void bita_ix( void )
{
	BYTE r;
	r = AREG&READB(eaddr);
	CLR_NZV; SET_NZ8(r);
}

/* $a6 LDA indexed -**0- */
static void lda_ix( void )
{
	AREG = READB(eaddr);
	CLR_NZV; SET_NZ8(AREG);
}

/* $a7 STA indexed -**0- */
static void sta_ix( void )
{
	CLR_NZV; SET_NZ8(AREG);
	WRITEB(eaddr,AREG);
}

/* $a8 EORA indexed -**0- */
static void eora_ix( void )
{
	AREG ^= READB(eaddr);
	CLR_NZV; SET_NZ8(AREG);
}

/* $a9 ADCA indexed ***** */
static void adca_ix( void )
{
	WORD t, r;

	t = READB(eaddr);
	r = AREG + t + (CCREG &0x01);
	CLR_HNZVC;
	SET_FLAGS8(AREG, t, r);
	SET_H(AREG, t, r);
	AREG = (BYTE)r;
}

/* $aA ORA indexed -**0- */
static void ora_ix( void )
{
	AREG |= READB(eaddr);
	CLR_NZV; SET_NZ8(AREG);
}

/* $aB ADDA indexed ***** */
static void adda_ix( void )
{
	WORD t, r;

	t = READB(eaddr);
	r = AREG + t;
	CLR_HNZVC;
	SET_FLAGS8(AREG, t, r);
	SET_H(AREG, t, r);
	AREG = (BYTE)r;
}

/* $aC CMPX (CMPY CMPS) indexed -**** */
static void cmpx_ix( void )
{
	DWORD t, r;

	t = READW(eaddr); t &= 0xffff;
	r = XREG - t;
	CLR_NZVC;
	SET_FLAGS16(XREG, t, r);
}

/* $10aC CMPY indexed -**** */
static void cmpy_ix( void )
{
	DWORD t, r;

	t = READW(eaddr); t &= 0xffff;
	r = YREG - t;
	CLR_NZVC;
	SET_FLAGS16(YREG, t, r);
}

/* $11aC CMPS indexed -**** */
static void cmps_ix( void )
{
	DWORD t, r;

	t = READW(eaddr); t &= 0xffff;
	r = SREG - t;
	CLR_NZVC;
	SET_FLAGS16(SREG, t, r);
}

/* $aD JSR indexed ----- */
static void jsr_ix( void )
{
	PUSHWORD(PCREG);
	PCREG = eaddr;
}

/* $aE LDX (LDY) indexed -**0- */
static void ldx_ix( void )
{
	XREG = READW(eaddr);
	CLR_NZV;
	SET_NZ16(XREG);
}

/* $10aE LDY indexed -**0- */
static void ldy_ix( void )
{
	YREG = READW(eaddr);
	CLR_NZV;
	SET_NZ16(YREG);
}

/* $aF STX (STY) indexed -**0- */
static void stx_ix( void )
{
	CLR_NZV;
	SET_NZ16(XREG);
	WRITEW(eaddr, XREG);
}

/* $10aF STY indexed -**0- */
static void sty_ix( void )
{
	CLR_NZV;
	SET_NZ16(YREG);
	WRITEW(eaddr, YREG);
}

/* $b0 SUBA extended ?**** */
static void suba_ex( void )
{
	WORD t, r;

	EXTBYTE(t);
	r = AREG - t;
	CLR_NZVC;
	SET_FLAGS8(AREG, t, r);
	AREG = (BYTE)r;
}

/* $b1 CMPA extended ?**** */
static void cmpa_ex( void )
{
	WORD t, r;

	EXTBYTE(t);
	r = AREG - t;
	CLR_NZVC;
	SET_FLAGS8(AREG, t, r);
}

/* $b2 SBCA extended ?**** */
static void sbca_ex( void )
{
	WORD t, r;

	EXTBYTE(t);
	r = AREG - t -(CCREG &0x01);
	CLR_NZVC;
	SET_FLAGS8(AREG, t, r);
	AREG = (BYTE)r;
}

/* $b3 SUBD (CMPD CMPU) extended -**** */
static void subd_ex( void )
{
	DWORD t, r;

	EXTWORD(t); t &= 0xffff;
	r = DREG - t;
	CLR_NZVC;
	SET_FLAGS16(DREG, t, r);
	DREG = (WORD)r;
}

/* $10b3 CMPD extended -**** */
static void cmpd_ex( void )
{
	DWORD t, r;

	EXTWORD(t); t &= 0xffff;
	r = DREG - t;
	CLR_NZVC;
	SET_FLAGS16(DREG, t, r);
}

/* $11b3 CMPU extended -**** */
static void cmpu_ex( void )
{
	DWORD r,b;
	EXTWORD(b); b &= 0xffff;
	r = UREG-b;
	CLR_NZVC; SET_FLAGS16(UREG,b,r);
}

/* $b4 ANDA extended -**0- */
static void anda_ex( void )
{
	BYTE t;
	EXTBYTE(t); AREG &= t;
	CLR_NZV; SET_NZ8(AREG);
}

/* $b5 BITA extended -**0- */
static void bita_ex( void )
{
	BYTE t,r;

	EXTBYTE(t);
	r = AREG & t;
	CLR_NZV;
	SET_NZ8(r);
}

/* $b6 LDA extended -**0- */
static void lda_ex( void )
{
	EXTBYTE(AREG);
	CLR_NZV;
	SET_NZ8(AREG);
}

/* $b7 STA extended -**0- */
static void sta_ex( void )
{
	CLR_NZV;
	SET_NZ8(AREG);
	EXTENDED;
	WRITEB(eaddr,AREG);
}

/* $b8 EORA extended -**0- */
static void eora_ex( void )
{
	BYTE t;

	EXTBYTE(t);
	AREG ^= t;
	CLR_NZV;
	SET_NZ8(AREG);
}

/* $b9 ADCA extended ***** */
static void adca_ex( void )
{
	WORD t, r;

	EXTBYTE(t);
	r = AREG + t + (CCREG &0x01);
	CLR_HNZVC;
	SET_FLAGS8(AREG, t ,r);
	SET_H(AREG, t, r);
	AREG = (BYTE)r;
}

/* $bA ORA extended -**0- */
static void ora_ex( void )
{
	BYTE t;
	EXTBYTE(t); AREG |= t;
	CLR_NZV; SET_NZ8(AREG);
}

/* $bB ADDA extended ***** */
static void adda_ex( void )
{
	WORD t, r;

	EXTBYTE(t);
	r = AREG + t;
	CLR_HNZVC;
	SET_FLAGS8(AREG, t, r);
	SET_H(AREG, t, r);
	AREG = (BYTE)r;
}

/* $bC CMPX (CMPY CMPS) extended -**** */
static void cmpx_ex( void )
{
	DWORD r,d,b;

	EXTWORD(b);
	b &= 0xffff;
	d = XREG;
	r = d - b;
	CLR_NZVC;
	SET_FLAGS16(d,b,r);
}

/* $10bC CMPY extended -**** */
static void cmpy_ex( void )
{
	DWORD r,d,b;

	EXTWORD(b);
	b &= 0xffff;
	d = YREG;
	r = d - b;
	CLR_NZVC;
	SET_FLAGS16(d,b,r);
}

/* $11bC CMPS extended -**** */
static void cmps_ex( void )
{
	DWORD r,d,b;

	EXTWORD(b);
	b &= 0xffff;
	d = SREG;
	r = d - b;
	CLR_NZVC;
	SET_FLAGS16(d,b,r);
}

/* $bD JSR extended ----- */
static void jsr_ex( void )
{
	EXTENDED;
	PUSHWORD(PCREG);
	PCREG = eaddr;
}

/* $bE LDX (LDY) extended -**0- */
static void ldx_ex( void )
{
	EXTWORD(XREG);
	CLR_NZV;
	SET_NZ16(XREG);
}

/* $10bE LDY extended -**0- */
static void ldy_ex( void )
{
	EXTWORD(YREG);
	CLR_NZV;
	SET_NZ16(YREG);
}

/* $bF STX (STY) extended -**0- */
static void stx_ex( void )
{
	CLR_NZV;
	SET_NZ16(XREG);
	EXTENDED;
	WRITEW(eaddr, XREG);
}

/* $10bF STY extended -**0- */
static void sty_ex( void )
{
	CLR_NZV;
	SET_NZ16(YREG);
	EXTENDED;
	WRITEW(eaddr,YREG);
}

/* $c0 SUBB immediate ?**** */
static void subb_im( void )
{
	WORD t,r;

	IMMBYTE(t);
	r = BREG - t;
	CLR_NZVC;
	SET_FLAGS8(BREG, t, r);
	BREG = (BYTE)r;
}

/* $c1 CMPB immediate ?**** */
static void cmpb_im( void )
{
	WORD t,r;

	IMMBYTE(t); 
	r = BREG - t;
	CLR_NZVC;
	SET_FLAGS8(BREG, t, r);
}

/* $c2 SBCB immediate ?**** */
static void sbcb_im( void )
{
	WORD t, r;

	IMMBYTE(t);
	r = BREG - t - (CCREG &0x01);
	CLR_NZVC;
	SET_FLAGS8(BREG, t, r);
	BREG = (BYTE)r;
}

/* $c3 ADDD immediate -**** */
static void addd_im( void )
{
	DWORD t, r;

	IMMWORD(t); t &= 0xffff;
	r = DREG + t;
	CLR_NZVC;
	SET_FLAGS16(DREG, t, r);
	DREG = (WORD)r;
}

/* $c4 ANDB immediate -**0- */
static void andb_im( void )
{
	BYTE t;

	IMMBYTE(t);
	BREG &= t;
	CLR_NZV;
	SET_NZ8(BREG);
}

/* $c5 BITB immediate -**0- */
static void bitb_im( void )
{
	BYTE t, r;

	IMMBYTE(t);
	r = BREG & t;
	CLR_NZV;
	SET_NZ8(r);
}

/* $c6 LDB immediate -**0- */
static void ldb_im( void )
{
	IMMBYTE(BREG);
	CLR_NZV;
	SET_NZ8(BREG);
}

/* is this a legal instruction? */
/* $c7 STB immediate -**0- */
static void stb_im( void )
{
}

/* $c8 EORB immediate -**0- */
static void eorb_im( void )
{
	BYTE t;

	IMMBYTE(t);
	BREG ^= t;
	CLR_NZV;
	SET_NZ8(BREG);
}

/* $c9 ADCB immediate ***** */
static void adcb_im( void )
{
	WORD t, r;

	IMMBYTE(t);
	r = BREG + t + (CCREG &0x01);
	CLR_HNZVC;
	SET_FLAGS8(BREG, t, r);
	SET_H(BREG, t, r);
	BREG = (BYTE)r;
}

/* $cA ORB immediate -**0- */
static void orb_im( void )
{
	BYTE t;

	IMMBYTE(t);
	BREG |= t;
	CLR_NZV;
	SET_NZ8(BREG);
}

/* $cB ADDB immediate ***** */
static void addb_im( void )
{
	WORD t, r;

	IMMBYTE(t);
	r = BREG + t;
	CLR_HNZVC;
	SET_FLAGS8(BREG, t, r);
	SET_H(BREG, t, r);
	BREG = (BYTE)r;
}

/* $cC LDD immediate -**0- */
static void ldd_im( void )
{
	WORD t;

	IMMWORD(t);
	DREG = t;
	CLR_NZV;
	SET_NZ16(t);
}

/* is this a legal instruction? */
/* $cD STD immediate -**0- */
static void std_im( void )
{
}

/* $cE LDU (LDS) immediate -**0- */
static void ldu_im( void )
{
	IMMWORD(UREG);
	CLR_NZV;
	SET_NZ16(UREG);
}

/* $10cE LDS immediate -**0- */
static void lds_im( void )
{
	IMMWORD(SREG);
	CLR_NZV;
	SET_NZ16(SREG);

	/* FM-7(Type-C ROM)̂ݒʗp */
	INTR |= INTR_SLOAD;
}

/* is this a legal instruction? */
/* $cF STU (STS) immediate -**0- */
static void stu_im( void )
{
}

/* is this a legal instruction? */
/* $10cF STS immediate -**0- */
static void sts_im( void )
{
}

/* $d0 SUBB direct ?**** */
static void subb_di( void )
{
	WORD t, r;

	DIRBYTE(t);
	r = BREG - t;
	CLR_NZVC;
	SET_FLAGS8(BREG, t, r);
	BREG = (BYTE)r;
}

/* $d1 CMPB direct ?**** */
static void cmpb_di( void )
{
	WORD t,r;
	DIRBYTE(t); r = BREG-t;
	CLR_NZVC; SET_FLAGS8(BREG,t,r);
}

/* $d2 SBCB direct ?**** */
static void sbcb_di( void )
{
	WORD t, r;

	DIRBYTE(t);
	r = BREG - t - (CCREG &0x01);
	CLR_NZVC;
	SET_FLAGS8(BREG, t, r);
	BREG = (BYTE)r;
}

/* $d3 ADDD direct -**** */
static void addd_di( void )
{
	DWORD t, r;

	DIRWORD(t); t &= 0xffff;
	r = DREG + t;
	CLR_NZVC;
	SET_FLAGS16(DREG, t, r);
	DREG = (WORD)r;
}

/* $d4 ANDB direct -**0- */
static void andb_di( void )
{
	BYTE t;
	DIRBYTE(t); BREG &= t;
	CLR_NZV; SET_NZ8(BREG);
}

/* $d5 BITB direct -**0- */
static void bitb_di( void )
{
	BYTE t,r;
	DIRBYTE(t); r = BREG&t;
	CLR_NZV; SET_NZ8(r);
}

/* $d6 LDB direct -**0- */
static void ldb_di( void )
{
	DIRBYTE(BREG);
	CLR_NZV; SET_NZ8(BREG);
}

/* $d7 STB direct -**0- */
static void stb_di( void )
{
	CLR_NZV; SET_NZ8(BREG);
	DIRECT; WRITEB(eaddr,BREG);
}

/* $d8 EORB direct -**0- */
static void eorb_di( void )
{
	BYTE t;
	DIRBYTE(t); BREG ^= t;
	CLR_NZV; SET_NZ8(BREG);
}

/* $d9 ADCB direct ***** */
static void adcb_di( void )
{
	WORD t, r;

	DIRBYTE(t);
	r = BREG + t + (CCREG &0x01);
	CLR_HNZVC;
	SET_FLAGS8(BREG, t, r);
	SET_H(BREG, t, r);
	BREG = (BYTE)r;
}

/* $dA ORB direct -**0- */
static void orb_di( void )
{
	BYTE t;
	DIRBYTE(t); BREG |= t;
	CLR_NZV; SET_NZ8(BREG);
}

/* $dB ADDB direct ***** */
static void addb_di( void )
{
	WORD t, r;

	DIRBYTE(t);
	r = BREG + t;
	CLR_HNZVC;
	SET_FLAGS8(BREG, t, r);
	SET_H(BREG, t, r);
	BREG = (BYTE)r;
}

/* $dC LDD direct -**0- */
static void ldd_di( void )
{
	WORD t;
	DIRWORD(t); DREG = t;
	CLR_NZV; SET_NZ16(t);
}

/* $dD STD direct -**0- */
static void std_di( void )
{
	DIRECT;
	CLR_NZV;
	SET_NZ16(DREG);
	WRITEW(eaddr, DREG);
}

/* $dE LDU (LDS) direct -**0- */
static void ldu_di( void )
{
	DIRWORD(UREG);
	CLR_NZV; SET_NZ16(UREG);
}

/* $10dE LDS direct -**0- */
static void lds_di( void )
{
	DIRWORD(SREG);
	CLR_NZV; SET_NZ16(SREG);
}

/* $dF STU (STS) direct -**0- */
static void stu_di( void )
{
	CLR_NZV;
	SET_NZ16(UREG);
	DIRECT;
	WRITEW(eaddr, UREG);
}

/* $10dF STS direct -**0- */
static void sts_di( void )
{
	CLR_NZV;
	SET_NZ16(SREG);
	DIRECT;
	WRITEW(eaddr, SREG);
}

/* $e0 SUBB indexed ?**** */
static void subb_ix( void )
{
	WORD t, r;

	t = READB(eaddr);
	r = BREG - t;
	CLR_NZVC;
	SET_FLAGS8(BREG, t, r);
	BREG = (BYTE)r;
}

/* $e1 CMPB indexed ?**** */
static void cmpb_ix( void )
{
	WORD t, r;

	t = READB(eaddr);
	r = BREG - t;
	CLR_NZVC;
	SET_FLAGS8(BREG, t, r);
}

/* $e2 SBCB indexed ?**** */
static void sbcb_ix( void )
{
	WORD t, r;

	t = READB(eaddr);
	r = BREG - t - (CCREG &0x01);
	CLR_NZVC;
	SET_FLAGS8(BREG, t, r);
	BREG = (BYTE)r;
}

/* $e3 ADDD indexed -**** */
static void addd_ix( void )
{
	DWORD t, r;

	t = READW(eaddr); t &= 0xffff;
	r = DREG + t;
	CLR_NZVC;
	SET_FLAGS16(DREG, t, r);
	DREG = (WORD)r;
}

/* $e4 ANDB indexed -**0- */
static void andb_ix( void )
{
	BREG &= READB(eaddr);
	CLR_NZV; SET_NZ8(BREG);
}

/* $e5 BITB indexed -**0- */
static void bitb_ix( void )
{
	BYTE r;
	r = BREG&READB(eaddr);
	CLR_NZV; SET_NZ8(r);
}

/* $e6 LDB indexed -**0- */
static void ldb_ix( void )
{
	BREG = READB(eaddr);
	CLR_NZV; SET_NZ8(BREG);
}

/* $e7 STB indexed -**0- */
static void stb_ix( void )
{
	CLR_NZV; SET_NZ8(BREG);
	WRITEB(eaddr,BREG);
}

/* $e8 EORB indexed -**0- */
static void eorb_ix( void )
{
	BREG ^= READB(eaddr);
	CLR_NZV; SET_NZ8(BREG);
}

/* $e9 ADCB indexed ***** */
static void adcb_ix( void )
{
	WORD t, r;

	t = READB(eaddr);
	r = BREG + t + (CCREG &0x01);
	CLR_HNZVC;
	SET_FLAGS8(BREG, t, r);
	SET_H(BREG, t, r);
	BREG = (BYTE)r;
}

/* $eA ORB indexed -**0- */
static void orb_ix( void )
{
	BREG |= READB(eaddr);
	CLR_NZV; SET_NZ8(BREG);
}

/* $eB ADDB indexed ***** */
static void addb_ix( void )
{
	WORD t, r;

	t = READB(eaddr);
	r = BREG + t;
	CLR_HNZVC;
	SET_FLAGS8(BREG, t, r);
	SET_H(BREG, t, r);
	BREG = (BYTE)r;
}

/* $eC LDD indexed -**0- */
static void ldd_ix( void )
{
	WORD t;
	t = READW(eaddr);
	CLR_NZV;
	SET_NZ16(t);
	DREG = t;
}

/* $eD STD indexed -**0- */
static void std_ix( void )
{
	CLR_NZV;
	SET_NZ16(DREG);
	WRITEW(eaddr, DREG);
}

/* $eE LDU (LDS) indexed -**0- */
static void ldu_ix( void )
{
	UREG = READW(eaddr);
	CLR_NZV;
	SET_NZ16(UREG);
}

/* $10eE LDS indexed -**0- */
static void lds_ix( void )
{
	SREG = READW(eaddr);
	CLR_NZV;
	SET_NZ16(SREG);
}

/* $eF STU (STS) indexed -**0- */
static void stu_ix( void )
{
	CLR_NZV;
	SET_NZ16(UREG);
	WRITEW(eaddr, UREG);
}

/* $10eF STS indexed -**0- */
static void sts_ix( void )
{
	CLR_NZV;
	SET_NZ16(SREG);
	WRITEW(eaddr, SREG);
}

/* $f0 SUBB extended ?**** */
static void subb_ex( void )
{
	WORD t, r;

	EXTBYTE(t);
	r = BREG - t;
	CLR_NZVC;
	SET_FLAGS8(BREG, t, r);
	BREG = (BYTE)r;
}

/* $f1 CMPB extended ?**** */
static void cmpb_ex( void )
{
	WORD t, r;

	EXTBYTE(t);
	r = BREG - t;
	CLR_NZVC;
	SET_FLAGS8(BREG, t, r);
}

/* $f2 SBCB extended ?**** */
static void sbcb_ex( void )
{
	WORD t, r;

	EXTBYTE(t);
	r = BREG - t - (CCREG &0x01);
	CLR_NZVC;
	SET_FLAGS8(BREG, t, r);
	BREG = (BYTE)r;
}

/* $f3 ADDD extended -**** */
static void addd_ex( void )
{
	DWORD t, r;

	EXTWORD(t); t &= 0xffff;
	r = DREG + t;
	CLR_NZVC;
	SET_FLAGS16(DREG ,t ,r);
	DREG = (WORD)r;
}

/* $f4 ANDB extended -**0- */
static void andb_ex( void )
{
	BYTE t;
	EXTBYTE(t); BREG &= t;
	CLR_NZV; SET_NZ8(BREG);
}

/* $f5 BITB extended -**0- */
static void bitb_ex( void )
{
	BYTE t,r;

	EXTBYTE(t);
	r = BREG & t;
	CLR_NZV;
	SET_NZ8(r);
}

/* $f6 LDB extended -**0- */
static void ldb_ex( void )
{
	EXTBYTE(BREG);
	CLR_NZV;
	SET_NZ8(BREG);
}

/* $f7 STB extended -**0- */
static void stb_ex( void )
{
	CLR_NZV;
	SET_NZ8(BREG);
	EXTENDED;
	WRITEB(eaddr,BREG);
}

/* $f8 EORB extended -**0- */
static void eorb_ex( void )
{
	BYTE t;
	EXTBYTE(t); BREG ^= t;
	CLR_NZV; SET_NZ8(BREG);
}

/* $f9 ADCB extended ***** */
static void adcb_ex( void )
{
	WORD t, r;

	EXTBYTE(t);
	r = BREG + t + (CCREG &0x01);
	CLR_HNZVC;
	SET_FLAGS8(BREG, t, r);
	SET_H(BREG, t, r);
	BREG = (BYTE)r;
}

/* $fA ORB extended -**0- */
static void orb_ex( void )
{
	BYTE t;
	EXTBYTE(t); BREG |= t;
	CLR_NZV; SET_NZ8(BREG);
}

/* $fB ADDB extended ***** */
static void addb_ex( void )
{
	WORD t, r;

	EXTBYTE(t);
	r = BREG + t;
	CLR_HNZVC;
	SET_FLAGS8(BREG, t, r);
	SET_H(BREG, t, r);
	BREG = (BYTE)r;
}

/* $fC LDD extended -**0- */
static void ldd_ex( void )
{
	WORD t;

	EXTWORD(t);
	DREG = t;
	CLR_NZV;
	SET_NZ16(t);
}

/* $fD STD extended -**0- */
static void std_ex( void )
{
	EXTENDED;
	CLR_NZV;
	SET_NZ16(DREG);
	WRITEW(eaddr, DREG);
}

/* $fE LDU (LDS) extended -**0- */
static void ldu_ex( void )
{
	EXTWORD(UREG);
	CLR_NZV;
	SET_NZ16(UREG);
}

/* $10fE LDS extended -**0- */
static void lds_ex( void )
{
	EXTWORD(SREG);
	CLR_NZV;
	SET_NZ16(SREG);
}

/* $fF STU (STS) extended -**0- */
static void stu_ex( void )
{
	CLR_NZV;
	SET_NZ16(UREG);
	EXTENDED;
	WRITEW(eaddr, UREG);
}

/* $10fF STS extended -**0- */
static void sts_ex( void )
{
	CLR_NZV;
	SET_NZ16(SREG);
	EXTENDED;
	WRITEW(eaddr, SREG);
}

/*-[ fBXpb` ]-------------------------------------------------------*/

/*
 *	AhX擾(5bit)
 */
static void cpu_ea_5bit(BYTE postb)
{
	switch (postb) {
		case 0x00: eaddr=XREG; break;
		case 0x01: eaddr=XREG+1; break;
		case 0x02: eaddr=XREG+2; break;
		case 0x03: eaddr=XREG+3; break;
		case 0x04: eaddr=XREG+4; break;
		case 0x05: eaddr=XREG+5; break;
		case 0x06: eaddr=XREG+6; break;
		case 0x07: eaddr=XREG+7; break;
		case 0x08: eaddr=XREG+8; break;
		case 0x09: eaddr=XREG+9; break;
		case 0x0A: eaddr=XREG+10; break;
		case 0x0B: eaddr=XREG+11; break;
		case 0x0C: eaddr=XREG+12; break;
		case 0x0D: eaddr=XREG+13; break;
		case 0x0E: eaddr=XREG+14; break;
		case 0x0F: eaddr=XREG+15; break;
		case 0x10: eaddr=XREG-16; break;
		case 0x11: eaddr=XREG-15; break;
		case 0x12: eaddr=XREG-14; break;
		case 0x13: eaddr=XREG-13; break;
		case 0x14: eaddr=XREG-12; break;
		case 0x15: eaddr=XREG-11; break;
		case 0x16: eaddr=XREG-10; break;
		case 0x17: eaddr=XREG-9; break;
		case 0x18: eaddr=XREG-8; break;
		case 0x19: eaddr=XREG-7; break;
		case 0x1A: eaddr=XREG-6; break;
		case 0x1B: eaddr=XREG-5; break;
		case 0x1C: eaddr=XREG-4; break;
		case 0x1D: eaddr=XREG-3; break;
		case 0x1E: eaddr=XREG-2; break;
		case 0x1F: eaddr=XREG-1; break;
		case 0x20: eaddr=YREG; break;
		case 0x21: eaddr=YREG+1; break;
		case 0x22: eaddr=YREG+2; break;
		case 0x23: eaddr=YREG+3; break;
		case 0x24: eaddr=YREG+4; break;
		case 0x25: eaddr=YREG+5; break;
		case 0x26: eaddr=YREG+6; break;
		case 0x27: eaddr=YREG+7; break;
		case 0x28: eaddr=YREG+8; break;
		case 0x29: eaddr=YREG+9; break;
		case 0x2A: eaddr=YREG+10; break;
		case 0x2B: eaddr=YREG+11; break;
		case 0x2C: eaddr=YREG+12; break;
		case 0x2D: eaddr=YREG+13; break;
		case 0x2E: eaddr=YREG+14; break;
		case 0x2F: eaddr=YREG+15; break;
		case 0x30: eaddr=YREG-16; break;
		case 0x31: eaddr=YREG-15; break;
		case 0x32: eaddr=YREG-14; break;
		case 0x33: eaddr=YREG-13; break;
		case 0x34: eaddr=YREG-12; break;
		case 0x35: eaddr=YREG-11; break;
		case 0x36: eaddr=YREG-10; break;
		case 0x37: eaddr=YREG-9; break;
		case 0x38: eaddr=YREG-8; break;
		case 0x39: eaddr=YREG-7; break;
		case 0x3A: eaddr=YREG-6; break;
		case 0x3B: eaddr=YREG-5; break;
		case 0x3C: eaddr=YREG-4; break;
		case 0x3D: eaddr=YREG-3; break;
		case 0x3E: eaddr=YREG-2; break;
		case 0x3F: eaddr=YREG-1; break;
		case 0x40: eaddr=UREG; break;
		case 0x41: eaddr=UREG+1; break;
		case 0x42: eaddr=UREG+2; break;
		case 0x43: eaddr=UREG+3; break;
		case 0x44: eaddr=UREG+4; break;
		case 0x45: eaddr=UREG+5; break;
		case 0x46: eaddr=UREG+6; break;
		case 0x47: eaddr=UREG+7; break;
		case 0x48: eaddr=UREG+8; break;
		case 0x49: eaddr=UREG+9; break;
		case 0x4A: eaddr=UREG+10; break;
		case 0x4B: eaddr=UREG+11; break;
		case 0x4C: eaddr=UREG+12; break;
		case 0x4D: eaddr=UREG+13; break;
		case 0x4E: eaddr=UREG+14; break;
		case 0x4F: eaddr=UREG+15; break;
		case 0x50: eaddr=UREG-16; break;
		case 0x51: eaddr=UREG-15; break;
		case 0x52: eaddr=UREG-14; break;
		case 0x53: eaddr=UREG-13; break;
		case 0x54: eaddr=UREG-12; break;
		case 0x55: eaddr=UREG-11; break;
		case 0x56: eaddr=UREG-10; break;
		case 0x57: eaddr=UREG-9; break;
		case 0x58: eaddr=UREG-8; break;
		case 0x59: eaddr=UREG-7; break;
		case 0x5A: eaddr=UREG-6; break;
		case 0x5B: eaddr=UREG-5; break;
		case 0x5C: eaddr=UREG-4; break;
		case 0x5D: eaddr=UREG-3; break;
		case 0x5E: eaddr=UREG-2; break;
		case 0x5F: eaddr=UREG-1; break;
		case 0x60: eaddr=SREG; break;
		case 0x61: eaddr=SREG+1; break;
		case 0x62: eaddr=SREG+2; break;
		case 0x63: eaddr=SREG+3; break;
		case 0x64: eaddr=SREG+4; break;
		case 0x65: eaddr=SREG+5; break;
		case 0x66: eaddr=SREG+6; break;
		case 0x67: eaddr=SREG+7; break;
		case 0x68: eaddr=SREG+8; break;
		case 0x69: eaddr=SREG+9; break;
		case 0x6A: eaddr=SREG+10; break;
		case 0x6B: eaddr=SREG+11; break;
		case 0x6C: eaddr=SREG+12; break;
		case 0x6D: eaddr=SREG+13; break;
		case 0x6E: eaddr=SREG+14; break;
		case 0x6F: eaddr=SREG+15; break;
		case 0x70: eaddr=SREG-16; break;
		case 0x71: eaddr=SREG-15; break;
		case 0x72: eaddr=SREG-14; break;
		case 0x73: eaddr=SREG-13; break;
		case 0x74: eaddr=SREG-12; break;
		case 0x75: eaddr=SREG-11; break;
		case 0x76: eaddr=SREG-10; break;
		case 0x77: eaddr=SREG-9; break;
		case 0x78: eaddr=SREG-8; break;
		case 0x79: eaddr=SREG-7; break;
		case 0x7A: eaddr=SREG-6; break;
		case 0x7B: eaddr=SREG-5; break;
		case 0x7C: eaddr=SREG-4; break;
		case 0x7D: eaddr=SREG-3; break;
		case 0x7E: eaddr=SREG-2; break;
		case 0x7F: eaddr=SREG-1; break;
	}
}

/*
 *	AhX擾iCfbNXAhbVOj
 */
static void cpu_calc_ea(void)
{
	BYTE postb;

	/* |XgoCg擾ATCNJE^ */
	postb = READB(PCREG++);

	/*  */
	if (postb < 0x80) {
		cpu_ea_5bit(postb);
		CYCLE++;
		return;
	}

	switch (postb) {
		case 0x80:	eaddr=XREG; XREG++; CYCLE+=2; break;
		case 0x81:	eaddr=XREG; XREG+=2; CYCLE+=3; break;
		case 0x82:	XREG--; eaddr=XREG; CYCLE+=2; break;
		case 0x83:	XREG-=2; eaddr=XREG; CYCLE+=3; break;
		case 0x84:	eaddr=XREG; break;
		case 0x85:	eaddr=XREG+SIGNED(BREG); CYCLE++; break;
		case 0x86:	eaddr=XREG+SIGNED(AREG); CYCLE++; break;
		case 0x87:	eaddr=0; break; /*ILLEGAL*/
		case 0x88:	IMMBYTE(eaddr);eaddr=XREG+SIGNED(eaddr); CYCLE++; break;
		case 0x89:	IMMWORD(eaddr);eaddr+=XREG; CYCLE+=4; break;
		case 0x8A:	eaddr=0;break; /*ILLEGAL*/
		case 0x8B:	eaddr=XREG+DREG; CYCLE+=4; break;
		case 0x8C:	IMMBYTE(eaddr);eaddr=PCREG+SIGNED(eaddr); CYCLE++; break;
		case 0x8D:	IMMWORD(eaddr);eaddr+=PCREG; CYCLE+=5; break;
		case 0x8E:	eaddr=0;break; /*ILLEGAL*/
		case 0x8F:	IMMWORD(eaddr); CYCLE+=5; break;
		case 0x90:	eaddr=XREG;XREG++;eaddr=READW(eaddr); CYCLE+=5; break;
		case 0x91:	eaddr=XREG;XREG+=2;eaddr=READW(eaddr); CYCLE+=6; break;
		case 0x92:	XREG--;eaddr=XREG;eaddr=READW(eaddr); CYCLE+=5; break;
		case 0x93:	XREG-=2;eaddr=XREG;eaddr=READW(eaddr); CYCLE+=6; break;
		case 0x94:	eaddr=XREG;eaddr=READW(eaddr); CYCLE+=3; break;
		case 0x95:	eaddr=XREG+SIGNED(BREG);eaddr=READW(eaddr); CYCLE+=4; break;
		case 0x96:	eaddr=XREG+SIGNED(AREG);eaddr=READW(eaddr); CYCLE+=4; break;
		case 0x97:	eaddr=0;break; /*ILLEGAL*/
		case 0x98:	IMMBYTE(eaddr);eaddr=XREG+SIGNED(eaddr);
					eaddr=READW(eaddr); CYCLE+=4; break;
		case 0x99:	IMMWORD(eaddr);eaddr+=XREG;eaddr=READW(eaddr); CYCLE+=7; break;
		case 0x9A:	eaddr=0;break; /*ILLEGAL*/
		case 0x9B:	eaddr=XREG+DREG;eaddr=READW(eaddr); CYCLE+=7; break;
		case 0x9C:	IMMBYTE(eaddr);eaddr=PCREG+SIGNED(eaddr);
					eaddr=READW(eaddr); CYCLE+=4; break;
		case 0x9D:	IMMWORD(eaddr);eaddr+=PCREG;eaddr=READW(eaddr); CYCLE+=8; break;
		case 0x9E:	eaddr=0;break; /*ILLEGAL*/
		case 0x9F:	IMMWORD(eaddr);eaddr=READW(eaddr); CYCLE+=8; break;
		case 0xA0:	eaddr=YREG;YREG++; CYCLE+=2; break;
		case 0xA1:	eaddr=YREG;YREG+=2; CYCLE+=3; break;
		case 0xA2:	YREG--;eaddr=YREG; CYCLE+=2; break;
		case 0xA3:	YREG-=2;eaddr=YREG; CYCLE+=3; break;
		case 0xA4:	eaddr=YREG;break;
		case 0xA5:	eaddr=YREG+SIGNED(BREG); CYCLE++; break;
		case 0xA6:	eaddr=YREG+SIGNED(AREG); CYCLE++; break;
		case 0xA7:	eaddr=0;break; /*ILLEGAL*/
		case 0xA8:	IMMBYTE(eaddr);eaddr=YREG+SIGNED(eaddr); CYCLE++; break;
		case 0xA9:	IMMWORD(eaddr);eaddr+=YREG; CYCLE+=4; break;
		case 0xAA:	eaddr=0;break; /*ILLEGAL*/
		case 0xAB:	eaddr=YREG+DREG; CYCLE+=4; break;
		case 0xAC:	IMMBYTE(eaddr);eaddr=PCREG+SIGNED(eaddr); CYCLE++; break;
		case 0xAD:	IMMWORD(eaddr);eaddr+=PCREG; CYCLE+=5; break;
		case 0xAE:	eaddr=0;break; /*ILLEGAL*/
		case 0xAF:	IMMWORD(eaddr); CYCLE+=5; break;
		case 0xB0:	eaddr=YREG;YREG++;eaddr=READW(eaddr); CYCLE+=5; break;
		case 0xB1:	eaddr=YREG;YREG+=2;eaddr=READW(eaddr); CYCLE+=6; break;
		case 0xB2:	YREG--;eaddr=YREG;eaddr=READW(eaddr); CYCLE+=5; break;
		case 0xB3:	YREG-=2;eaddr=YREG;eaddr=READW(eaddr); CYCLE+=6; break;
		case 0xB4:	eaddr=YREG;eaddr=READW(eaddr); CYCLE+=3; break;
		case 0xB5:	eaddr=YREG+SIGNED(BREG);eaddr=READW(eaddr); CYCLE+=4; break;
		case 0xB6:	eaddr=YREG+SIGNED(AREG);eaddr=READW(eaddr); CYCLE+=4; break;
		case 0xB7:	eaddr=0;break; /*ILLEGAL*/
		case 0xB8:	IMMBYTE(eaddr);eaddr=YREG+SIGNED(eaddr);
					eaddr=READW(eaddr); CYCLE+=4; break;
		case 0xB9:	IMMWORD(eaddr);eaddr+=YREG;eaddr=READW(eaddr); CYCLE+=7; break;
		case 0xBA:	eaddr=0;break; /*ILLEGAL*/
		case 0xBB:	eaddr=YREG+DREG;eaddr=READW(eaddr); CYCLE+=7; break;
		case 0xBC:	IMMBYTE(eaddr);eaddr=PCREG+SIGNED(eaddr);
					eaddr=READW(eaddr); CYCLE+=4; break;
		case 0xBD:	IMMWORD(eaddr);eaddr+=PCREG;eaddr=READW(eaddr); CYCLE+=8; break;
		case 0xBE:	eaddr=0;break; /*ILLEGAL*/
		case 0xBF:	IMMWORD(eaddr);eaddr=READW(eaddr); CYCLE+=8; break;
		case 0xC0:	eaddr=UREG;UREG++; CYCLE+=2; break;
		case 0xC1:	eaddr=UREG;UREG+=2; CYCLE+=3; break;
		case 0xC2:	UREG--;eaddr=UREG; CYCLE+=2; break;
		case 0xC3:	UREG-=2;eaddr=UREG; CYCLE+=3; break;
		case 0xC4:	eaddr=UREG;break;
		case 0xC5:	eaddr=UREG+SIGNED(BREG); CYCLE++; break;
		case 0xC6:	eaddr=UREG+SIGNED(AREG); CYCLE++; break;
		case 0xC7:	eaddr=0;break; /*ILLEGAL*/
		case 0xC8:	IMMBYTE(eaddr);eaddr=UREG+SIGNED(eaddr); CYCLE++; break;
		case 0xC9:	IMMWORD(eaddr);eaddr+=UREG; CYCLE+=4; break;
		case 0xCA:	eaddr=0;break; /*ILLEGAL*/
		case 0xCB:	eaddr=UREG+DREG; CYCLE+=4; break;
		case 0xCC:	IMMBYTE(eaddr);eaddr=PCREG+SIGNED(eaddr); CYCLE++; break;
		case 0xCD:	IMMWORD(eaddr);eaddr+=PCREG; CYCLE+=5; break;
		case 0xCE:	eaddr=0;break; /*ILLEGAL*/
		case 0xCF:	IMMWORD(eaddr); CYCLE+=5; break;
		case 0xD0:	eaddr=UREG;UREG++;eaddr=READW(eaddr); CYCLE+=5; break;
		case 0xD1:	eaddr=UREG;UREG+=2;eaddr=READW(eaddr); CYCLE+=6; break;
		case 0xD2:	UREG--;eaddr=UREG;eaddr=READW(eaddr); CYCLE+=5; break;
		case 0xD3:	UREG-=2;eaddr=UREG;eaddr=READW(eaddr); CYCLE+=6; break;
		case 0xD4:	eaddr=UREG;eaddr=READW(eaddr); CYCLE+=3; break;
		case 0xD5:	eaddr=UREG+SIGNED(BREG);eaddr=READW(eaddr); CYCLE+=4; break;
		case 0xD6:	eaddr=UREG+SIGNED(AREG);eaddr=READW(eaddr); CYCLE+=4; break;
		case 0xD7:	eaddr=0;break; /*ILLEGAL*/
		case 0xD8:	IMMBYTE(eaddr);eaddr=UREG+SIGNED(eaddr);
					eaddr=READW(eaddr); CYCLE+=4; break;
		case 0xD9:	IMMWORD(eaddr);eaddr+=UREG;eaddr=READW(eaddr); CYCLE+=7; break;
		case 0xDA:	eaddr=0;break; /*ILLEGAL*/
		case 0xDB:	eaddr=UREG+DREG;eaddr=READW(eaddr); CYCLE+=7; break;
		case 0xDC:	IMMBYTE(eaddr);eaddr=PCREG+SIGNED(eaddr);
					eaddr=READW(eaddr); CYCLE+=4; break;
		case 0xDD:	IMMWORD(eaddr);eaddr+=PCREG;eaddr=READW(eaddr); CYCLE+=8; break;
		case 0xDE:	eaddr=0;break; /*ILLEGAL*/
		case 0xDF:	IMMWORD(eaddr);eaddr=READW(eaddr); CYCLE+=8; break;
		case 0xE0:	eaddr=SREG;SREG++; CYCLE+=2; break;
		case 0xE1:	eaddr=SREG;SREG+=2; CYCLE+=3; break;
		case 0xE2:	SREG--;eaddr=SREG; CYCLE+=2; break;
		case 0xE3:	SREG-=2;eaddr=SREG; CYCLE+=3; break;
		case 0xE4:	eaddr=SREG;break;
		case 0xE5:	eaddr=SREG+SIGNED(BREG); CYCLE++; break;
		case 0xE6:	eaddr=SREG+SIGNED(AREG); CYCLE++; break;
		case 0xE7:	eaddr=0;break; /*ILLEGAL*/
		case 0xE8:	IMMBYTE(eaddr);eaddr=SREG+SIGNED(eaddr); CYCLE++; break;
		case 0xE9:	IMMWORD(eaddr);eaddr+=SREG; CYCLE+=4; break;
		case 0xEA:	eaddr=0;break; /*ILLEGAL*/
		case 0xEB:	eaddr=SREG+DREG; CYCLE+=4; break;
		case 0xEC:	IMMBYTE(eaddr);eaddr=PCREG+SIGNED(eaddr); CYCLE++; break;
		case 0xED:	IMMWORD(eaddr);eaddr+=PCREG; CYCLE+=5; break;
		case 0xEE:	eaddr=0;break; /*ILLEGAL*/
		case 0xEF:	IMMWORD(eaddr); CYCLE+=5; break;
		case 0xF0:	eaddr=SREG;SREG++;eaddr=READW(eaddr); CYCLE+=5; break;
		case 0xF1:	eaddr=SREG;SREG+=2;eaddr=READW(eaddr); CYCLE+=6; break;
		case 0xF2:	SREG--;eaddr=SREG;eaddr=READW(eaddr); CYCLE+=5; break;
		case 0xF3:	SREG-=2;eaddr=SREG;eaddr=READW(eaddr); CYCLE+=6; break;
		case 0xF4:	eaddr=SREG;eaddr=READW(eaddr); CYCLE+=3; break;
		case 0xF5:	eaddr=SREG+SIGNED(BREG);eaddr=READW(eaddr); CYCLE+=4; break;
		case 0xF6:	eaddr=SREG+SIGNED(AREG);eaddr=READW(eaddr); CYCLE+=4; break;
		case 0xF7:	eaddr=0;break; /*ILLEGAL*/
		case 0xF8:	IMMBYTE(eaddr);eaddr=SREG+SIGNED(eaddr);
					eaddr=READW(eaddr); CYCLE+=4; break;
		case 0xF9:	IMMWORD(eaddr);eaddr+=SREG;eaddr=READW(eaddr); CYCLE+=7; break;
		case 0xFA:	eaddr=0;break; /*ILLEGAL*/
		case 0xFB:	eaddr=SREG+DREG;eaddr=READW(eaddr); CYCLE+=7; break;
		case 0xFC:	IMMBYTE(eaddr);eaddr=PCREG+SIGNED(eaddr);
					eaddr=READW(eaddr); CYCLE+=4; break;
		case 0xFD:	IMMWORD(eaddr);eaddr+=PCREG;eaddr=READW(eaddr); CYCLE+=8; break;
		case 0xFE:	eaddr=0;break; /*ILLEGAL*/
		case 0xFF:	IMMWORD(eaddr);eaddr=READW(eaddr); CYCLE+=8; break;
	}
}

/*
 *	`
 */
static void illegal(void)
{
}

/*
 *	CPUPߎs
 */
static void cpu_execline(void)
{
	BYTE opc;

	/* IyR[h擾 */
	opc = READB(PCREG++);

	/* PoCg߂`FbNAEAvZ */
	CYCLE = cycles1[opc];
	if (CYCLE != 0x00ff) {
		if (CYCLE & 0x0080) {
			CYCLE &= 0x007f;
			cpu_calc_ea();
		}
		switch (opc) {
			case 0x00: neg_di(); break;
			case 0x01: neg_di(); break;
			case 0x02: ngc_di(); break;
			case 0x03: com_di(); break;
			case 0x04: lsr_di(); break;
			case 0x05: lsr_di(); break;
			case 0x06: ror_di(); break;
			case 0x07: asr_di(); break;
			case 0x08: asl_di(); break;
			case 0x09: rol_di(); break;
			case 0x0a: dec_di(); break;
			case 0x0b: dcc_di(); break;
			case 0x0c: inc_di(); break;
			case 0x0d: tst_di(); break;
			case 0x0e: jmp_di(); break;
			case 0x0f: clr_di(); break;
			case 0x10: illegal(); break;
			case 0x11: illegal(); break;
			case 0x12: nop(); break;
			case 0x13: sync(); break;
			case 0x14: illegal(); break;
			case 0x15: illegal(); break;
			case 0x16: lbra(); break;
			case 0x17: lbsr(); break;
			case 0x18: aslcc(); break;
			case 0x19: daa(); break;
			case 0x1a: orcc(); break;
			case 0x1b: nop(); break;
			case 0x1c: andcc(); break;
			case 0x1d: sex(); break;
			case 0x1e: exg(); break;
			case 0x1f: tfr(); break;
			case 0x20: bra(); break;
			case 0x21: brn(); break;
			case 0x22: bhi(); break;
			case 0x23: bls(); break;
			case 0x24: bcc(); break;
			case 0x25: bcs(); break;
			case 0x26: bne(); break;
			case 0x27: beq(); break;
			case 0x28: bvc(); break;
			case 0x29: bvs(); break;
			case 0x2a: bpl(); break;
			case 0x2b: bmi(); break;
			case 0x2c: bge(); break;
			case 0x2d: blt(); break;
			case 0x2e: bgt(); break;
			case 0x2f: ble(); break;
			case 0x30: leax(); break;
			case 0x31: leay(); break;
			case 0x32: leas(); break;
			case 0x33: leau(); break;
			case 0x34: pshs(); break;
			case 0x35: puls(); break;
			case 0x36: pshu(); break;
			case 0x37: pulu(); break;
			case 0x38: andcc(); break;
			case 0x39: rts(); break;
			case 0x3a: abx(); break;
			case 0x3b: rti(); break;
			case 0x3c: cwai(); break;
			case 0x3d: mul(); break;
			case 0x3e: rst(); break;
			case 0x3f: swi(); break;
			case 0x40: nega(); break;
			case 0x41: nega(); break;
			case 0x42: ngca(); break;
			case 0x43: coma(); break;
			case 0x44: lsra(); break;
			case 0x45: lsra(); break;
			case 0x46: rora(); break;
			case 0x47: asra(); break;
			case 0x48: asla(); break;
			case 0x49: rola(); break;
			case 0x4a: deca(); break;
			case 0x4b: dcca(); break;
			case 0x4c: inca(); break;
			case 0x4d: tsta(); break;
			case 0x4e: clca(); break;
			case 0x4f: clra(); break;
			case 0x50: negb(); break;
			case 0x51: negb(); break;
			case 0x52: ngcb(); break;
			case 0x53: comb(); break;
			case 0x54: lsrb(); break;
			case 0x55: lsrb(); break;
			case 0x56: rorb(); break;
			case 0x57: asrb(); break;
			case 0x58: aslb(); break;
			case 0x59: rolb(); break;
			case 0x5a: decb(); break;
			case 0x5b: dccb(); break;
			case 0x5c: incb(); break;
			case 0x5d: tstb(); break;
			case 0x5e: clcb(); break;
			case 0x5f: clrb(); break;
			case 0x60: neg_ix(); break;
			case 0x61: neg_ix(); break;
			case 0x62: ngc_ix(); break;
			case 0x63: com_ix(); break;
			case 0x64: lsr_ix(); break;
			case 0x65: lsr_ix(); break;
			case 0x66: ror_ix(); break;
			case 0x67: asr_ix(); break;
			case 0x68: asl_ix(); break;
			case 0x69: rol_ix(); break;
			case 0x6a: dec_ix(); break;
			case 0x6b: dcc_ix(); break;
			case 0x6c: inc_ix(); break;
			case 0x6d: tst_ix(); break;
			case 0x6e: jmp_ix(); break;
			case 0x6f: clr_ix(); break;
			case 0x70: neg_ex(); break;
			case 0x71: neg_ex(); break;
			case 0x72: ngc_ex(); break;
			case 0x73: com_ex(); break;
			case 0x74: lsr_ex(); break;
			case 0x75: lsr_ex(); break;
			case 0x76: ror_ex(); break;
			case 0x77: asr_ex(); break;
			case 0x78: asl_ex(); break;
			case 0x79: rol_ex(); break;
			case 0x7a: dec_ex(); break;
			case 0x7b: dcc_ex(); break;
			case 0x7c: inc_ex(); break;
			case 0x7d: tst_ex(); break;
			case 0x7e: jmp_ex(); break;
			case 0x7f: clr_ex(); break;
			case 0x80: suba_im(); break;
			case 0x81: cmpa_im(); break;
			case 0x82: sbca_im(); break;
			case 0x83: subd_im(); break;
			case 0x84: anda_im(); break;
			case 0x85: bita_im(); break;
			case 0x86: lda_im(); break;
			case 0x87: flag8_im(); break;
			case 0x88: eora_im(); break;
			case 0x89: adca_im(); break;
			case 0x8a: ora_im(); break;
			case 0x8b: adda_im(); break;
			case 0x8c: cmpx_im(); break;
			case 0x8d: bsr(); break;
			case 0x8e: ldx_im(); break;
			case 0x8f: flag16_im(); break;
			case 0x90: suba_di(); break;
			case 0x91: cmpa_di(); break;
			case 0x92: sbca_di(); break;
			case 0x93: subd_di(); break;
			case 0x94: anda_di(); break;
			case 0x95: bita_di(); break;
			case 0x96: lda_di(); break;
			case 0x97: sta_di(); break;
			case 0x98: eora_di(); break;
			case 0x99: adca_di(); break;
			case 0x9a: ora_di(); break;
			case 0x9b: adda_di(); break;
			case 0x9c: cmpx_di(); break;
			case 0x9d: jsr_di(); break;
			case 0x9e: ldx_di(); break;
			case 0x9f: stx_di(); break;
			case 0xa0: suba_ix(); break;
			case 0xa1: cmpa_ix(); break;
			case 0xa2: sbca_ix(); break;
			case 0xa3: subd_ix(); break;
			case 0xa4: anda_ix(); break;
			case 0xa5: bita_ix(); break;
			case 0xa6: lda_ix(); break;
			case 0xa7: sta_ix(); break;
			case 0xa8: eora_ix(); break;
			case 0xa9: adca_ix(); break;
			case 0xaa: ora_ix(); break;
			case 0xab: adda_ix(); break;
			case 0xac: cmpx_ix(); break;
			case 0xad: jsr_ix(); break;
			case 0xae: ldx_ix(); break;
			case 0xaf: stx_ix(); break;
			case 0xb0: suba_ex(); break;
			case 0xb1: cmpa_ex(); break;
			case 0xb2: sbca_ex(); break;
			case 0xb3: subd_ex(); break;
			case 0xb4: anda_ex(); break;
			case 0xb5: bita_ex(); break;
			case 0xb6: lda_ex(); break;
			case 0xb7: sta_ex(); break;
			case 0xb8: eora_ex(); break;
			case 0xb9: adca_ex(); break;
			case 0xba: ora_ex(); break;
			case 0xbb: adda_ex(); break;
			case 0xbc: cmpx_ex(); break;
			case 0xbd: jsr_ex(); break;
			case 0xbe: ldx_ex(); break;
			case 0xbf: stx_ex(); break;
			case 0xc0: subb_im(); break;
			case 0xc1: cmpb_im(); break;
			case 0xc2: sbcb_im(); break;
			case 0xc3: addd_im(); break;
			case 0xc4: andb_im(); break;
			case 0xc5: bitb_im(); break;
			case 0xc6: ldb_im(); break;
			case 0xc7: flag8_im(); break;
			case 0xc8: eorb_im(); break;
			case 0xc9: adcb_im(); break;
			case 0xca: orb_im(); break;
			case 0xcb: addb_im(); break;
			case 0xcc: ldd_im(); break;
			case 0xcd: illegal(); break;
			case 0xce: ldu_im(); break;
			case 0xcf: flag16_im(); break;
			case 0xd0: subb_di(); break;
			case 0xd1: cmpb_di(); break;
			case 0xd2: sbcb_di(); break;
			case 0xd3: addd_di(); break;
			case 0xd4: andb_di(); break;
			case 0xd5: bitb_di(); break;
			case 0xd6: ldb_di(); break;
			case 0xd7: stb_di(); break;
			case 0xd8: eorb_di(); break;
			case 0xd9: adcb_di(); break;
			case 0xda: orb_di(); break;
			case 0xdb: addb_di(); break;
			case 0xdc: ldd_di(); break;
			case 0xdd: std_di(); break;
			case 0xde: ldu_di(); break;
			case 0xdf: stu_di(); break;
			case 0xe0: subb_ix(); break;
			case 0xe1: cmpb_ix(); break;
			case 0xe2: sbcb_ix(); break;
			case 0xe3: addd_ix(); break;
			case 0xe4: andb_ix(); break;
			case 0xe5: bitb_ix(); break;
			case 0xe6: ldb_ix(); break;
			case 0xe7: stb_ix(); break;
			case 0xe8: eorb_ix(); break;
			case 0xe9: adcb_ix(); break;
			case 0xea: orb_ix(); break;
			case 0xeb: addb_ix(); break;
			case 0xec: ldd_ix(); break;
			case 0xed: std_ix(); break;
			case 0xee: ldu_ix(); break;
			case 0xef: stu_ix(); break;
			case 0xf0: subb_ex(); break;
			case 0xf1: cmpb_ex(); break;
			case 0xf2: sbcb_ex(); break;
			case 0xf3: addd_ex(); break;
			case 0xf4: andb_ex(); break;
			case 0xf5: bitb_ex(); break;
			case 0xf6: ldb_ex(); break;
			case 0xf7: stb_ex(); break;
			case 0xf8: eorb_ex(); break;
			case 0xf9: adcb_ex(); break;
			case 0xfa: orb_ex(); break;
			case 0xfb: addb_ex(); break;
			case 0xfc: ldd_ex(); break;
			case 0xfd: std_ex(); break;
			case 0xfe: ldu_ex(); break;
			case 0xff: stu_ex(); break;
		}
	}
	else {
		/* 0x10䂩`FbNAEAvZ */
		if (opc == 0x10) {
			opc = READB(PCREG++);
			CYCLE = cycles2[opc];
			if (CYCLE & 0x0080) {
				CYCLE &= 0x007f;
				cpu_calc_ea();
			}
			switch(opc) {
				case 0x21: lbrn(); break;
				case 0x22: lbhi(); break;
				case 0x23: lbls(); break;
				case 0x24: lbcc(); break;
				case 0x25: lbcs(); break;
				case 0x26: lbne(); break;
				case 0x27: lbeq(); break;
				case 0x28: lbvc(); break;
				case 0x29: lbvs(); break;
				case 0x2a: lbpl(); break;
				case 0x2b: lbmi(); break;
				case 0x2c: lbge(); break;
				case 0x2d: lblt(); break;
				case 0x2e: lbgt(); break;
				case 0x2f: lble(); break;
				case 0x3f: swi2(); break;
				case 0x83: cmpd_im(); break;
				case 0x8c: cmpy_im(); break;
				case 0x8e: ldy_im(); break;
				case 0x8f: sty_im(); break; /* ILLEGAL? */
				case 0x93: cmpd_di(); break;
				case 0x9c: cmpy_di(); break;
				case 0x9e: ldy_di(); break;
				case 0x9f: sty_di(); break;
				case 0xa3: cmpd_ix(); break;
				case 0xac: cmpy_ix(); break;
				case 0xae: ldy_ix(); break;
				case 0xaf: sty_ix(); break;
				case 0xb3: cmpd_ex(); break;
				case 0xbc: cmpy_ex(); break;
				case 0xbe: ldy_ex(); break;
				case 0xbf: sty_ex(); break;
				case 0xce: lds_im(); break;
				case 0xcf: sts_im(); break; /* ILLEGAL? */
				case 0xde: lds_di(); break;
				case 0xdf: sts_di(); break;
				case 0xee: lds_ix(); break;
				case 0xef: sts_ix(); break;
				case 0xfe: lds_ex(); break;
				case 0xff: sts_ex(); break;
				default: illegal(); break;
				}
			}
		else {
			opc = READB(PCREG++);
			CYCLE = cycles2[opc];
			if (CYCLE & 0x0080) {
				CYCLE &= 0x007f;
				cpu_calc_ea();
			}
			switch (opc) {
				case 0x3f: swi3(); break;
				case 0x83: cmpu_im(); break;
				case 0x8c: cmps_im(); break;
				case 0x93: cmpu_di(); break;
				case 0x9c: cmps_di(); break;
				case 0xa3: cmpu_ix(); break;
				case 0xac: cmps_ix(); break;
				case 0xb3: cmpu_ex(); break;
				case 0xbc: cmps_ex(); break;
				default: illegal(); break;
			}
		}
	}
}

/*-[ Rg[ ]-------------------------------------------------------*/

/*
 *	NMI荞݂
 */
static void cpu_nmi(void)
{
	/* CWAIΉ */
	INTR |= INTR_CWAI_OUT;

	/* EtOZbgAWX^ޔ */
	CCREG |= 0x80;

	PUSHWORD(PCREG);
	PUSHWORD(UREG);
	PUSHWORD(YREG);
	PUSHWORD(XREG);
	PUSHBYTE(DPREG);
	PUSHBYTE(BREG);
	PUSHBYTE(AREG);
	PUSHBYTE(CCREG);

	/* NMIs */
	CCREG |= 0x50;
	PCREG = READW(0xfffc);

	/* tO~낷 */
	INTR &= ~INTR_NMI;
}

/*
 *	FIRQ荞݂
 */
static void cpu_firq(void)
{
	if (INTR & INTR_CWAI_IN) {
		/* CWAIsBE=1ōs */
		CCREG |= 0x80;

		/* WX^ޔ */
		PUSHWORD(PCREG);
		PUSHWORD(UREG);
		PUSHWORD(YREG);
		PUSHWORD(XREG);
		PUSHBYTE(DPREG);
		PUSHBYTE(BREG);
		PUSHBYTE(AREG);
		PUSHBYTE(CCREG);
	}
	else {
		/* EtO~낷 */
		CCREG &= 0x7f;

		/* WX^ޔ */
		PUSHWORD(PCREG);
		PUSHBYTE(CCREG);
	}

	/* FIRQs */
	CCREG |= 0x50;
	PCREG = READW(0xfff6);

	/* tO~낷 */
	INTR &= ~INTR_FIRQ;
}

/*
 *	IRQ荞݂
 */
static void cpu_irq(void)
{
	/* CWAIΉ */
	INTR |= INTR_CWAI_OUT;

	/* EtOZbg */
	CCREG |= 0x80;

	/* WX^ޔ */
	PUSHWORD(PCREG);
	PUSHWORD(UREG);
	PUSHWORD(YREG);
	PUSHWORD(XREG);
	PUSHBYTE(DPREG);
	PUSHBYTE(BREG);
	PUSHBYTE(AREG);
	PUSHBYTE(CCREG);

	/* IRQs */
	CCREG |= 0x10;
	PCREG = READW(0xfff8);

	/* tO~낷 */
	INTR &= ~INTR_IRQ;
}

/*
 *	CPUZbg
 */
static void cpu_reset(void)
{
	CCREG = 0x50;
	AREG = 0;
	BREG = 0;
	DPREG = 0;
	XREG = 0;
	YREG = 0;
	UREG = 0;
	SREG = 0;
	INTR = 0;
	CYCLE = 0;
	TOTAL = 0;

	PCREG = (WORD)((READB(0xfffe) << 8) + READB(0xffff));
}

/*
 *	CPUs
 */
static void cpu_exec(void)
{
	/* NMI荞݃`FbN */
	if (INTR & INTR_NMI) {
		INTR |= INTR_SYNC_OUT;
		if (INTR & INTR_SLOAD) {
			cpu_nmi();
			cpu_execline();
			TOTAL += CYCLE;
			return;
		}
	}

	/* FIRQ荞݃`FbN */
	if (INTR & INTR_FIRQ) {
		INTR |= INTR_SYNC_OUT;
		if (!(CCREG & 0x40)) {
			cpu_firq();
			cpu_execline();
			TOTAL += CYCLE;
			return;
		}
	}

	/* IRQ荞݃`FbN */
	if (INTR & INTR_IRQ) {
		INTR |= INTR_SYNC_OUT;
		if (!(CCREG & 0x10)) {
			cpu_irq();
			cpu_execline();
			TOTAL += CYCLE;
			return;
		}
	}

	/* 荞݂Ȃ */
	cpu_execline();
	TOTAL += CYCLE;
	return;
}
