/*
 *	FM-7 EMULATOR "XM7"
 *
 *	Copyright (C) 1999,2000 ohD(ytanaka@ipc-tokai.or.jp)
 *	[ CCPU ]
 */

#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include "xm7.h"
#include "device.h"
#include "subctrl.h"
#include "ttlpalet.h"
#include "fdc.h"
#include "mainetc.h"
#include "multipag.h"
#include "kanji.h"
#include "tapelp.h"
#include "opn.h"
#include "gui.h"

#ifdef _WIN32
#include <crtdbg.h>
#endif

/*
 *	O[o [N
 */
BYTE *mainram_a;						/* RAM (\RAM)   $8000 */
BYTE *mainram_b;						/* RAM (RAM)   $7C80 */
BYTE *basic_rom;						/* ROM (F-BASIC) $7C00 */
BYTE *main_io;							/* CCPU I/O $0100 */
BYTE *bootbas_rom;						/* BOOT (BASIC)  $0200 */
BYTE *bootdos_rom;						/* BOOT (DOS)    $0200 */
BOOL basicrom_en;						/* BASIC ROMCl[utO */

/*
 *	CCPU
 *	
 */
BOOL mainmem_init(void)
{
	/* RAM */
	mainram_a = (BYTE *)malloc(0x8000);
	if (mainram_a == NULL) {
		return FALSE;
	}
	mainram_b = (BYTE *)malloc(0x7c80);
	if (mainram_b == NULL) {
		return FALSE;
	}

	/* BASIC ROM, I/O, BOOT ROM */
	basic_rom = (BYTE *)malloc(0x7c00);
	if (basic_rom == NULL) {
		return FALSE;
	}
	main_io = (BYTE *)malloc(0x0100);
	if (main_io == NULL) {
		return FALSE;
	}
	bootbas_rom = (BYTE *)malloc(0x0200);
	if (bootbas_rom == NULL) {
		return FALSE;
	}
	bootdos_rom = (BYTE *)malloc(0x0200);
	if (bootdos_rom == NULL) {
		return FALSE;
	}

	/* ROMt@Cǂݍ */
	if (!file_load(FBASIC_ROM, basic_rom, 0x7c00)) {
		return FALSE;
	}
	if (!file_load(BOOTBAS_ROM, bootbas_rom, 0x0200)) {
		return FALSE;
	}
	if (!file_load(BOOTDOS_ROM, bootdos_rom, 0x0200)) {
		return FALSE;
	}

	/* I/Oԏ */
	memset(main_io, 0xff, 0x0100);

	/* ROMpb` */
	bootbas_rom[0x1fe] = 0xfe;
	bootbas_rom[0x1ff] = 0x00;
	bootdos_rom[0x1fe] = 0xfe;
	bootdos_rom[0x1ff] = 0x00;

	return TRUE;
}

/*
 *	CCPU
 *	N[Abv
 */
void mainmem_cleanup(void)
{
	ASSERT(mainram_a);
	ASSERT(mainram_b);
	ASSERT(basic_rom);
	ASSERT(main_io);
	ASSERT(bootbas_rom);
	ASSERT(bootdos_rom);

	/* rŎsꍇl */
	if (mainram_a) {
		free(mainram_a);
	}
	if (mainram_b) {
		free(mainram_b);
	}
	if (basic_rom) {
		free(basic_rom);
	}
	if (main_io) {
		free(main_io);
	}
	if (bootbas_rom) {
		free(bootbas_rom);
	}
	if (bootdos_rom) {
		free(bootdos_rom);
	}
}

/*
 *	CCPU
 *	Zbg
 */
void mainmem_reset(void)
{
	/* N[h */
	if (bootfast_flag) {
		if (bootbas_rom[0x14f] == 0x26) {
			bootbas_rom[0x14f] = 0x21;
		}
		if (bootdos_rom[0x151] == 0x26) {
			bootdos_rom[0x151] = 0x21;
		}
	}
	else {
		if (bootbas_rom[0x14f] == 0x21) {
			bootbas_rom[0x14f] = 0x26;
		}
		if (bootdos_rom[0x151] == 0x21) {
			bootdos_rom[0x151] = 0x26;
		}
	}
}

/*
 *	CCPU
 *	PoCg擾
 */
BYTE mainmem_readb(WORD addr)
{
	BYTE dat;

	/* CRAM(\) */
	if (addr < 0x8000) {
		return mainram_a[addr];
	}

	/* BASIC ROM or CRAM() */
	if ((addr >= 0x8000) && (addr < 0xfc00)) {
		if (basicrom_en) {
			return basic_rom[addr - 0x8000];
		}
		else {
			return mainram_b[addr - 0x8000];
		}
	}

	/* CROM̒ */
	if ((addr >= 0xfc00) && (addr < 0xfc80)) {
		return mainram_b[addr - 0x8000];
	}

	/* LRAM */
	if ((addr >= 0xfc80) && (addr < 0xfd00)) {
		if (subhalt_flag) {
			return shared_ram[(WORD)(addr - 0xfc80)];
		}
		else {
			return 0xff;
		}
	}

	/* u[gROM */
	if (addr >= 0xfe00) {
		if (boot_mode == BOOT_BASIC) {
			return bootbas_rom[addr - 0xfe00];
		}
		else {
	 		return bootdos_rom[addr - 0xfe00];
		}
	}

	/*
	 *	I/O
	 */
	if (mainetc_readb(addr, &dat)) {
		return dat;
	}
	if (ttlpalet_readb(addr, &dat)) {
		return dat;
	}
	if (subctrl_readb(addr, &dat)) {
		return dat;
	}
	if (multipag_readb(addr, &dat)) {
		return dat;
	}
	if (fdc_readb(addr, &dat)) {
		return dat;
	}
	if (kanji_readb(addr, &dat)) {
		return dat;
	}
	if (tapelp_readb(addr, &dat)) {
		return dat;
	}
	if (opn_readb(addr, &dat)) {
		return dat;
	}

	/* `GA */
#if defined(_WIN32) && defined(_DEBUG)
	_CrtDbgReport(_CRT_WARN, NULL, (int)NULL, 0, "AhXǂݍ %04x\n", addr);
#endif
	return 0xff;
}

/*
 *	CCPU
 *	PoCg擾(I/OȂ)
 */
BYTE mainmem_readbnio(WORD addr)
{
	/* CRAM(\) */
	if (addr < 0x8000) {
		return mainram_a[addr];
	}

	/* BASIC ROM or CRAM() */
	if ((addr >= 0x8000) && (addr < 0xfc00)) {
		if (basicrom_en) {
			return basic_rom[addr - 0x8000];
		}
		else {
			return mainram_b[addr - 0x8000];
		}
	}

	/* CROM̒ */
	if ((addr >= 0xfc00) && (addr < 0xfc80)) {
		return mainram_b[addr - 0x8000];
	}

	/* LRAM */
	if ((addr >= 0xfc80) && (addr < 0xfd00)) {
		if (subhalt_flag) {
			return shared_ram[(WORD)(addr - 0xfc80)];
		}
		else {
			return 0xff;
		}
	}

	/* u[gROM */
	if (addr >= 0xfe00) {
		if (boot_mode == BOOT_BASIC) {
			return bootbas_rom[addr - 0xfe00];
		}
		else {
			return bootdos_rom[addr - 0xfe00];
		}
	}

	/* I/O */
	ASSERT((addr >= 0xfd00) && (addr < 0xfe00));
	return main_io[addr - 0xfd00];
}

/*
 *	CCPU
 *	PoCg
 */
void mainmem_writeb(WORD addr, BYTE dat)
{
	/* CRAM(\) */
	if (addr < 0x8000) {
		mainram_a[addr] = dat;
		return;
	}

	/* BASIC ROM or CRAM() */
	if ((addr >= 0x8000) && (addr < 0xfc00)) {
		if (basicrom_en) {
			/* ROMRCBBIOSĂяoP[X */
			return;
		}
		else {
			mainram_b[addr - 0x8000] = dat;
			return;
		}
	}

	/* CROM̒ */
	if ((addr >= 0xfc00) && (addr < 0xfc80)) {
		mainram_b[addr - 0x8000] = dat;
		return;
	}

	/* LRAM */
	if ((addr >= 0xfc80) && (addr < 0xfd00)) {
		if (subhalt_flag) {
			shared_ram[(WORD)(addr - 0xfc80)] = dat;
			return;
		}
		else {
			/* BASE09΍ */
			return;
		}
	}

	/* u[gROM */
	if ((addr >= 0xfe00) && (addr < 0xffe0)) {
		ASSERT(FALSE);
		return;
	}

	/* u[g[NRAMAxN^ */
	if ((addr >= 0xffe0) && (addr < 0xfffe)) {
		bootbas_rom[(WORD)(addr - 0xfe00)] = dat;
		bootdos_rom[(WORD)(addr - 0xfe00)] = dat;
		return;
	}
	if (addr >= 0xfffe) {
		/* UihDEViIII */
		return;
	}

	/*
	 *	I/O
	 */
	ASSERT((addr >= 0xfd00) && (addr < 0xfe00));
	main_io[(WORD)(addr - 0xfd00)] = dat;
	
	if (mainetc_writeb(addr, dat)) {
		return;
	}
	if (ttlpalet_writeb(addr, dat)) {
		return;
	}
	if (subctrl_writeb(addr, dat)) {
		return;
	}
	if (multipag_writeb(addr, dat)) {
		return;
	}
	if (fdc_writeb(addr, dat)) {
		return;
	}
	if (kanji_writeb(addr, dat)) {
		return;
	}
	if (tapelp_writeb(addr, dat)) {
		return;
	}
	if (opn_writeb(addr, dat)) {
		return;
	}

#if defined(_WIN32) && defined(_DEBUG)
	_CrtDbgReport(_CRT_WARN, NULL, (int)NULL, 0, "AhX %04x\n", addr);
#endif
	return;
}

/*
 *	CCPU
 *	Z[u
 */
BOOL mainmem_save(int fileh)
{
	if (!file_write(fileh, mainram_a, 0x4000)) {
		return FALSE;
	}
	if (!file_write(fileh, &mainram_a[0x4000], 0x4000)) {
		return FALSE;
	}

	if (!file_write(fileh, mainram_b, 0x7c80)) {
		return FALSE;
	}

#if 0
	if (!file_write(fileh, basic_rom, 0x7c00)) {
		return FALSE;
	}
#endif

	if (!file_write(fileh, main_io, 0x100)) {
		return FALSE;
	}

	if (!file_write(fileh, bootbas_rom, 0x200)) {
		return FALSE;
	}

	if (!file_write(fileh, bootdos_rom, 0x200)) {
		return FALSE;
	}

	if (!file_bool_write(fileh, basicrom_en)) {
		return FALSE;
	}

	return TRUE;
}

/*
 *	CCPU
 *	[h
 */
BOOL mainmem_load(int fileh, int ver)
{
	/* o[W`FbN */
	if (ver > 1) {
		return FALSE;
	}

	if (!file_read(fileh, mainram_a, 0x4000)) {
		return FALSE;
	}
	if (!file_read(fileh, &mainram_a[0x4000], 0x4000)) {
		return FALSE;
	}

	if (!file_read(fileh, mainram_b, 0x7c80)) {
		return FALSE;
	}

#if 0
	if (!file_read(fileh, basic_rom, 0x7c00)) {
		return FALSE;
	}
#endif

	if (!file_read(fileh, main_io, 0x100)) {
		return FALSE;
	}

	if (!file_read(fileh, bootbas_rom, 0x200)) {
		return FALSE;
	}

	if (!file_read(fileh, bootdos_rom, 0x200)) {
		return FALSE;
	}

	if (!file_bool_read(fileh, &basicrom_en)) {
		return FALSE;
	}

	return TRUE;
}
