/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
#include <sys/param.h>
#include <sys/time.h>
#include <sys/types.h>
#ifdef _AIX
#include <sys/select.h>
#endif
#include <sys/socket.h>
#include <errno.h>
#include <netdb.h>
#include <xmc.h>
#include <xmclib.h>
#include <xmcp.h>

#define COMMON
#include "common.h"

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 256
#endif

#define BUFINC		512
#define BUFMAX		4096

static XmcEventList *eqfree;

XmcEventList *
Xmc_eventlist_new
   VOID
{
   XmcEventList *elp;

   if (eqfree) {
      elp = eqfree;
      eqfree = elp->next;
   }
   else
      elp = (XmcEventList *)malloc(sizeof(XmcEventList));

   return elp;
}

void
Xmc_eventlist_free
   AL((elp))
   DB XmcEventList *elp
   DE
{
   elp->next = eqfree;
   eqfree = elp;
}

void
Xmc_event_push
   AL((muxp, ep))
   DB Mux *muxp
   DD xmcEvent *ep
   DE
{
   XmcEventList *elp;

   if ((elp = Xmc_eventlist_new()) == 0) {
      warn("Xmc_event_push: cannot get a new eventlist element!\n");
      return;
   }
   Xmc_event_convert(ep, &elp->event);
   if (muxp->qtail)
      muxp->qtail->next = elp;
   else
      muxp->qhead = elp;
   muxp->qtail = elp;
   elp->next = 0;
   muxp->qlen++;
}

int
Xmc_event_shift
   AL((muxp, event))
   DB Mux *muxp
   DD XmcEvent *event
   DE
{
   XmcEventList *elp;

   if (elp = muxp->qhead) {
      *event = elp->event;
      muxp->qhead = elp->next;
      if (muxp->qhead == 0)
         muxp->qtail = 0;
      Xmc_eventlist_free(elp);
      muxp->qlen--;
      return 1;
   }
   return 0;
}

void
Xmc_event_convert
   AL((ep, eventp))
   DB xmcEvent *ep
   DD XmcEvent *eventp
   DE
{
   switch (ep->u.code) {
      case DisplayInEvent:
      case DisplayRefusedEvent:
      case DisplayOutEvent:
      case DisplayTagEvent:
      case ModeFloorEvent:
      case ModeSeatEvent:
      case ModeViewEvent:
      case PointerGrabEvent:
      case PointerNoGrabEvent:
      case PointerUngrabEvent:
      case KeyboardGrabEvent:
      case KeyboardNoGrabEvent:
      case KeyboardUngrabEvent:
      case ShareSelectionsEvent:
      case UnshareSelectionsEvent:
         eventp->display.type = ep->display.code;
         eventp->display.serial = Xmc_seq2serial(ep->display.seqNo);
         eventp->display.id = ep->display.dispID;
         break;
      case TptrAssignEvent:
         eventp->tptrassign.type = ep->tptrAssign.code;
         eventp->tptrassign.serial = Xmc_seq2serial(ep->tptrAssign.seqNo);
         eventp->tptrassign.id = ep->tptrAssign.dispID;
         eventp->tptrassign.tptrID = ep->tptrAssign.tpID;
         break;
      case TptrHideEvent:
      case TptrShowEvent:
         eventp->tptrvis.type = ep->tptr.code;
         eventp->tptrvis.serial = Xmc_seq2serial(ep->tptr.seqNo);
         eventp->tptrvis.tptrID = ep->tptr.tpID;
         break;
      case ConfigModeEvent:
         eventp->config.type = ep->config.code;
         eventp->config.serial = Xmc_seq2serial(ep->config.seqNo);
         eventp->config.mode = ep->config.mode;
         break;
      case SeatKeyPressEvent:
      case SeatKeyReleaseEvent:
         eventp->key.type = ep->keyButton.code;
         eventp->key.serial = Xmc_seq2serial(ep->keyButton.seqNo);
         eventp->key.id = ep->keyButton.dispID;
         eventp->key.keycode = ep->keyButton.detail;
         eventp->key.state = ep->keyButton.state;
         eventp->key.time = ep->keyButton.time;
         eventp->key.window = ep->keyButton.event;
         eventp->key.subwindow = ep->keyButton.child;
         eventp->key.x_root = ep->keyButton.rootX;
         eventp->key.y_root = ep->keyButton.rootY;
         eventp->key.x = ep->keyButton.eventX;
         eventp->key.y = ep->keyButton.eventY;
         break;
      case SeatButtonPressEvent:
      case SeatButtonReleaseEvent:
         eventp->button.type = ep->keyButton.code;
         eventp->button.serial = Xmc_seq2serial(ep->keyButton.seqNo);
         eventp->button.id = ep->keyButton.dispID;
         eventp->button.button = ep->keyButton.detail;
         eventp->button.state = ep->keyButton.state;
         eventp->button.time = ep->keyButton.time;
         eventp->button.window = ep->keyButton.event;
         eventp->button.subwindow = ep->keyButton.child;
         eventp->button.x_root = ep->keyButton.rootX;
         eventp->button.y_root = ep->keyButton.rootY;
         eventp->button.x = ep->keyButton.eventX;
         eventp->button.y = ep->keyButton.eventY;
         break;
      default:
         warn("Xmc_event_convert: not an event!\n");
         break;
   }
}

ulong_t
Xmc_seq2serial
   AL((seqNo))
   DB ushort_t seqNo
   DE
{
   return (ulong_t)seqNo;
}

ulong_t
Xmc_newId
   AL((muxp))
   DB Mux *muxp
   DE
{
   static ulong_t id;

   return muxp->base | (++id << muxp->shift) & muxp->mask;
}

void
Xmc_zerobuf
   AL((bp))
   DB XmcBuf *bp
   DE
{
   bp->n = 0;
   bp->sz = 0;
   bp->bp = 0;
   bp->cp = 0;
}

void
Xmc_clearbuf
   AL((bp, n))
   DB XmcBuf *bp
   DD int n
   DE
{
   if (n == 0 || n >= bp->n) {
      bp->n = 0;
      bp->cp = bp->bp;
   }
   else {
      bp->n -= n;
      bp->cp += n;
   }
}

int
Xmc_bufadj
   AL((bp, n))
   DB XmcBuf *bp
   DD int n
   DE
{
   register int i, sz;
   register char *cp;

   if (n > bp->sz - bp->n) {		/* buffer too small? */
      sz = RUP(bp->n + n, BUFINC);

      if (sz > BUFMAX)
         return -1;
      else {
         if ((cp = (char *)malloc(sz)) == 0)
            return -1;
         if (bp->n) {
            for (i=0; i<bp->n; i++)
               cp[i] = bp->cp[i];
         }
         free(bp->bp);
         bp->bp = bp->cp = cp;
         bp->sz = sz;
      }
   }					/* too fragmented? */
   else {
      for (i=0; i<bp->n; i++)
         bp->bp[i] = bp->cp[i];
      bp->cp = bp->bp;
   }
   return 0;
}

void
Xmc_freebuf
   AL((bp))
   DB XmcBuf *bp
   DE
{
   if (bp->bp)
      free(bp->bp);

   Xmc_zerobuf(bp);
}

char *
Xmc_allocout
   AL((muxp, n))
   DB Mux *muxp
   DD int n
   DE
{
   register XmcBuf *bp = &muxp->outbuf;
   register char *cp;

   if (n > buf_free(bp))
      if (Xmc_bufadj(bp, n)) {
         Xmc_flush(muxp);		/* too much, write it out */
         if (Xmc_bufadj(bp, n))		/* still no-go?  error */
            return 0;
      }
   cp = bp->cp + bp->n;
   bp->n += n;
   return cp;
}

void
Xmc_inclear
   AL((muxp, n))
   DB Mux *muxp
   DD int n
   DE
{
   Xmc_clearbuf(&muxp->inbuf, n);
}

int
Xmc_flush
   AL((muxp))
   DB Mux *muxp
   DE
{
   register int r, t;
   register XmcBuf *bp = &muxp->outbuf;
   fd_set wfds;
   
   if (buf_active(bp))
      for (t=0; t<buf_active(bp); t+=r)
         switch (r = write(muxp->fd, buf_data(bp)+t, buf_active(bp)-t)) {
            case -1:
               if (errno == EWOULDBLOCK) {
                  FD_ZERO(&wfds);
                  FD_SET(muxp->fd, &wfds);
                  select(muxp->fd+1, 0, &wfds, 0, 0);	/* block */
                  r = 0;
               }
               else {
                  perror("write to socket");
                  return -1;
               }
            case 0:
               return -1;
         }
   Xmc_clearbuf(bp, 0);

   return 0;
}

void
Xmc_sync
   AL((muxp))
   DB Mux *muxp
   DE
{
   Xmc_flush(muxp);
   Xmc_read(muxp, 1);
}

/*
**	Xmc_read
**
**	Ensure that at least n bytes are in the incoming buffer.  Blocks.
*/
char *
Xmc_read
   AL((muxp, n))
   DB Mux *muxp
   DD int n
   DE
{
   register int r, t, max;
   register char *cp;
   register XmcBuf *bp = &muxp->inbuf;
   fd_set rfds;

   if (n <= buf_active(bp))	/* is there enough in the buffer? */
      return buf_data(bp);

   t = n - buf_active(bp);
   if (t > buf_free(bp))	/* is the buffer large enough? */
      if (Xmc_bufadj(bp, t))
         return 0;

   max = buf_free(bp);		/* fill the buffer, if we can */

   for (t=0; t<max; t+=r) {
      switch (r = read(muxp->fd, buf_next(bp)+t, max-t)) {
         case -1:
            if (errno == EWOULDBLOCK) {
               if (t >= n)
                  break;	/* loop exit */

               FD_ZERO(&rfds);
               FD_SET(muxp->fd, &rfds);
               select(muxp->fd+1, &rfds, 0, 0, 0);	/* block */
               r = 0;
               continue;	/* iterate */
            }
            else {
               Xmc_io_error(muxp);
               return 0;
            }
         case 0:
            Xmc_io_error(muxp);
            return 0;
         default:
            continue;		/* iterate */
      }
      break;
   }
   buf_setactive(bp, buf_active(bp) + t);
   return buf_data(bp);
}

/*
**	Xmc_read_noblock
**
**	Non-blocking read.  Returns the number of bytes read.
*/
int
Xmc_read_noblock
   AL((muxp))
   DB Mux *muxp
   DE
{
   register int nfree, r, t;
   register XmcBuf *bp = &muxp->inbuf;

   nfree = buf_free(bp);
   if (nfree == 0) {
      /*
      ** bump it a small amount, just enough so we make progress
      */
      if (Xmc_bufadj(bp, 16)) {
         return 0;	/* not good enough - this is an error TODO */
      }
      nfree = buf_free(bp);
   }
   for (t=0; t<nfree; t+=r) {
      switch (r = read(muxp->fd, buf_next(bp)+t, nfree-t)) {
         case -1:
            if (errno == EWOULDBLOCK) {
               r = 0;
               break;	/* loop exit */
            }
            /* fall through... */
         case 0:
            Xmc_io_error(muxp);
            return 0;
         default:
            continue;	/* iterate */
      }
      break;
   }
   buf_setactive(bp, buf_active(bp) + t);
   return t;
}

/*
**	Xmc_queue_events
**
**	Process the contents of the incoming buffer.  Enqueues events
**	and forwards errors.  If rpp is non-zero, stops when
**	a reply or an error with the current outgoing sequence number
**	is encountered.  It is an error to encounter a reply if
**	need_reply is zero.
**
**	Returns non-zero if a reply (or error-reply) was encountered.
**	*rpp is set to point to the reply or zero if error.
**
**	This routine does no i/o.
*/
int
Xmc_queue_events
   AL((muxp, rpp))
   DB Mux *muxp
   DD xmcReply **rpp	/* reply pointer return parameter */
   DE
{
   register ushort_t seqno;
   register int sz;
   register xmcReply *rp;
   register XmcBuf *bp = &muxp->inbuf;

   while (buf_active(bp) >= sz_xmcReply) {
      rp = (xmcReply *)buf_data(bp);

      switch (rp->reply) {
         case XMC_Error:
            sz = sz_xmcError;
            break;
         case XMC_Reply:
            sz = rp->length;
            break;
         default:		/* Event */
            sz = sz_xmcEvent;
            break;
      }
      if (buf_active(bp) < sz)
         break;				/* exit loop */
      /*
      **  Verify that incoming sequence number satisfies
      **	muxp->inseqno <= rp->seqno <= muxp->outseqno
      **
      **  Handle (short) integer wraparound, too.
      */
      if (	(muxp->inseqno <= rp->seqNo) +
		(rp->seqNo <= muxp->outseqno) +
		(muxp->outseqno < muxp->inseqno) == 2)
         muxp->inseqno = rp->seqNo;
      else
         warn("lost XMC sequence number in reply type 0x%x\n", rp->reply);

      /*
      **  process it
      */
      switch (rp->reply) {
         case XMC_Error:
            seqno = rp->seqNo;

            Xmc_error(muxp, (xmcError *)rp);
            Xmc_inclear(muxp, sz_xmcError);

            if (rpp && muxp->outseqno == seqno) {
               *rpp = 0;
               return 1;			/* error reply, return */
            }
            break;
         case XMC_Reply:
            if (rpp) {
               if (muxp->outseqno != rp->seqNo)
                  warn("bad sequence number in reply!\n");
               *rpp = rp;
            }
            else
               warn("unexpected reply!\n");

            return 1;				/* got reply, return */
         default:
            Xmc_event_push(muxp, (xmcEvent *)rp);
            Xmc_inclear(muxp, sz_xmcEvent);
            break;
      }
   }
   return 0;
}

/*
**	Xmc_reply
**
**	Get next reply.  Process errors and events that intervene.
**	If the request whose reply is pending caused an error, process
**	it and return zero (signalling that no reply is coming).
*/
xmcReply *
Xmc_reply
   AL((muxp))
   DB Mux *muxp
   DE
{
   xmcReply *rp;

   while (Xmc_read(muxp, sz_xmcReply))
      if (Xmc_queue_events(muxp, &rp))
         return rp;

   return 0;
}

int
Xmc_event
   AL((muxp, eventp))
   DB Mux *muxp
   DD XmcEvent *eventp
   DE
{
   register xmcEvent *ep;

   while (muxp->qlen == 0 && Xmc_read(muxp, sz_xmcReply))
      if (Xmc_queue_events(muxp, 0))
         warn("Xmc_event: got a reply - can't happen!\n");

   if (Xmc_event_shift(muxp, eventp))
      return 0;
   else
      return -1;
}

int
Xmc_parse_display
   AL((display, hostp, dnop, scrp, decnetp))
   DB char *display
   DD char **hostp
   DD int *dnop
   DD int *scrp
   DD int *decnetp
   DE
{
   register int n, i;
   register char *cp;
   int decnet = 0;
   int dno, scrno;
   char nbuf[16];
   static char hostbuf[MAXHOSTNAMELEN];

   if (display == 0 || *display == '\0')
      if ((display = (char *)getenv("DISPLAY")) == 0)
         return -1;
						/* find colon */
   for (cp = display; *cp && *cp != ':'; cp++);
   if (*cp == 0)
      return -1;
						/* find hostname */
   n = cp - display;
   if (n >= MAXHOSTNAMELEN)
      return -1;
   for (i=0; i<n; i++)
      hostbuf[i] = display[i];
   hostbuf[i] = '\0';
						/* DECnet? */
   if (*++cp == ':') {
#ifdef DNETCONN
      decnet = 1;
      cp++;
#else
      return -1;
#endif
   }
   else
      decnet = 0;
						/* find display number */
   for (display=cp; *cp && *cp != '.'; cp++);
   n = cp - display;
   if (n == 0 || n > sizeof(nbuf) - 1) 	/* sanity check */
      return -1;
   for (i=0; i<n; i++)
      if (isdigit(display[i]))
         nbuf[i] = display[i];
      else
         return -1;
   nbuf[i] = '\0';
   dno = atoi(nbuf);

   if (scrp)			/* optional screen number */
      if (*cp == '\0')
         scrno = 0;
      else {
         for (display= ++cp; *cp; cp++);
         n = cp - display;
         if (n == 0 || n > sizeof(nbuf) - 1)
            return -1;
         for (i=0; i<n; i++)
            if (isdigit(display[i]))
               nbuf[i] = display[i];
            else
               return -1;
         nbuf[i] = '\0';
         scrno = atoi(nbuf);
      }

   if (n = strlen(hostbuf)) {
      if ((cp = (char *)malloc(n + 1)) == 0)
         return 0;
      strcpy(cp, hostbuf);
      *hostp = cp;
   }
   else
      *hostp = 0;
   *dnop = dno;
   if (scrp)
      *scrp = scrno;
   *decnetp = decnet;

   return 0;
}

char *
Xmc_host_me
   VOID
{
   register char *cp;
   char hostbuf[MAXHOSTNAMELEN];

   if (gethostname(hostbuf, MAXHOSTNAMELEN)) {
      perror("gethostname");
      return 0;
   }
   if ((cp = (char *)malloc(strlen(hostbuf) + 1)) == 0)
      return 0;
   strcpy(cp, hostbuf);
   return cp;
}

/* TODO allocate addr - no statics! */
int
Xmc_host_addr
   AL((hostname, famp, lenp, addrp))
   DB char *hostname
   DD int *famp
   DD int *lenp
   DD char **addrp
   DE
{
   register char *cp;
   static ulong_t iaddr;
   static struct hostent *hp;

   if (isdigit(*hostname)) {
      for (cp=hostname; *cp && (isdigit(*cp) || *cp == '.'); cp++);
      if (*cp == '\0') {	/* it's an inet number */
         if ((long)(iaddr = inet_addr(hostname)) != -1) {
            *famp = AddrFamInternet;
            *lenp = sizeof(iaddr);
            *addrp = (char *)&iaddr;
            return 0;
         }
      }
   }
   if (hp = gethostbyname(hostname)) {
      *famp = Xmc_family_utox(hp->h_addrtype);
      *lenp = hp->h_length;
      *addrp = *hp->h_addr_list;
      return 0;
   }
   return -1;
}

ushort_t
Xmc_family_utox
   AL((family))
   DB ushort_t family
   DE
{
   switch (family) {
#ifdef AF_UNIX
      case AF_UNIX:	return AddrFamLocal;
#endif
#ifdef AF_INET
      case AF_INET:	return AddrFamInternet;
#endif
#ifdef AF_DECnet
      case AF_DECnet:	return AddrFamDECnet;
#endif
#ifdef AF_CHAOS
      case AF_CHAOS:	return AddrFamChaos;
#endif
      default:		return 0xffff;
   }
}

ushort_t
Xmc_family_xtou
   AL((family))
   DB ushort_t family
   DE
{
   switch (family) {
#ifdef AF_UNIX
      case AddrFamLocal:	return AF_UNIX;
#endif
#ifdef AF_INET
      case AddrFamInternet:	return AF_INET;
#endif
#ifdef AF_DECnet
      case AddrFamDECnet:	return AF_DECnet;
#endif
#ifdef AF_CHAOS
      case AddrFamChaos:	return AF_CHAOS;
#endif
      default:			return 0xffff;
   }
}
