/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   access.c								*
*									*
*	Implements host access control.					*
*									*
************************************************************************/
#include <stdio.h>
#include <sys/param.h>
#include <malloc.h>
#ifdef SVR4
#include <sys/utsname.h>
#include <string.h>
#else
#include <strings.h>
#endif
#include <errno.h>

#include <X11/Xproto.h>
#include <X11/X.h>

#include "xmx.h"
#include "incl/access.pvt.h"

typedef struct _hosta_t {
	int			access;		/* can modify access list */
	u16_t			family;
	u16_t			length;
	char *			address;
	struct _hosta_t *	next;
}hosta_t;

static int		access = 1;	/* on or off */
static hosta_t *	list;		/* access control list */
static u16_t		nlist;		/* number of hosts in list */

/************************************************************************
*									*
*   access_reset							*
*									*
************************************************************************/
void
access_reset
   AL((display, authon))
   DB u16_t display
   DD int authon		/* is authorization turned on? */
   DE
{
   register u8_t glob;
   FILE *fp;
   hosta_t *hp, *tp;
   u16_t family, length;
   char *cp, *address;
   char file[32];
   char buf[MAXHOSTNAMELEN];
							/*LINTED*/
   for (hp=list; tp=hp; hp=hp->next)
      free((char *)tp);
   list = (hosta_t *)0;

   sprintf(file, "/etc/X%d.hosts", display);
   if (fp = fopen(file, "r")) {
      while (fgets(buf, MAXHOSTNAMELEN, fp)) {
         if (cp = (char *)index(buf, '\n'))
            *cp = '\0';
         glob = opt.glob;
         opt.glob = 0;
         host_pattern(buf);
         if (host_match(&family, &length, &address))
            (void) access_add(family, length, address);
         opt.glob = glob;
      }
      if (ferror(fp))
         warn("%s: error opening or reading file\n", file);
   }
   if (!authon)
      (void) access_add(me.family, me.length, me.address);
}

/************************************************************************
*									*
*   access_get								*
*									*
************************************************************************/
int
access_get
   VOID
{
   return access;
}

/************************************************************************
*									*
*   access_set								*
*									*
************************************************************************/
void
access_set
   AL((cp, flag))
   DB client_t *cp
   DD int flag
   DE
{
   if (! host_isme(cp->addr.family, cp->addr.length, cp->addr.address)) {
      proto_Error(cp, BadAccess, 0L, 0, X_SetAccessControl);
      return;
   }
   access = flag ? 1 : 0;
}

/************************************************************************
*									*
*   access_change							*
*									*
************************************************************************/
int
access_change
   AL((cp, p))
   DB client_t *cp
   DD xChangeHostsReq *p
   DE
{
   if (! host_isme(cp->addr.family, cp->addr.length, cp->addr.address)) {
      proto_Error(cp, BadAccess, 0L, 0, X_ChangeHosts);
      return -1;
   }
   if (p->mode == HostInsert) {
      if (access_add(p->hostFamily, p->hostLength, (char *)(p+1))) {
         proto_Error(cp, BadAlloc, 0L, 0, X_ChangeHosts);
         return -1;
      }
   }
   else
      if (access_remove(p->hostFamily, p->hostLength, (char *)(p+1))) {
         proto_Error(cp, BadValue, 0L, 0, X_ChangeHosts);
         return -1;
      }

   return 0;
}

/************************************************************************
*									*
*   access_add								*
*									*
************************************************************************/
int
access_add
   AL((family, length, address))
   DB u16_t family
   DD u16_t length
   DD char *address
   DE
{
   register hosta_t *hp, *lp;

   lp = (hosta_t *)0;
   for (hp=list; hp; hp=hp->next) {
      if (	family == hp->family &&
		length == hp->length &&
		bcmp(address, hp->address, (int)length) == 0)
         return 0;
      lp = hp;
   }
   if (MALLOC(hp, hosta_t *, sizeof(hosta_t) + length))
      return 1;
   hp->family = family;
   hp->length = length;
   hp->address = (char *)(hp + 1);
   (void) bcopy(address, hp->address, (int)length);
   hp->next = (hosta_t *)0;
   if (lp)
      lp->next = hp;
   else
      list = hp;

   nlist++;

   return 0;
}

/************************************************************************
*									*
*   access_check							*
*									*
************************************************************************/
int
access_check
   AL((family, length, address))
   DB u16_t family
   DD u16_t length
   DD char *address
   DE
{
   register hosta_t *hp;

   if (access == 0)
      return 1;

   if (opt.auth == 0 && family == FamilyLocal)
      return 1;

   for (hp=list; hp; hp=hp->next) {
      if (	family == hp->family &&
		length == hp->length &&
		bcmp(address, hp->address, (int)length) == 0)
         return 1;
   }
   return 0;
}

/************************************************************************
*									*
*   access_remove							*
*									*
************************************************************************/
int
access_remove
   AL((family, length, address))
   DB u16_t family
   DD u16_t length
   DD char *address
   DE
{
   register hosta_t *hp, *lp;

   lp = (hosta_t *)0;
   for (hp=list; hp; hp=hp->next) {
      if (	family == hp->family &&
		length == hp->length &&
		bcmp(address, hp->address, (int)length) == 0) {
         if (lp)
            lp->next = hp->next;
         else
            list = hp->next;
         free((char *)hp);
         nlist--;
         break;
      }
      lp = hp;
   }
   return hp ? 0 : -1;
}

/************************************************************************
*									*
*   access_list								*
*									*
*	Write a ListHosts reply into the given buffer.			*
*									*
************************************************************************/
void
access_list
   AL((cp))
   DB client_t *cp
   DE
{
   register u32_t totlen = 0;
   register hosta_t *hp;

   for (hp=list; hp; hp=hp->next)
      totlen += hp->length + PAD(hp->length);

   proto_ListHostsReply(cp, (u8_t)access, totlen, nlist);

   for (hp=list; hp; hp=hp->next)
      proto_HostEntry(hp->family, hp->length, hp->address);

   zb_client_queue();
}
