/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   args.c								*
*									*
*									*
************************************************************************/
#include <stdio.h>
#ifdef SVR4
#include <sys/utsname.h>
#include <string.h>
#else
#include <strings.h>
#endif
#include <ctype.h>
#include <stdlib.h>
#include <xmc.h>
#include "xmx.h"
#include "incl/args.pvt.h"

#ifndef RGB_DB
#define RGB_DB "/usr/lib/X11/rgb"
#endif

#define USAGE	"usage: %s [:displaynum] [-option...]\n"

static char *options[]	= {
       "-a #		ignored\n",
       "-af filename	read command line arguments from filename\n",
       "-auth path	select authorization file\n",
       "bc		ignored\n",
       "-bs		disable any backing store support\n",
       "-c		ignored\n",
       "c #		ignored\n",
       "-cc #		sets the visual of root windows of color screens\n",
       "-co filename	sets name of RGB color database\n",
       "-ctp		make the default telepointer constant\n",
       "-dc		delay calculation of virtual configuration\n",
       "-debug levs	sets debug message level ('-debug ?' lists them)\n",
       "-display disp..	same as -floor, but implies -nodpy\n",
       "-dpi #		sets the resolution of the screen in dots per inch\n",
       "-f #		ignored\n",
       "-fc font	ignored\n",
       "-floor disp...	the following displays are to have control\n",
       "-fn font	ignored\n",
       "-fp path	ignored\n",
       "-geometry geom	size and position of virtual root window\n",
       "-glob		do hostname wildcarding\n",
       "-help		print message with these options\n",
       "-I		ignore remaining command line arguments\n",
       "-install	bypass window manager and install colormaps\n",
       "-ld #		ignored\n",
       "-ls #		ignored\n",
       "-logo		ignored\n",
       "-mdpynum #	display number on which to listen for XMC protocol\n",
       "-maxcols #	maximum number of root colors to allocate\n",
       "-nodpy		ignore DISPLAY environment variable\n",
       "nologo		ignored\n",
       "-notp		no default telepointer\n",
       "-noresize	don't allow resizing of virtual root windows\n",
       "-noxmc		don't listen for multiplexor control clients\n",
       "-owncmap	don't share root colormaps\n",
       "-p #		ignored\n",
       "-quiet		suppress information and warning messages\n",
       "-r		ignored\n",
       "r		ignored\n",
       "-s #		ignored\n",
       "-sto #		sets default server connection timeout in seconds\n",
       "-su		disables save under support\n",
       "-seat disp...	the following displays may signal for attention\n",
       "-t #		sets pointer acceleration threshold\n",
       "-to #		sets default connection timeout in seconds\n",
       "ttyxx		ignored\n",
       "v		ignored\n",
       "-v		ignored\n",
       "-wm		forces default backing store to WhenMapped\n",
       "-view disp...	the following displays may only watch\n",
       "-x extension	ignored\n",
       (char *)0};

/*
**	private data
*/
static char *	command;
static int	argc;
static char **	argv;
static FILE *	fp = 0;

/************************************************************************
*									*
*   args_get								*
*									*
*	Parse command line.  Honor all standard X server options and	*
*	add a few of our own.						*
*									*
************************************************************************/
#define EQ(A)	strcmp(arg,(A))==0

void
args_get
   AL((ac, av))
   DB int ac
   DD char **av
   DE
{
   register int i;
   register u32_t did = 1;	/* display id */
   register u8_t mode = 0;
   register int nodpy = 0;
   register char *arg, *cp;
   char *host, *name, *address;
   int nhosts;
   char *hosts[512];
   u8_t modes[512];
   u16_t family, display, screen, length;
   rid_t window;

   command = *av;
   argc = ac - 1;
   argv = av + 1;
   if (argc && **argv == ':') {		/* optional display num */
      argc--;
      opt.dpy = atoi(1+*argv++);
   }
   else
      opt.dpy = DEFAULTDISPLAY;
						/* defaults */
   debug = D_ERRORS;

   opt.co = RGB_DB;
   opt.ctp = 0;
   opt.glob = 0;
   opt.install = 0;
   opt.owncmap = 0;
   opt.notp = 0;	/* yes, telepointer by default */
   opt.mdpynum = -1;	/* impossible display */
   opt.resize = 1;
   opt.sto = 60;	/* 60 secs default connection timeout */
   opt.to = 60;		/* 60 secs default connection timeout */
   opt.xmc = 1;

   for (nhosts=0; argc;) {
      if (fp == 0) {
         argc--;
         arg = *argv++;
      }
      else if ((arg = tok(fp)) == 0) {
         fclose(fp);
         fp = 0;
         continue;
      }
      if (*arg == '-')
         switch (*(arg+1)) {
            case 'a':
               if (EQ("-a")) {
                  (void) opt_num_arg();
                  continue;
               }
               if (EQ("-af")) {
                  if ((fp = fopen(cp=opt_str_arg(), "r")) == 0)
                     warn("%s: cannot open\n", cp);
                  continue;
               }
               if (EQ("-auth")) {
                  opt.auth = opt_str_arg();
                  continue;
               }
               break;
            case 'b':
               if (EQ("-bs")) {
                  opt.bs = 1;
                  continue;
               }
               break;
            case 'c':
               if (EQ("-cc")) {
                  continue;
               }
               if (EQ("-co")) {
                  opt.co = opt_str_arg();
                  continue;
               }
               if (EQ("-ctp")) {
                  opt.ctp = 1;
                  continue;
               }
               break;
            case 'd':
               if (EQ("-dc")) {
                  config_mode = ConfigDelay;
                  continue;
               }
               if (EQ("-debug")) {
                  debug_set(opt_debug_arg());
#ifndef DEBUG
                  if (debug > D_ERRORS)
                     warn("not compiled for debugging\n");
#endif
                  continue;
               }
               if (EQ("-display")) {	/* same as -floor, but implies nodpy */
                  nodpy = 1;
                  mode = Floor;
                  continue;
               }
               if (EQ("-dpi")) {
                  opt.dpi = (u8_t)opt_num_arg();
                  continue;
               }
               break;
            case 'f':
               if (EQ("-fc")) {
                  continue;
               }
               if (EQ("-floor")) {
                  mode = Floor;
                  continue;
               }
               if (EQ("-fn")) {
                  continue;
               }
               if (EQ("-fp")) {
                  continue;
               }
               break;
            case 'g':
               if (EQ("-geometry")) {
                  if ((opt.geomp = util_parse_geom(cp=opt_str_arg())) == 0)
                     warn("bad geometry '%s'\n", cp);
                  continue;
               }
               if (EQ("-glob")) {	/* do hostname wildcarding */
                  opt.glob = 1;
                  continue;
               }
               break;
            case 'h':
               if (EQ("-help"))
                  usage(1);
               break;
            case 'I':
               if (EQ("-I")) {		/* ignore remaining args */
                  if (fp) {
                     fclose(fp);
                     fp = 0;
                  }
                  else
                     argc = 0;
                  continue;
               }
               break;
            case 'i':
               if (EQ("-install")) {
                  opt.install = 1;
                  continue;
               }
               break;
            case 'm':
               if (EQ("-maxcols")) {	/* max number of root colormap cells */
                  if ((opt.maxcolsp = parse_max(cp=opt_str_arg())) == 0)
                     warn("bad maxcols '%s'\n", cp);
                  continue;
               }
               if (EQ("-mdpynum")) {
                  opt.mdpynum = (int)opt_num_arg();
                  continue;
               }
               break;
            case 'n':
               if (EQ("-nodpy")) {	/* ignore DISPLAY env var */
                  nodpy = 1;
                  continue;
               }
               if (EQ("-notp")) {	/* no pointer display */
                  opt.notp = 1;
                  continue;
               }
               if (EQ("-noresize")) {	/* no window resize */
                  opt.resize = 0;
                  continue;
               }
               if (EQ("-noxmc")) {	/* no xmx control clients */
                  opt.xmc = 0;
                  continue;
               }
               break;
            case 'o':
               if (EQ("-owncmap")) {
                  opt.owncmap = 1;
                  continue;
               }
               break;
            case 'q':
               if (EQ("-quiet")) {	/* implies -debug 0 */
                  debug = 0;
                  opt.quiet = 1;
                  continue;
               }
               break;
            case 's':
               if (EQ("-sto")) {
                  opt.sto = (u16_t)opt_num_arg();
                  continue;
               }
               if (EQ("-seat")) {
                  mode = Seat;
                  continue;
               }
               if (EQ("-su")) {
                  opt.su = 1;
                  continue;
               }
               break;
            case 't':
               if (EQ("-to")) {
                  opt.to = (u16_t)opt_num_arg();
                  continue;
               }
               break;
            case 'v':
               if (EQ("-view")) {
                  mode = View;
                  continue;
               }
               break;
            case 'w':
               if (EQ("-wm")) {
                  opt.wm = 1;
                  continue;
               }
               break;
         }
      else if (mode == 0)
         switch (*arg) {
            case 'b':
               if (EQ("bc")) {
                  continue;
               }
               break;
            case 'c':
               if (EQ("c")) {
                  (void) opt_num_arg();
                  continue;
               }
               break;
            case 'n':
               if (EQ("nologo")) {
                  continue;
               }
               break;
            case 'r':
               if (EQ("r")) {
                  continue;
               }
               break;
            case 't':
               if (strlen(arg) == 5 && strncmp(arg, "tty", 3) == 0) {
                  continue;
               }
               break;
            case 'v':
               if (EQ("v")) {
                  continue;
               }
               break;
         }
      else {
         hosts[nhosts] = arg;
         modes[nhosts] = mode;
         nhosts++;
         continue;
      }
      usage(1);	/* error */
   }

   /* more defaults */

   if (opt.geomp == 0)
      opt.geomp = util_parse_geom("");		/* use defaults */
   /*
   **  by default, always set aside some color cells, but set
   **  aside more if we are sharing the root colormap
   */
   if (opt.maxcolsp == 0) {
      if (opt.owncmap)
         opt.maxcolsp = parse_max("7.0/8.0");
      else
         opt.maxcolsp = parse_max("5.0/8.0");
   }
   if (opt.mdpynum < 0)
      opt.mdpynum = opt.dpy;

   if (!nodpy && (cp = (char *)getenv("DISPLAY"))) {
      hosts[nhosts] = cp;
      modes[nhosts] = Floor;
      nhosts++;
   }
   for (i=0; i<nhosts; i++)
      if (util_parse_display(hosts[i], &host, &family,
					&display, &screen, &window)==0) {
         host_pattern(host);
         if (name = host_match(&family, &length, &address))
            (void) server_alloc(did++, name, strlen(name), family, length,
			address, display, screen, modes[i], window, 0);
      }
}
#undef EQ

/*
**   usage
**
**	Print usage message and options.
*/
static void
usage
   AL((status))
   DB int status
   DE
{
   register i;

   fprintf(stderr, USAGE, command);
   for (i=0; options[i]; i++)
      fprintf(stderr, options[i]);
   exit(status);
}

/*
**   opt_num_arg -- return a numeric argument or print usage and quit
*/
static int
opt_num_arg
   VOID
{
   char *rv;

   if (fp && (rv = tok(fp)) == 0) {
      fclose(fp);
      fp = 0;
   }
   if (fp == 0) {
      argc--;
      rv = *argv++;
   }
   if (rv && isdigit(*rv))
      return atoi(rv);

   usage(1);
   /*NOTREACHED*/
}

/*
**   parse_max -- return a max_t argument or print usage and quit
*/
static max_t *
parse_max
   AL((str))
   DB char *str
   DE
{
   int num, den;
   char buf[64], *cp;
   static max_t max;

   if (str && isdigit(*str)) {
      (void)strncpy(buf, str, 63);
      cp = index(buf, '/');
      if (cp) {
         *cp = '\0';
         num = atoi(buf);
         *cp++ = '/';
         den = atoi(cp);
         if (den && num <= den) {
            max.abs = 0;
            max.r = (double)num / (double)den;
            return &max;
         }
         else
            warn("bad fractional max: %s\n", str);
      }
      else {
         max.abs = 1;
         max.n = atoi(str);
         return &max;
      }
   }
   return 0;	/* bad parse */
}

/*
**   opt_str_arg -- return a string argument or print usage and quit
*/
static char *
opt_str_arg
   VOID
{
   char *rv;

   if (fp)
      if (rv = tok(fp))
         return rv;
      else {
         fclose(fp);
         fp = 0;
      }
   if (argc--)
      return *argv++;

   usage(1);
   /*NOTREACHED*/
}

/*
**   opt_debug_arg -- return a numeric debug argument or print usage and quit
*/
static int
opt_debug_arg
   VOID
{
   int i, level;
   char *rv;
   static char *debugopts[]	= {
       "1	trace incoming client X protocol\n",
       "2	trace outgoing virtual client X protocol\n",
       "3	trace incoming server X protocol\n",
       "4	trace outgoing virtual server X protocol\n",
       "a	trace access/authority operations\n",
       "A	trace server add operations\n",
       "b	trace buffer management operations\n",
       "c	trace chunk operations\n",
       "C	trace color operations\n",
       "d	trace protocol details\n",
       "e	print errors (default)\n",
       "E	trace exposure queue operations\n",
       "f	trace focus processing\n",
       "g	trace grab processing\n",
       "h	trace hash table operations\n",
       "i	trace image processing operations\n",
       "k	trace key and pointer mapping operations\n",
       "l	trace list operations\n",
       "m	trace pixel mapping\n",
       "n	trace networking operations\n",
       "p	trace all X protocol (same as 1234)\n",
       "P	painful tracing\n",
       "q	trace protocol to and from query server\n",
       "r	print 'soft' reply errors\n",
       "s	trace sequence number mapping operations\n",
       "S	trace selection processing\n",
       "t	trace request translation processing\n",
       "v	trace virtual configuration operations\n",
       "V	run internal verification checks\n",
       "w	trace byte swapping operations\n",
       "x	trace XMC protocol\n",
       "?	print this message\n",
       (char *)0};

   if (fp && (rv = tok(fp)) == 0) {
      fclose(fp);
      fp = 0;
   }
   if (fp == 0) {
      argc--;
      rv = *argv++;
   }
   if (rv) {
      level = 0;
      while (*rv) {
         switch (*rv) {
            case '1':	level += D_PROTO1;	break;
            case '2':	level += D_PROTO2;	break;
            case '3':	level += D_PROTO3;	break;
            case '4':	level += D_PROTO4;	break;
            case 'a':	level += D_AUTH;	break;
            case 'A':	level += D_RESET;	break;
            case 'b':	level += D_BUFFERS;	break;
            case 'c':	level += D_CHUNK;	break;
            case 'C':	level += D_COLOR;	break;
            case 'd':	level += D_PROTOd;	break;
            case 'e':	level += D_ERRORS;	break;
            case 'E':	level += D_EXPOS;	break;
            case 'f':	level += D_FOCUS;	break;
            case 'g':	level += D_GRABS;	break;
            case 'h':	level += D_HASH;	break;
            case 'i':	level += D_IMAGE;	break;
            case 'k':	level += D_KEYMAP;	break;
            case 'l':	level += D_LIST;	break;
            case 'm':	level += D_PMAP;	break;
            case 'n':	level += D_NETWORK;	break;
            case 'p':	level += D_PROTO;	break;
            case 'P':	level += D_PAINFUL;	break;
            case 'q':	level += D_PROTOq;	break;
            case 'r':	level += D_REPLY;	break;
            case 's':	level += D_SM;		break;
            case 'S':	level += D_SELECT;	break;
            case 't':	level += D_PTC;		break;
            case 'v':	level += D_VCONF;	break;
            case 'V':	level += D_VERIFY;	break;
            case 'w':	level += D_SWAP;	break;
            case 'x':	level += D_PROTOx;	break;
            case '?':
               fprintf(stderr, "debug options:\n");
               for (i=0; debugopts[i]; i++)
                  fprintf(stderr, debugopts[i]);
               exit(0);
            default:
               warn("unknown debug option '%c'\n", *rv);
               usage(1);
         }
         rv++;
      }
      return level;
   }
   usage(1);
   /*NOTREACHED*/
}

/************************************************************************
*									*
*   tok									*
*									*
*	Returns a static buffer containing the next token in the file	*
*	specified by fp.  A token is any contiguous group of ascii	*
*	codes excluding whitespace (space, tab, newline) and comments.	*
*	Comments are "sharp sign (#) to end of line."			*
*									*
*	Returns zero on end-of-file or error.  Ferror(3) can be used	*
*	to determine if an error occurred.				*
*									*
************************************************************************/
#define MAXTOKLEN (MAXHOSTNAMELEN+32)
static char *
tok
   AL((fp))
   DB FILE *fp
   DE
{
   register int i, ch;
   register int comment = 0;
   static char buf[MAXTOKLEN];

   for (i=0; i<MAXTOKLEN;)
      switch (ch = getc(fp)) {
         case '\n':
            comment = 0;
         case ' ':
         case '\t':
            if (i && !comment) {
               buf[i] = '\0';
               return buf;
            }
            break;
         case '#':
            comment = 1;
            break;
         case EOF:
            if (ferror(fp) || i==0)
               return 0;
            else {
               buf[i] = '\0';
               return buf;
            }
            break;
         default:
            if (!comment)
               buf[i++] = (char)ch;
            break;
      }
   return (char *)err(0, "tok: token too long\n");
}
#undef MAXTOKLEN
