/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   cblock.c								*
*									*
*	Routines for traversing an X protocol connection block.		*
*	Only useful for traversing one block at a time.			*
*									*
************************************************************************/
#include <stdlib.h>
#include <X11/Xproto.h>
#include "xmx.h"
#include "incl/cblock.pvt.h"

static xConnSetup *cur_setup = 0;
static xPixmapFormat *cur_format = 0;
static xWindowRoot *cur_root = 0;
static xDepth *cur_depth = 0;
static xVisualType *cur_visual = 0;

static int vendor_len;
static int nformats;
static int nroots;
static int ndepths;
static int nvisuals;
static int totvisuals;	/* count of all visuals per root */

static int formats_left;
static int roots_left;
static int depths_left;
static int visuals_left;

/************************************************************************
*									*
*   cblock_prefix							*
*									*
*	Return a pointer to the xConnSetupPrefix structure within the	*
*	given connection block.						*
*									*
************************************************************************/
xConnSetupPrefix *
cblock_prefix
   AL((cp))
   DB char *cp
   DE
{
   return (xConnSetupPrefix *)cp;
}

/************************************************************************
*									*
*   cblock_setup							*
*									*
*	Return a pointer to the xConnSetup structure within the given	*
*	connection block.  Also sets the current connection block as	*
*	a side effect.							*
*									*
************************************************************************/
xConnSetup *
cblock_setup
   AL((cp))
   DB char *cp
   DE
{
   cur_setup = (xConnSetup *)(cp + sz_xConnSetupPrefix);
   vendor_len = cur_setup->nbytesVendor;
   nformats = cur_setup->numFormats;
   nroots = cur_setup->numRoots;

   cblock_format(1);
   cblock_root(1);

   return cur_setup;
}

/************************************************************************
*									*
*   cblock_vstring							*
*									*
*	Return the vender string from the current connection block.	*
*	The string is copied into a static buffer and is null-		*
*	terminated.							*
*									*
************************************************************************/
char *
cblock_vstring
   VOID
{
   static u16_t ncp = 0;
   static char *cp = 0;
   static char buf[512];

   if (vendor_len >= ncp)
      if (vendor_len >= 512) {	/* unlikely... */
         if (cp && cp != buf)
            free(cp);
         if (MALLOC(cp, char *, vendor_len))
            return "";
         ncp = vendor_len;
      }
      else {
         cp = buf;
         ncp = 512;
      }
   bcopy((char *)(cur_setup+1), cp, vendor_len);
   cp[vendor_len] = '\0';

   return cp;
}

/************************************************************************
*									*
*   cblock_format							*
*									*
*	Return the next format in the current List of Formats.		*
*	Returns zero when the end of the list is reached, and resets	*
*	to the beginning.  If the argument passed is non-zero, resets	*
*	to the beginning.						*
*									*
************************************************************************/
xPixmapFormat *
cblock_format
   AL((rewind))
   DB int rewind
   DE
{
   if (rewind || formats_left == 0) {
      cur_format = 0;
      formats_left = nformats;
   }
   else {
      if (cur_format)
         cur_format++;
      else if (cur_setup && nformats)
         cur_format = (xPixmapFormat *)((char *)(cur_setup + 1) +
					vendor_len + PAD(vendor_len));
      formats_left--;
   }
   return cur_format;
}

/************************************************************************
*									*
*   cblock_root								*
*									*
*	Return the next screen in the current List of Screens.		*
*	Returns zero when the end of the list is reached, and resets	*
*	to the beginning.  If the argument passed is non-zero, resets	*
*	to the beginning.						*
*									*
************************************************************************/
xWindowRoot *
cblock_root
   AL((rewind))
   DB int rewind
   DE
{
   if (rewind || roots_left == 0) {
      cur_root = 0;
      roots_left = nroots;
   }
   else {
      if (cur_root) {
         while (depths_left)
            (void)cblock_depth(0);
         cur_root = (xWindowRoot *)((char *)(cur_root + 1) + 
			ndepths * sz_xDepth +
			totvisuals * sz_xVisualType);
      }
      else if (cur_setup && nroots)
         cur_root = (xWindowRoot *)((char *)(cur_setup + 1) +
					vendor_len + PAD(vendor_len) +
					nformats * sz_xPixmapFormat);
      ndepths = cur_root->nDepths;
      roots_left--;
   }
   cblock_depth(1);
   return cur_root;
}

/************************************************************************
*									*
*   cblock_depth							*
*									*
*	Return the next depth in the current List of Depths.		*
*	Returns zero when the end of the list is reached, and resets	*
*	to the beginning.  If the argument passed is non-zero, resets	*
*	to the beginning.						*
*									*
************************************************************************/
xDepth *
cblock_depth
   AL((rewind))
   DB int rewind
   DE
{
   if (rewind || depths_left == 0) {
      cur_depth = 0;
      depths_left = ndepths;
      totvisuals = 0;
   }
   else {
      if (cur_depth)
         cur_depth = (xDepth *)((char *)(cur_depth + 1) +
					nvisuals * sz_xVisualType);
      else if (cur_root && ndepths)
         cur_depth = (xDepth *)(cur_root + 1);

      nvisuals = cur_depth->nVisuals;
      totvisuals += nvisuals;
      depths_left--;
   }
   cblock_visual(1);
   return cur_depth;
}

/************************************************************************
*									*
*   cblock_visual							*
*									*
*	Return the next visual in the current List of Visuals.		*
*	Returns zero when the end of the list is reached, and resets	*
*	to the beginning.  If the argument passed is non-zero, resets	*
*	to the beginning.						*
*									*
************************************************************************/
xVisualType *
cblock_visual
   AL((rewind))
   DB int rewind
   DE
{
   if (rewind || visuals_left == 0) {
      cur_visual = 0;
      visuals_left = nvisuals;
   }
   else {
      if (cur_visual)
         cur_visual++;
      else if (cur_depth && nvisuals)
         cur_visual = (xVisualType *)(cur_depth + 1);
      visuals_left--;
   }
   return cur_visual;
}
