/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   cursor.c								*
*									*
************************************************************************/
#include <X11/Xproto.h>
#include <X11/X.h>
#include "xmx.h"
#include "df.h"
#include "cx.h"
#include "rx.h"
#include "zb.h"
#include "res.h"
#include "incl/cursor.pvt.h"

static cursor_t *cursors;

/************************************************************************
*									*
*   cursor_create							*
*									*
************************************************************************/
int
cursor_create
   AL((cp, p, chpp))
   DB client_t *cp
   DD xCreateCursorReq *p
   DD chunk_t **chpp
   DE
{
   cursor_t *crp;
   pixmap_t *spxp, *mpxp;

   if ((spxp = (pixmap_t *)hash_data(vmap, p->source)) == 0) {
      proto_Error(cp, BadPixmap, p->source, 0, X_CreateCursor);
      return -1;
   }
   if (p->mask) {
      if ((mpxp = (pixmap_t *)hash_data(vmap, p->mask)) == 0) {
         proto_Error(cp, BadPixmap, p->mask, 0, X_CreateCursor);
         return -1;
      }
      if (	spxp->dwb.width != mpxp->dwb.width ||
		spxp->dwb.height != mpxp->dwb.height) {
         proto_Error(cp, BadMatch, p->mask, 0, X_CreateCursor);
         return -1;
      }
   }
   else
      mpxp = 0;

   if (MALLOC(crp, cursor_t *, sizeof(cursor_t))) {
      proto_Error(cp, BadAlloc, 0, 0, X_CreateCursor);
      return -1;
   }
   if ((crp->vid = hash_add_client_id(p->cid, (char *)crp)) == 0) {
      free(crp);
      proto_Error(cp, BadIDChoice, p->cid, 0, X_CreateCursor);
      return -1;
   }
   crp->res.client = cp;
   crp->cid = p->cid;
   crp->foreRed = p->foreRed;
   crp->foreGreen = p->foreGreen;
   crp->foreBlue = p->foreBlue;
   crp->backRed = p->backRed;
   crp->backGreen = p->backGreen;
   crp->backBlue = p->backBlue;
   crp->isglyph = 0;
   crp->u.pixmap.spxp = 0;
   crp->u.pixmap.mpxp = 0;
   crp->u.pixmap.x = p->x;
   crp->u.pixmap.y = p->y;
   crp->tpimp = tptr_create_image(spxp, mpxp, p->x, p->y, chpp);

   pixmap_assign(&crp->u.pixmap.spxp, spxp);
   pixmap_assign(&crp->u.pixmap.mpxp, mpxp);

   crp->refs = 1;

   crp->last = 0;
   crp->next = cursors;
   if (crp->next)
      crp->next->last = crp;
   cursors = crp;

   cp->refs++;

   return 0;
}

/************************************************************************
*									*
*   cursor_create_glyph							*
*									*
************************************************************************/
int
cursor_create_glyph
   AL((cp, p, chpp))
   DB client_t *cp
   DD xCreateGlyphCursorReq *p
   DD chunk_t **chpp
   DE
{
   cursor_t *crp;
   font_t *sfp, *mfp;

   if ((sfp = (font_t *)hash_data(vmap, p->source)) == 0) {
      proto_Error(cp, BadFont, p->source, 0, X_CreateGlyphCursor);
      return -1;
   }
   if (p->mask) {
      if ((mfp = (font_t *)hash_data(vmap, p->mask)) == 0) {
         proto_Error(cp, BadFont, p->mask, 0, X_CreateGlyphCursor);
         return -1;
      }
   }
   else
      mfp = 0;

   if (MALLOC(crp, cursor_t *, sizeof(cursor_t))) {
      proto_Error(cp, BadAlloc, 0, 0, X_CreateGlyphCursor);
      return -1;
   }
   if ((crp->vid = hash_add_client_id(p->cid, (char *)crp)) == 0) {
      free(crp);
      proto_Error(cp, BadIDChoice, p->cid, 0, X_CreateGlyphCursor);
      return -1;
   }
   crp->res.client = cp;
   crp->cid = p->cid;
   crp->foreRed = p->foreRed;
   crp->foreGreen = p->foreGreen;
   crp->foreBlue = p->foreBlue;
   crp->backRed = p->backRed;
   crp->backGreen = p->backGreen;
   crp->backBlue = p->backBlue;
   crp->isglyph = 1;
   crp->u.glyph.sfp = 0;
   crp->u.glyph.mfp = 0;
   crp->u.glyph.csrc = p->sourceChar;
   crp->u.glyph.cmsk = p->maskChar;
   crp->tpimp = tptr_create_glyph_image(sfp,
					mfp,
					p->sourceChar,
					p->maskChar,
					chpp);
   font_assign(&crp->u.glyph.sfp, sfp);
   font_assign(&crp->u.glyph.mfp, mfp);

   crp->refs = 1;

   crp->last = 0;
   crp->next = cursors;
   if (crp->next)
      crp->next->last = crp;
   cursors = crp;

   cp->refs++;

   return 0;
}

/************************************************************************
*									*
*   cursor_free								*
*									*
************************************************************************/
int
cursor_free
   AL((cp, p))
   DB client_t *cp
   DD xResourceReq *p
   DE
{
   cursor_t *crp;

   if ((crp = (cursor_t *)hash_data(vmap, p->id)) == 0) {
      proto_Error(cp, BadCursor, p->id, 0, X_FreeCursor);
      return -1;
   }
   free_cursor(crp);

   return 0;
}

/************************************************************************
*									*
*   cursor_client_death							*
*									*
*	Does not free telepointer stuff TODO.				*
*									*
************************************************************************/
void
cursor_client_death
   AL((client))
   DB client_t *client
   DE
{
   register rid_t cid;
   register cursor_t *crp, *ncrp;

   for (ncrp=cursors; crp=ncrp;) {
      ncrp = ncrp->next;
      if (client_cmp(client, crp->res.client)) {
         cid = crp->cid;
         if (free_cursor(crp) == 0)
            proto_FreeCursor(cid);
         else {
            client->refs--;
            crp->res.client = 0;
         }
      }
   }
}

/************************************************************************
*									*
*   cursor_reset							*
*									*
************************************************************************/
void
cursor_reset
   VOID
{
   register rid_t cid;
   register cursor_t *crp, *ncrp;

   for (ncrp=cursors; crp=ncrp;) {
      ncrp = ncrp->next;
      cid = crp->cid;
      if (free_cursor(crp) == 0)
         proto_FreeCursor(cid);
   }
}

/************************************************************************
*									*
*   cursor_ketchup							*
*									*
************************************************************************/
void
cursor_ketchup
   VOID
{
   register cursor_t *crp;

   for (crp=cursors; crp; crp=crp->next)
      if (crp->isglyph)
         proto_CreateGlyphCursor(crp->cid,
				crp->u.glyph.sfp->cid,
				crp->u.glyph.mfp ? crp->u.glyph.mfp->cid : 0,
				crp->u.glyph.csrc,
				crp->u.glyph.cmsk,
				crp->foreRed, crp->foreGreen, crp->foreBlue,
				crp->backRed, crp->backGreen, crp->backBlue);
      else
         proto_CreateCursor(	crp->cid,
				crp->tpimp->sbmid,
				crp->tpimp->mbmid,
				crp->foreRed, crp->foreGreen, crp->foreBlue,
				crp->backRed, crp->backGreen, crp->backBlue,
				crp->u.pixmap.x,
				crp->u.pixmap.y);
}

/************************************************************************
*									*
*   cursor_pixmap_ketchup						*
*									*
************************************************************************/
void
cursor_pixmap_ketchup
   VOID
{
   register cursor_t *crp;

   for (crp=cursors; crp; crp=crp->next)
      tptr_draw_pixmaps(crp->tpimp);
}

/************************************************************************
*									*
*   cursor_get_images							*
*									*
************************************************************************/
void
cursor_get_images
   AL((dest))
   DB int dest
   DE
{
   register cursor_t *crp;

   for (crp=cursors; crp; crp=crp->next) {
      proto_GetImage(		XYPixmap,
				crp->tpimp->sbmid,
				0, 0,
				crp->tpimp->width,
				crp->tpimp->height,
				1);
      df_put_i(dest);
      df_put_p((void *)crp->tpimp);
      df_put_i(0);	/* source */
      rx_queue(zrxqp, X_GetImage, 0, zb_seqno(), tptr_image_reply);

      if (crp->tpimp->mbmid) {
         proto_GetImage(	XYPixmap,
				crp->tpimp->mbmid,
				0, 0,
				crp->tpimp->width,
				crp->tpimp->height,
				1);
         df_put_i(dest);
         df_put_p((void *)crp->tpimp);
         df_put_i(1);	/* mask */
         rx_queue(zrxqp, X_GetImage, 0, zb_seqno(), tptr_image_reply);
      }
   }
}

/************************************************************************
*									*
*   cursor_ref								*
*   cursor_deref							*
*   cursor_assign							*
*									*
************************************************************************/
void
cursor_ref
   AL((crp))
   DB cursor_t *crp
   DE
{
   crp->refs++;
}

void
cursor_deref
   AL((crp))
   DB cursor_t *crp
   DE
{
   register chunk_t *chp;
 
   if (crp->refs == 1) {
      zb_dest(ZD_ALLSERV, 0);
      proto_FreeCursor(crp->cid);
   }
   free_cursor(crp);
}

void
cursor_assign
   AL((crpp, crp))
   DB cursor_t **crpp
   DD cursor_t *crp
   DE
{
   if (*crpp != crp) {  
      if (*crpp)
         cursor_deref(*crpp);
      if (crp)
         cursor_ref(crp);
      *crpp = crp;
   }
}


static int
free_cursor
   AL((crp))
   DB cursor_t *crp
   DE
{
   register int i;

   if (--crp->refs == 0) {
      if (crp->last)
         crp->last->next = crp->next;
      if (cursors == crp)
         cursors = crp->next;
      if (crp->next)
         crp->next->last = crp->last;
      hash_mark(vmap, crp->cid);
      crp->res.client->refs--;
      if (crp->isglyph) {
         font_deref(crp->u.glyph.sfp);
         if (crp->u.glyph.mfp)
            font_deref(crp->u.glyph.mfp);
      }
      else {
         pixmap_deref(crp->u.pixmap.spxp);
         if (crp->u.pixmap.mpxp)
            pixmap_deref(crp->u.pixmap.mpxp);
#ifdef TODO
         /*
         **	not so simple - may have multiple refs, also must
         **	handle glyph cursors in a more unified way
         **
         **	these routines do not exist
         */
         cliz_free_tptr_resources(crp->tpimsp);
         tptr_free_image_source(crp->tpimsp);
#endif
      }
      free(crp);
      return 0;
   }
   return -1;
}
