/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   debug.c								*
*									*
*	Some routines for debugging.					*
*									*
************************************************************************/
#include <stdlib.h>

#include "xmx.h"
#include "incl/debug.pvt.h"

/************************************************************************
*									*
*   debug_set								*
*									*
************************************************************************/
void
debug_set
   AL((level))
   DB int level
   DE
{
   debug = level;
}

/************************************************************************
*									*
*   debug_malloc							*
*   debug_calloc							*
*   debug_free								*
*   debug_memverify							*
*   debug_memscan							*
*   debug_memprint							*
*									*
************************************************************************/

#define HEAPMAPSZ	10000

typedef struct _heapmap_t {
   char *	beg;
   char *	end;
}heapmap_t;

#ifdef MALLOC_DEBUG
static heapmap_t hmap[HEAPMAPSZ];
static int hmax = -1;
#endif

static void
addheapmap
   AL((ptr, sz))
   DB char *ptr
   DD uint_t sz
   DE
{
   register int i;

#ifdef MALLOC_DEBUG
   for (i=0; i<HEAPMAPSZ; i++)
      if (hmap[i].beg == 0) {
         hmap[i].beg = ptr;
         hmap[i].end = ptr + sz;
         break;
      }
   if (i == HEAPMAPSZ) {
      warn("addheapmap: heapmap size exceeded\n");
      abort();
   }
   else if (i > hmax)
      hmax = i;
#endif
}

/*
**	encloses each allocated block in markers which can be checked
**	as an indicator of heap integrity.
*/
char *
debug_malloc
   AL((sz))
   DB uint_t sz
   DE
{
   register uint_t len;
   register char *ptr, *cp;

#ifdef malloc
#undef malloc
#endif
   if (sz == 0) abort();

#ifdef MALLOC_DEBUG
   ptr = (char *)malloc(sz + 8);
   cp = ptr;
   cp[0] = 0x55;
   cp[1] = 0x55;
   cp[2] = 0x55;
   cp[3] = 0x55;
   ptr += 4;
   cp = ptr + sz;
   cp[0] = 0x55;
   cp[1] = 0x55;
   cp[2] = 0x55;
   cp[3] = 0x55;
   addheapmap(ptr, sz);
#else
   ptr = (char *)malloc(sz);
#endif
/*
   fprintf(stderr, "0x%x = malloc(%d)\n", ptr, sz);
*/
   return ptr;
}

char *
debug_calloc
   AL((n, sz))
   DB uint_t n
   DD uint_t sz
   DE
{
   register int len;
   register char *ptr;

#ifdef calloc
#undef calloc
#endif
#ifdef MALLOC_DEBUG
   len = n * sz;
   ptr = debug_malloc(len);
   bzero(ptr, len);
#else
   ptr = (char *)calloc(n, sz);
#endif
/*
   fprintf(stderr, "0x%x = calloc(%d, %d)\n", ptr, n, sz);
*/
   return ptr;
}

void
debug_free
   AL((ptr))
   DB char *ptr
   DE
{
   int n;

#ifdef free
#undef free
#endif

#ifdef MALLOC_DEBUG
   switch (debug_memverify(ptr, &n)) {
      case -1:
         warn("debug_free: freeing unallocated block 0x%x\n", ptr);
         abort();
         /*NOTREACHED*/
      case 0:
         debug_memscan();
         hmap[n].beg = 0;
         break;
      case 1:
         warn("debug_free: 0x%x is inside the block 0x%x, sz %d\n",
				ptr, hmap[n].beg, hmap[n].end - hmap[n].beg);
         abort();
   }
   ptr -= 4;
#endif

   free(ptr);
/*
   fprintf(stderr, "free(0x%x)\n", ptr);
*/
}

int
debug_memverify
   AL((ptr, n))
   DB char *ptr
   DD int *n
   DE
{
   register int i;

#ifdef MALLOC_DEBUG
   for (i=0; i<=hmax; i++)
      if (hmap[i].beg)
         if (hmap[i].beg == ptr) {
            *n = i;
            return 0;		/* found */
         }
         else if (ptr > hmap[i].beg && ptr < hmap[i].end) {
            *n = i;
            return 1;		/* inside a block */
         }
   if (i > hmax)
      return -1;		/* not found */
#endif

   return 0;
}

static int
verify_block
   AL((n))
   DB int n
   DE
{
   register char *ptr, *cp;

#ifdef MALLOC_DEBUG
   ptr = hmap[n].beg;

   cp = ptr - 4;
   if (	cp[0] != 0x55 || cp[1] != 0x55 || cp[2] != 0x55 || cp[3] != 0x55)
      return err(-1, "verify_block: block 0x%x, left edge violation\n", ptr);
   cp = hmap[n].end;
   if (	cp[0] != 0x55 || cp[1] != 0x55 || cp[2] != 0x55 || cp[3] != 0x55)
      return err(-1, "verify_block: block 0x%x, right edge violation\n", ptr);
#endif

   return 0;
}

/*
**	scan all allocated blocks and verify their integrity
*/
void
debug_memscan
   VOID
{
   register int i;

#ifdef MALLOC_DEBUG
   for (i=0; i<=hmax; i++)
      if (hmap[i].beg)
         if (verify_block(i))
            abort();
#endif
}

/*
**	handy for use from a debugger
*/
void
debug_memprint
   AL((ptr))
   DB char *ptr
   DE
{
   int n;

#ifdef MALLOC_DEBUG
   switch (debug_memverify(ptr, &n)) {
      case -1:
         warn("debug_memprint: 0x%x not allocated\n", ptr);
         break;
      case 0:
         warn("debug_memprint: 0x%x allocated, size %d\n", ptr,
						hmap[n].end - hmap[n].beg);
         (void) verify_block(n);
         break;
      case 1:
         warn("debug_memprint: 0x%x inside block 0x%x, sz %d\n",
				ptr, hmap[n].beg, hmap[n].end - hmap[n].beg);
         (void) verify_block(n);
         break;
   }
#else
   warn("debug_memprint: not compiled -DMALLOC_DEBUG\n");
#endif
}

/************************************************************************
*									*
*   debug_conn_type_str							*
*									*
************************************************************************/
char *
debug_conn_type_str
   AL((type))
   DB int type
   DE
{
   static char buf[32];

   switch (type) {
      case CLOSED:	return "CLOSED";
      case XPORT:	return "XPORT";
      case CLIENT:	return "CLIENT";
      case SERVER:	return "SERVER";
      case XMCPORT:	return "XMCPORT";
      case XMC:		return "XMC";
      case XDMCP:	return "XDMCP";
      default:
         sprintf(buf, "<%d>", type);
         return buf;
   }
}

/************************************************************************
*									*
*   debug_vcstate_str							*
*									*
************************************************************************/
char *
debug_vcstate_str
   AL((state))
   DB int state
   DE
{
   static char buf[32];

   switch (state) {
      case VC_MUSH:	return "VC_MUSH";
      case VC_INFLUX:	return "VC_INFLUX";
      case VC_FIXED:	return "VC_FIXED";
      default:
         sprintf(buf, "<%d>", state);
         return buf;
   }
}

/************************************************************************
*									*
*   debug_server_state_str						*
*									*
*	Convert a server state to a string name.			*
*									*
************************************************************************/
char *
debug_server_state_str
   AL((state))
   DB u16_t state
   DE
{
   static char buf[16];

   switch (state) {
      case S_ZOMBIE:		return "S_ZOMBIE";
      case S_NAMED:		return "S_NAMED";
      case S_INPROGRESS:	return "S_INPROGRESS";
      case S_BLOCKED:		return "S_BLOCKED";
      case S_POKED:		return "S_POKED";
      case S_KETCHUP:		return "S_KETCHUP";
      case S_READY:		return "S_READY";
      default:
         sprintf(buf, "<%d>", state);
         return buf;
   }
}

/************************************************************************
*									*
*   debug_buf_type_str							*
*									*
************************************************************************/
char *
debug_buf_type_str
   AL((type))
   DB u8_t type
   DE
{
   static char buf[16];

   switch (type) {
      case B_STATIC:		return "B_STATIC";
      case B_FREEONWRITE:	return "B_FREEONWRITE";
      default:
         sprintf(buf, "<%d>", type);
         return buf;
   }
}

/************************************************************************
*									*
*   debug_queue_dest_str						*
*									*
************************************************************************/
char *
debug_queue_dest_str
   AL((dest))
   DB u16_t dest
   DE
{
   static char buf[16];

   switch (dest) {
      case Q_NONE:		return "Q_NONE";
      case Q_XSERVER:		return "Q_XSERVER";
      case Q_XCLIENT:		return "Q_XCLIENT";
      case Q_XMCCLIENT:		return "Q_XMCCLIENT";
      default:
         sprintf(buf, "<%d>", dest);
         return buf;
   }
}

/************************************************************************
*									*
*   debug_proto_type_str						*
*									*
************************************************************************/
char *
debug_proto_type_str
   AL((type))
   DB u8_t type
   DE
{
   static char buf[16];

   switch (type) {
      case P_NONE:		return "P_NONE";
      case P_REQUEST:		return "P_REQUEST";
      case P_REPLY:		return "P_REPLY";
      case P_ERROR:		return "P_ERROR";
      case P_EVENT:		return "P_EVENT";
      case P_IMAGE:		return "P_IMAGE";
      case P_EXT:		return "P_EXT";
      default:
         sprintf(buf, "<%d>", type);
         return buf;
   }
}
