/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   dprx.c								*
*									*
*	Routines for printing X protocol structures.			*
*									*
************************************************************************/
#include <stdlib.h>
#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/Xproto.h>
#include <X11/X.h>
#include <X11/Xatom.h>

#include "xmx.h"
#include "ptc.h"
#include "incl/dprx.pvt.h"

static char *empty_string = "";

/************************************************************************
*									*
*   dprx_stackmode_str							*
*									*
************************************************************************/
char *
dprx_stackmode_str
   AL((mode))
   DB u8_t mode
   DE
{
   static char buf[32];

   switch (mode) {
      case Above:	return "Above";
      case Below:	return "Below";
      case TopIf:	return "TopIf";
      case BottomIf:	return "BottomIf";
      case Opposite:	return "Opposite";
      default:
         sprintf(buf, "<%d>", mode);
         return buf;
   }
}

char *
dprx_notify_detail_str
   AL((detail))
   DB u8_t detail
   DE
{
   static char buf[32];

   switch (detail) {
      case NotifyAncestor:		return "NotifyAncestor";
      case NotifyVirtual:		return "NotifyVirtual";
      case NotifyInferior:		return "NotifyInferior";
      case NotifyNonlinear:		return "NotifyNonlinear";
      case NotifyNonlinearVirtual:	return "NotifyNonlinearVirtual";
      case NotifyPointer:		return "NotifyPointer";
      case NotifyPointerRoot:		return "NotifyPointerRoot";
      case NotifyDetailNone:		return "NotifyDetailNone";
      default:
         sprintf(buf, "<%d>", detail);
         return buf;
   }
}

/************************************************************************
*									*
*   dprx_event_str							*
*									*
*	X_Reply and X_Error are included because they are always	*
*	distinct from event codes.					*
*									*
************************************************************************/
char *
dprx_event_str
   AL((type))
   DB u8_t type
   DE
{
   switch (type & 0x7f) {
      case X_Reply:			return "X_Reply";
      case X_Error:			return "X_Error";
      case KeyPress:			return "KeyPress";
      case KeyRelease:			return "KeyRelease";
      case ButtonPress:			return "ButtonPress";
      case ButtonRelease:		return "ButtonRelease";
      case MotionNotify:		return "MotionNotify";
      case EnterNotify:			return "EnterNotify";
      case LeaveNotify:			return "LeaveNotify";
      case FocusIn:			return "FocusIn";
      case FocusOut:			return "FocusOut";
      case KeymapNotify:		return "KeymapNotify";
      case Expose:			return "Expose";
      case GraphicsExpose:		return "GraphicsExpose";
      case NoExpose:			return "NoExpose";
      case VisibilityNotify:		return "VisibilityNotify";
      case CreateNotify:		return "CreateNotify";
      case DestroyNotify:		return "DestroyNotify";
      case UnmapNotify:			return "UnmapNotify";
      case MapNotify:			return "MapNotify";
      case MapRequest:			return "MapRequest";
      case ReparentNotify:		return "ReparentNotify";
      case ConfigureNotify:		return "ConfigureNotify";
      case ConfigureRequest:		return "ConfigureRequest";
      case GravityNotify:		return "GravityNotify";
      case ResizeRequest:		return "ResizeRequest";
      case CirculateNotify:		return "CirculateNotify";
      case CirculateRequest:		return "CirculateRequest";
      case PropertyNotify:		return "PropertyNotify";
      case SelectionClear:		return "SelectionClear";
      case SelectionRequest:		return "SelectionRequest";
      case SelectionNotify:		return "SelectionNotify";
      case ColormapNotify:		return "ColormapNotify";
      case ClientMessage:		return "ClientMessage";
      case MappingNotify:		return "MappingNotify";
      default:				return "<unknown event>";
   }
}

/************************************************************************
*									*
*   dprx_event_window_str						*
*									*
*	If the event has an event window field, print it.		*
*									*
************************************************************************/
rid_t
dprx_event_window
   AL((ep))
   DB xEvent *ep
   DE
{
   switch (ep->u.u.type & 0x7f) {
      case KeyPress:		return ep->u.keyButtonPointer.event;
      case KeyRelease:		return ep->u.keyButtonPointer.event;
      case ButtonPress:		return ep->u.keyButtonPointer.event;
      case ButtonRelease:	return ep->u.keyButtonPointer.event;
      case MotionNotify:	return ep->u.keyButtonPointer.event;
      case EnterNotify:		return ep->u.enterLeave.event;
      case LeaveNotify:		return ep->u.enterLeave.event;
      case FocusIn:		return ep->u.focus.window;
      case FocusOut:		return ep->u.focus.window;
      case Expose:		return ep->u.expose.window;
      case VisibilityNotify:	return ep->u.visibility.window;
      case CreateNotify:	return ep->u.createNotify.window;
      case DestroyNotify:	return ep->u.destroyNotify.window;
      case UnmapNotify:		return ep->u.unmapNotify.window;
      case MapNotify:		return ep->u.mapNotify.window;
      case MapRequest:		return ep->u.mapRequest.window;
      case ReparentNotify:	return ep->u.reparent.window;
      case ConfigureNotify:	return ep->u.configureNotify.window;
      case ConfigureRequest:	return ep->u.configureRequest.window;
      case GravityNotify:	return ep->u.gravity.window;
      case ResizeRequest:	return ep->u.resizeRequest.window;
      case CirculateNotify:	return ep->u.circulate.window;
      case PropertyNotify:	return ep->u.property.window;
      case SelectionClear:	return ep->u.selectionClear.window;
      case SelectionRequest:	return ep->u.selectionRequest.requestor;
      case SelectionNotify:	return ep->u.selectionNotify.requestor;
      case ColormapNotify:	return ep->u.colormap.window;
      case ClientMessage:	return ep->u.clientMessage.window;
      default:			return 0;
   }
}

/************************************************************************
*									*
*   dprx_event_mask_str							*
*									*
*	Convert an event mask to a string.				*
*									*
************************************************************************/
char *
dprx_event_mask_str
   AL((mask))
   DB mask_t mask
   DE
{
   register int n = 0;
   register mask_t m, tm;
   static char buf[256];

   buf[0] = '(';
   buf[1] = '\0';

   for (tm=mask, m=1; tm; tm>>=1, m<<=1)
      if (tm & 1) {
         if (n++)
            strcat(buf, "|");
         switch (m) {
            case KeyPressMask:		strcat(buf, "KeyPress"); break;
            case KeyReleaseMask:	strcat(buf, "KeyRelease"); break;
            case ButtonPressMask:	strcat(buf, "ButtonPress"); break;
            case ButtonReleaseMask:	strcat(buf, "ButtonRelease"); break;
            case EnterWindowMask:	strcat(buf, "EnterWindow"); break;
            case LeaveWindowMask:	strcat(buf, "LeaveWindow"); break;
            case PointerMotionMask:	strcat(buf, "PointerMotion"); break;
            case PointerMotionHintMask:	strcat(buf, "PointerMotionHint"); break;
            case Button1MotionMask:	strcat(buf, "Button1Motion"); break;
            case Button2MotionMask:	strcat(buf, "Button2Motion"); break;
            case Button3MotionMask:	strcat(buf, "Button3Motion"); break;
            case Button4MotionMask:	strcat(buf, "Button4Motion"); break;
            case Button5MotionMask:	strcat(buf, "Button5Motion"); break;
            case ButtonMotionMask:	strcat(buf, "ButtonMotion"); break;
            case KeymapStateMask:	strcat(buf, "KeymapState"); break;
            case ExposureMask:		strcat(buf, "Exposure"); break;
            case VisibilityChangeMask:	strcat(buf, "VisibilityChange"); break;
            case StructureNotifyMask:	strcat(buf, "StructureNotify"); break;
            case ResizeRedirectMask:	strcat(buf, "ResizeRedirect"); break;
            case SubstructureNotifyMask:
				strcat(buf, "SubstructureNotify"); break;
            case SubstructureRedirectMask:
				strcat(buf, "SubstructureRedirect"); break;
            case FocusChangeMask:	strcat(buf, "FocusChange"); break;
            case PropertyChangeMask:	strcat(buf, "PropertyChange"); break;
            case ColormapChangeMask:	strcat(buf, "ColormapChange"); break;
            case OwnerGrabButtonMask:	strcat(buf, "OwnerGrabButton"); break;
         }
      }
   strcat(buf, ")");
   return buf;
}

/************************************************************************
*									*
*   dprx_notify_mode_str						*
*									*
************************************************************************/
char *
dprx_notify_mode_str
   AL((mode))
   DB u8_t mode
   DE
{
   static char buf[32];

   switch (mode) {
      case NotifyNormal:	return "NotifyNormal";
      case NotifyGrab:		return "NotifyGrab";
      case NotifyUngrab:	return "NotifyUngrab";
      case NotifyWhileGrabbed:	return "NotifyWhileGrabbed";
      default:
         sprintf(buf, "<unknown mode(%d)>", mode);
         return buf;
   }
}

/************************************************************************
*									*
*   dprx_allowevents_mode_str						*
*									*
************************************************************************/
char *
dprx_allowevents_mode_str
   AL((mode))
   DB u8_t mode
   DE
{
   static char buf[32];

   switch (mode) {
      case AsyncPointer:	return "AsyncPointer";
      case SyncPointer:		return "SyncPointer";
      case ReplayPointer:	return "ReplayPointer";
      case AsyncKeyboard:	return "AsyncKeyboard";
      case SyncKeyboard:	return "SyncKeyboard";
      case ReplayKeyboard:	return "ReplayKeyboard";
      case AsyncBoth:		return "AsyncBoth";
      case SyncBoth:		return "SyncBoth";
      default:
         sprintf(buf, "<unknown mode(%d)>", mode);
         return buf;
   }
}

/************************************************************************
*									*
*   dprx_addrfam_str							*
*									*
************************************************************************/
char *
dprx_addrfam_str
   AL((family))
   DB u16_t family
   DE
{
   static char buf[32];

   switch (family) {
      case FamilyInternet:	return "FamilyInternet";
      case FamilyDECnet:	return "FamilyDECnet";
      case FamilyChaos:		return "FamilyChaos";
      case FamilyLocal:		return "FamilyLocal";
      case FamilyWild:		return "FamilyWild";
      default:
         sprintf(buf, "FamilyBAD(%d)", family);
         return buf;
   }
}

/************************************************************************
*									*
*   dprx_errorCode_str							*
*									*
************************************************************************/
char *
dprx_errorCode_str
   AL((errorCode))
   DB int errorCode
   DE
{
   switch (errorCode) {
      case BadRequest:		return "BadRequest";
      case BadValue:		return "BadValue";
      case BadWindow:		return "BadWindow";
      case BadPixmap:		return "BadPixmap";
      case BadAtom:		return "BadAtom";
      case BadCursor:		return "BadCursor";
      case BadFont:		return "BadFont";
      case BadMatch:		return "BadMatch";
      case BadDrawable:		return "BadDrawable";
      case BadAccess:		return "BadAccess";
      case BadAlloc:		return "BadAlloc";
      case BadColor:		return "BadColor";
      case BadGC:		return "BadGC";
      case BadIDChoice:		return "BadIDChoice";
      case BadName:		return "BadName";
      case BadLength:		return "BadLength";
      case BadImplementation:	return "BadImplementation";
      default:			return "<unknown errorcode>";
   }
}

/************************************************************************
*									*
*   dprx_reqType_str							*
*									*
************************************************************************/
char *
dprx_reqType_str
   AL((reqType))
   DB int reqType
   DE
{
   switch (reqType) {
      case X_CreateWindow:		return "CreateWindow";
      case X_ChangeWindowAttributes:	return "ChangeWindowAttributes";
      case X_GetWindowAttributes:	return "GetWindowAttributes";
      case X_DestroyWindow:		return "DestroyWindow";
      case X_DestroySubwindows:		return "DestroySubwindows";
      case X_ChangeSaveSet:		return "ChangeSaveSet";
      case X_ReparentWindow:		return "ReparentWindow";
      case X_MapWindow:			return "MapWindow";
      case X_MapSubwindows:		return "MapSubwindows";
      case X_UnmapWindow:		return "UnmapWindow";
      case X_UnmapSubwindows:		return "UnmapSubwindows";
      case X_ConfigureWindow:		return "ConfigureWindow";
      case X_CirculateWindow:		return "CirculateWindow";
      case X_GetGeometry:		return "GetGeometry";
      case X_QueryTree:			return "QueryTree";
      case X_InternAtom:		return "InternAtom";
      case X_GetAtomName:		return "GetAtomName";
      case X_ChangeProperty:		return "ChangeProperty";
      case X_DeleteProperty:		return "DeleteProperty";
      case X_GetProperty:		return "GetProperty";
      case X_ListProperties:		return "ListProperties";
      case X_SetSelectionOwner:		return "SetSelectionOwner";
      case X_GetSelectionOwner:		return "GetSelectionOwner";
      case X_ConvertSelection:		return "ConvertSelection";
      case X_SendEvent:			return "SendEvent";
      case X_GrabPointer:		return "GrabPointer";
      case X_UngrabPointer:		return "UngrabPointer";
      case X_GrabButton:		return "GrabButton";
      case X_UngrabButton:		return "UngrabButton";
      case X_ChangeActivePointerGrab:	return "ChangeActivePointerGrab";
      case X_GrabKeyboard:		return "GrabKeyboard";
      case X_UngrabKeyboard:		return "UngrabKeyboard";
      case X_GrabKey:			return "GrabKey";
      case X_UngrabKey:			return "UngrabKey";
      case X_AllowEvents:		return "AllowEvents";
      case X_GrabServer:		return "GrabServer";
      case X_UngrabServer:		return "UngrabServer";
      case X_QueryPointer:		return "QueryPointer";
      case X_GetMotionEvents:		return "GetMotionEvents";
      case X_TranslateCoords:		return "TranslateCoords";
      case X_WarpPointer:		return "WarpPointer";
      case X_SetInputFocus:		return "SetInputFocus";
      case X_GetInputFocus:		return "GetInputFocus";
      case X_QueryKeymap:		return "QueryKeymap";
      case X_OpenFont:			return "OpenFont";
      case X_CloseFont:			return "CloseFont";
      case X_QueryFont:			return "QueryFont";
      case X_QueryTextExtents:		return "QueryTextExtents";
      case X_ListFonts:			return "ListFonts";
      case X_ListFontsWithInfo:		return "ListFontsWithInfo";
      case X_SetFontPath:		return "SetFontPath";
      case X_GetFontPath:		return "GetFontPath";
      case X_CreatePixmap:		return "CreatePixmap";
      case X_FreePixmap:		return "FreePixmap";
      case X_CreateGC:			return "CreateGC";
      case X_ChangeGC:			return "ChangeGC";
      case X_CopyGC:			return "CopyGC";
      case X_SetDashes:			return "SetDashes";
      case X_SetClipRectangles:		return "SetClipRectangles";
      case X_FreeGC:			return "FreeGC";
      case X_ClearArea:			return "ClearArea";
      case X_CopyArea:			return "CopyArea";
      case X_CopyPlane:			return "CopyPlane";
      case X_PolyPoint:			return "PolyPoint";
      case X_PolyLine:			return "PolyLine";
      case X_PolySegment:		return "PolySegment";
      case X_PolyRectangle:		return "PolyRectangle";
      case X_PolyArc:			return "PolyArc";
      case X_FillPoly:			return "FillPoly";
      case X_PolyFillRectangle:		return "PolyFillRectangle";
      case X_PolyFillArc:		return "PolyFillArc";
      case X_PutImage:			return "PutImage";
      case X_GetImage:			return "GetImage";
      case X_PolyText8:			return "PolyText8";
      case X_PolyText16:		return "PolyText16";
      case X_ImageText8:		return "ImageText8";
      case X_ImageText16:		return "ImageText16";
      case X_CreateColormap:		return "CreateColormap";
      case X_FreeColormap:		return "FreeColormap";
      case X_CopyColormapAndFree:	return "CopyColormapAndFree";
      case X_InstallColormap:		return "InstallColormap";
      case X_UninstallColormap:		return "UninstallColormap";
      case X_ListInstalledColormaps:	return "ListInstalledColormaps";
      case X_AllocColor:		return "AllocColor";
      case X_AllocNamedColor:		return "AllocNamedColor";
      case X_AllocColorCells:		return "AllocColorCells";
      case X_AllocColorPlanes:		return "AllocColorPlanes";
      case X_FreeColors:		return "FreeColors";
      case X_StoreColors:		return "StoreColors";
      case X_StoreNamedColor:		return "StoreNamedColor";
      case X_QueryColors:		return "QueryColors";
      case X_LookupColor:		return "LookupColor";
      case X_CreateCursor:		return "CreateCursor";
      case X_CreateGlyphCursor:		return "CreateGlyphCursor";
      case X_FreeCursor:		return "FreeCursor";
      case X_RecolorCursor:		return "RecolorCursor";
      case X_QueryBestSize:		return "QueryBestSize";
      case X_QueryExtension:		return "QueryExtension";
      case X_ListExtensions:		return "ListExtensions";
      case X_ChangeKeyboardMapping:	return "ChangeKeyboardMapping";
      case X_GetKeyboardMapping:	return "GetKeyboardMapping";
      case X_ChangeKeyboardControl:	return "ChangeKeyboardControl";
      case X_GetKeyboardControl:	return "GetKeyboardControl";
      case X_Bell:			return "Bell";
      case X_ChangePointerControl:	return "ChangePointerControl";
      case X_GetPointerControl:		return "GetPointerControl";
      case X_SetScreenSaver:		return "SetScreenSaver";
      case X_GetScreenSaver:		return "GetScreenSaver";
      case X_ChangeHosts:		return "ChangeHosts";
      case X_ListHosts:			return "ListHosts";
      case X_SetAccessControl:		return "SetAccessControl";
      case X_SetCloseDownMode:		return "SetCloseDownMode";
      case X_KillClient:		return "KillClient";
      case X_RotateProperties:		return "RotateProperties";
      case X_ForceScreenSaver:		return "ForceScreenSaver";
      case X_SetPointerMapping:		return "SetPointerMapping";
      case X_GetPointerMapping:		return "GetPointerMapping";
      case X_SetModifierMapping:	return "SetModifierMapping";
      case X_GetModifierMapping:	return "GetModifierMapping";
      case X_NoOperation:		return "NoOperation";
      default:				return "<unknown request>";
   }
}

/************************************************************************
*									*
*   dprx_winval_str							*
*									*
************************************************************************/
char *
dprx_winval_str
   AL((val))
   DB int val
   DE
{
   switch (val) {
      case CWBackPixmap:		return "CWBackPixmap";
      case CWBackPixel:			return "CWBackPixel";
      case CWBorderPixmap:		return "CWBorderPixmap";
      case CWBorderPixel:		return "CWBorderPixel";
      case CWBitGravity:		return "CWBitGravity";
      case CWWinGravity:		return "CWWinGravity";
      case CWBackingStore:		return "CWBackingStore";
      case CWBackingPlanes:		return "CWBackingPlanes";
      case CWBackingPixel:		return "CWBackingPixel";
      case CWOverrideRedirect:		return "CWOverrideRedirect";
      case CWSaveUnder:			return "CWSaveUnder";
      case CWEventMask:			return "CWEventMask";
      case CWDontPropagate:		return "CWDontPropagate";
      case CWColormap:			return "CWColormap";
      case CWCursor:			return "CWCursor";
      default:				return "<unknown winval>";
   }
}


/************************************************************************
*									*
*   dprx_visual_class_str						*
*									*
************************************************************************/
char *
dprx_visual_class_str
   AL((class))
   DB u8_t class
   DE
{
   switch (class) {
      case StaticGray:		return "StaticGray";
      case GrayScale:		return "GrayScale";
      case StaticColor:		return "StaticColor";
      case PseudoColor:		return "PseudoColor";
      case TrueColor:		return "TrueColor";
      case DirectColor:		return "DirectColor";
      default:			return "<unknown visual class>";
   }
}
/************************************************************************
*									*
*   dprx_pixel_str							*
*									*
************************************************************************/
char *
dprx_pixel_str
   AL((pxl))
   DB pixel_t pxl
   DE
{
   static char buf[32];

   sprintf(buf, "%d (%x)", pxl, pxl);

   return buf;
}

/************************************************************************
*									*
*   dprx_ConnClientPrefix						*
*									*
************************************************************************/
void
dprx_ConnClientPrefix
   AL((p))
   DB xConnClientPrefix *p
   DE
{
   register int i;
   register char *cp;

   warn("xConnClientPrefix\n");
   warn("	byteOrder_____________ %02x\n", p->byteOrder);
   warn("	majorVersion__________ %d\n", p->majorVersion);
   warn("	minorVersion__________ %d\n", p->minorVersion);
   warn("	nbytesAuthProto_______ %d\n", p->nbytesAuthProto);
   warn("	nbytesAuthString______ %d\n", p->nbytesAuthString);
   cp = (char *)(p+1);
   warn("	<protocol name>_______\"");
   for (i=0; i<(int)p->nbytesAuthProto; i++)
      warn("%c", (unsigned char)cp[i]);
   warn("\"\n");

   cp += p->nbytesAuthProto + PAD(p->nbytesAuthProto);
   warn("	<protocol data>_______");
   for (i=0; i<(int)p->nbytesAuthString; i++)
      warn("%02x", (unsigned char)cp[i]);
   warn("\n");
}

/************************************************************************
*									*
*   dprx_*								*
*									*
*	Routines to print the contents of particular structures.	*
*									*
************************************************************************/
void
dprx_ConnSetupPrefix
   AL((prefix))
   DB xConnSetupPrefix *prefix
   DE
{
   char buf[128];

   warn("ConnSetupPrefix\n");
   warn("	success___________________ %s\n",
					prefix->success ? "TRUE" : "FALSE");
   warn("	majorVersion.minorVersion_ %d.%d\n", prefix->majorVersion,
						prefix->minorVersion);
   warn("	length____________________ %d\n", prefix->length);
   if (prefix->success == 0) {
      bcopy((char *)(prefix+1), buf, prefix->lengthReason);
      buf[prefix->lengthReason] = '\0';
      warn("	reason____________________\"%s\"\n", buf);
   }
}

void
dprx_ConnSetup
   AL((setup))
   DB xConnSetup *setup
   DE
{
   warn("ConnSetup\n");
   warn("	release___________________ %08x\n", setup->release);
   warn("	ridBase.ridMask___________ %08x.%08x\n",	setup->ridBase,
							setup->ridMask);
   warn("	motionBufferSize__________ %d\n", setup->motionBufferSize);
   warn("	maxRequestSize____________ %d\n", setup->maxRequestSize);
   warn("	numRoots__________________ %d\n", setup->numRoots);
   warn("	numFormats________________ %d\n", setup->numFormats);
   warn("	imageByteOrder____________ %s\n",
					setup->imageByteOrder ? "MSB":"LSB");
   warn("	bitmapBitOrder____________ %s\n",
					setup->bitmapBitOrder ? "MSB":"LSB");
   warn("	bitmapScanlineUnit________ %d\n", setup->bitmapScanlineUnit);
   warn("	bitmapScanlinePad_________ %d\n", setup->bitmapScanlinePad);
   warn("	minKeyCode.maxKeyCode_____ %d.%d\n",	setup->minKeyCode,
							setup->maxKeyCode);
}

void
dprx_vender
   AL((str))
   DB char *str
   DE
{
   warn("vender\n");
   warn("	\"%s\"\n", str);
}

void
dprx_PixmapFormat
   AL((format))
   DB xPixmapFormat *format
   DE
{
   warn("PixmapFormat\n");
   warn("	depth_____________________ %d\n", format->depth);
   warn("	bitsPerPixel______________ %d\n", format->bitsPerPixel);
   warn("	scanLinePad_______________ %d\n", format->scanLinePad);
}

void
dprx_WindowRoot
   AL((root))
   DB xWindowRoot *root
   DE
{
   warn("WindowRoot\n");
   warn("	windowID__________________ %08x\n", root->windowId);
   warn("	defaultColormap___________ %08x\n", root->defaultColormap);
   warn("	whitePixel.blackPixel_____ %d.%d\n",	root->whitePixel,
							root->blackPixel);
   warn("	currentInputMask__________ %08x\n", root->currentInputMask);
   warn("	pixWidth.pixHeight________ %d.%d\n",	root->pixWidth,
							root->pixHeight);
   warn("	mmWidth.mmHeight__________ %d.%d\n",	root->mmWidth,
							root->mmHeight);
   warn("	minInstalledMaps__________ %d\n", root->minInstalledMaps);
   warn("	maxInstalledMaps__________ %d\n", root->maxInstalledMaps);
   warn("	rootVisualID______________ %08x\n", root->rootVisualID);
   warn("	backingStore______________ %d", root->backingStore);
   switch (root->backingStore) {
      case 0:  warn(" (Never)\n");			break;
      case 1:  warn(" (WhenMapped)\n");			break;
      case 2:  warn(" (Always)\n");			break;
      default: warn(" (%d)\n", root->backingStore);	break;
   }
   warn("	saveUnders________________ %s\n",
					root->saveUnders ? "TRUE" : "FALSE");
   warn("	rootDepth_________________ %d\n", root->rootDepth);
   warn("	nDepths___________________ %d\n", root->nDepths);
}

void
dprx_Depth
   AL((depth))
   DB xDepth *depth
   DE
{
   warn("Depth\n");
   warn("	depth_____________________ %d\n", depth->depth);
   warn("	nVisuals__________________ %d\n", depth->nVisuals);
}

void
dprx_VisualType
   AL((visual))
   DB xVisualType *visual
   DE
{
   warn("VisualType\n");
   warn("	visualID__________________ %08x\n",
						visual->visualID);
   warn("	class_____________________ %d (%s)\n",
						visual->class,
					dprx_visual_class_str(visual->class));
   warn("	bitsPerRGB________________ %d\n",
						visual->bitsPerRGB);
   warn("	colormapEntries___________ %d\n",
						visual->colormapEntries);
   warn("	redMask.greenMask.blueMask %08x.%08x.%08x\n",
						visual->redMask,
						visual->greenMask,
						visual->blueMask);
}

void
dprx_connsetup
   AL((cp))
   DB char *cp
   DE
{
   register xPixmapFormat *fp;
   register xWindowRoot *rp;
   register xDepth *dp;
   register xVisualType *vp;

   dprx_ConnSetupPrefix(cblock_prefix(cp));
   dprx_ConnSetup(cblock_setup(cp));
   dprx_vender(cblock_vstring());
   while (fp = cblock_format(0))
      dprx_PixmapFormat(fp);
   while (rp = cblock_root(0)) {
      dprx_WindowRoot(rp);
      while (dp = cblock_depth(0)) {
         dprx_Depth(dp);
         while (vp = cblock_visual(0))
            dprx_VisualType(vp);
      }
   }
}

void
dprx_textitem8
   AL((p))
   DB u8_t *p
   DE
{
   int m = (int)*p;
   char buf[257];

   if (m == 255)
      warn("<font-shift>");
   else {
      p += 2;
      bcopy(p, buf, m);
      buf[m] = '\0';
      warn(buf);
   }
}

void
dprx_textitem16
   AL((p))
   DB u8_t *p
   DE
{
   register int i;
   int m = (int)*p;
   char buf[257];

   if (m == 255)
      warn("<font-shift>");
   else {
      p += 2;
      for (i=0; i<m; i++)
         buf[i] = (u8_t)((u16_t *)p)[i];
      buf[m] = '\0';
      warn(buf);
   }
}


void
dprx_request
   AL((rp, seqno))
   DB xReq *rp
   DD s32_t seqno
   DE
{
   register int i;

   warn("%s", dprx_reqType_str(rp->reqType));
   if (seqno >= 0)
      warn(" [%d]", seqno);
   warn("\n\tlength________ %d\n", rp->length);

   switch (rp->reqType) {
      case X_CreateWindow:
         {  register xCreateWindowReq *p = (xCreateWindowReq *)rp;
            warn("\tdepth_________ %d\n", p->depth);
            warn("\twid___________ 0x%08x\n", p->wid);
            warn("\tparent________ 0x%08x\n", p->parent);
            warn("\tx_____________ %d\n", p->x);
            warn("\ty_____________ %d\n", p->y);
            warn("\twidth_________ %d\n", p->width);
            warn("\theight________ %d\n", p->height);
            warn("\tborderWidth___ %d\n", p->borderWidth);
            warn("\tclass_________ %d\n", p->class);
            warn("\tvisual________ 0x%08x\n", p->visual);
            warn("\tmask__________ 0x%08x\n", p->mask);
            dprx_window_vals(p->mask, (u32_t *)(p+1));
         }
         break;
      case X_ChangeWindowAttributes:
         {  register xChangeWindowAttributesReq *p =
					(xChangeWindowAttributesReq *)rp;
            warn("\twindow________ 0x%08x\n", p->window);
            warn("\tvalueMask_____ 0x%08x\n", p->valueMask);
            dprx_window_vals(p->valueMask, (u32_t *)(p+1));
         }
         break;
      case X_ChangeSaveSet:
         {  register xChangeSaveSetReq *p = (xChangeSaveSetReq *)rp;
            warn("\tmode__________ %d\n", p->mode);
            warn("\twindow________ 0x%08x\n", p->window);
         }
         break;
      case X_ReparentWindow:
         {  register xReparentWindowReq *p = (xReparentWindowReq *)rp;
            warn("\twindow________ 0x%08x\n", p->window);
            warn("\tparent________ 0x%08x\n", p->parent);
            warn("\tx_____________ %d\n", p->x);
            warn("\ty_____________ %d\n", p->y);
         }
         break;
      case X_ConfigureWindow:
         {  register xConfigureWindowReq *p = (xConfigureWindowReq *)rp;
            warn("\twindow________ 0x%08x\n", p->window);
            warn("\tmask__________ 0x%04x\n", p->mask);
            dprx_config_vals(p->mask, (u32_t *)(p+1));
         }
         break;
      case X_CirculateWindow:
         {  register xCirculateWindowReq *p = (xCirculateWindowReq *)rp;
            warn("\tdirection_____ %d\n", p->direction);
            warn("\twindow________ 0x%08x\n", p->window);
         }
         break;
      case X_InternAtom:
         {  register xInternAtomReq *p = (xInternAtomReq *)rp;
            char buf[1024];

            warn("\tonlyIfExists__ %d\n", p->onlyIfExists);
            bcopy((char *)(p+1), buf, p->nbytes);
            buf[p->nbytes] = '\0';
            warn("\t<string>______\"%s\"\n", buf);
         }
         break;
      case X_ChangeProperty:
         {  register xChangePropertyReq *p = (xChangePropertyReq *)rp;
            int len;
            char *cp, buf[1024];
            warn("\tmode__________ %d\n", p->mode);
            warn("\twindow________ 0x%08x\n", p->window);
            if (atom_name(p->property, &cp, 0))
               cp = empty_string;
            warn("\tproperty______ %d [%s]\n", p->property, cp);
            if (atom_name(p->type, &cp, 0))
               cp = empty_string;
            warn("\ttype__________ %d [%s]\n", p->type, cp);
            warn("\tformat________ %d\n", p->format);
            warn("\tnUnits________ %d\n", p->nUnits);
            if (p->type == XA_STRING && p->format == 8) {
               strncpy(buf, (char *)(p+1), len=MIN(p->nUnits,1023));
               buf[len] = '\0';
               warn("\t\t\"%s\"\n", buf);
            }
            else if (p->format == 16 && p->nUnits)
               warn("\t\t0x%04x...\n", *(u16_t *)(p+1));
            else if (p->format == 32 && p->nUnits)
               warn("\t\t0x%08x...\n", *(u32_t *)(p+1));
         }
         break;
      case X_DeleteProperty:
         {  register xDeletePropertyReq *p = (xDeletePropertyReq *)rp;
            char *cp;
            warn("\twindow________ 0x%08x\n", p->window);
            if (atom_name(p->property, &cp, 0))
               cp = empty_string;
            warn("\tproperty______ %d [%s]\n", p->property, cp);
         }
         break;
      case X_GetProperty:
         {  register xGetPropertyReq *p = (xGetPropertyReq *)rp;
            char *cp;
            warn("\tdelete________ %d\n", p->delete);
            warn("\twindow________ 0x%08x\n", p->window);
            if (atom_name(p->property, &cp, 0))
               cp = empty_string;
            warn("\tproperty______ %d [%s]\n", p->property, cp);
            if (atom_name(p->type, &cp, 0))
               cp = empty_string;
            warn("\ttype__________ %d [%s]\n", p->type, cp);
            warn("\tlongOffset____ %d\n", p->longOffset);
            warn("\tlongLength____ %d\n", p->longLength);
         }
         break;
      case X_SetSelectionOwner:
         {  register xSetSelectionOwnerReq *p = (xSetSelectionOwnerReq *)rp;
            char *cp;
            warn("\twindow________ 0x%08x\n", p->window);
            if (atom_name(p->selection, &cp, 0))
               cp = empty_string;
            warn("\tselection_____ %d [%s]\n", p->selection, cp);
            warn("\ttime__________ %u\n", p->time);
         }
         break;
      case X_ConvertSelection:
         {  register xConvertSelectionReq *p = (xConvertSelectionReq *)rp;
            char *cp;
            warn("\trequestor_____ 0x%08x\n", p->requestor);
            if (atom_name(p->selection, &cp, 0))
               cp = empty_string;
            warn("\tselection_____ %d [%s]\n", p->selection, cp);
            if (atom_name(p->target, &cp, 0))
               cp = empty_string;
            warn("\ttarget________ %d [%s]\n", p->target, cp);
            if (atom_name(p->property, &cp, 0))
               cp = empty_string;
            warn("\tproperty______ %d [%s]\n", p->property, cp);
            warn("\ttime__________ %u\n", p->time);
         }
         break;
      case X_SendEvent:
         {  register xSendEventReq *p = (xSendEventReq *)rp;
            warn("\tpropagate_____ %d\n", p->propagate);
            warn("\tdestination___ 0x%08x\n", p->destination);
            warn("\teventMask_____ 0x%08x %s\n", p->eventMask,
					dprx_event_mask_str(p->eventMask));
            dprx_event(&p->event);
         }
         break;
      case X_GrabPointer:
         {  register xGrabPointerReq *p = (xGrabPointerReq *)rp;
            warn("\townerEvents___ %d\n", p->ownerEvents);
            warn("\tgrabWindow____ 0x%08x\n", p->grabWindow);
            warn("\teventMask_____ 0x%08x %s\n", p->eventMask,
					dprx_event_mask_str(p->eventMask));
            warn("\tpointerMode___ %d\n", p->pointerMode);
            warn("\tkeyboardMode__ %d\n", p->keyboardMode);
            warn("\tconfineTo_____ 0x%08x\n", p->confineTo);
            warn("\tcursor________ 0x%08x\n", p->cursor);
            warn("\ttime__________ %u\n", p->time);
         }
         break;
      case X_GrabButton:
         {  register xGrabButtonReq *p = (xGrabButtonReq *)rp;
            warn("\townerEvents___ %d\n", p->ownerEvents);
            warn("\tgrabWindow____ 0x%08x\n", p->grabWindow);
            warn("\teventMask_____ 0x%08x %s\n", p->eventMask,
					dprx_event_mask_str(p->eventMask));
            warn("\tpointerMode___ %d\n", p->pointerMode);
            warn("\tkeyboardMode__ %d\n", p->keyboardMode);
            warn("\tconfineTo_____ 0x%08x\n", p->confineTo);
            warn("\tcursor________ 0x%08x\n", p->cursor);
            warn("\tbutton________ %d\n", p->button);
            warn("\tmodifiers_____ 0x%04x\n", p->modifiers);
         }
         break;
      case X_UngrabButton:
         {  register xUngrabButtonReq *p = (xUngrabButtonReq *)rp;
            warn("\tbutton________ %d\n", p->button);
            warn("\tgrabWindow____ 0x%08x\n", p->grabWindow);
            warn("\tmodifiers_____ 0x%04x\n", p->modifiers);
         }
         break;
      case X_ChangeActivePointerGrab:
         {  register xChangeActivePointerGrabReq *p =
					(xChangeActivePointerGrabReq *)rp;
            warn("\tcursor________ 0x%08x\n", p->cursor);
            warn("\ttime__________ %u\n", p->time);
            warn("\teventMask_____ 0x%08x %s\n", p->eventMask,
					dprx_event_mask_str(p->eventMask));

         }
         break;
      case X_GrabKeyboard:
         {  register xGrabKeyboardReq *p = (xGrabKeyboardReq *)rp;
            warn("\townerEvents___ %d\n", p->ownerEvents);
            warn("\tgrabWindow____ 0x%08x\n", p->grabWindow);
            warn("\ttime__________ %u\n", p->time);
            warn("\tpointerMode___ %d\n", p->pointerMode);
            warn("\tkeyboardMode__ %d\n", p->keyboardMode);
         }
         break;
      case X_GrabKey:
         {  register xGrabKeyReq *p = (xGrabKeyReq *)rp;
            warn("\townerEvents___ %d\n", p->ownerEvents);
            warn("\tgrabWindow____ 0x%08x\n", p->grabWindow);
            warn("\tmodifiers_____ 0x%04x\n", p->modifiers);
            warn("\tkey___________ %d\n", p->key);
            warn("\tpointerMode___ %d\n", p->pointerMode);
            warn("\tkeyboardMode__ %d\n", p->keyboardMode);
         }
         break;
      case X_UngrabKey:
         {  register xUngrabKeyReq *p = (xUngrabKeyReq *)rp;
            warn("\tkey___________ %d\n", p->key);
            warn("\tgrabWindow____ 0x%08x\n", p->grabWindow);
            warn("\tmodifiers_____ 0x%04x\n", p->modifiers);
         }
         break;
      case X_AllowEvents:
         {  register xAllowEventsReq *p = (xAllowEventsReq *)rp;
            warn("\tmode__________ %d\n", p->mode);
            warn("\ttime__________ %u\n", p->time);
         }
         break;
      case X_GetMotionEvents:
         {  register xGetMotionEventsReq *p = (xGetMotionEventsReq *)rp;
            warn("\twindow________ 0x%08x\n", p->window);
            warn("\tstart_________ 0x%08x\n", p->start);
            warn("\tstop__________ 0x%08x\n", p->stop);
         }
         break;
      case X_TranslateCoords:
         {  register xTranslateCoordsReq *p = (xTranslateCoordsReq *)rp;
            warn("\tsrcWid________ 0x%08x\n", p->srcWid);
            warn("\tdstWid________ 0x%08x\n", p->dstWid);
            warn("\tsrcX__________ %d\n", p->srcX);
            warn("\tsrcY__________ %d\n", p->srcY);
         }
         break;
      case X_WarpPointer:
         {  register xWarpPointerReq *p = (xWarpPointerReq *)rp;
            warn("\tsrcWid________ 0x%08x\n", p->srcWid);
            warn("\tdstWid________ 0x%08x\n", p->dstWid);
            warn("\tsrcX__________ %d\n", p->srcX);
            warn("\tsrcY__________ %d\n", p->srcY);
            warn("\tsrcWidth______ %d\n", p->srcWidth);
            warn("\tsrcHeight_____ %d\n", p->srcHeight);
            warn("\tdstX__________ %d\n", p->dstX);
            warn("\tdstY__________ %d\n", p->dstY);
         }
         break;
      case X_SetInputFocus:
         {  register xSetInputFocusReq *p = (xSetInputFocusReq *)rp;
            warn("\trevertTo______ %d\n", p->revertTo);
            warn("\tfocus_________ 0x%08x\n", p->focus);
            warn("\ttime__________ %u\n", p->time);
         }
         break;
      case X_OpenFont:
         {  register xOpenFontReq *p = (xOpenFontReq *)rp;
            char buf[1024];

            warn("\tfid___________ 0x%08x\n", p->fid);
            bcopy((char *)(p+1), buf, p->nbytes);
            buf[p->nbytes] = '\0';
            warn("\t<string>______\"%s\"\n", buf);
         }
         break;
      case X_QueryTextExtents:
         {  register xQueryTextExtentsReq *p = (xQueryTextExtentsReq *)rp;
            warn("\toddLength_____ %d\n", p->oddLength);
            warn("\tfid___________ 0x%08x\n", p->fid);
         }
         break;
      case X_ListFonts:
      case X_ListFontsWithInfo:
         {  register xListFontsReq *p = (xListFontsReq *)rp;
            char buf[1024];

            warn("\tmaxNames______ %d\n", p->maxNames);
            bcopy((char *)(p+1), buf, p->nbytes);
            buf[p->nbytes] = '\0';
            warn("\t<string>______\"%s\"\n", buf);
         }
         break;
      case X_SetFontPath:
         {  register xSetFontPathReq *p = (xSetFontPathReq *)rp;
            warn("\tnFonts________ %d\n", p->nFonts);
         }
         break;
      case X_CreatePixmap:
         {  register xCreatePixmapReq *p = (xCreatePixmapReq *)rp;
            warn("\tdepth_________ %d\n", p->depth);
            warn("\tpid___________ 0x%08x\n", p->pid);
            warn("\tdrawable______ 0x%08x\n", p->drawable);
            warn("\twidth_________ %d\n", p->width);
            warn("\theight________ %d\n", p->height);
         }
         break;
      case X_CreateGC:
         {  register xCreateGCReq *p = (xCreateGCReq *)rp;
            warn("\tgc____________ 0x%08x\n", p->gc);
            warn("\tdrawable______ 0x%08x\n", p->drawable);
            warn("\tmask__________ 0x%08x\n", p->mask);
            dprx_gc_vals(p->mask, (u32_t *)(p+1));
         }
         break;
      case X_ChangeGC:
         {  register xChangeGCReq *p = (xChangeGCReq *)rp;
            warn("\tgc____________ 0x%08x\n", p->gc);
            warn("\tmask__________ 0x%08x\n", p->mask);
            dprx_gc_vals(p->mask, (u32_t *)(p+1));
         }
         break;
      case X_CopyGC:
         {  register xCopyGCReq *p = (xCopyGCReq *)rp;
            warn("\tsrcGC_________ 0x%08x\n", p->srcGC);
            warn("\tdstGC_________ 0x%08x\n", p->dstGC);
            warn("\tmask__________ 0x%08x\n", p->mask);
         }
         break;
      case X_SetDashes:
         {  register xSetDashesReq *p = (xSetDashesReq *)rp;
            warn("\tgc____________ 0x%08x\n", p->gc);
            warn("\tdashOffset____ %d\n", p->dashOffset);
            warn("\tnDashes_______ %d\n", p->nDashes);
         }
         break;
      case X_SetClipRectangles:
         {  register xSetClipRectanglesReq *p = (xSetClipRectanglesReq *)rp;
            warn("\tordering______ %d\n", p->ordering);
            warn("\tgc____________ 0x%08x\n", p->gc);
            warn("\txOrigin_______ %d\n", p->xOrigin);
            warn("\tyOrigin_______ %d\n", p->yOrigin);
         }
         break;
      case X_ClearArea:
         {  register xClearAreaReq *p = (xClearAreaReq *)rp;
            warn("\texposures_____ %d\n", p->exposures);
            warn("\twindow________ 0x%08x\n", p->window);
            warn("\tx_____________ %d\n", p->x);
            warn("\ty_____________ %d\n", p->y);
            warn("\twidth_________ %d\n", p->width);
            warn("\theight________ %d\n", p->height);
         }
         break;
      case X_CopyArea:
         {  register xCopyAreaReq *p = (xCopyAreaReq *)rp;
            warn("\tsrcDrawable___ 0x%08x\n", p->srcDrawable);
            warn("\tdstDrawable___ 0x%08x\n", p->dstDrawable);
            warn("\tgc____________ 0x%08x\n", p->gc);
            warn("\tsrcX__________ %d\n", p->srcX);
            warn("\tsrcY__________ %d\n", p->srcY);
            warn("\tdstX__________ %d\n", p->dstX);
            warn("\tdstY__________ %d\n", p->dstY);
            warn("\twidth_________ %d\n", p->width);
            warn("\theight________ %d\n", p->height);
         }
         break;
      case X_CopyPlane:
         {  register xCopyPlaneReq *p = (xCopyPlaneReq *)rp;
            warn("\tsrcDrawable___ 0x%08x\n", p->srcDrawable);
            warn("\tdstDrawable___ 0x%08x\n", p->dstDrawable);
            warn("\tgc____________ 0x%08x\n", p->gc);
            warn("\tsrcX__________ %d\n", p->srcX);
            warn("\tsrcY__________ %d\n", p->srcY);
            warn("\tdstX__________ %d\n", p->dstX);
            warn("\tdstY__________ %d\n", p->dstY);
            warn("\twidth_________ %d\n", p->width);
            warn("\theight________ %d\n", p->height);
            warn("\tbitPlane______ 0x%08x\n", p->bitPlane);
         }
         break;
      case X_PolyPoint:
      case X_PolyLine:
         {  register xPolyPointReq *p = (xPolyPointReq *)rp;
            warn("\tcoordMode_____ %d\n", p->coordMode);
            warn("\tdrawable______ 0x%08x\n", p->drawable);
            warn("\tgc____________ 0x%08x\n", p->gc);
         }
         break;
      case X_PolySegment:
      case X_PolyArc:
      case X_PolyRectangle:
      case X_PolyFillRectangle:
      case X_PolyFillArc:
         {  register xPolySegmentReq *p = (xPolySegmentReq *)rp;
            warn("\tdrawable______ 0x%08x\n", p->drawable);
            warn("\tgc____________ 0x%08x\n", p->gc);
         }
         break;
      case X_FillPoly:
         {  register xFillPolyReq *p = (xFillPolyReq *)rp;
            warn("\tdrawable______ 0x%08x\n", p->drawable);
            warn("\tgc____________ 0x%08x\n", p->gc);
            warn("\tshape_________ %d\n", p->shape);
            warn("\tcoordMode_____ %d\n", p->coordMode);
         }
         break;
      case X_PutImage:
         {  register xPutImageReq *p = (xPutImageReq *)rp;
            warn("\tformat________ %d\n", p->format);
            warn("\tdrawable______ 0x%08x\n", p->drawable);
            warn("\tgc____________ 0x%08x\n", p->gc);
            warn("\twidth_________ %d\n", p->width);
            warn("\theight________ %d\n", p->height);
            warn("\tdstX__________ %d\n", p->dstX);
            warn("\tdstY__________ %d\n", p->dstY);
            warn("\tleftPad_______ %d\n", p->leftPad);
            warn("\tdepth_________ %d\n", p->depth);
         }
         break;
      case X_GetImage:
         {  register xGetImageReq *p = (xGetImageReq *)rp;
            warn("\tformat________ %d\n", p->format);
            warn("\tdrawable______ 0x%08x\n", p->drawable);
            warn("\tx_____________ %d\n", p->x);
            warn("\ty_____________ %d\n", p->y);
            warn("\twidth_________ %d\n", p->width);
            warn("\theight________ %d\n", p->height);
            warn("\tplaneMask_____ %d\n", p->planeMask);
         }
         break;
      case X_PolyText8:
      case X_PolyText16:
         {  register xPolyTextReq *p = (xPolyTextReq *)rp;
            warn("\tdrawable______ 0x%08x\n", p->drawable);
            warn("\tgc____________ 0x%08x\n", p->gc);
            warn("\tx_____________ %d\n", p->x);
            warn("\ty_____________ %d\n", p->y);
            warn("\t\"");
            switch (rp->reqType) {
               case X_PolyText8:  dprx_textitem8((u8_t *)(p+1));	break;
               case X_PolyText16: dprx_textitem16((u8_t *)(p+1));	break;
            }
            warn("\"...\n");
         }
         break;
      case X_ImageText8:
      case X_ImageText16:
         {  register xImageTextReq *p = (xImageTextReq *)rp;
            register char *cp;
            register short *sp;

            warn("\tdrawable______ 0x%08x\n", p->drawable);
            warn("\tgc____________ 0x%08x\n", p->gc);
            warn("\tx_____________ %d\n", p->x);
            warn("\ty_____________ %d\n", p->y);
            warn("\tnChars________ %d\n", p->nChars);
            warn("\t\"");
            if (rp->reqType == X_ImageText8)
               for (cp = (char *)(p+1), i=0; i<(int)p->nChars; i++)
                  warn("%c", cp[i]);
            else
               for (sp = (short *)(p+1), i=0; i<(int)p->nChars; i++)
                  warn("%c", sp[i]);
            warn("\"\n");
         }
         break;
      case X_CreateColormap:
         {  register xCreateColormapReq *p = (xCreateColormapReq *)rp;
            warn("\talloc_________ %d\n", p->alloc);
            warn("\tmid___________ 0x%08x\n", p->mid);
            warn("\twindow________ 0x%08x\n", p->window);
            warn("\tvisual________ 0x%08x\n", p->visual);
         }
         break;
      case X_CopyColormapAndFree:
         {  register xCopyColormapAndFreeReq *p = (xCopyColormapAndFreeReq *)rp;
            warn("\tmid___________ 0x%08x\n", p->mid);
            warn("\tsrcCmap_______ 0x%08x\n", p->srcCmap);
         }
         break;
      case X_AllocColor:
         {  register xAllocColorReq *p = (xAllocColorReq *)rp;
            warn("\tcmap__________ 0x%08x\n", p->cmap);
            warn("\tred.green.blue %04x.%04x.%04x\n", p->red,p->green,p->blue);
         }
         break;
      case X_AllocNamedColor:
         {  register xAllocNamedColorReq *p = (xAllocNamedColorReq *)rp;
         char buf[1024];

            warn("\tcmap__________ 0x%08x\n", p->cmap);
            bcopy((char *)(p+1), buf, p->nbytes);
            buf[p->nbytes] = '\0';
            warn("\t<string>______\"%s\"\n", buf);
         }
         break;
      case X_AllocColorCells:
         {  register xAllocColorCellsReq *p = (xAllocColorCellsReq *)rp;
            warn("\tcontiguous____ %d\n", p->contiguous);
            warn("\tcmap__________ 0x%08x\n", p->cmap);
            warn("\tcolors________ %d\n", p->colors);
            warn("\tplanes________ %d\n", p->planes);
         }
         break;
      case X_AllocColorPlanes:
         {  register xAllocColorPlanesReq *p = (xAllocColorPlanesReq *)rp;
            warn("\tcontiguous____ %d\n", p->contiguous);
            warn("\tcmap__________ 0x%08x\n", p->cmap);
            warn("\tcolors________ %d\n", p->colors);
            warn("\tred.green.blue %04x.%04x.%04x\n", p->red,p->green,p->blue);
         }
         break;
      case X_FreeColors:
         {  register xFreeColorsReq *p = (xFreeColorsReq *)rp;
            warn("\tcmap___________ 0x%08x\n", p->cmap);
            warn("\tplaneMask______ 0x%08x\n", p->planeMask);
            dprx_pixels(p->length - sz_xFreeColorsReq/4, (pixel_t *)(p+1));
         }
         break;
      case X_StoreColors:
         {  register xStoreColorsReq *p = (xStoreColorsReq *)rp;
            register xColorItem *c;
            register int n;
            warn("\tcmap__________ 0x%08x\n", p->cmap);
            n = (int)(p->length - sz_xStoreColorsReq/4)/(sz_xColorItem/4);
            c = (xColorItem *)(p+1);
            for (i=0; i<n; i++) {
               warn("\t\tpixel_________ %s\n", dprx_pixel_str(c[i].pixel));
               warn("\t\tred.green.blue %04x.%04x.%04x\n",
					c[i].red, c[i].green, c[i].blue);
               warn("\t\tflags_________ 0x%02x\n", c[i].flags);
            }
         }
         break;
      case X_StoreNamedColor:
         {  register xStoreNamedColorReq *p = (xStoreNamedColorReq *)rp;
            char buf[1024];

            warn("\tflags_________ 0x%02x\n", p->flags);
            warn("\tcmap__________ 0x%08x\n", p->cmap);
            warn("\tpixel_________ %s\n", dprx_pixel_str(p->pixel));
            bcopy((char *)(p+1), buf, p->nbytes);
            buf[p->nbytes] = '\0';
            warn("\t<string>______\"%s\"\n", buf);
         }
         break;
      case X_QueryColors:
         {  register xQueryColorsReq *p = (xQueryColorsReq *)rp;
            warn("\tcmap__________ 0x%08x\n", p->cmap);
            dprx_pixels(p->length-(sz_xQueryColorsReq/4),
							(pixel_t *)(p+1));
         }
         break;
      case X_LookupColor:
         {  register xLookupColorReq *p = (xLookupColorReq *)rp;
            char buf[1024];

            warn("\tcmap__________ 0x%08x\n", p->cmap);
            bcopy((char *)(p+1), buf, p->nbytes);
            buf[p->nbytes] = '\0';
            warn("\t<string>______\"%s\"\n", buf);
         }
         break;
      case X_CreateCursor:
         {  register xCreateCursorReq *p = (xCreateCursorReq *)rp;
            warn("\tcid___________ 0x%08x\n", p->cid);
            warn("\tsource________ 0x%08x\n", p->source);
            warn("\tmask__________ 0x%08x\n", p->mask);
            warn("\tforeRed_______ %d\n", p->foreRed);
            warn("\tforeGreen_____ %d\n", p->foreGreen);
            warn("\tforeBlue______ %d\n", p->foreBlue);
            warn("\tbackRed_______ %d\n", p->backRed);
            warn("\tbackGreen_____ %d\n", p->backGreen);
            warn("\tbackBlue______ %d\n", p->backBlue);
            warn("\tx_____________ %d\n", p->x);
            warn("\ty_____________ %d\n", p->y);
         }
         break;
      case X_CreateGlyphCursor:
         {  register xCreateGlyphCursorReq *p = (xCreateGlyphCursorReq *)rp;
            warn("\tcid___________ 0x%08x\n", p->cid);
            warn("\tsource________ 0x%08x\n", p->source);
            warn("\tmask__________ 0x%08x\n", p->mask);
            warn("\tsourceChar____ %d\n", p->sourceChar);
            warn("\tmaskChar______ %d\n", p->maskChar);
            warn("\tforeRed_______ %d\n", p->foreRed);
            warn("\tforeGreen_____ %d\n", p->foreGreen);
            warn("\tforeBlue______ %d\n", p->foreBlue);
            warn("\tbackRed_______ %d\n", p->backRed);
            warn("\tbackGreen_____ %d\n", p->backGreen);
            warn("\tbackBlue______ %d\n", p->backBlue);
         }
         break;
      case X_RecolorCursor:
         {  register xRecolorCursorReq *p = (xRecolorCursorReq *)rp;
            warn("\tcursor________ 0x%08x\n", p->cursor);
            warn("\tforeRed_______ %d\n", p->foreRed);
            warn("\tforeGreen_____ %d\n", p->foreGreen);
            warn("\tforeBlue______ %d\n", p->foreBlue);
            warn("\tbackRed_______ %d\n", p->backRed);
            warn("\tbackGreen_____ %d\n", p->backGreen);
            warn("\tbackBlue______ %d\n", p->backBlue);
         }
         break;
      case X_QueryBestSize:
         {  register xQueryBestSizeReq *p = (xQueryBestSizeReq *)rp;
            warn("\tclass_________ %d\n", p->class);
            warn("\tdrawable______ 0x%08x\n", p->drawable);
            warn("\twidth_________ %d\n", p->width);
            warn("\theight________ %d\n", p->height);
         }
         break;
      case X_QueryExtension:
         {  register xQueryExtensionReq *p = (xQueryExtensionReq *)rp;
            char buf[1024];

            bcopy((char *)(p+1), buf, p->nbytes);
            buf[p->nbytes] = '\0';
            warn("\t<string>______\"%s\"\n", buf);
         }
         break;
      case X_SetModifierMapping:
         {  register xSetModifierMappingReq *p =
						(xSetModifierMappingReq *)rp;
            warn("\tnumKeyPerModifier_ %d\n", p->numKeyPerModifier);
         }
         break;
      case X_SetPointerMapping:
         {  register xSetPointerMappingReq *p =
						(xSetPointerMappingReq *)rp;
            warn("\tnElts_________ %d\n", p->nElts);
         }
         break;
      case X_GetKeyboardMapping:
         {  register xGetKeyboardMappingReq *p =
						(xGetKeyboardMappingReq *)rp;
            warn("\tfirstKeyCode__ %d\n", p->firstKeyCode);
            warn("\tcount_________ %d\n", p->count);
         }
         break;
      case X_ChangeKeyboardMapping:
         {  register xChangeKeyboardMappingReq *p =
					(xChangeKeyboardMappingReq *)rp;
            warn("\tkeyCodes______ %d\n", p->keyCodes);
            warn("\tfirstKeyCode__ %d\n", p->firstKeyCode);
            warn("\tkeySymsPerKeyCode_ %d\n", p->keySymsPerKeyCode);
         }
         break;
      case X_ChangeKeyboardControl:
         {  register xChangeKeyboardControlReq *p =
					(xChangeKeyboardControlReq *)rp;
            warn("\tmask__________ 0x%08x\n", p->mask);
         }
         break;
      case X_Bell:
         {  register xBellReq *p = (xBellReq *)rp;
            warn("\tpercent_______ %d\n", p->percent);
         }
         break;
      case X_ChangePointerControl:
         {  register xChangePointerControlReq *p =
					(xChangePointerControlReq *)rp;
            warn("\taccelNum______ %d\n", p->accelNum);
            warn("\taccelDenum____ %d\n", p->accelDenum);
            warn("\tthreshold_____ %d\n", p->threshold);
            warn("\tdoAccel_______ %d\n", p->doAccel);
            warn("\tdoThresh______ %d\n", p->doThresh);
         }
         break;
      case X_SetScreenSaver:
         {  register xSetScreenSaverReq *p = (xSetScreenSaverReq *)rp;
            warn("\ttimeout_______ %d\n", p->timeout);
            warn("\tinterval______ %d\n", p->interval);
            warn("\tpreferBlank___ %d\n", p->preferBlank);
            warn("\tallowExpose___ %d\n", p->allowExpose);
         }
         break;
      case X_ChangeHosts:
         {  register xChangeHostsReq *p = (xChangeHostsReq *)rp;
            warn("\tmode__________ %d\n", p->mode);
            warn("\thostFamily____ %d\n", p->hostFamily);
         }
         break;
      case X_ListHosts:
         {  /* register xListHostsReq *p = (xListHostsReq *)rp; */
         }
         break;
      case X_SetAccessControl:
      case X_SetCloseDownMode:
      case X_ForceScreenSaver:
         {  register xChangeModeReq *p = (xChangeModeReq *)rp;
            warn("\tmode__________ %d\n", p->mode);
         }
         break;
      case X_RotateProperties:
         {  register xRotatePropertiesReq *p = (xRotatePropertiesReq *)rp;
            warn("\twindow________ 0x%08x\n", p->window);
            warn("\tnAtoms________ %d\n", p->nAtoms);
            warn("\tnPositions____ %d\n", p->nPositions);
         }
         break;
      case X_GetWindowAttributes:
      case X_DestroyWindow:
      case X_DestroySubwindows:
      case X_MapWindow:
      case X_MapSubwindows:
      case X_UnmapWindow:
      case X_UnmapSubwindows:
      case X_GetGeometry:
      case X_QueryTree:
      case X_GetAtomName:
      case X_ListProperties:
      case X_GetSelectionOwner:
      case X_UngrabPointer:
      case X_UngrabKeyboard:
      case X_QueryPointer:
      case X_CloseFont:
      case X_QueryFont:
      case X_FreePixmap:
      case X_FreeGC:
      case X_FreeColormap:
      case X_InstallColormap:
      case X_UninstallColormap:
      case X_ListInstalledColormaps:
      case X_FreeCursor:
      case X_KillClient:
         {  register xResourceReq *p = (xResourceReq *)rp;
            warn("\tid____________ 0x%08x\n", p->id);
         }
         break;
      case X_NoOperation:
         {  register xReq *p = (xReq *)rp;
            warn("\tlength_________ %d\n", p->length);
         }
         break;
   }
}

void
dprx_window_vals
   AL((mask, vals))
   DB mask_t mask
   DD u32_t *vals
   DE
{
   register mask_t m, tm;

   for (m=1,tm=mask; tm; m<<=1, tm>>=1)
      if (tm & 1) {
         switch (m) {
            case CWBackPixmap:
               warn("\t\tCWBackPixmap_______ 0x%08x\n", *vals); break;
            case CWBackPixel:
               warn("\t\tCWBackPixel________ %s\n",dprx_pixel_str(*vals));break;
            case CWBorderPixmap:
               warn("\t\tCWBorderPixmap_____ 0x%08x\n", *vals); break;
            case CWBorderPixel:
               warn("\t\tCWBorderPixel______ %s\n",dprx_pixel_str(*vals));break;
            case CWBitGravity:
               warn("\t\tCWBitGravity_______ %d\n", *vals); break;
            case CWWinGravity:
               warn("\t\tCWWinGravity_______ %d\n", *vals); break;
            case CWBackingStore:
               warn("\t\tCWBackingStore_____ %d\n", *vals); break;
            case CWBackingPlanes:
               warn("\t\tCWBackingPlanes____ 0x%08x\n", *vals); break;
            case CWBackingPixel:
               warn("\t\tCWBackingPixel_____ %s\n",dprx_pixel_str(*vals));break;
            case CWOverrideRedirect:
               warn("\t\tCWOverrideRedirect_ %d\n", *vals); break;
            case CWSaveUnder:
               warn("\t\tCWSaveUnder________ %d\n", *vals); break;
            case CWEventMask:
               warn("\t\tCWEventMask________ 0x%08x %s\n", *vals,
				dprx_event_mask_str(*vals)); break;
            case CWDontPropagate:
               warn("\t\tCWDontPropagate____ 0x%08x\n", *vals); break;
            case CWColormap:
               warn("\t\tCWColormap_________ 0x%08x\n", *vals); break;
            case CWCursor:
               warn("\t\tCWCursor___________ 0x%08x\n", *vals); break;
         }
         vals++;
      }
}

void
dprx_config_vals
   AL((mask, vals))
   DB mask_t mask
   DD u32_t *vals
   DE
{
   register mask_t m, tm;

   for (m=1,tm=mask; tm; m<<=1, tm>>=1)
      if (tm & 1) {
         switch (m) {
            case CWX:
               warn("\t\tCWX________________ %d\n", *vals); break;
            case CWY:
               warn("\t\tCWY________________ %d\n", *vals); break;
            case CWWidth:
               warn("\t\tCWWidth____________ %d\n", *vals); break;
            case CWHeight:
               warn("\t\tCWHeight___________ %d\n", *vals); break;
            case CWBorderWidth:
               warn("\t\tCWBorderWidth______ %d\n", *vals); break;
            case CWSibling:
               warn("\t\tCWSibling__________ 0x%08x\n", *vals); break;
            case CWStackMode:
               warn("\t\tCWStackMode________ %s\n",
				dprx_stackmode_str((u8_t)*vals));
               break;
         }
         vals++;
      }
}

void
dprx_gc_vals
   AL((mask, vals))
   DB mask_t mask
   DD u32_t *vals
   DE
{
   register mask_t m, tm;

   for (m=1,tm=mask; tm; m<<=1, tm>>=1)
      if (tm & 1) {
         switch (m) {
            case GCFunction:
               warn("\t\tGCFunction__________ %d\n", *vals); break;
            case GCPlaneMask:
               warn("\t\tGCPlaneMask_________ 0x%08x\n", *vals); break;
            case GCForeground:
               warn("\t\tGCForeground________ %s\n", dprx_pixel_str(*vals));
               break;
            case GCBackground:
               warn("\t\tGCBackground________ %s\n", dprx_pixel_str(*vals));
               break;
            case GCLineWidth:
               warn("\t\tGCLineWidth_________ %d\n", *vals); break;
            case GCLineStyle:
               warn("\t\tGCLineStyle_________ %d\n", *vals); break;
            case GCCapStyle:
               warn("\t\tGCCapStyle__________ %d\n", *vals); break;
            case GCJoinStyle:
               warn("\t\tGCJoinStyle_________ %d\n", *vals); break;
            case GCFillStyle:
               warn("\t\tGCFillStyle_________ %d\n", *vals); break;
            case GCFillRule:
               warn("\t\tGCFillRule__________ %d\n", *vals); break;
            case GCTile:
               warn("\t\tGCTile______________ 0x%08x\n", *vals); break;
            case GCStipple:
               warn("\t\tGCStipple___________ 0x%08x\n", *vals); break;
            case GCTileStipXOrigin:
               warn("\t\tGCTileStipXOrigin___ %d\n", *vals); break;
            case GCTileStipYOrigin:
               warn("\t\tGCTileStipYOrigin___ %d\n", *vals); break;
            case GCFont:
               warn("\t\tGCFont______________ 0x%08x\n", *vals); break;
            case GCSubwindowMode:
               warn("\t\tGCSubwindowMode_____ %d\n", *vals); break;
            case GCGraphicsExposures:
               warn("\t\tGCGraphicsExposures_ %d\n", *vals); break;
            case GCClipXOrigin:
               warn("\t\tGCClipXOrigin_______ %d\n", *vals); break;
            case GCClipYOrigin:
               warn("\t\tGCClipYOrigin_______ %d\n", *vals); break;
            case GCClipMask:
               warn("\t\tGCClipMask__________ 0x%08x\n", *vals); break;
            case GCDashOffset:
               warn("\t\tGCDashOffset________ %d\n", *vals); break;
            case GCDashList:
               warn("\t\tGCDashList__________ %d\n", *vals); break;
            case GCArcMode:
               warn("\t\tGCArcMode___________ %d\n", *vals); break;
         }
         vals++;
      }
}

void
dprx_pixels
   AL((n, pp))
   DB int n
   DD pixel_t *pp
   DE
{
   register int i;

   for (i=0; i<n; i++)
      warn("\t\tpixel%03d___ %s\n", i, dprx_pixel_str(*pp++));
}

void
dprx_masks
   AL((n, mp))
   DB int n
   DD mask_t *mp
   DE
{
   register int i;

   for (i=0; i<n; i++)
      warn("\t\tmask%03d____ 0x%08x\n", i, *mp++);
}

void
dprx_rgbs
   AL((n, sp))
   DB int n
   DD u16_t *sp
   DE
{
   register int i;

   for (i=0; i<n; i++) {
      warn("\t\tr.g.b%03d___ %04x.%04x.%04x\n", i, sp[0], sp[1], sp[2]);
      sp += 4;
   }
}

void
dprx_charinfos
   AL((n, p))
   DB int n
   DD xCharInfo *p
   DE
{
   register int i;

   for (i=0; i<n; i++) {
      warn("\t\tCharInfo%03d\n", i);
      warn("\t\t\tleftSideBearing___ %d\n", p->leftSideBearing);
      warn("\t\t\trightSideBearing__ %d\n", p->rightSideBearing);
      warn("\t\t\tcharacterWidth____ %d\n", p->characterWidth);
      warn("\t\t\tascent____________ %d\n", p->ascent);
      warn("\t\t\tdescent___________ %d\n", p->descent);
      warn("\t\t\tattributes________ 0x%04x\n", p->attributes);
      p++;
   }
}

void
dprx_fontprops
   AL((n, fp))
   DB int n
   DD xFontProp *fp
   DE
{
   register int i;

   for (i=0; i<n; i++) {
      warn("\t\tFontProp%03d________ %d.0x%08x\n", i, fp->name, fp->value);
      fp++;
   }
}

void
dprx_chunk_beg
   AL((chp))
   DB chunk_t *chp
   DE
{
   warn("[CHUNK-> [0x%x] off[%d] sz[%d] refs[%d] bp[0x%x]\n",
				chp, chp->off, chp->sz, chp->refs, chp->bp);
}

void
dprx_chunk_end
   VOID
{
   warn("<-CHUNK]\n");
}

void
dprx_request_chunk
   AL((chp, smp))
   DB chunk_t *chp
   DD sm_t *smp		/* use this to get dest seqno */
   DE
{
   register int left, len;
   register u8_t clinum;
   register u16_t cseq, sseq;
   register char *cp;
   register u32_t type;

   dprx_chunk_beg(chp);

   if (chp->lpp) {
      len = MIN(chp->lpp->nmore, buf_chunksize(chp));
      warn("<<%d bytes of data>>\n", len);
   }
   else {
      warn("clinum[%d] cseq[%d-%d]\n", ptc_clinum(chp), ptc_seqno(chp),
					ptc_seqno(chp) + ptc_count(chp) - 1);
      sseq = sm_mapping(smp, ptc_clinum(chp), ptc_seqno(chp));
      cp = buf_data(chp);
      left = buf_chunksize(chp);
      for (; left>0; left-=len, cp+=len) {
         if ((len = ((xReq *)cp)->length * 4) && left >= len)
            dprx_request((xReq *)cp, sseq);
         /*
         **  hack for partial PutImages
         */
         else if (((xReq *)cp)->reqType==X_PutImage && left>=sz_xPutImageReq)
            dprx_request((xReq *)cp, sseq);
         else {
            warn("[%d extra bytes (want %d)]\n", left, len);
            break;
         }
         sseq++;
      }
   }
   dprx_chunk_end();
}

void
dprx_reply
   AL((rp, major, minor))
   DB xGenericReply *rp
   DD u8_t major
   DD u16_t minor
   DE
{
   switch (rp->type) {
      case X_Error:
         dprx_error((xError *)rp);
         return;
      case X_Reply:
         break;
      default:
         dprx_event((xEvent *)rp);
         return;
   }
   warn("%sReply\n", dprx_reqType_str(major));

   warn("\tsequenceNumber _____ %d\n", rp->sequenceNumber);
   warn("\tlength _____________ %d\n", rp->length);
   switch (major) {
      case X_GetWindowAttributes:
         {  register xGetWindowAttributesReply *p =
					(xGetWindowAttributesReply *)rp;
            warn("\tbackingStore _______ %d\n", p->backingStore);
            warn("\tvisualID ___________ 0x%08x\n", p->visualID);
            warn("\tclass ______________ %d\n", p->class);
            warn("\tbitGravity _________ %d\n", p->bitGravity);
            warn("\twinGravity _________ %d\n", p->winGravity);
            warn("\tbackingBitPlanes ___ %d\n", p->backingBitPlanes);
            warn("\tbackingPixel _______ %s\n", dprx_pixel_str(p->backingPixel));
            warn("\tsaveUnder __________ %d\n", p->saveUnder);
            warn("\tmapInstalled _______ %d\n", p->mapInstalled);
            warn("\tmapState ___________ %d\n", p->mapState);
            warn("\toverride ___________ %d\n", p->override);
            warn("\tcolormap ___________ 0x%08x\n", p->colormap);
            warn("\tallEventMasks ______ 0x%08x\n", p->allEventMasks);
            warn("\tyourEventMask ______ 0x%08x\n", p->yourEventMask);
            warn("\tdoNotPropagateMask _ 0x%04x\n", p->doNotPropagateMask);
         }
         break;
      case X_GetGeometry:
         {  register xGetGeometryReply *p = (xGetGeometryReply *)rp;
            warn("\tdepth ______________ %d\n", p->depth);
            warn("\troot _______________ 0x%08x\n", p->root);
            warn("\tx __________________ %d\n", p->x);
            warn("\ty __________________ %d\n", p->y);
            warn("\twidth ______________ %d\n", p->width);
            warn("\theight _____________ %d\n", p->height);
            warn("\tborderWidth ________ %d\n", p->borderWidth);
         }
         break;
      case X_QueryTree:
         {  register xQueryTreeReply *p = (xQueryTreeReply *)rp;
            warn("\troot _______________ 0x%08x\n", p->root);
            warn("\tparent _____________ 0x%08x\n", p->parent);
            warn("\tnChildren __________ %d\n", p->nChildren);
         }
         break;
      case X_InternAtom:
         {  register xInternAtomReply *p = (xInternAtomReply *)rp;
            warn("\tatom _______________ %d\n", p->atom);
         }
         break;
      case X_GetAtomName:
         {  register xGetAtomNameReply *p = (xGetAtomNameReply *)rp;
            warn("\tnameLength _________ %d\n", p->nameLength);
         }
         break;
      case X_GetProperty:
         {  register xGetPropertyReply *p = (xGetPropertyReply *)rp;
            warn("\tformat _____________ %d\n", p->format);
            warn("\tpropertyType _______ %d\n", p->propertyType);
            warn("\tbytesAfter _________ %d\n", p->bytesAfter);
            warn("\tnItems _____________ %d\n", p->nItems);
         }
         break;
      case X_ListProperties:
         {  register xListPropertiesReply *p = (xListPropertiesReply *)rp;
            warn("\tnProperties ________ %d\n", p->nProperties);
         }
         break;
      case X_GetSelectionOwner:
         {  register xGetSelectionOwnerReply *p =
					(xGetSelectionOwnerReply *)rp;
            warn("\towner ______________ 0x%08x\n", p->owner);
         }
         break;
      case X_GrabPointer:
         {  register xGrabPointerReply *p = (xGrabPointerReply *)rp;
            warn("\tstatus _____________ %d\n", p->status);
         }
         break;
      case X_GrabKeyboard:
         {  register xGrabKeyboardReply *p = (xGrabKeyboardReply *)rp;
            warn("\tstatus _____________ %d\n", p->status);
         }
         break;
      case X_QueryPointer:
         {  register xQueryPointerReply *p = (xQueryPointerReply *)rp;
            warn("\tsameScreen _________ %d\n", p->sameScreen);
            warn("\troot _______________ 0x%08x\n", p->root);
            warn("\tchild ______________ 0x%08x\n", p->child);
            warn("\trootX ______________ %d\n", p->rootX);
            warn("\trootY ______________ %d\n", p->rootY);
            warn("\twinX _______________ %d\n", p->winX);
            warn("\twinY _______________ %d\n", p->winY);
            warn("\tmask _______________ 0x%04x\n", p->mask);
         }
         break;
      case X_GetMotionEvents:
         {  register xGetMotionEventsReply *p = (xGetMotionEventsReply *)rp;
            warn("\tnEvents ____________ %d\n", p->nEvents);
         }
         break;
      case X_TranslateCoords:
         {  register xTranslateCoordsReply *p = (xTranslateCoordsReply *)rp;
            warn("\tsameScreen _________ %d\n", p->sameScreen);
            warn("\tchild ______________ 0x%08x\n", p->child);
            warn("\tdstX _______________ %d\n", p->dstX);
            warn("\tdstY _______________ %d\n", p->dstY);
         }
         break;
      case X_GetInputFocus:
         {  register xGetInputFocusReply *p = (xGetInputFocusReply *)rp;
            warn("\trevertTo ___________ %d\n", p->revertTo);
            warn("\tfocus ______________ 0x%08x\n", p->focus);
         }
         break;
      case X_QueryKeymap:
         {  /* register xQueryKeymapReply *p = (xQueryKeymapReply *)rp; */
         }
         break;
      case X_QueryFont:
         {  register xQueryFontReply *p = (xQueryFontReply *)rp;
            warn("\tminBounds\n");
            dprx_charinfos(1, &p->minBounds);
            warn("\tmaxBounds\n");
            dprx_charinfos(1, &p->maxBounds);
            warn("\tminCharOrByte2 _____ %d\n", p->minCharOrByte2);
            warn("\tmaxCharOrByte2 _____ %d\n", p->maxCharOrByte2);
            warn("\tdefaultChar ________ %d\n", p->defaultChar);
            warn("\tnFontProps _________ %d\n", p->nFontProps);
            warn("\tdrawDirection ______ %d\n", p->drawDirection);
            warn("\tminByte1 ___________ %d\n", p->minByte1);
            warn("\tmaxByte1 ___________ %d\n", p->maxByte1);
            warn("\tallCharsExist ______ %d\n", p->allCharsExist);
            warn("\tfontAscent _________ %d\n", p->fontAscent);
            warn("\tfontDescent ________ %d\n", p->fontDescent);
            warn("\tnCharInfos _________ %d\n", p->nCharInfos);
            dprx_fontprops(p->nFontProps, (xFontProp *)(p+1));
            D_CALL2(D_PROTOd, dprx_charinfos, p->nCharInfos,
			(xCharInfo *)((xFontProp *)(p+1)+p->nFontProps));
         }
         break;
      case X_QueryTextExtents:
         {  register xQueryTextExtentsReply *p =
					(xQueryTextExtentsReply *)rp;
            warn("\tdrawDirection ______ %d\n", p->drawDirection);
            warn("\tfontAscent _________ %d\n", p->fontAscent);
            warn("\tfontDescent ________ %d\n", p->fontDescent);
            warn("\toverallAscent ______ %d\n", p->overallAscent);
            warn("\toverallDescent _____ %d\n", p->overallDescent);
            warn("\toverallWidth _______ %d\n", p->overallWidth);
            warn("\toverallLeft ________ %d\n", p->overallLeft);
            warn("\toverallRight _______ %d\n", p->overallRight);
         }
         break;
      case X_ListFonts:
         {  register xListFontsReply *p = (xListFontsReply *)rp;
            warn("\tnFonts _____________ %d\n", p->nFonts);
         }
         break;
      case X_ListFontsWithInfo:
         {  register xListFontsWithInfoReply *p =
					(xListFontsWithInfoReply *)rp;
            register int i;
            register char *cp;

            warn("\tnameLength _________ %d\n", p->nameLength);
            if (p->nameLength) {
               warn("\tminBounds\n");
               dprx_charinfos(1, &p->minBounds);
               warn("\tmaxBounds\n");
               dprx_charinfos(1, &p->maxBounds);
               warn("\tminCharOrByte2 _____ %d\n", p->minCharOrByte2);
               warn("\tmaxCharOrByte2 _____ %d\n", p->maxCharOrByte2);
               warn("\tdefaultChar ________ %d\n", p->defaultChar);
               warn("\tnFontProps _________ %d\n", p->nFontProps);
               warn("\tdrawDirection ______ %d\n", p->drawDirection);
               warn("\tminByte1 ___________ %d\n", p->minByte1);
               warn("\tmaxByte1 ___________ %d\n", p->maxByte1);
               warn("\tallCharsExist ______ %d\n", p->allCharsExist);
               warn("\tfontAscent _________ %d\n", p->fontAscent);
               warn("\tfontDescent ________ %d\n", p->fontDescent);
               warn("\tnReplies ___________ %d\n", p->nReplies);
               cp = (char *)(p+1);
               dprx_fontprops(p->nFontProps, (xFontProp *)cp);
               cp += p->nFontProps * sz_xFontProp;
               warn("\t\"");
               for (i=0; i<(int)p->nameLength; i++)
                  warn("%c", (unsigned char)cp[i]);
               warn("\"\n");
            }
         }
         break;
      case X_GetFontPath:
         {  register xGetFontPathReply *p = (xGetFontPathReply *)rp;
            warn("\tnPaths _____________ %d\n", p->nPaths);
         }
         break;
      case X_GetImage:
         {  register xGetImageReply *p = (xGetImageReply *)rp;
            warn("\tdepth ______________ %d\n", p->depth);
            warn("\tvisual _____________ 0x%08x\n", p->visual);
         }
         break;
      case X_ListInstalledColormaps:
         {  register xListInstalledColormapsReply *p =
					(xListInstalledColormapsReply *)rp;
            warn("\tnColormaps _________ %d\n", p->nColormaps);
         }
         break;
      case X_AllocColor:
         {  register xAllocColorReply *p = (xAllocColorReply *)rp;
            warn("\tred.green.blue______ %04x.%04x.%04x\n",
						p->red, p->green, p->blue);
            warn("\tpixel ______________ %s\n", dprx_pixel_str(p->pixel));
         }
         break;
      case X_AllocNamedColor:
         {  register xAllocNamedColorReply *p = (xAllocNamedColorReply *)rp;
            warn("\tpixel ______________ %s\n", dprx_pixel_str(p->pixel));
            warn("\texactRed ___________ %d\n", p->exactRed);
            warn("\texactGreen _________ %d\n", p->exactGreen);
            warn("\texactBlue __________ %d\n", p->exactBlue);
            warn("\tscreenRed __________ %d\n", p->screenRed);
            warn("\tscreenGreen ________ %d\n", p->screenGreen);
            warn("\tscreenBlue _________ %d\n", p->screenBlue);
         }
         break;
      case X_AllocColorCells:
         {  register xAllocColorCellsReply *p = (xAllocColorCellsReply *)rp;
            warn("\tnPixels ____________ %d\n", p->nPixels);
            warn("\tnMasks _____________ %d\n", p->nMasks);
            dprx_pixels(p->nPixels, (pixel_t *)(p+1));
            dprx_masks(p->nMasks,
				(mask_t *)((pixel_t *)(p+1)+p->nPixels));
         }
         break;
      case X_AllocColorPlanes:
         {  register xAllocColorPlanesReply *p =
					(xAllocColorPlanesReply *)rp;
            warn("\tnPixels ____________ %d\n", p->nPixels);
            warn("\tredMask ____________ %08x\n", p->redMask);
            warn("\tgreenMask __________ %08x\n", p->greenMask);
            warn("\tblueMask ___________ %08x\n", p->blueMask);
         }
         break;
      case X_QueryColors:
         {  register xQueryColorsReply *p = (xQueryColorsReply *)rp;
            warn("\tnColors ____________ %d\n", p->nColors);
            dprx_rgbs(p->nColors, (u16_t *)(p+1));
         }
         break;
      case X_LookupColor:
         {  register xLookupColorReply *p = (xLookupColorReply *)rp;
            warn("\texactRed ___________ %d\n", p->exactRed);
            warn("\texactGreen _________ %d\n", p->exactGreen);
            warn("\texactBlue __________ %d\n", p->exactBlue);
            warn("\tscreenRed __________ %d\n", p->screenRed);
            warn("\tscreenGreen ________ %d\n", p->screenGreen);
            warn("\tscreenBlue _________ %d\n", p->screenBlue);
         }
         break;
      case X_QueryBestSize:
         {  register xQueryBestSizeReply *p = (xQueryBestSizeReply *)rp;
            warn("\twidth ______________ %d\n", p->width);
            warn("\theight _____________ %d\n", p->height);
         }
         break;
      case X_QueryExtension:
         {  register xQueryExtensionReply *p = (xQueryExtensionReply *)rp;
            warn("\tpresent ____________ %d\n", p->present);
            warn("\tmajor_opcode _______ %d\n", p->major_opcode);
            warn("\tfirst_event ________ %d\n", p->first_event);
            warn("\tfirst_error ________ %d\n", p->first_error);
         }
         break;
      case X_ListExtensions:
         {  register xListExtensionsReply *p = (xListExtensionsReply *)rp;
            warn("\tnExtensions ________ %d\n", p->nExtensions);
         }
         break;
      case X_GetKeyboardMapping:
         {  register xGetKeyboardMappingReply *p =
						(xGetKeyboardMappingReply *)rp;
            warn("\tkeySymsPerKeyCode __ %d\n", p->keySymsPerKeyCode);
         }
         break;
      case X_GetKeyboardControl:
         {  register xGetKeyboardControlReply *p =
						(xGetKeyboardControlReply *)rp;
            warn("\tglobalAutoRepeat ___ %d\n", p->globalAutoRepeat);
            warn("\tledMask ____________ %d\n", p->ledMask);
            warn("\tkeyClickPercent ____ %d\n", p->keyClickPercent);
            warn("\tbellPercent ________ %d\n", p->bellPercent);
            warn("\tbellPitch __________ %d\n", p->bellPitch);
            warn("\tbellDuration _______ %d\n", p->bellDuration);
         }
         break;
      case X_GetPointerControl:
         {  register xGetPointerControlReply *p =
					(xGetPointerControlReply *)rp;
            warn("\taccelNumerator _____ %d\n", p->accelNumerator);
            warn("\taccelDenominator ___ %d\n", p->accelDenominator);
            warn("\tthreshold __________ %d\n", p->threshold);
         }
         break;
      case X_GetScreenSaver:
         {  register xGetScreenSaverReply *p = (xGetScreenSaverReply *)rp;
            warn("\ttimeout ____________ %d\n", p->timeout);
            warn("\tinterval ___________ %d\n", p->interval);
            warn("\tpreferBlanking _____ %d\n", p->preferBlanking);
            warn("\tallowExposures _____ %d\n", p->allowExposures);
         }
         break;
      case X_ListHosts:
         {  register xListHostsReply *p = (xListHostsReply *)rp;
            warn("\tenabled ____________ %d\n", p->enabled);
            warn("\tnHosts _____________ %d\n", p->nHosts);
         }
         break;
      case X_SetPointerMapping:
         {  register xSetPointerMappingReply *p =
					(xSetPointerMappingReply *)rp;
            warn("\tsuccess ____________ %d\n", p->success);
         }
         break;
      case X_GetPointerMapping:
         {  register xGetPointerMappingReply *p =
					(xGetPointerMappingReply *)rp;
            warn("\tnElts ______________ %d\n", p->nElts);
         }
         break;
      case X_SetModifierMapping:
         {  register xSetModifierMappingReply *p =
						(xSetModifierMappingReply *)rp;
            warn("\tsuccess ____________ %d\n", p->success);
         }
         break;
      case X_GetModifierMapping:
         {  register xGetModifierMappingReply *p =
						(xGetModifierMappingReply *)rp;
            warn("\tnumKeyPerModifier __ %d\n", p->numKeyPerModifier);
         }
         break;
   }
}

void
dprx_event
   AL((ep))
   DB xEvent *ep
   DE
{
   char buf[8];

   warn("%s%s\n", dprx_event_str(ep->u.u.type),
			(ep->u.u.type & 0x80) ? " (SendEvent)" : "");
   warn("\tsequenceNumber_ %d\n", ep->u.u.sequenceNumber);
   switch (ep->u.u.type & 0x7f) {
      case KeyPress:
      case KeyRelease:
         warn("\ttime___________ %u\n", ep->u.keyButtonPointer.time);
         warn("\troot___________ 0x%08x\n", ep->u.keyButtonPointer.root);
         warn("\tevent__________ 0x%08x\n", ep->u.keyButtonPointer.event);
         warn("\tchild__________ 0x%08x\n", ep->u.keyButtonPointer.child);
         warn("\trootX__________ %d\n", ep->u.keyButtonPointer.rootX);
         warn("\trootY__________ %d\n", ep->u.keyButtonPointer.rootY);
         warn("\teventX_________ %d\n", ep->u.keyButtonPointer.eventX);
         warn("\teventY_________ %d\n", ep->u.keyButtonPointer.eventY);
         if (isprint(ep->u.u.detail))
            sprintf(buf, " '%c'", ep->u.u.detail);
         else
            buf[0] = '\0';
         warn("\tdetail_________ %d%s\n", ep->u.u.detail, buf);
         warn("\tstate__________ 0x%04x\n", ep->u.keyButtonPointer.state);
         break;
      case ButtonPress:
      case ButtonRelease:
      case MotionNotify:
         warn("\ttime___________ %u\n", ep->u.keyButtonPointer.time);
         warn("\troot___________ 0x%08x\n", ep->u.keyButtonPointer.root);
         warn("\tevent__________ 0x%08x\n", ep->u.keyButtonPointer.event);
         warn("\tchild__________ 0x%08x\n", ep->u.keyButtonPointer.child);
         warn("\trootX__________ %d\n", ep->u.keyButtonPointer.rootX);
         warn("\trootY__________ %d\n", ep->u.keyButtonPointer.rootY);
         warn("\teventX_________ %d\n", ep->u.keyButtonPointer.eventX);
         warn("\teventY_________ %d\n", ep->u.keyButtonPointer.eventY);
         warn("\tdetail_________ %d\n", ep->u.u.detail);
         warn("\tstate__________ 0x%04x\n", ep->u.keyButtonPointer.state);
         break;
      case EnterNotify:
      case LeaveNotify:
         warn("\ttime___________ %u\n", ep->u.enterLeave.time);
         warn("\troot___________ 0x%08x\n", ep->u.enterLeave.root);
         warn("\tevent__________ 0x%08x\n", ep->u.enterLeave.event);
         warn("\tchild__________ 0x%08x\n", ep->u.enterLeave.child);
         warn("\trootX__________ %d\n", ep->u.enterLeave.rootX);
         warn("\trootY__________ %d\n", ep->u.enterLeave.rootY);
         warn("\teventX_________ %d\n", ep->u.enterLeave.eventX);
         warn("\teventY_________ %d\n", ep->u.enterLeave.eventY);
         warn("\tstate__________ 0x%04x\n", ep->u.enterLeave.state);
         warn("\tmode___________ %d\n", ep->u.enterLeave.mode);
         warn("\tdetail_________ %d (%s)\n", ep->u.u.detail,
				dprx_notify_detail_str(ep->u.u.detail));
         warn("\tflags__________ 0x%02x\n", ep->u.enterLeave.flags);
         break;
      case FocusIn:
      case FocusOut:
         warn("\tdetail_________ %d (%s)\n", ep->u.u.detail,
				dprx_notify_detail_str(ep->u.u.detail));
         warn("\twindow_________ 0x%08x\n", ep->u.focus.window);
         warn("\tmode___________ %d (%s)\n", ep->u.focus.mode,
				dprx_notify_mode_str(ep->u.focus.mode));
         break;
      case KeymapNotify:
         break;
      case Expose:
         warn("\twindow_________ 0x%08x\n", ep->u.expose.window);
         warn("\tx______________ %d\n", ep->u.expose.x);
         warn("\ty______________ %d\n", ep->u.expose.y);
         warn("\twidth__________ %d\n", ep->u.expose.width);
         warn("\theight_________ %d\n", ep->u.expose.height);
         warn("\tcount__________ %d\n", ep->u.expose.count);
         break;
      case GraphicsExpose:
         warn("\tdrawable_______ 0x%08x\n", ep->u.graphicsExposure.drawable);
         warn("\tx______________ %d\n", ep->u.graphicsExposure.x);
         warn("\ty______________ %d\n", ep->u.graphicsExposure.y);
         warn("\twidth__________ %d\n", ep->u.graphicsExposure.width);
         warn("\theight_________ %d\n", ep->u.graphicsExposure.height);
         warn("\tmajorEvent_____ %d\n", ep->u.graphicsExposure.majorEvent);
         warn("\tminorEvent_____ %d\n", ep->u.graphicsExposure.minorEvent);
         warn("\tcount__________ %d\n", ep->u.graphicsExposure.count);
         break;
      case NoExpose:
         warn("\tdrawable_______ 0x%08x\n", ep->u.noExposure.drawable);
         warn("\tmajorEvent_____ %d\n", ep->u.noExposure.majorEvent);
         warn("\tminorEvent_____ %d\n", ep->u.noExposure.minorEvent);
         break;
      case VisibilityNotify:
         warn("\twindow_________ 0x%08x\n", ep->u.visibility.window);
         break;
      case CreateNotify:
         warn("\tparent_________ 0x%08x\n", ep->u.createNotify.parent);
         warn("\twindow_________ 0x%08x\n", ep->u.createNotify.window);
         warn("\tx______________ %d\n", ep->u.createNotify.x);
         warn("\ty______________ %d\n", ep->u.createNotify.y);
         warn("\twidth__________ %d\n", ep->u.createNotify.width);
         warn("\theight_________ %d\n", ep->u.createNotify.height);
         warn("\tborderWidth____ %d\n", ep->u.createNotify.borderWidth);
         break;
      case DestroyNotify:
         warn("\tevent__________ 0x%08x\n", ep->u.destroyNotify.event);
         warn("\twindow_________ 0x%08x\n", ep->u.destroyNotify.window);
         break;
      case UnmapNotify:
         warn("\tevent__________ 0x%08x\n", ep->u.unmapNotify.event);
         warn("\twindow_________ 0x%08x\n", ep->u.unmapNotify.window);
         break;
      case MapNotify:
         warn("\tevent__________ 0x%08x\n", ep->u.mapNotify.event);
         warn("\twindow_________ 0x%08x\n", ep->u.mapNotify.window);
         break;
      case MapRequest:
         warn("\tparent_________ 0x%08x\n", ep->u.mapRequest.parent);
         warn("\twindow_________ 0x%08x\n", ep->u.mapRequest.window);
         break;
      case ReparentNotify:
         warn("\tevent__________ 0x%08x\n", ep->u.reparent.event);
         warn("\twindow_________ 0x%08x\n", ep->u.reparent.window);
         warn("\tparent_________ 0x%08x\n", ep->u.reparent.parent);
         warn("\tx______________ %d\n", ep->u.reparent.x);
         warn("\ty______________ %d\n", ep->u.reparent.y);
         break;
      case ConfigureNotify:
         warn("\tevent__________ 0x%08x\n", ep->u.configureNotify.event);
         warn("\twindow_________ 0x%08x\n", ep->u.configureNotify.window);
         warn("\taboveSibling___ 0x%08x\n", ep->u.configureNotify.aboveSibling);
         warn("\tx______________ %d\n", ep->u.configureNotify.x);
         warn("\ty______________ %d\n", ep->u.configureNotify.y);
         warn("\twidth__________ %d\n", ep->u.configureNotify.width);
         warn("\theight_________ %d\n", ep->u.configureNotify.height);
         warn("\tborderWidth____ %d\n", ep->u.configureNotify.borderWidth);
         break;
      case ConfigureRequest:
         warn("\tparent_________ 0x%08x\n", ep->u.configureRequest.parent);
         warn("\twindow_________ 0x%08x\n", ep->u.configureRequest.window);
         warn("\tsibling________ 0x%08x\n", ep->u.configureRequest.sibling);
         warn("\tx______________ %d\n", ep->u.configureRequest.x);
         warn("\ty______________ %d\n", ep->u.configureRequest.y);
         warn("\twidth__________ %d\n", ep->u.configureRequest.width);
         warn("\theight_________ %d\n", ep->u.configureRequest.height);
         warn("\tborderWidth____ %d\n", ep->u.configureRequest.borderWidth);
         warn("\tvalueMask______ 0x%04x\n", ep->u.configureRequest.valueMask);
         break;
      case GravityNotify:
         warn("\tevent__________ 0x%08x\n", ep->u.gravity.event);
         warn("\twindow_________ 0x%08x\n", ep->u.gravity.window);
         warn("\tx______________ %d\n", ep->u.gravity.x);
         warn("\ty______________ %d\n", ep->u.gravity.y);
         break;
      case ResizeRequest:
         warn("\twindow_________ 0x%08x\n", ep->u.resizeRequest.window);
         warn("\twidth__________ %d\n", ep->u.resizeRequest.width);
         warn("\theight_________ %d\n", ep->u.resizeRequest.height);
         break;
      case CirculateNotify:
      case CirculateRequest:
         warn("\tevent__________ 0x%08x\n", ep->u.circulate.event);
         warn("\twindow_________ 0x%08x\n", ep->u.circulate.window);
         warn("\tparent_________ 0x%08x\n", ep->u.circulate.parent);
         break;
      case PropertyNotify:
         warn("\twindow_________ 0x%08x\n", ep->u.property.window);
         warn("\tatom___________ %d\n", ep->u.property.atom);
         warn("\ttime___________ %u\n", ep->u.property.time);
         break;
      case SelectionClear:
         warn("\ttime___________ %u\n", ep->u.selectionClear.time);
         warn("\twindow_________ 0x%08x\n", ep->u.selectionClear.window);
         warn("\tatom___________ %d\n", ep->u.selectionClear.atom);
         break;
      case SelectionRequest:
         warn("\ttime___________ %u\n", ep->u.selectionRequest.time);
         warn("\towner__________ 0x%08x\n", ep->u.selectionRequest.owner);
         warn("\trequestor______ 0x%08x\n", ep->u.selectionRequest.requestor);
         warn("\tselection______ %d\n", ep->u.selectionRequest.selection);
         warn("\ttarget_________ %d\n", ep->u.selectionRequest.target);
         warn("\tproperty_______ %d\n", ep->u.selectionRequest.property);
         break;
      case SelectionNotify:
         warn("\ttime___________ %u\n", ep->u.selectionNotify.time);
         warn("\trequestor______ 0x%08x\n", ep->u.selectionNotify.requestor);
         warn("\tselection______ %d\n", ep->u.selectionNotify.selection);
         warn("\ttarget_________ %d\n", ep->u.selectionNotify.target);
         warn("\tproperty_______ %d\n", ep->u.selectionNotify.property);
         break;
      case ColormapNotify:
         warn("\twindow_________ 0x%08x\n", ep->u.colormap.window);
         warn("\tcolormap_______ 0x%08x\n", ep->u.colormap.colormap);
         warn("\tnew____________ %d\n", ep->u.colormap.new);
         warn("\tstate__________ %d (%s)\n", ep->u.colormap.state,
			ep->u.colormap.state ? "Installed" : "Uninstalled");
         break;
      case ClientMessage:
         warn("\twindow_________ 0x%08x\n", ep->u.clientMessage.window);
         warn("\tdetail_________ %d\n", ep->u.u.detail);
         switch (ep->u.u.detail) {
            case 8:
               warn("\ttype___________ %d\n", ep->u.clientMessage.u.b.type);
               break;
            case 16:
               warn("\ttype___________ %d\n",ep->u.clientMessage.u.s.type);
               warn("\tshorts0________ %d\n",ep->u.clientMessage.u.s.shorts0);
               warn("\tshorts1________ %d\n",ep->u.clientMessage.u.s.shorts1);
               warn("\tshorts2________ %d\n",ep->u.clientMessage.u.s.shorts2);
               warn("\tshorts3________ %d\n",ep->u.clientMessage.u.s.shorts3);
               warn("\tshorts4________ %d\n",ep->u.clientMessage.u.s.shorts4);
               warn("\tshorts5________ %d\n",ep->u.clientMessage.u.s.shorts5);
               warn("\tshorts6________ %d\n",ep->u.clientMessage.u.s.shorts6);
               warn("\tshorts7________ %d\n",ep->u.clientMessage.u.s.shorts7);
               warn("\tshorts8________ %d\n",ep->u.clientMessage.u.s.shorts8);
               warn("\tshorts9________ %d\n",ep->u.clientMessage.u.s.shorts9);
               break;
            case 32:
               warn("\ttype___________ %d\n", ep->u.clientMessage.u.l.type);
               warn("\tlongs0_________ %d\n", ep->u.clientMessage.u.l.longs0);
               warn("\tlongs1_________ %d\n", ep->u.clientMessage.u.l.longs1);
               warn("\tlongs2_________ %d\n", ep->u.clientMessage.u.l.longs2);
               warn("\tlongs3_________ %d\n", ep->u.clientMessage.u.l.longs3);
               warn("\tlongs4_________ %d\n", ep->u.clientMessage.u.l.longs4);
               break;
         }
         break;
      case MappingNotify:
         break;
      default:
         {
            register u32_t *p = (u32_t *)ep;
            warn("\tlong0__________ 0x%08x\n", p[0]);
            warn("\tlong1__________ 0x%08x\n", p[1]);
            warn("\tlong2__________ 0x%08x\n", p[2]);
            warn("\tlong3__________ 0x%08x\n", p[3]);
            warn("\tlong4__________ 0x%08x\n", p[4]);
            warn("\tlong5__________ 0x%08x\n", p[5]);
            warn("\tlong6__________ 0x%08x\n", p[6]);
            warn("\tlong7__________ 0x%08x\n", p[7]);
         }
         /* ext_dprx_event(ep); */
         break;
   }
}

void
dprx_error
   AL((ep))
   DB xError *ep
   DE
{
   warn("%s\n", dprx_errorCode_str(ep->errorCode));
   warn("\tsequenceNumber __ %d\n", ep->sequenceNumber);
   warn("\tmajorCode _______ %d (%s)\n", ep->majorCode,
					dprx_reqType_str(ep->majorCode));
   warn("\tminorCode _______ %d\n", ep->minorCode);
   warn("\tresourceID ______ 0x%08x\n", ep->resourceID);
}
