/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   ebuf.c								*
*									*
*	Event buffer data structure and methods.			*
*	Store keyboard or pointer events and play them back.		*
*									*
************************************************************************/
#include <X11/X.h>
#define NEED_EVENTS
#include <X11/Xproto.h>
#include "xmx.h"
#include "incl/ebuf.pvt.h"

/*
**  These get malloc'd and free'd, but we always keep one around, so
**  pick a big-ish number.
*/
#define CHVSZ	1024

typedef struct _ev_t {
   server_t *		sp;
   chunk_t *		chp;
}ev_t;

typedef struct _ebuf_t {
   ev_t			ev[CHVSZ];
   int			nevents;
   struct _ebuf_t *	next;
}ebuf_t;

static ebuf_t *headp, *tailp;	/* head and tail of buffer list */
static int first = -1;		/* index of first event */
static int next = CHVSZ;	/* index of next spot */

static int playing;		/* ebuf_play in progress? */

/************************************************************************
*									*
*   ebuf_ready								*
*									*
*	Expression to protect ebuf_play.  This must always be		*
*	checked before ebuf_play is called.				*
*									*
************************************************************************/
int
ebuf_ready
   VOID
{
   return (first >= 0 && playing == 0);
}

/************************************************************************
*									*
*   ebuf_store								*
*									*
*	Store an event.							*
*									*
************************************************************************/
void
ebuf_store
   AL((sp, chp))
   DB server_t *sp
   DD chunk_t *chp
   DE
{
   register ebuf_t *ebufp;

   if (next == CHVSZ) {
      if (MALLOC(ebufp, ebuf_t *, sizeof(ebuf_t)))
         return;	/* throw out later events... */
      next = 0;
      ebufp->nevents = 0;
      ebufp->next = 0;
      if (tailp)
         tailp->next = ebufp;
      else
         headp = ebufp;	/* this should happen only once */

      tailp = ebufp;
   }
   if (first < 0)
      first = next;

   tailp->ev[next].sp = sp;
   tailp->ev[next].chp = chp;
   tailp->nevents++;

   chp->refs++;		/* keep chunk from being cleared */

   next++;
}

/************************************************************************
*									*
*   ebuf_clear								*
*									*
*	Clear all events.						*
*									*
************************************************************************/
void
ebuf_clear
   VOID
{
   register ebuf_t *nextp, *lastp;

   if (headp) {
      /*
      **  free all but one buffer
      */
      for (nextp=headp->next; lastp=nextp;) {
         nextp = nextp->next;
         free(lastp);
      }
      tailp = headp;
      tailp->nevents = 0;
      tailp->next = 0;
      next = 0;
      first = -1;
   }
}

/************************************************************************
*									*
*   ebuf_play								*
*									*
*	Reprocess and deliver all events of the given type(s).  If a	*
*	device is frozen or freezes as we go along, its events will be	*
*	put right back in the buffer, so we must be sure to play only	*
*	those events that were in the queue when we started.		*
*									*
************************************************************************/
void
ebuf_play
   VOID
{
   register int n, index, stop;
   register uint_t mask;
   register int iskey, isbutton;
   register u8_t type;
   register ebuf_t *ebufp, *lastp, *nextp;
   register xEvent *ep;
   window_t *wp;

#ifdef DEBUG
   if (first < 0) {			/* nothing to play back, bye */
      warn("ebuf_play: sanity - ebuf is empty!\n");
      return;
   }
   if (playing) {			/* avoid recursion */
      warn("ebuf_play: sanity - ebuf recursion!\n");
      return;
   }
#endif

   if ((mask = inp_playmask()) == 0)	/* devices both frozen */
      return;

   playing = 1;

   /*
   **  There's no stopping point until we get to the tail.
   */
   stop = headp == tailp ? next : -1;
   /*
   **  This loop plays back the events.
   */
   for (ebufp=headp, index=first; index != stop; index++) {
      if (index == CHVSZ) {
         index = 0;
         ebufp = ebufp->next;
         if (ebufp == tailp)
            stop = next;
      }
      if (ebufp->ev[index].chp) {
         ep = (xEvent *)buf_data(ebufp->ev[index].chp);
         type = ep->u.u.type & 0x7f;
         iskey = (type == KeyPress || type == KeyRelease);
         isbutton = (type == ButtonPress || type == ButtonRelease ||
						type == MotionNotify);

         if (	mask & KeyPressMask && iskey ||
		mask & ButtonPressMask && isbutton) {
            /*
            **  The event window and coordinates are no longer valid,
            **  and must be recalculated.  This is somewhat costly.
            */
            inp_root_translate(	ep->u.keyButtonPointer.rootX,
				ep->u.keyButtonPointer.rootY,
				&wp,
				&ep->u.keyButtonPointer.eventX,
				&ep->u.keyButtonPointer.eventY);
            ep->u.keyButtonPointer.event = wp->cid;
            if (iskey)
               inp_key(ebufp->ev[index].sp, wp, ebufp->ev[index].chp, type);
            else /* isbutton */
               inp_pointer(ebufp->ev[index].sp, wp, ebufp->ev[index].chp, type);
         }
         buf_clear(ebufp->ev[index].chp);
         ebufp->ev[index].chp = 0;
         ebufp->nevents--;
         /*
         **  Calls to inp_key and inp_pointer above can cause those
         **  devices to freeze or unfreeze, so check and respond to
         **  such changes here.
         */
	 if ((mask = inp_playmask()) == 0)
	    break;	/* devices are both frozen again, stop */
      }
   }
   /*
   **  Delete the buffers with no events in them from the list.  If there
   **  are no events in any, keep one.
   */
   for (lastp=0, nextp=headp; ebufp=nextp;) {
      nextp = nextp->next;
      if (ebufp->nevents == 0 && (nextp || lastp))
         free(ebufp);
      else {
         if (lastp)
            lastp->next = ebufp;
         else
            headp = ebufp;
         lastp = ebufp;
      }
   }
   if (lastp)
      tailp = lastp;
   else
      headp = tailp = ebufp;
   tailp->next = 0;
   /*
   **  Finally, reset first and next.
   */
   if (headp->nevents) {
      /*
      **  Wind back next.
      */
      while (next && tailp->ev[next-1].chp == 0)
         next--;
      /*
      **  Search backward for first.
      */
      if (headp == tailp)
         first = next;	/* must be positive */
      else
         first = CHVSZ;

      for (n=headp->nevents; n;)
         if (headp->ev[--first].chp)
            n--;
   }
   else {	/* queue is empty */
      first = -1;
      next = 0;
   }
   playing = 0;
}
