/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   focus.c								*
*									*
************************************************************************/
#define NEED_EVENTS
#include <X11/Xproto.h>
#include <X11/X.h>
#include "xmx.h"
#include "df.h"
#include "ptc.h"
#include "incl/focus.pvt.h"

/* focuswp is defined in xmx.h */
static int reverto;
static etime_t ftime;

static int grabbed;	/* boolean, is focus grabbed? */

/*
**	Previous focus, saved during a grab
*/
static window_t *focuswp_g;
static int reverto_g;

/************************************************************************
*									*
*   focus_set								*
*									*
************************************************************************/
void
focus_set
   AL((cp, p))
   DB client_t *cp
   DD xSetInputFocusReq *p
   DE
{
   register window_t *wp;
   etime_t reqtime;

   switch (p->focus) {
      case None:
         wp = 0;
         break;
      case PointerRoot:
         wp = vscreen.wp;
         break;
      default:
         if ((wp = (window_t *)hash_data(vmap, p->focus)) == 0) {
            proto_Error(cp, BadWindow, p->focus, 0, X_SetInputFocus);
            return;
         }
         break;
   }
   DEBUG4(D_FOCUS, "focus_set: cp [%d] wp [0x%08x/\"%s\"] time [%u]\n",
				cp->fd, wp, prop_wm_name(wp), p->time);
   time_update(0);

   if (p->time) {
      time_convert(p->time, &reqtime);
      if (	time_cmp(reqtime, ftime) < 0 ||
		time_cmp(reqtime, vtime) > 0) {
         DEBUG3(D_FOCUS, ">bad time focus[%s] < req[%s] < now[%s], ignoring\n",
		time_str(&ftime), time_str(&reqtime), time_str(&vtime));
         return;
      }
   }
   focus_setwindow(	wp,
			p->time ? &reqtime : &vtime,
			p->revertTo,
			grabbed ? NotifyWhileGrabbed : NotifyNormal);
}

/************************************************************************
*									*
*   focus_get								*
*									*
************************************************************************/
void
focus_get
   AL((cp))
   DB client_t *cp
   DE
{
   register rid_t fid;

   if (focuswp == vscreen.wp)
      fid = PointerRoot;
   else if (focuswp)
      fid = focuswp->cid;
   else 
      fid = None;

   proto_GetInputFocusReply(cp, reverto, fid);
}

/************************************************************************
*									*
*   focus_setwindow							*
*									*
************************************************************************/
void
focus_setwindow
   AL((wp, tp, nreverto, mode))
   DB window_t *wp
   DD etime_t *tp
   DD int nreverto
   DD u8_t mode
   DE
{
   register window_t *ow;

   if (wp == 0 || wp == vscreen.wp)
      reverto = RevertToNone;
   else
      reverto = nreverto;

   ow = focuswp;
   focus_assign(wp, tp);

   if (ow != focuswp)
      focus_out_in(ow, mode);
}

/************************************************************************
*									*
*   focus_grab								*
*									*
*	Set the focus as the result of a grab.				*
*									*
************************************************************************/
void
focus_grab
   AL((wp, tp))
   DB window_t *wp
   DD etime_t *tp
   DE
{
   register window_t *ow;

   DEBUG3(D_FOCUS, "focus_grab: wp [0x%08x/\"%s\"] time [%s]\n",
				wp, prop_wm_name(wp), time_str(tp));
   if (grabbed == 0) {
      focuswp_g = focuswp;
      reverto_g = reverto;

      grabbed = 1;
   }
   focus_setwindow(wp, tp, RevertToNone, NotifyGrab);
}


/************************************************************************
*									*
*   focus_ungrab							*
*									*
*	Change the focus to what it was prior to a keyboard grab.	*
*									*
************************************************************************/
void
focus_ungrab
   AL((tp))
   DB etime_t *tp
   DE
{
   register window_t *ow;

   if (grabbed) {
      DEBUG1(D_FOCUS, "focus_ungrab: time [%s]\n", time_str(tp));
      ow = focuswp;
      if (tp == 0) {
         time_update(0);
         tp = &vtime;
      }
      focus_assign(focuswp_g, tp);
      reverto = reverto_g;

      if (ow != focuswp)
         focus_out_in(ow, NotifyUngrab);

      grabbed = 0;
   }
}

/************************************************************************
*									*
*   focus_revert							*
*									*
*	Called when the focus window becomes not visible.		*
*	May be called during a grab (huh?).				*
*									*
************************************************************************/
void
focus_revert
   VOID
{
   register window_t *ow;

   if (grabbed) {
      DEBUG0(D_FOCUS, "focus_revert: ungrabbing\n");
      time_update(0);
      focus_ungrab(&vtime);
   }
   else {
      ow = focuswp;
      switch (reverto) {
         case RevertToNone:
            focuswp = 0;
            break;
         case RevertToPointerRoot:
            reverto = None;
            focuswp = vscreen.wp;
            break;
         case RevertToParent:
            reverto = None;
            focuswp = focuswp->parent;
            break;
      }
      DEBUG2(D_FOCUS, "focus_revert: reverting to wp [0x%08x/\"%s\"]\n",
					focuswp, prop_wm_name(focuswp));
      if (ow != focuswp)
         focus_out_in(ow, NotifyNormal);
   }
}

/************************************************************************
*									*
*   focus_assign							*
*									*
*	Set the current focus window.  Updates the last focus change	*
*	time.  If tp is zero, the current time is used.			*
*									*
************************************************************************/
void
focus_assign
   AL((wp, tp))
   DB window_t *wp
   DD etime_t *tp
   DE
{
   if (tp) {
      time_assign(*tp, ftime);
   }
   else {
      time_update(0);
      time_assign(vtime, ftime);
   }
   focuswp = wp;
}

/************************************************************************
*									*
*   focus_out_in							*
*									*
*	This routine and those below handle all focus event		*
*	generation.  The new focuswp should already have been set when	*
*	this is called.  Don't call this if the focus has not really	*
*	changed, in some cases that will cause spurious events to be	*
*	generated.							*
*									*
************************************************************************/
void
focus_out_in
   AL((ofocuswp, mode))
   DB window_t *ofocuswp
   DD u8_t mode
   DE
{
   register int direction;
   register window_t *wp;

   /*
   **  linear or nonlinear?     (positive is up, negative is down)
   */
   if (ofocuswp && ofocuswp != vscreen.wp && focuswp && focuswp != vscreen.wp)
      if (ofocuswp->level > focuswp->level) {
         for (wp=ofocuswp; wp->level > focuswp->level; wp=wp->parent);
         direction = (wp == focuswp) ? 1 : 0;
      }
      else {
         for (wp=focuswp; wp->level > ofocuswp->level; wp=wp->parent);
         direction = (wp == ofocuswp) ? -1 : 0;
      }
   else
      direction = 0;

   out_pointerup(ofocuswp, mode);
   inout(ofocuswp, focuswp, 0, 0, mode, direction);
   in_pointerdown(ofocuswp, mode);
}

/*
**	Recursively generate focus in/out events per the X protocol spec.
**
**	This routine climbs the window heirarchy beginning at both of
**	the two windows involved.  As "out" windows are encountered,
**	FocusOut events are generated.  As the recursion unwinds, previously
**	encountered "in" windows generate FocusIn events.  In this way
**	a single recursive traversal generates all events in the correct
**	order.
*/
static void
inout
   AL((ow, nw, outs, ins, mode, direction))
   DB window_t *ow
   DD window_t *nw
   DD int outs
   DD int ins
   DD u8_t mode
   DD int direction
   DE
{
   if (ow == 0) {
      if (nw) {			/* None -> window */
         /*
         **  climb ins to None
         */
         inout(ow, nw->parent, outs, ins+1, mode, direction);
         in(nw, mode, ins, direction);
      }
      else
         /*
         **  top of recursion to or from None, issue in or out for None
         */
         if (outs == 0)
            out(ow, mode, outs, direction);	/* final None -> window */
         else if (ins == 0)
            in(nw, mode, ins, direction);	/* final window -> None */
   }
   else if (nw == 0) {		/* window -> None */
      /*
      **  climb outs to None
      */
      out(ow, mode, outs, direction);
      inout(ow->parent, nw, outs+1, ins, mode, direction);
   }
   else {			/* window -> window */
      if (ow->level > nw->level) {
         /*
         **  climb the outs' side
         */
         out(ow, mode, outs, direction);
         inout(ow->parent, nw, outs+1, ins, mode, direction);
      }
      else if (ow->level < nw->level) {
         /*
         **  climb the ins' side
         */
         inout(ow, nw->parent, outs, ins+1, mode, direction);
         in(nw, mode, ins, direction);
      }
      else if (ow != nw) {
         /*
         **  climb both outs and ins, looking for common root
         */
         out(ow, mode, outs, direction);
         inout(ow->parent, nw->parent, outs+1, ins+1, mode, direction);
         in(nw, mode, ins, direction);
      }
      else if (ow == vscreen.wp)	/* && ow == nw */
         if (outs == 0 || ins == 0) {		/* (but never both...) */
            /*
            **  root is old or new focus
            */
            out(ow, mode, outs, direction);
            in(nw, mode, ins, direction);
         }
   }
}

/*
**	Generate FocusOut events for a window.  This routine handles all
**	permutations of FocusOut events a window might trigger.
*/
static void
out
   AL((wp, mode, cnt, direction))
   DB window_t *wp	/* old focus window, or an ancestor */
   DD u8_t mode		/* Normal, Grab, Ungrab, WhileGrabbed */
   DD int cnt		/* Virtual if non-zero */
   DD int direction	/* up, down or non-linear */
   DE
{
   if (wp) {
      if (wp->xmask->mask & FocusChangeMask)
         if (cnt)
            if (direction)
               event_FocusOut(wp, mode, NotifyVirtual);
            else
               event_FocusOut(wp, mode, NotifyNonlinearVirtual);
         else
            if (wp == vscreen.wp)
               event_FocusOut(wp, mode, NotifyPointerRoot);
            else
               if (direction > 0)
                  event_FocusOut(wp, mode, NotifyAncestor);
               else if (direction < 0)
                  event_FocusOut(wp, mode, NotifyInferior);
               else
                  event_FocusOut(wp, mode, NotifyNonlinear);
   }
   else
      if (cnt == 0)
         if (vscreen.wp->xmask->mask & FocusChangeMask)
            event_FocusOut(vscreen.wp, mode, NotifyDetailNone);
}

/*
**	Generate FocusIn events for a window.  This routine handles all
**	permutations of FocusIn events a window might trigger.
*/
static void
in
   AL((wp, mode, cnt, direction))
   DB window_t *wp	/* new focus window, or an ancestor */
   DD u8_t mode		/* Normal, Grab, Ungrab, WhileGrabbed */
   DD int cnt		/* Virtual if non-zero */
   DD int direction	/* up, down or non-linear */
   DE
{
   register int i;

   if (wp) {
      if (wp->xmask->mask & FocusChangeMask)
         if (cnt)
            if (direction)
               event_FocusIn(wp, mode, NotifyVirtual);
            else
               event_FocusIn(wp, mode, NotifyNonlinearVirtual);
         else
            if (wp == vscreen.wp)
               event_FocusIn(wp, mode, NotifyPointerRoot);
            else
               if (direction < 0)
                  event_FocusIn(wp, mode, NotifyAncestor);
               else if (direction > 0)
                  event_FocusIn(wp, mode, NotifyInferior);
               else
                  event_FocusIn(wp, mode, NotifyNonlinear);
   }
   else
      if (cnt == 0)
         if (vscreen.wp->xmask->mask & FocusChangeMask)
            event_FocusIn(vscreen.wp, mode, NotifyDetailNone);
}

/*
**  out_pointerup
**
**	Generate FocusOut events with detail Pointer upward from the
**	pointer window to the old focus window (including the old focus
**	window only if it is also the root window).  The pointer window
**	must be an inferior of the old focus window but not an inferior
**	of the new focus window (if it is both, it is not considered
**	to be losing the focus).
*/
static void
out_pointerup
   AL((ofocuswp, mode))
   DB window_t *ofocuswp
   DD u8_t mode
   DE
{
   register int okay;
   register u16_t level;
   register window_t *wp;

   /*
   **  There can be no out events if there is no old focus window
   */
   if (ofocuswp == 0)
      return;
   /*
   **  Ensure that the pointer window is a descendent of the old
   **  focus window, but not a descendent of the new focus window
   **  (unless the new focus window is the root)
   */
   level = focuswp ? MIN(ofocuswp->level, focuswp->level) : 0;
   okay = 0;
   for (wp=ptrwp; wp && wp->level >= level; wp=wp->parent) {
      if (wp == ofocuswp)
         okay = 1;
      else if (wp == focuswp)
         return;
   }
   if (!okay)
      return;
   /*
   **  Generate the events
   */
   for (wp=ptrwp; wp != ofocuswp; wp=wp->parent)
      event_FocusOut(wp, mode, NotifyPointer);
   /*
   **  include the old focus window only when it is the root
   */
   if (ofocuswp == vscreen.wp)
      event_FocusOut(ofocuswp, mode, NotifyPointer);
}

/*
**  in_pointerup
**
**	Generate FocusIn events with detail Pointer downward from the
**	new focus window to the pointer window (including the new focus
**	window only if it is also the root window).  The pointer window
**	must be an inferior of the new focus window but not an inferior
**	of the old focus window (if it is both, it is not considered
**	to be gaining the focus).
*/
static void
in_pointerdown
   AL((ofocuswp, mode))
   DB window_t *ofocuswp
   DD u8_t mode
   DE
{
   register int okay;
   register u16_t level;
   register window_t *wp;

   /*
   **  There can be no in events if there is no new focus window
   */
   if (focuswp == 0)
      return;
   /*
   **  Ensure that the pointer window is a descendent of the new
   **  focus window, but not a descendent of the old focus window
   **  (unless the old focus window is the root)
   */
   level = ofocuswp ? MIN(ofocuswp->level, focuswp->level) : 0;
   okay = 0;
   for (wp=ptrwp; wp && wp->level >= level; wp=wp->parent) {
      if (wp == focuswp)
         okay = 1;
      else if (wp == ofocuswp)
         return;
   }
   if (!okay)
      return;
   /*
   **  Generate the events recursively
   */
   (void) in_pointerdown_rec(ptrwp, mode);
}

/*
**  Climb from ptrwp to focuswp and generate events on the way back
*/
static void
in_pointerdown_rec
   AL((wp, mode))
   DB window_t *wp
   DD u8_t mode
   DE
{
   if (wp == focuswp) {
      /*
      **  include the focus window only when it is the root
      */
      if (wp != vscreen.wp)
         return;
   }
   else
      in_pointerdown_rec(wp->parent, mode);

   event_FocusIn(wp, mode, NotifyPointer);
}
