/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   host.c								*
*									*
*	Routines for host name processing.				*
*									*
************************************************************************/
#include <sys/param.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>

#include "xmx.h"
#include "incl/host.pvt.h"

static int inetno = 0;
static char *pattern;

/************************************************************************
*									*
*   host_me								*
*									*
*	Establish identity of local machine.				*
*									*
************************************************************************/
void
host_me
   VOID
{
   static char host[MAXHOSTNAMELEN];
   struct hostent *hp;

   if (gethostname(host, MAXHOSTNAMELEN))
      pquit(1, "gethostname");
   hostnamelen = strlen(host);
   if (MALLOC(hostname, char *, hostnamelen+1))		/* set global */
      quit(-1, "host_me: fatal memory allocation error\n");
   (void)strcpy(hostname, host);
   if ((hp = gethostbyname(host)) == 0)
      pquit(1, "%s: fatal identity crisis\n", host);

   if (MALLOC(me.address, char *, hp->h_length))
      quit(-1, "host_me: fatal memory allocation error\n");

   me.family = util_fam_utox((u16_t)hp->h_addrtype);
   me.length = (u16_t)hp->h_length;
   bcopy(*hp->h_addr_list, me.address, hp->h_length);
}

/************************************************************************
*									*
*   host_isme								*
*									*
*	Check an address and see if it's ours.				*
*									*
************************************************************************/
int
host_isme
   AL((family, length, address))
   DB u16_t family
   DD u16_t length
   DD char *address
   DE
{
   if (family == FamilyLocal ||
       (	family == me.family &&
		length == me.length &&
		bcmp(address, me.address, length) == 0))
      return 1;

   return 0;
}

/************************************************************************
*									*
*   host_pattern							*
*									*
*	Set the current pattern for host database searching.		*
*	Don't call this routine unless you call host_match after it.	*
*									*
************************************************************************/
void
host_pattern
   AL((pat))
   DB char *pat
   DE
{
   register char *cp;

   pattern = pat;
   for (cp=pat; isdigit(*cp) || *cp == '.'; cp++);
   inetno = (*cp == '\0') ? 1 : 0;
   if (opt.glob && !inetno)
      sethostent(1);
}

/************************************************************************
*									*
*   host_match								*
*									*
*	Returns the hostname and address of the next host in the	*
*	hosts database that matches the current pattern.  Returns	*
*	zero if there is no match.					*
*									*
************************************************************************/
char *
host_match
   AL((family, length, address))
   DB u16_t *family
   DD u16_t *length
   DD char **address
   DE
{
   char *name = (char *)0;
   struct hostent *hp;
   static char **alias = (char **)0;
   static unsigned long iaddr;

   if (pattern)
      if (*pattern == 0) {
         *family = FamilyLocal;
         *length = me.length;
         *address = me.address;
         name = pattern;
         pattern = (char *)0;
      }
      else if (inetno) {
         if ((long)(iaddr = inet_addr(pattern)) != -1) {
            *family = FamilyInternet;
            *length = sizeof(iaddr);
            *address = (char *)&iaddr;
            name = pattern;
         }
         pattern = (char *)0;
      }
      else if (opt.glob)
         for (;;) {
            if (alias && *alias)
               name = *alias++;
            else if (hp = gethostent()) {
               name = (char *)hp->h_name;
               alias = hp->h_aliases;
            }
            else {
               pattern = name = (char *)0;
               endhostent();
               break;
            }
            if (util_match(pattern, name)) {
               *family = util_fam_utox(hp->h_addrtype);
               *length = hp->h_length;
               *address = *hp->h_addr_list;
               name = (char *)hp->h_name;
               alias = (char **)0;
               break;
            }
         }
      else if (hp = gethostbyname(pattern)) {
         *family = util_fam_utox(hp->h_addrtype);
         *length = hp->h_length;
         *address = *hp->h_addr_list;
         name = (char *)hp->h_name;
         pattern = (char *)0;
      }
   return name;
}
