/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   iswap.c								*
*									*
*	These routines perform bit and byte swapping while copying	*
*	images from one buffer to another.  It is assumed that all	*
*	buffers are padded to 32 bits (whether or not they need to	*
*	be), so that we needn't worry about referencing memory		*
*	locations outside the buffers.					*
*									*
************************************************************************/
#include "iswap.h"
#include "xmx.h"
#include "incl/iswap.pvt.h"

#define	INCR(bytes,pad)	(((pad)-(bytes)%(pad))%(pad))

/*
**	just copy - should only be used if padding differs
*/
void
iswap_cpy
   AL((src, dst, spad, dpad, bpp, w, h))
   DB u8_t *src
   DD u8_t *dst
   DD int spad
   DD int dpad
   DD int bpp
   DD int w
   DD int h
   DE
{
   register int bytes;
   register int dinc, sinc;
   u8_t *sp, *dp;

   bytes = (w*bpp + 7) / 8;
   dpad /= 8;
   spad /= 8;
   dinc = bytes + INCR(bytes,dpad);
   sinc = bytes + INCR(bytes,spad);
   sp = src;
   dp = dst;
   for (; h>0; h--) {	/* for each scanline */
      bcopy(sp, dp, bytes);
      dp += dinc;
      sp += sinc;
   }
}
/*
**	swap 4 bit units (nibbles) within 8 bit units
*/
void
iswap_nib
   AL((src, dst, spad, dpad, bpp, w, h))
   DB u8_t *src
   DD u8_t *dst
   DD int spad
   DD int dpad
   DD int bpp
   DD int w
   DD int h
   DE
{
   register int i, bytes;
   register int dinc, sinc;
   u8_t *sp, *dp;

   bytes = (w*bpp + 7) / 8;
   dpad /= 8;
   spad /= 8;
   dinc = INCR(bytes,dpad);
   sinc = INCR(bytes,spad);
   sp = src;
   dp = dst;
   for (; h>0; h--) {	/* for each scanline */
      for (i=0; i<bytes; i++)		/* for each bytes */
         *dp++ = rnt[*sp++];
      dp += dinc;
      sp += sinc;
   }
}
/*
**	swap 16 bit units (words) within 32 bit units
*/
void
iswap___w
   AL((src, dst, spad, dpad, bpp, w, h))
   DB u8_t *src
   DD u8_t *dst
   DD int spad
   DD int dpad
   DD int bpp
   DD int w
   DD int h
   DE
{
   register int i, bits, bytes, longs;
   register int dinc, sinc;
   u8_t *sp, *dp;

   bits = w * bpp;
   bytes = (bits + 7) / 8;
   longs = (bits + 31) / 32;
   dpad /= 8;
   spad /= 8;
   dinc = INCR(bytes,dpad) - INCR(bytes,4);
   sinc = INCR(bytes,spad) - INCR(bytes,4);
   sp = src;
   dp = dst;
   for (; h>0; h--) {	/* for each scanline */
      for (i=0; i<longs; i++) {	/* for each long */
         dp[0] = sp[2];
         dp[1] = sp[3];
         dp[2] = sp[0];
         dp[3] = sp[1];
         dp += 4;
         sp += 4;
      }
      dp += dinc;
      sp += sinc;
   }
}
/*
**	swap 8 bit units (bytes) within 16 bit units
*/
void
iswap__b_
   AL((src, dst, spad, dpad, bpp, w, h))
   DB u8_t *src
   DD u8_t *dst
   DD int spad
   DD int dpad
   DD int bpp
   DD int w
   DD int h
   DE
{
   register int i, bits, bytes, shorts;
   register int dinc, sinc;
   u8_t *sp, *dp;

   bits = w * bpp;
   bytes = (bits + 7) / 8;
   shorts = (bits + 15) / 16;
   dpad /= 8;
   spad /= 8;
   dinc = INCR(bytes,dpad) - INCR(bytes,2);
   sinc = INCR(bytes,spad) - INCR(bytes,2);
   sp = src;
   dp = dst;
   for (; h>0; h--) {	/* for each scanline */
      for (i=0; i<shorts; i++) {	/* for each short */
         dp[0] = sp[1];
         dp[1] = sp[0];
         dp += 2;
         sp += 2;
      }
      dp += dinc;
      sp += sinc;
   }
}
/*
**	swap 8 bit units (bytes) within 24 bit units
*/
void
iswap__b3
   AL((src, dst, spad, dpad, bpp, w, h))
   DB u8_t *src
   DD u8_t *dst
   DD int spad
   DD int dpad
   DD int bpp
   DD int w
   DD int h
   DE
{
   register int i, bits, bytes, triples;
   register int dinc, sinc;
   u8_t *sp, *dp;

   bits = w * bpp;
   bytes = (bits + 7) / 8;
   triples = (bits + 23) / 24;
   dpad /= 8;
   spad /= 8;
   dinc = INCR(bytes,dpad) - INCR(bytes,3);
   sinc = INCR(bytes,spad) - INCR(bytes,3);
   sp = src;
   dp = dst;
   for (; h>0; h--) {	/* for each scanline */
      for (i=0; i<triples; i++) {	/* for each 3 bytes */
         dp[0] = sp[2];
         dp[1] = sp[1];
         dp[2] = sp[0];
         dp += 3;
         sp += 3;
      }
      dp += dinc;
      sp += sinc;
   }
}
/*
**	swap bytes and words
*/
void
iswap__bw
   AL((src, dst, spad, dpad, bpp, w, h))
   DB u8_t *src
   DD u8_t *dst
   DD int spad
   DD int dpad
   DD int bpp
   DD int w
   DD int h
   DE
{
   register int i, bits, bytes, longs;
   register int dinc, sinc;
   u8_t *sp, *dp;

   bits = w * bpp;
   bytes = (bits + 7) / 8;
   longs = (bits + 31) / 32;
   dpad /= 8;
   spad /= 8;
   dinc = INCR(bytes,dpad) - INCR(bytes,4);
   sinc = INCR(bytes,spad) - INCR(bytes,4);
   sp = src;
   dp = dst;
   for (; h>0; h--) {	/* for each scanline */
      for (i=0; i<longs; i++) {	/* for each long */
         dp[0] = sp[3];
         dp[1] = sp[2];
         dp[2] = sp[1];
         dp[3] = sp[0];
         dp += 4;
         sp += 4;
      }
      dp += dinc;
      sp += sinc;
   }
}
/*
**	reverse bits within bytes
*/
void
iswap_r__
   AL((src, dst, spad, dpad, bpp, w, h))
   DB u8_t *src
   DD u8_t *dst
   DD int spad
   DD int dpad
   DD int bpp
   DD int w
   DD int h
   DE
{
   register int i, bytes;
   register int dinc, sinc;
   u8_t *sp, *dp;

   bytes = (w*bpp + 7) / 8;
   dpad /= 8;
   spad /= 8;
   dinc = INCR(bytes,dpad);
   sinc = INCR(bytes,spad);
   sp = src;
   dp = dst;
   for (; h>0; h--) {	/* for each scanline */
      for (i=0; i<bytes; i++)		/* for each byte */
         *dp++ = rbt[*sp++];
      dp += dinc;
      sp += sinc;
   }
}
/*
**	reverse bits and swap words
*/
void
iswap_r_w
   AL((src, dst, spad, dpad, bpp, w, h))
   DB u8_t *src
   DD u8_t *dst
   DD int spad
   DD int dpad
   DD int bpp
   DD int w
   DD int h
   DE
{
   register int i, bits, bytes, longs;
   register int dinc, sinc;
   u8_t *sp, *dp;

   bits = w * bpp;
   bytes = (bits + 7) / 8;
   longs = (bits + 31) / 32;
   dpad /= 8;
   spad /= 8;
   dinc = INCR(bytes,dpad) - INCR(bytes,4);
   sinc = INCR(bytes,spad) - INCR(bytes,4);
   sp = src;
   dp = dst;
   for (; h>0; h--) {	/* for each scanline */
      for (i=0; i<longs; i++) {	/* for each long */
         dp[0] = rbt[sp[2]];
         dp[1] = rbt[sp[3]];
         dp[2] = rbt[sp[0]];
         dp[3] = rbt[sp[1]];
         dp += 4;
         sp += 4;
      }
      dp += dinc;
      sp += sinc;
   }
}
/*
**	reverse bits and swap bytes
*/
void
iswap_rb_
   AL((src, dst, spad, dpad, bpp, w, h))
   DB u8_t *src
   DD u8_t *dst
   DD int spad
   DD int dpad
   DD int bpp
   DD int w
   DD int h
   DE
{
   register int i, bits, bytes, shorts;
   register int dinc, sinc;
   u8_t *sp, *dp;

   bits = w * bpp;
   bytes = (bits + 7) / 8;
   shorts = (bits + 15) / 16;
   dpad /= 8;
   spad /= 8;
   dinc = INCR(bytes,dpad) - INCR(bytes,2);
   sinc = INCR(bytes,spad) - INCR(bytes,2);
   sp = src;
   dp = dst;
   for (; h>0; h--) {	/* for each scanline */
      for (i=0; i<shorts; i++) {	/* for each short */
         dp[0] = rbt[sp[1]];
         dp[1] = rbt[sp[0]];
         dp += 2;
         sp += 2;
      }
      dp += dinc;
      sp += sinc;
   }
}
/*
**	reverse bits and swap bytes and words
*/
void
iswap_rbw
   AL((src, dst, spad, dpad, bpp, w, h))
   DB u8_t *src
   DD u8_t *dst
   DD int spad
   DD int dpad
   DD int bpp
   DD int w
   DD int h
   DE
{
   register int i, bits, bytes, longs;
   register int dinc, sinc;
   u8_t *sp, *dp;

   bits = w * bpp;
   bytes = (bits + 7) / 8;
   longs = (bits + 31) / 32;
   dpad /= 8;
   spad /= 8;
   dinc = INCR(bytes,dpad) - INCR(bytes,4);
   sinc = INCR(bytes,spad) - INCR(bytes,4);
   sp = src;
   dp = dst;
   for (; h>0; h--) {	/* for each scanline */
      for (i=0; i<longs; i++) {	/* for each long */
         dp[0] = rbt[sp[3]];
         dp[1] = rbt[sp[2]];
         dp[2] = rbt[sp[1]];
         dp[3] = rbt[sp[0]];
         dp += 4;
         sp += 4;
      }
      dp += dinc;
      sp += sinc;
   }
}
