/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   kpm.c								*
*									*
*	Implement keyboard/pointer mapping.				*
*									*
************************************************************************/
#include <X11/X.h>
#define NEED_EVENTS
#define NEED_REPLIES
#include <X11/Xproto.h>

#include "xmx.h"
#include "df.h"
#include "cx.h"
#include "rx.h"
#include "zb.h"
#include "kpm.h"
#include "incl/kpm.pvt.h"

static vkeymap;			/* index of current key mapping */
static int kmapsz;		/* size of vector of alternate key mappings */
static kmap_t **kmaps;		/* vector of alternate key mappings */
static int nbuttons;		/* number of buttons in virtual map */

static uint_t dirty;		/* has keymapping changed? (bitmask) */
static int mindirty;		/* min keycode changed */
static int maxdirty;		/* max keycode changed */

/*
**  keycode hash table - lookup keycodes by their first keysym
*/
static khash_t *khash[KEYCODESZ];

/************************************************************************
*									*
*   kpm_new_ktm								*
*   kpm_free_ktm							*
*									*
************************************************************************/
ktm_t *
kpm_new_ktm
   VOID
{
   register int i;
   register ktm_t *ktmp;
   
   if (MALLOC(ktmp, ktm_t *, sizeof(ktm_t)))
      return 0;

   DEBUG1(D_KEYMAP, "kpm_new_ktm: ktmp [0x%08x]\n", ktmp);
   
   ktmp->vkm = 0;
   for (i=0; i<KEYCODESZ; i++) {
      ktmp->keymap[i] = 0;
      ktmp->modmap[i] = 0;
   }
   ktmp->butmapsz = 0;
   ktmp->butmap = 0;

   return ktmp;
}

void
kpm_free_ktm
   AL((ktmp))
   DB ktm_t *ktmp
   DE
{
   DEBUG1(D_KEYMAP, "kpm_free_ktm: ktmp [0x%08x]\n", ktmp);
   free(ktmp);
}

/************************************************************************
*									*
*   kpm_set_keys							*
*   kpm_set_mods							*
*   kpm_set_buttons							*
*									*
*	The bottom line is, you can't do any of these.  If you want to	*
*	change any of these mappings, you have to connect to the local	*
*	X server to do it - xmx will not "broadcast" any of these	*
*	requests.  Unfortunately, there isn't always a way to ignore	*
*	these requests cleanly.						*
*									*
*	ChangeKeyboardMapping always succeeds (modulo alloc or		*
*	out-of-range errors, which we won't resort to).  It also	*
*	always generates a MappingNotify event.  So we ignore the	*
*	change, and send the expected event, albeit without the		*
*	expected changes.						*
*									*
*	SetModifierMapping replies to the client, and one possible	*
*	reply is Failed, so we do that in all cases.			*
*									*
*	SetPointerMapping replies to the client, but the only choices	*
*	we have are Success or Busy.  So we report success, though	*
*	we in fact ignore the changes here, too.			*
*									*
************************************************************************/
void
kpm_set_keys
   AL((cp, p))
   DB client_t *cp
   DD xChangeKeyboardMappingReq *p
   DE
{
   DEBUG3(D_KEYMAP, "kpm_set_keys: client [%d] first [%d] count [%d]\n",
					cp->fd, p->firstKeyCode, p->keyCodes);
   event_MappingNotify(MappingKeyboard, p->firstKeyCode, p->keyCodes);
}

void
kpm_set_mods
   AL((cp, p))
   DB client_t *cp
   DD xSetModifierMappingReq *p
   DE
{
   DEBUG1(D_KEYMAP, "kpm_set_mods: client [%d]\n", cp->fd);
   proto_SetModifierMappingReply(cp, MappingFailed);
}

void
kpm_set_buttons
   AL((cp, p))
   DB client_t *cp
   DD xSetPointerMappingReq *p
   DE
{
   DEBUG1(D_KEYMAP, "kpm_set_buttons: client [%d]\n", cp->fd);
   proto_SetPointerMappingReply(cp, MappingSuccess);
   /*
   **  it's an error to return success and not generate a MappingNotify
   */
   event_MappingNotify(MappingPointer, 0, 0);
}

/************************************************************************
*									*
*   kpm_get_keys							*
*   kpm_get_mods							*
*   kpm_get_buttons							*
*									*
************************************************************************/
void
kpm_get_keys
   AL((cp, p))
   DB client_t *cp
   DD xGetKeyboardMappingReq *p
   DE
{
   register int nspc;

   DEBUG3(D_KEYMAP, "kpm_get_keys: client [%d] first [%d] count [%d]\n",
					cp->fd, p->firstKeyCode, p->count);
   nspc = kmaps[vkeymap]->nspc;
   proto_GetKeyboardMappingReply(cp, nspc, p->count,
			&kmaps[vkeymap]->keymap[p->firstKeyCode * nspc]);
}

void
kpm_get_mods
   AL((cp, p))
   DB client_t *cp
   DD xReq *p
   DE
{
   DEBUG1(D_KEYMAP, "kpm_get_mods: client [%d]\n", cp->fd);
   proto_GetModifierMappingReply(	cp,
					kmaps[vkeymap]->ncpm,
					kmaps[vkeymap]->modmap);
}

void
kpm_get_buttons
   AL((cp, p))
   DB client_t *cp
   DD xReq *p
   DE
{
   register int i;
   static int butmapsz;
   static u8_t *butmap;

   DEBUG1(D_KEYMAP, "kpm_get_buttons: client [%d]\n", cp->fd);
   
   if (butmapsz < nbuttons) {
      if (butmap)
         free(butmap);
      butmapsz = RUP(nbuttons, 16);
      if (MALLOC(butmap, u8_t *, butmapsz))
         butmapsz = 0;
      else
         for (i=0; i<butmapsz; i++)
            butmap[i] = i;
   }
   proto_GetPointerMappingReply(cp, nbuttons, butmap);
}

/************************************************************************
*									*
*   kpm_update_keys							*
*   kpm_update_mods							*
*   kpm_update_buttons							*
*									*
*	Request keyboard/pointer mapping information from the current	*
*	server destination.						*
*									*
*	This needs to postpone processing of incoming key and button	*
*	events until the new mapping is sucked in. (TODO)		*
*									*
************************************************************************/
void
kpm_update_keys
   AL((minkey, keycnt))
   DB int minkey
   DD int keycnt
   DE
{
   proto_GetKeyboardMapping(minkey, keycnt);
   df_put_i(minkey);
   df_put_i(keycnt);
   rx_queue(zrxqp, X_GetKeyboardMapping, 0, zb_seqno(), keymapping_reply);
}

static void
keymapping_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   register xGetKeyboardMappingReply *rp =
				(xGetKeyboardMappingReply *)buf_data(chp);
   register int start = df_get_i(dfp);
   register int count = df_get_i(dfp);

   kpm_merge_keys(	sp->smap.ktmp,
			start,
			count,
			rp->keySymsPerKeyCode,
			(ksym_t *)(rp + 1));
   if (dirty)
      clean_dirt();
}

void
kpm_update_mods
   VOID
{
   proto_GetModifierMapping();
   rx_queue(zrxqp, X_GetModifierMapping, 0, zb_seqno(), modmapping_reply);
}

static void
modmapping_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   register xGetModifierMappingReply *rp =
				(xGetModifierMappingReply *)buf_data(chp);

   kpm_merge_mods(sp->smap.ktmp, rp->numKeyPerModifier, (kcode_t *)(rp + 1));
   if (dirty)
      clean_dirt();
}

void
kpm_update_buttons
   VOID
{
   proto_GetPointerMapping();
   rx_queue(zrxqp, X_GetPointerMapping, 0, zb_seqno(), ptrmapping_reply);
}

static void
ptrmapping_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   register xGetPointerMappingReply *rp =
				(xGetPointerMappingReply *)buf_data(chp);

   kpm_merge_buttons(sp->smap.ktmp, rp->nElts, (u8_t *)(rp + 1));
   if (dirty)
      clean_dirt();
}

/************************************************************************
*									*
*   kpm_merge_keys							*
*   kpm_merge_mods							*
*   kpm_merge_buttons							*
*									*
************************************************************************/
void
kpm_merge_keys
   AL((ktmp, start, count, nspc, kmp))
   DB ktm_t *ktmp
   DD int start
   DD int count
   DD int nspc
   DD ksym_t *kmp
   DE
{
   register int i, skc;

   DEBUG5(D_KEYMAP, "%s: ktmp [0x%08x] start [%d] count [%d] nspc [%d]\n",
			"kpm_merge_keys", ktmp, start, count, nspc);
   skc = start;
   for (i=0; i<count; i++) {
      canon_syms(nspc, &kmp[i*nspc]);
      add_keymapping(ktmp, skc, nspc, &kmp[i*nspc], ktmp->modmap[skc]);
      skc++;
   }
   update_modmap();
}

void
kpm_merge_mods
   AL((ktmp, ncpm, modmap))
   DB ktm_t *ktmp
   DD int ncpm
   DD kcode_t *modmap
   DE
{
   register int i, j, nspc;
   register kcode_t vkc;
   register kmap_t *tmap;
   kcode_t tmodmap[KEYCODESZ];
   static int vecsz;
   static ksym_t *ksvec;

   DEBUG2(D_KEYMAP, "kpm_merge_mods: ktmp [0x%08x] ncpm [%d]\n", ktmp, ncpm);
   nspc = kmaps[ktmp->vkm] ? kmaps[ktmp->vkm]->nspc : 0;

   if (vecsz < nspc) {
      if (ksvec)
         free(ksvec);
      if (MALLOC(ksvec, ksym_t *, nspc * sizeof(ksym_t)))
         return;
   }
   /*
   **  first, build new modmap
   */
   for (i=FIRSTKEYCODE; i<KEYCODESZ; i++)
      tmodmap[i] = 0;
   for (i=0; i<8; i++)
      for (j=0; j<ncpm; j++)
         if (modmap[i*ncpm+j])
            tmodmap[modmap[i*ncpm+j]] |= (1<<i);
   if (ncpm > kmaps[ktmp->vkm]->ncpm)
      kmaps[ktmp->vkm]->ncpm = ncpm;
   /*
   **  compare with old one, adjusting mappings as needed
   */
   for (i=FIRSTKEYCODE; i<KEYCODESZ; i++)
      if (ktmp->modmap[i] != tmodmap[i]) {
         /*
         **  if this change in mod affects an existing mapping, adjust it
         */
         if (vkc = ktmp->keymap[i]) {
            tmap = kmaps[ktmp->vkm];
            /*
            **  make a copy of the keysyms, since they may get wiped
            */
            bcopy(	(char *)&tmap->keymap[vkc*nspc],
			(char *)ksvec,
			nspc * sizeof(ksym_t));
            del_mapping(ktmp->vkm, vkc);
            ktmp->keymap[i] = 0;

            add_keymapping(ktmp, i, nspc, ksvec, tmodmap[i]);
         }
         ktmp->modmap[i] = tmodmap[i];
      }

   update_modmap();
}

void
kpm_merge_buttons
   AL((ktmp, n, map))
   DB ktm_t *ktmp
   DD int n
   DD u8_t *map
   DE
{
   register int i;

   DEBUG2(D_KEYMAP, "kpm_merge_buttons: ktmp [0x%08x] n [%d]\n", ktmp, n);
   if (n+1 > ktmp->butmapsz) {
      if (ktmp->butmap)
         free(ktmp->butmap);
      if (MALLOC(ktmp->butmap, u8_t *, n+1))
         return;
      ktmp->butmap[0] = 0;	/* never used */
   }
   if (n > nbuttons) {	/* need to generate a MappingNotify event here? */
      nbuttons = n;
   }
   for (i=1; i<=n; i++)
      ktmp->butmap[i] = map[i];
   
   for (; i<ktmp->butmapsz; i++)
      ktmp->butmap[i] = i;
}

/************************************************************************
*									*
*   kpm_map_key								*
*   kpm_map_button							*
*									*
************************************************************************/
kcode_t
kpm_map_key
   AL((ktmp, keycode))
   DB ktm_t *ktmp
   DD kcode_t keycode
   DE
{
   DEBUG3(D_KEYMAP, "kpm_map_key: ktmp [0x%08x] [%d->%d]\n",
					ktmp, keycode, ktmp->keymap[keycode]);
   if (vkeymap != ktmp->vkm) {
      /*
      **  Alert clients that we are using a different key mapping.
      */
      switch_vkeymap(ktmp->vkm);
   }
   return ktmp->keymap[keycode];
}

u8_t
kpm_map_button
   AL((ktmp, button))
   DB ktm_t *ktmp
   DD u8_t button
   DE
{
   DEBUG3(D_KEYMAP, "kpm_map_button: ktmp [0x%08x] [%d->%d]\n", ktmp, button,
		button <= ktmp->butmapsz ? ktmp->butmap[button] : button);
   if (button <= ktmp->butmapsz)
      return ktmp->butmap[button];
   else
      return button;
}

/*
**  add_keymapping
**
**	Establish a server keycode to virtual keycode mapping.
**	Creates a virtual keymap (kmap) if needed, but only after
**	first trying to fold this keymapping into an existing
**	keymap.  This routine may cause a server to use a
**	different virtual keymap (so don't expect ktmp->vkm
**	to remain unchanged after this routine is called).
*/
static void
add_keymapping
   AL((ktmp, skc, nspc, ksvec, modmask))
   DB ktm_t *ktmp	/* server's mapping info */
   DD kcode_t skc	/* server keycode to map */
   DD int nspc		/* number of symbols in ksvec */
   DD ksym_t *ksvec	/* vector of keysyms to which skc should map */
   DD u8_t modmask	/* bitmask of modifiers applied to skc */
   DE
{
   register int vkm, first, next;
   register int newfg = 0;
   register kcode_t vkc;
   register kmap_t **tmaps;

#ifdef DEBUG
   if (debug & D_KEYMAP) {
      warn("kpm.c/add_keymapping: ktmp [0x%08x] skc [%d] nspc [%d]\n",
							ktmp, skc, nspc);
      print_kprint(nspc, ksvec, modmask);
   }
#endif
   /*
   **  Consider first the virtual keymap that the server is using (if
   **  any).  If that doesn't work, try the others in order from zero.
   */
   next = 0;
   vkm = first = ktmp->vkm;
   for (;;) {
      /*
      **  kmaps list too short?
      */
      if (vkm == kmapsz) {
         if (CALLOC(tmaps, kmap_t **, (kmapsz + VKMAPINC), sizeof(kmap_t *)))
            return;
         if (kmaps) {
            bcopy(kmaps, tmaps, kmapsz * sizeof(kmap_t **));
            free(kmaps);
         }
         kmapsz += VKMAPINC;
         kmaps = tmaps;
      }
      /*
      **  do we need a new kmap?
      */
      if (kmaps[vkm] == 0) {
         if (newfg) {   /* avoid infinite malloc loop - just in case */
            warn("kpm.c/add_keymapping: failed - this can't happen\n");
            return;
         }
         kmaps[vkm] = new_kmap(nspc);
         newfg = 1;
      }
      /*
      **  If the server is not using this kmap, make it so.
      */
      if (vkm == ktmp->vkm || move_server(ktmp, vkm) == 0) {
         /*
         **  is there already a server entry for this keycode?
         */
         if (vkc = ktmp->keymap[skc]) {
            if (match(vkm, vkc, nspc, ksvec, modmask) == 0)
               return;		/* all done */
            /*
            **  if not a match, get rid of it
            */
            del_mapping(vkm, vkc);
            ktmp->keymap[skc] = 0;
         }
         if (vkc = add_mapping(vkm, nspc, ksvec, modmask)) {
            ktmp->keymap[skc] = vkc;
            return;		/* success */
         }
      }
      /*
      **  no?  try another kmap
      */
      while ((vkm = next++) == first);
   }
}

static void
update_modmap
   VOID
{
   register int i, j;
   register int ncpm;
   register kmap_t *tmap;
   int modix[8] = {0,0,0,0,0,0,0,0};

   DEBUG1(D_KEYMAP, "kpm.c/update_modmap: vkeymap [%d]\n", vkeymap);
   tmap = kmaps[vkeymap];
   ncpm = tmap->ncpm;
   if (ncpm * 8 > tmap->modmapsz) {
      if (tmap->modmapsz)
         free(tmap->modmap);
      tmap->modmapsz = (ncpm + 4) * 8;	/* add 4 to leave room for growth */
      if (MALLOC(tmap->modmap, kcode_t *, tmap->modmapsz))
         return;
   }
   for (i=0; i<KEYCODESZ; i++)
      if (tmap->krefs[i].modmask)
         for (j=0; j<8; j++)
            if (tmap->krefs[i].modmask & 1<<j && modix[j] < ncpm)
               tmap->modmap[j*ncpm+modix[j]++] = (kcode_t)i;
   /*
   **  Fill up unused map entries with zeros
   */
   for (j=0; j<8; j++)
      while (modix[j] < ncpm)
         tmap->modmap[j*ncpm+modix[j]++] = 0;
}

static int
move_server
   AL((ktmp, vkm))
   DB ktm_t *ktmp
   DD int vkm
   DE
{
   register int nspc, skc;
   register kcode_t vkc, nvkc;
   register kmap_t *tmap;
   kcode_t keymap[KEYCODESZ];

   DEBUG3(D_KEYMAP, "kpm.c/move_server: ktmp [0x%08x] vkm [%d->%d]\n",
					ktmp, ktmp->vkm, vkm);
   tmap = kmaps[ktmp->vkm];	/* source */
   nspc = tmap->nspc;

   for (skc=FIRSTKEYCODE; skc<KEYCODESZ; skc++) {
      if (vkc = ktmp->keymap[skc]) {
         if (nvkc = add_mapping(vkm, nspc, &tmap->keymap[vkc*nspc],
						tmap->krefs[vkc].modmask))
            keymap[skc] = nvkc;
         else {
            /*
            **  no good, undo and return failure
            */
            for (skc--; skc>=FIRSTKEYCODE; skc--)
               if (nvkc = keymap[skc])
                  del_mapping(vkm, nvkc);
            return -1;
         }
      }
      else
         keymap[skc] = 0;
   }
   /*
   **  success - commit move
   */
   for (skc=FIRSTKEYCODE; skc<KEYCODESZ; skc++)
      ktmp->keymap[skc] = keymap[skc];
   ktmp->vkm = vkm;

   return 0;
}

/*
**  add_mapping
**
**	Actually add a mapping to a virtual keymap.  Returns the
**	virtual keycode if successful, or zero otherwise.
*/
static kcode_t
add_mapping
   AL((vkm, nspc, ksvec, modmask))
   DB int vkm
   DD int nspc
   DD ksym_t *ksvec
   DD u8_t modmask
   DE
{
   register int i, vnspc;
   register kcode_t vkc;

   vnspc = kmaps[vkm]->nspc;

   if (vnspc < nspc) {
      for (i=vnspc; i<nspc; i++)
         if (ksvec[i])
            break;
      if (i < nspc)	/* need to enlarge vkeymap */
         if (kmaps[vkm]->nfree)
            change_nspc(vkm, nspc);
         else
            return 0;	/* no room */
      else
         nspc = vnspc;	/* just ignore trailing null keysyms */
   }
   /*
   **  if the exact entry does not already exist,
   **  add a new entry if there's room for it
   */
   if ((vkc = khash_lookup(vkm, nspc, ksvec, modmask)) == 0) {
      if (kmaps[vkm]->minfree < KEYCODESZ) {
         vkc = kmaps[vkm]->minfree;
         for (i=vkc+1; i<KEYCODESZ; i++)
            if (kmaps[vkm]->krefs[i].refs == 0)
               break;
         kmaps[vkm]->minfree = i;
         kmaps[vkm]->nfree--;
         if (vkc > kmaps[vkm]->maxref)
            kmaps[vkm]->maxref = vkc;

         dirty_vkeymap(vkm, vkc, nspc, ksvec, modmask);
         set_kprint(vkm, vkc, nspc, ksvec, modmask);
         khash_add(vkm, vkc, ksvec);
      }
      else
         return 0;	/* no room */
   }
   /*
   **  increment ref, set server mapping
   */
   kmaps[vkm]->krefs[vkc].refs++;

   return vkc;
}

static void
switch_vkeymap
   AL((vkm))
   DB int vkm
   DE
{
   dirty = 0;
   mindirty = maxdirty = 0;
   vkeymap = vkm;
   event_MappingNotify(MappingKeyboard, FIRSTKEYCODE, kmaps[vkeymap]->maxref);
   event_MappingNotify(MappingModifier, 0, 0);
}

static void
dirty_vkeymap
   AL((vkm, vkc, nspc, ksvec, modmask))
   DB int vkm
   DD int vkc
   DD int nspc
   DD ksym_t *ksvec
   DD u8_t modmask
   DE
{
   register int ndirty = 0;
   register u8_t omodmask;
   
   omodmask = kmaps[vkm]->krefs[vkc].modmask;
   if (modmask == omodmask)
      ndirty |= 1<<MappingKeyboard;
   else if (match(vkm, vkc, nspc, ksvec, omodmask) == 0)
      ndirty |= 1<<MappingModifier;
   else
      ndirty |= 1<<MappingKeyboard | 1<<MappingModifier;
   
   if (ndirty & 1<<MappingKeyboard) {
      if (vkc < mindirty || mindirty == 0)
         mindirty = vkc;
      if (vkc > maxdirty)
         maxdirty = vkc;
   }
   dirty |= ndirty;
}

/*
**  clean_dirt
**
**	Dirt is cleaned by notifying all X clients that one or
**	more of the mappings has changed.
*/
static void
clean_dirt
   VOID
{
   if (dirty & 1<<MappingKeyboard) {
      event_MappingNotify(MappingKeyboard, mindirty, maxdirty);
      mindirty = maxdirty = 0;
   }
   if (dirty & 1<<MappingModifier)
      event_MappingNotify(MappingModifier, 0, 0);
   if (dirty & 1<<MappingPointer)
      event_MappingNotify(MappingPointer, 0, 0);
   
   dirty = 0;
}

/*
**  change_nspc
**
**	Change the number of keysyms per keycode in the key mapping.
**	Allocates a new table, copies the old data over and frees the
**	old table.  If the new nspc is smaller, symbols at higher
**	indices will be lost (it's assumed they're all zero).
**
**	This routine could be used to reduce the size of a keymapping
**	table when the extra keysyms are no longer needed - after a
**	server leaves the session, for instance. (TODO)
*/
static void
change_nspc
   AL((vkm, nspc))
   DB int vkm
   DD int nspc
   DE
{
   register int i, j, onspc, mnspc;
   register kmap_t *tmap;
   register ksym_t *keymap;
   register ksym_t *ksvec, *oksvec;

   tmap = kmaps[vkm];
   onspc = tmap->nspc;
   mnspc = MIN(nspc, onspc);

   if (CALLOC(keymap, ksym_t *, nspc*KEYCODESZ, sizeof(ksym_t))) {
      warn("kpm.c/change_nspc: I won't recover from this\n");
      return;
   }
   for (i=FIRSTKEYCODE; i<=tmap->maxref; i++) {
      oksvec = &tmap->keymap[i*onspc];
      ksvec = &keymap[i*nspc];
      for (j=0; j<mnspc; j++)
         ksvec[j] = oksvec[j];
   }
   free(tmap->keymap);
   tmap->keymap = keymap;
   tmap->nspc = nspc;
}

/*
**  del_mapping
**
**	Decrement reference count of the given virtual mapping.  If
**	the count reaches zero, free the mapping.
*/
static void
del_mapping
   AL((vkm, vkc))
   DB int vkm
   DD kcode_t vkc
   DE
{
   register int nspc, kc;
   register ksym_t *ksvec;

   switch (kmaps[vkm]->krefs[vkc].refs) {
      case 0:
         break;
      case 1:
         kmaps[vkm]->krefs[vkc].refs = 0;
         if (vkc < kmaps[vkm]->minfree)
            kmaps[vkm]->minfree = vkc;
         if (vkc == kmaps[vkm]->maxref) {
            for (kc=vkc-1; kc && kmaps[vkm]->krefs[kc].refs == 0; kc--);
            kmaps[vkm]->maxref = kc;
         }
         kmaps[vkm]->nfree++;

         nspc = kmaps[vkm]->nspc;
         ksvec = &kmaps[vkm]->keymap[vkc*nspc];
         khash_del(vkm, vkc, ksvec);
         clear_kprint(vkm, vkc);
         break;
      default:
         kmaps[vkm]->krefs[vkc].refs--;
         break;
   }
}

/*
**  canon_syms
**
**	Put the keysym vector into canonical form.
**	We choose always the simplest expression of the key.
*/
static void
canon_syms
   AL((nspc, ksvec))
   DB int nspc
   DD ksym_t *ksvec
   DE
{
   if (ksvec[0]) {
      if (nspc >= 2)
	 canon_group(&ksvec[0]);
      if (nspc >= 4) {
	 canon_group(&ksvec[2]);
         /*
         **  if group 1 and group 2 are identical, clear group 2
         */
         if (ksvec[0] == ksvec[2] && ksvec[1] == ksvec[3])
            ksvec[2] = ksvec[3] = 0;
      }
   }
}

/*
**  canon_group
**
**	Put the symbol group into canonical form.
*/
static void
canon_group
   AL((ksvec))
   DB ksym_t *ksvec	/* must be at least 2 elements long */
   DE
{
   /*
   **  if both syms are the same, clear the second
   */
   if (ksvec[0] == ksvec[1])
      ksvec[1] = 0;
   else if (isascii(ksvec[0]) && isascii(ksvec[1]))
      /*
      **  if they are a lowercase/uppercase ascii pair,
      **  clear the uppercase one.
      */
      if (isupper(ksvec[1]) && ksvec[0] == tolower(ksvec[1]))
         ksvec[1] = 0;
      /*
      **  if they are an uppercase/null pair, make the first
      **  one lowercase.
      */
      else if (isupper(ksvec[0]) && ksvec[1] == 0)
         ksvec[0] = tolower(ksvec[0]);
}

/*
**  set_kprint
**
**	Set the "keyprint" of the given virtual keycode.  The keyprint
**	is just the keysyms and modifiers associated with it.
*/
static void
set_kprint
   AL((vkm, vkc, nspc, ksvec, modmask))
   DB int vkm
   DD kcode_t vkc
   DD int nspc
   DD ksym_t *ksvec
   DD u8_t modmask
   DE
{
   register int vnspc;

   vnspc = kmaps[vkm]->nspc;
   if (nspc > vnspc)
      nspc = vnspc;
   kmaps[vkm]->krefs[vkc].modmask = modmask;
   bcopy(	(char *)ksvec,
		(char *)&kmaps[vkm]->keymap[vkc*vnspc],
		nspc * sizeof(ksym_t));
   if (nspc < vnspc)
      bzero((char *)&kmaps[vkm]->keymap[vkc*vnspc+nspc], vnspc - nspc);
}

/*
**  clear_kprint
**
**	Clear the "keyprint" of the given virtual keycode.
*/
static void
clear_kprint
   AL((vkm, vkc))
   DB int vkm
   DD kcode_t vkc
   DE
{
   register int nspc;

   nspc = kmaps[vkm]->nspc;
   kmaps[vkm]->krefs[vkc].modmask = 0;
   bzero((char *)&kmaps[vkm]->keymap[vkc*nspc], nspc * sizeof(ksym_t));
}

static void
print_kprint
   AL((nspc, ksvec, modmask))
   DB int nspc
   DD ksym_t *ksvec
   DD u8_t modmask
   DE
{
   register int i;
   
   warn("->keyprint");
   for (i=0; i<nspc; i++)
      warn("%c%08x", i ? '/' : ' ', ksvec[i]);
   warn(" mods %02x\n", modmask);
}

static void
print_vkeymap
   AL((vkm))
   DB int vkm
   DE
{
   register int i, nspc;
   register kmap_t *tmap;

   tmap = kmaps[vkm];
   nspc = tmap->nspc;

   warn("vkm: %d", vkm);
   warn(" minfree [%d] nfree [%d] modmapsz [%d] nspc [%d] ncpm [%d]\n",
	tmap->minfree, tmap->nfree, tmap->modmapsz, tmap->nspc, tmap->ncpm);

   for (i=FIRSTKEYCODE; i<KEYCODESZ; i++) {
      warn("   %03d refs[%d]", i, tmap->krefs[i].refs);
      print_kprint(nspc, &tmap->keymap[i*nspc], tmap->krefs[i].modmask);
   }
}

static void
print_modmap
   AL((ncpm, modmap))
   DB int ncpm
   DD kcode_t *modmap
   DE
{
   register int i, j;
   char *modnames[] =
	{"shift", "lock", "control", "mod1", "mod2", "mod3", "mod4", "mod5"};

   for (i=0; i<8; i++) {
      warn("%s\t", modnames[i]);
      for (j=0; j<ncpm; j++)
         if (modmap[i*ncpm+j]) {
            if (j)
               warn(", ");
            warn("0x%x", modmap[i*ncpm+j]);
         }
      warn("\n");
   }
}

/*
**  match
**
**	Compare the "keyprint" of a virtual keymap entry with the
**	one given.  The length of the keysym lists may differ, if
**	they do, missing keysyms are taken to be zero.
**	Returns 0 if it matches, -1 otherwise.
*/
static int
match
   AL((vkm, vkc, nspc, ksvec, modmask))
   DB int vkm
   DD kcode_t vkc
   DD int nspc
   DD ksym_t *ksvec
   DD u8_t modmask
   DE
{
   register int i;
   register int lnspc, bnspc;	/* little, big */
   register ksym_t *lksvec, *bksvec;

   if (nspc < kmaps[vkm]->nspc) {
      lnspc = nspc;
      lksvec = ksvec;
      bnspc = kmaps[vkm]->nspc;
      bksvec = &kmaps[vkm]->keymap[vkc*bnspc];
   }
   else {
      bnspc = nspc;
      bksvec = ksvec;
      lnspc = kmaps[vkm]->nspc;
      lksvec = &kmaps[vkm]->keymap[vkc*lnspc];
   }
   for (i=0; i<lnspc; i++)
      if (lksvec[i] != bksvec[i])
         return -1;
   for (; i<bnspc; i++)
      if (bksvec[i] != 0)
         return -1;

   if (kmaps[vkm]->krefs[vkc].modmask != modmask)
      return -1;

   return 0;
}

static kmap_t *
new_kmap
   AL((nspc))
   DB int nspc
   DE
{
   register kmap_t *kmap;

   if (MALLOC(kmap, kmap_t *, sizeof(kmap_t)))
      return 0;

   kmap->maxref = 0;
   kmap->minfree = FIRSTKEYCODE;
   kmap->nfree = MAXKEYCODES;
   kmap->modmapsz = 0;
   kmap->nspc = nspc;
   kmap->ncpm = 0;

   if (CALLOC(kmap->keymap, ksym_t *, nspc*KEYCODESZ, sizeof(ksym_t))) {
      free(kmap);
      return 0;
   }
   kmap->modmap = 0;

   bzero(kmap->krefs, KEYCODESZ*sizeof(kref_t));
   bzero(kmap->khash, KEYCODESZ*sizeof(khash_t));
   
   return kmap;
}

void
free_kmap
   AL((kmap))
   DB kmap_t *kmap
   DE
{
   free(kmap->keymap);
   if (kmap->modmapsz)
      free(kmap->modmap);
   free(kmap);
}

/*
**  khash_lookup
**
**	Look up a keycode using it's "keyprint".
*/
static kcode_t
khash_lookup
   AL((vkm, nspc, ksvec, modmask))
   DB int vkm
   DD int nspc
   DD ksym_t *ksvec
   DD u8_t modmask
   DE
{
   register int i;
   register kcode_t vkc;
   register khash_t *khp;

   khp = &kmaps[vkm]->khash[ksvec[0]%KEYCODESZ];

   for (; khp; khp=khp->next)
      for (i=0; i<NBUCKET; i++)
         if (vkc = khp->kc[i]) {
            if (match(vkm, vkc, nspc, ksvec, modmask) == 0)
               return vkc;
         }
         else
            return 0;	/* not found */

   return 0;	/* not found */
}

/*
**  khash_add
**
**	Add a keycode to the given keysym hash table.
*/
static void
khash_add
   AL((vkm, vkc, ksvec))
   DB int vkm
   DD kcode_t vkc
   DD ksym_t *ksvec
   DE
{
   register int i;
   register khash_t *khp, *lkhp;
   register kcode_t kc;

   khp = &kmaps[vkm]->khash[ksvec[0]%KEYCODESZ];

   for (; khp; khp=khp->next) {
      lkhp = khp;
      for (i=0; i<NBUCKET; i++)
         if (khp->kc[i] == 0) {
            khp->kc[i] = vkc;
            return;
         }
         else if (khp->kc[i] == vkc)
            return;
   }
   /*
   **  create a new bucket in the hash table
   */
   if (MALLOC(khp, khash_t *, sizeof(khash_t)))
      return;
   khp->next = 0;
   bzero(khp->kc, NBUCKET * sizeof(kcode_t));
   lkhp->next = khp;

   khp->kc[0] = vkc;
}

/*
**  khash_del
**
**	Delete a keycode from the given keysym hash table.
*/
static void
khash_del
   AL((vkm, vkc, ksvec))
   DB int vkm
   DD kcode_t vkc
   DD ksym_t *ksvec
   DE
{
   register int i, j;
   register khash_t *khp, *lkhp;

   khp = &kmaps[vkm]->khash[ksvec[0]%KEYCODESZ];

   for (; lkhp=khp; khp=khp->next) {
      for (i=0; i<NBUCKET; i++) {
         if (khp->kc[i] == 0)
            return;		/* not found, bye */
         else if (khp->kc[i] == vkc)
            break;		/* found it, break both loops */
      }
      if (i < NBUCKET)
         break;
   }
   /*
   **  slide bucket values left
   */
   j = 0;
   do {
      if (j) {
         lkhp->kc[j] = khp->kc[0];
         if (khp->kc[1] == 0) {	/* bucket no longer needed, nuke it */
            lkhp->next = 0;
            free(khp);
            return;
         }
         i = 0;
      }
      for (j=i++; i<NBUCKET;)
         khp->kc[j++] = khp->kc[i++];
   }while (lkhp=khp, khp=khp->next);
}
