/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   list.c								*
*									*
*	Efficient variable length lists.				*
*									*
************************************************************************/
#include "xmx.h"
#include "incl/list.pvt.h"

static list_t *lfree;	/* list_t free list */
static lblk_t *blkfree;	/* lblk_t free list */

/************************************************************************
*									*
*   list_new								*
*									*
************************************************************************/
list_t *
list_new
   VOID
{
   register list_t *lp;
   register lblk_t *blkp;

   if (lfree) {
      lp = lfree;
      lfree = lfree->next;
   }
   else {
      if (MALLOC(lp, list_t *, sizeof(list_t)))
         return 0;
      /*
      **  there is always a block
      */
      if ((blkp = new_block()) == 0) {
         free(lp);
         return 0;
      }
      lp->head = blkp;
   }
   lp->head->next = 0;
   lp->tail = lp->head;
   lp->hix = lp->tix = 0;
   lp->nval = 0;

   return lp;
}

/************************************************************************
*									*
*   list_free								*
*									*
************************************************************************/
void
list_free
   AL((lp))
   DB list_t *lp
   DE
{
   lblk_t *blkp, *last;

   for (blkp=lp->head->next; last=blkp;) {
      blkp = blkp->next;
      free_block(last);
   }
   /*
   **  lp->head is okay, everything else dangles...
   */
   lp->next = lfree;
   lfree = lp;
}

/************************************************************************
*									*
*   list_free_freelists							*
*									*
************************************************************************/
void
list_free_freelists
   VOID
{
   register list_t *lp, *llp;
   register lblk_t *blkp, *lblkp;

   for (blkp=blkfree; lblkp=blkp;) {
      blkp = blkp->next;
      free(lblkp);
   }
   blkfree = 0;

   for (lp=lfree; llp=lp;) {
      lp = lp->next;
      free(llp->head);
      free(llp);
   }
   lfree = 0;
}

/************************************************************************
*									*
*   list_bump								*
*									*
*	Add (allocate) a list block to the end of a list structure.	*
*									*
************************************************************************/
int
list_bump
   AL((lp))
   DB list_t *lp
   DE
{
   register lblk_t *blkp;

   if ((blkp = new_block()) == 0)
      return -1;

   lp->tail->next = blkp;
   lp->tail = blkp;
   blkp->next = 0;

   return 0;
}

/************************************************************************
*									*
*   list_drop								*
*									*
*	Drop (free) a list block from the head of a list structure.	*
*									*
************************************************************************/
void
list_drop
   AL((lp))
   DB list_t *lp
   DE
{
   register lblk_t *blkp;

   blkp = lp->head;
   lp->head = blkp->next;
   free_block(blkp);
}

/************************************************************************
*									*
*   list_clear								*
*									*
************************************************************************/
void
list_clear
   AL((lp, n))
   DB list_t *lp
   DD int n		/* number of values to clear, all if zero */
   DE
{
   register int ncontig;
   lblk_t *blkp, *last;

   if (n && n < lp->nval) {	/* clear just n values */
      lp->nval -= n;
      ncontig = LISTSZ - lp->hix;
      while (n >= ncontig) {
         n -= ncontig;
         list_drop(lp);
         ncontig = LISTSZ;
      }
      lp->hix = n;
   }
   else {			/* clear it all */
      for (blkp=lp->head->next; last=blkp;) {
         blkp = blkp->next;
         free_block(last);
      }
      lp->head->next = 0;
      lp->tail = lp->head;
      lp->hix = lp->tix = 0;
      lp->nval = 0;
   }
}

/************************************************************************
*									*
*   list_push								*
*									*
*	This is identical to the list_put macro.			*
*									*
************************************************************************/
void
list_push
   AL((lp, val))
   DB list_t *lp
   DD u32_t val
   DE
{
   if (lp->tix == LISTSZ) {
      list_bump(lp);
      lp->tix = 0;
   }
   lp->tail->v[lp->tix++] = val;
   lp->nval++;
}

/************************************************************************
*									*
*   list_first								*
*   list_next								*
*									*
*	Walk a list non-destructively.					*
*									*
************************************************************************/
static u16_t cix;
static lblk_t *cblkp;

void
list_reset
   AL((lp))
   DB list_t *lp
   DE
{
   cix = lp->hix;
   cblkp = lp->head;
}

u32_t
list_next
   AL((lp))
   DB list_t *lp
   DE
{
   register u32_t val;

   val = cblkp->v[cix++];

   if (cix == LISTSZ) {
      cblkp = cblkp->next;
      cix = 0;
   }
   return val;
}

/************************************************************************
*									*
*   list_get								*
*									*
*	Retrieve the next value in the list, deleting it.		*
*									*
************************************************************************/
u32_t
list_get
   AL((lp))
   DB list_t *lp
   DE
{
   register u32_t val;

   switch (lp->nval) {
      case 0:
         val = 0;
         warn("list_get: list is empty!\n");
         break;
      case 1:				/* if last, clear list */
         val = lp->head->v[lp->hix];
         lp->hix = 0;
         lp->nval = 0;
         break;
      default:
         val = lp->head->v[lp->hix++];

         if (lp->hix == LISTSZ) {
            list_drop(lp);
            lp->hix = 0;
         }
         lp->nval--;
         break;
   }
   return val;
}

/************************************************************************
*									*
*   list_move								*
*									*
*	Move all data from a list into a buffer, leaving the list	*
*	cleared.							*
*									*
************************************************************************/
void
list_move
   AL((lp, bp))
   DB list_t *lp
   DD buffer_t *bp
   DE
{
   register int n, beg, cnt, max;
   register lblk_t *blkp, *last;

   beg = lp->hix;
   blkp = lp->head;

   n = lp->nval;
   cnt = MIN(n, LISTSZ) - beg;

   while (n) {
      buf_put(bp, (char *)&blkp->v[beg], cnt * 4);
      n -= cnt;

      if (blkp->next == 0)
         break;			/* loop exit */

      last = blkp;
      blkp = blkp->next;
      free_block(last);

      cnt = MIN(n, LISTSZ);
      beg = 0;
   }
   lp->head = lp->tail = blkp;
   lp->hix = lp->tix = 0;
   lp->nval = 0;
}

/*
**   new_block
*/
static lblk_t *
new_block
   VOID
{
   register lblk_t *blkp;

   if (blkfree) {
      blkp = blkfree;
      blkfree = blkfree->next;
   }
   else if (MALLOC(blkp, lblk_t *, sizeof(lblk_t)))
      return 0;

   return blkp;
}

/*
**   free_block
*/
static void
free_block
   AL((blkp))
   DB lblk_t *blkp
   DE
{
   blkp->next = blkfree;
   blkfree = blkp;
}
