/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   pixmap.c								*
*									*
************************************************************************/
#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/Xproto.h>
#include <X11/X.h>
#include "xmx.h"
#include "df.h"
#include "cx.h"
#include "rx.h"
#include "zb.h"
#include "res.h"
#include "incl/pixmap.pvt.h"

static pixmap_t *pixmaps;

/************************************************************************
*									*
*   pixmap_create							*
*									*
************************************************************************/
int
pixmap_create
   AL((cp, p))
   DB client_t *cp
   DD xCreatePixmapReq *p
   DE
{
   register int i;
   register drawable_t *dp;
   register pixmap_t *pxp;

   if ((dp = (drawable_t *)hash_data(vmap, p->drawable)) == 0) {
      proto_Error(cp, BadDrawable, p->drawable, 0, X_CreatePixmap);
      return -1;
   }
   if (p->width == 0 || p->height == 0) {
      proto_Error(cp, BadValue, 0, 0, X_CreatePixmap);
      return -1;
   }
   for (i=0; i<vscreen.ndepths; i++)
      if (vscreen.dv[i].depth == p->depth)
         break;
   if (i == vscreen.ndepths) {		/* depth is not supported */
      proto_Error(cp, BadValue, p->depth, 0, X_CreatePixmap);
      return -1;
   }
   if (MALLOC(pxp, pixmap_t *, sizeof(pixmap_t))) {
      proto_Error(cp, BadAlloc, 0, 0, X_CreatePixmap);
      return -1;
   }
   if ((pxp->vid = hash_add_client_id(p->pid, (char *)pxp)) == 0) {
      free(pxp);
      proto_Error(cp, BadIDChoice, p->pid, 0, X_CreatePixmap);
      return -1;
   }
   pxp->dwb.res.client = cp;
   pxp->dwb.type = W_PIXMAP;
   pxp->dwb.depth = p->depth;
   pxp->dwb.width = p->width;
   pxp->dwb.height = p->height;
   pxp->cid = p->pid;
   pxp->refs = 1;
   pxp->cached = 0;
   pxp->imp = 0;

   pxp->last = 0;
   pxp->next = pixmaps;
   if (pxp->next)
      pxp->next->last = pxp;
   pixmaps = pxp;

   cp->refs++;
   return 0;
}

/************************************************************************
*									*
*   pixmap_free								*
*									*
************************************************************************/
int
pixmap_free
   AL((cp, p))
   DB client_t *cp
   DD xResourceReq *p
   DE
{
   pixmap_t *pxp;

   if ((pxp = (pixmap_t *)hash_data(vmap, p->id)) == 0) {
      proto_Error(cp, BadPixmap, p->id, 0, X_FreePixmap);
      return -1;
   }
   return free_pixmap(pxp);
}

/************************************************************************
*									*
*   pixmap_client_death							*
*									*
************************************************************************/
void
pixmap_client_death
   AL((client))
   DB client_t *client
   DE
{
   register rid_t cid;
   register pixmap_t *pxp, *npxp;

   for (npxp=pixmaps; pxp=npxp;) {
      npxp = npxp->next;
      if (client_cmp(client, pxp->dwb.res.client)) {
         cid = pxp->cid;
         if (free_pixmap(pxp) == 0)
            proto_FreePixmap(cid);
         else {
            client->refs--;
            pxp->dwb.res.client = 0;
         }
      }
   }
}

/************************************************************************
*									*
*   pixmap_ref								*
*   pixmap_deref							*
*   pixmap_assign							*
*									*
************************************************************************/
void
pixmap_ref
   AL((pxp))
   DB pixmap_t *pxp
   DE
{
   pxp->refs++;
}

void
pixmap_deref
   AL((pxp))
   DB pixmap_t *pxp
   DE
{
   register chunk_t *chp;
 
   if (pxp->refs == 1) {
      zb_dest(ZD_ALLSERV, 0);
      proto_FreePixmap(pxp->cid);
   }
   free_pixmap(pxp);
}

void
pixmap_assign
   AL((pxpp, pxp))
   DB pixmap_t **pxpp
   DD pixmap_t *pxp
   DE
{
   if (*pxpp != pxp) {  
      if (*pxpp)
         pixmap_deref(*pxpp);
      if (pxp)
         pixmap_ref(pxp);
      *pxpp = pxp;
   }
}

/************************************************************************
*									*
*   pixmap_ketchup							*
*									*
************************************************************************/
void
pixmap_ketchup
   VOID
{
   register pixmap_t *pxp;

   for (pxp=pixmaps; pxp; pxp=pxp->next)
      proto_CreatePixmap(	pxp->cid,
				vscreen.shellroot,
				pxp->dwb.width,
				pxp->dwb.height,
				pxp->dwb.depth);
}

/************************************************************************
*									*
*   pixmap_reset							*
*									*
************************************************************************/
void
pixmap_reset
   VOID
{
   register rid_t cid;
   register pixmap_t *pxp, *npxp;

   for (npxp=pixmaps; pxp=npxp;) {
      npxp = npxp->next;
      cid = pxp->cid;
      if (free_pixmap(pxp) == 0)
         proto_FreePixmap(cid);
   }
}

/************************************************************************
*									*
*   pixmap_put_requests							*
*									*
*	Generate a list of chunks containing X protocol requests	*
*	to draw the contents of all cached pixmaps.			*
*									*
*	Caching is not yet implemented.					*
*									*
************************************************************************/
int
pixmap_put_requests
   AL((bp, chpp))
   DB buffer_t *bp
   DD chunk_t **chpp
   DE
{
   return 0;
}

/************************************************************************
*									*
*   pixmap_get_images							*
*									*
*	Generate X protocol requests to retrieve all pixmap images	*
*	not currently cached.						*
*									*
*	Some images are too big to fit in a single request, so they	*
*	are broken up into regions and transmitted in separate		*
*	requests.							*
*									*
************************************************************************/
void
pixmap_get_images
   AL((dest))
   DB int dest
   DE
{
   register pixmap_t *pxp;
   register pp_t *pp;
   register u16_t totwidth, totheight;
   u16_t width, savewidth, height;

   for (pxp=pixmaps; pxp; pxp=pxp->next)
      if (pxp->cached == 0) {
         width = pxp->dwb.width;
         height = pxp->dwb.height;
         image_bite_size(pxp->dwb.depth, &width, &height);
         savewidth = width;

         for (totheight=0; totheight < pxp->dwb.height; totheight+=height) {
            if (totheight + height > pxp->dwb.height)
               height = pxp->dwb.height - totheight;

            width = savewidth;
            for (totwidth=0; totwidth < pxp->dwb.width; totwidth+=width) {
               if (totwidth + width > pxp->dwb.width)
                  width = pxp->dwb.width - totwidth;

               proto_GetImage(	ZPixmap,
				pxp->cid,
				(s16_t)totwidth, (s16_t)totheight,
				width, height,
				(mask_t)( (1<<pxp->dwb.depth) - 1));
               df_put_i(dest);
               df_put_p((void *)pxp);
               df_put_i((int)totwidth);		/* x */
               df_put_i((int)totheight);	/* y */
               df_put_i((int)width);		/* width */
               df_put_i((int)height);		/* height */
               rx_queue(zrxqp, X_GetImage, 0, zb_seqno(), pixmap_image_reply);
            }
         }
      }
}

/************************************************************************
*									*
*   pixmap_image_reply							*
*									*
*	Respond to an incoming GetImageReply by formulating a		*
*	PutImage of the same, and setting up for the arrival of the	*
*	actual image data.						*
*									*
************************************************************************/
void
pixmap_image_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   register int i;
   register pp_t *pp;
   register int dest = (int)df_get_i(dfp);
   register pixmap_t *pxp = (pixmap_t *)df_get_p(dfp);
   register s16_t x = (s16_t)df_get_i(dfp);
   register s16_t y = (s16_t)df_get_i(dfp);
   register u16_t width = (u16_t)df_get_i(dfp);
   register u16_t height = (u16_t)df_get_i(dfp);
   register xGetImageReply *rp = (xGetImageReply *)buf_data(chp);
   register rid_t gid;

   if (rp->type == X_Error) {
      warn("pixmap_image_reply: request failed\n");
      dprx_error((xError *)rp);
      return;
   }
   /*
   **   find an appropriate gc
   */
   for (i=0; i<vscreen.ndepths; i++)
      if (vscreen.dv[i].depth == pxp->dwb.depth) {
         gid = vscreen.dv[i].gc;
         break;
      }

   zb_dest(dest, 0);

   proto_PutImage(ZPixmap, pxp->cid, gid, x,y, width,height, 0, pxp->dwb.depth);
   pp = pp_image(rp->length * 4, sp->smap.iordv[pxp->dwb.depth], width,
	 							pxp->dwb.depth);
   pp_assign(sp->pp, pp);		/* use pp for incoming data */

   df_put_i(dest);
   rx_push(	zrxqp,
		X_GetImage,
		0,
		seqno,
		pixmap_image_reply_data);
   zb_queue(pp);			/* ...and outgoing tag! */
}

/************************************************************************
*									*
*   pixmap_image_reply_data						*
*									*
*	Take all or part of the image returned as part of a GetImage	*
*	reply and send it to the given destination.  (as part of a	*
*	partial PutImage of the same image).				*
*									*
************************************************************************/
void
pixmap_image_reply_data
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   register int dest = (int)df_get_i(dfp);

   /*
   **  lpp of incoming chunk matches the tpp of previous outgoing
   **  chunk, so we can just turn around and send it out.  the server
   **  is already tagged with tpp (if set, for the next incoming
   **  packet).
   */
   if (chp->tpp) {
      df_put_i(dest);
      rx_push(zrxqp, X_GetImage, 0, seqno, pixmap_image_reply_data);
   }
   zb_dest(dest, 0);
   zb_queue_chunk(chp);
}

/************************************************************************
*									*
*   pixmap_set_image							*
*									*
************************************************************************/
int
pixmap_set_image
   AL((pxp, iord, depth, width, height, data))
   DB pixmap_t *pxp
   DD iord_t iord
   DD u8_t depth
   DD u16_t width
   DD u16_t height
   DD char *data
   DE
{
   register int len;

   if (pxp->imp)
      pixmap_free_image(pxp);

   if (MALLOC(pxp->imp, image_t *, sizeof(image_t)))
      return -1;

   pxp->imp->iord = iord;

   len = image_length(iord, width, height, 0, depth);

   if (MALLOC(pxp->imp->data, char *, len)) {
      pixmap_free_image(pxp);
      return -1;
   }
   bcopy(data, pxp->imp->data, len);

   return 0;
}

/************************************************************************
*									*
*   pixmap_free_image							*
*									*
************************************************************************/
void
pixmap_free_image
   AL((pxp))
   DB pixmap_t *pxp
   DE
{
   if (pxp->imp->data)
      free(pxp->imp->data);
   free(pxp->imp);
   pxp->imp = 0;
}

static int
free_pixmap
   AL((pxp))
   DB pixmap_t *pxp
   DE
{
   if (--pxp->refs == 0) {
      if (pxp->last)
         pxp->last->next = pxp->next;
      if (pixmaps == pxp)
         pixmaps = pxp->next;
      if (pxp->next)
         pxp->next->last = pxp->last;
      hash_mark(vmap, pxp->cid);
      pxp->dwb.res.client->refs--;
      if (pxp->imp)
         pixmap_free_image(pxp);
      free(pxp);
      return 0;
   }
   return -1;
}
