/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   prop.c								*
*									*
************************************************************************/
#include <X11/X.h>
#define NEED_EVENTS
#include <X11/Xproto.h>
#include <X11/Xatom.h>
#include "xmx.h"
#include "df.h"
#include "cx.h"
#include "incl/prop.pvt.h"

typedef struct _prop_t {
   atom_t		name;
   atom_t		type;
   u16_t		format;
   u16_t		nbytes;
   u16_t		sz;
   char *		dp;
   struct _prop_t *	next;
}prop_xxx;	/* prop_t defined in xmx.h */

/************************************************************************
*									*
*   prop_change								*
*									*
*	Note: property data is always allocated in multiples of 4	*
*	bytes.								*
*									*
************************************************************************/
int
prop_change
   AL((cp, windowId, name, type, format, mode, nunits, dp))
   DB client_t *cp
   DD rid_t windowId
   DD atom_t name
   DD atom_t type
   DD int format
   DD int mode
   DD int nunits
   DD char *dp
   DE
{
   register int i, sz, nbytes;
   register char *strp, *tp;
   register window_t *wp;
   register prop_t *pp, *lp = 0;

   if (		(wp = (window_t *)hash_data(vmap, windowId)) == 0 ||
		wp->dwb.type != W_WINDOW) {
      proto_Error(cp, BadWindow, windowId, 0, X_ChangeProperty);
      return -1;
   }
   for (pp=wp->props; pp; lp=pp, pp=pp->next)
      if (name == pp->name)
         break;

   if (pp == 0) {
      if (MALLOC(pp, prop_t *, sizeof(prop_t)))
         return -1;
      pp->name = name;
      pp->type = type;
      pp->format = format;
      pp->nbytes = 0;
      pp->sz = 0;
      pp->dp = 0;
      pp->next = 0;

      if (lp)
         lp->next = pp;
      else
         wp->props = pp;
   }
   nbytes = format ? nunits * (format/8) : 0;
   sz = (mode == PropModeReplace) ? nbytes : nbytes + pp->nbytes;
   sz = RUP(sz, 4);
   if (sz > (int)pp->sz) {
      if (MALLOC(strp, char *, sz))
         return -1;
      pp->sz = sz;
      if (pp->dp) {
         if (mode == PropModePrepend) {
            if (pp->nbytes)
               bcopy(pp->dp, strp + nbytes, pp->nbytes);
         }
         else if (mode == PropModeAppend)
            if (pp->nbytes)
               bcopy(pp->dp, strp, pp->nbytes);

         free(pp->dp);
      }
      pp->dp = strp;
   }
   else if (mode == PropModePrepend && pp->nbytes) {
      strp = pp->dp + pp->nbytes - 1;
      tp = pp->dp + pp->nbytes + nbytes - 1;
      for (i=pp->nbytes; i; i--)
         *tp-- = *strp--;
   }
   if (mode == PropModeAppend)
      bcopy(dp, pp->dp + pp->nbytes, nbytes);
   else
      bcopy(dp, pp->dp, nbytes);

   pp->nbytes = (mode == PropModeReplace) ? nbytes : pp->nbytes + nbytes;
   pp->type = type;
   pp->format = format;

   time_update(0);
   event_PropertyNotify(wp, name, time_stamp(vtime), PropertyNewValue);

   return 0;
}

/************************************************************************
*									*
*   prop_delete								*
*									*
************************************************************************/
void
prop_delete
   AL((cp, windowId, name))
   DB client_t *cp
   DD rid_t windowId
   DD atom_t name
   DE
{
   register window_t *wp;
   register prop_t *pp, *np, *lp = 0;

   if (		(wp = (window_t *)hash_data(vmap, windowId)) == 0 ||
		wp->dwb.type != W_WINDOW) {
      proto_Error(cp, BadWindow, windowId, 0, X_DeleteProperty);
      return;
   }
   for (np=wp->props; pp=np; lp=pp) {
      np = np->next;
      if (name == pp->name) {
         time_update(0);
         event_PropertyNotify(wp, name, time_stamp(vtime), PropertyDelete);

         if (lp)
            lp->next = pp->next;
         else
            wp->props = pp->next;

         if (pp->dp)
            free(pp->dp);

         free(pp);
      }
   }
}

/************************************************************************
*									*
*   prop_get								*
*									*
************************************************************************/
void
prop_get
   AL((cp, windowId, atom, type, offlong, nlong, delete))
   DB client_t *cp
   DD rid_t windowId
   DD atom_t atom
   DD atom_t type
   DD int offlong
   DD int nlong
   DD int delete
   DE
{
   register int after;
   register int off, nbytes;
   register window_t *wp;
   register prop_t *pp, *lp = 0;

   if (		(wp = (window_t *)hash_data(vmap, windowId)) == 0 ||
		wp->dwb.type != W_WINDOW) {
      proto_Error(cp, BadWindow, windowId, 0, X_GetProperty);
      return;
   }
   for (pp=wp->props; pp; lp=pp, pp=pp->next)
      if (atom == pp->name) {
         if (type == AnyPropertyType || type == pp->type) {
            off = offlong * 4;
            nbytes = nlong * 4;
            nbytes = MIN((int)pp->nbytes - off, nbytes);

            if (nbytes < 0)		/* return a Value error */
               proto_Error(cp, BadValue, offlong, 0, X_GetProperty);

            else {			/* match - return propery */
               proto_GetPropertyReply(	cp,
					pp->type,
					pp->format,
					after = pp->nbytes - off - nbytes,
					nbytes,
					pp->dp + off);

               if (delete && after == 0) {	/* delete if no bytes after */
                  time_update(0);
                  event_PropertyNotify(wp, atom, time_stamp(vtime),
								PropertyDelete);
                  if (lp)
                     lp->next = pp->next;
                  else
                     wp->props = pp->next;

                  if (pp->dp)
                     free(pp->dp);

                  free(pp);
               }
            }
         }
         else				/* type mismatch, return type */
            proto_GetPropertyReply(	cp,
					pp->type,
					pp->format,
					pp->nbytes,
					0, 0);
         return;
      }
   if (atom_name(atom, 0, 0))
      proto_Error(cp, BadAtom, atom, 0, X_GetProperty);
   else
      proto_GetPropertyReply(cp, 0, 0, 0, 0, 0);
}

/************************************************************************
*									*
*   prop_wm_name							*
*									*
*	Return a static copy of the WM_NAME of a window.  For		*
*	debugging only.							*
*									*
************************************************************************/
char *
prop_wm_name
   AL((wp))
   DB window_t *wp
   DE
{
   register int len;
   register prop_t *pp;
   static char buf[64];

   if (wp == 0)
      return "<null>";
   if (wp == vscreen.wp)
      return "<root>";
   for (pp=wp->props; pp; pp=pp->next)
      if (pp->name == XA_WM_NAME) {
	 len = MIN(pp->nbytes, 63);
         bcopy(pp->dp, buf, len);
	 buf[pp->nbytes] = '\0';
	 return buf;
      }
   return "<no name>";
}

/************************************************************************
*									*
*   prop_rotate								*
*									*
************************************************************************/
void
prop_rotate
   AL((cp, windowId, delta, n, names))
   DB client_t *cp
   DD rid_t windowId
   DD int delta
   DD int n
   DD atom_t *names
   DE
{
   register int i, j, l;
   register int found;
   register window_t *wp;
   register prop_t *pp, *last;

   if (		(wp = (window_t *)hash_data(vmap, windowId)) == 0 ||
		wp->dwb.type != W_WINDOW) {
      proto_Error(cp, BadWindow, windowId, 0, X_RotateProperties);
      return;
   }
   pp = 0;
   for (i=0; i<n; i++) {	/* check for errors */
      for (j=0; j<n; j++)
         if (i!=j && names[i] == names[j]) {
            /* Match error */; 
            return;
         }
      found = 0;
      last = pp;
      for (pp=pp?pp->next:wp->props; pp != last; pp=pp?pp->next:wp->props)
         if (pp && names[i] == pp->name) {
            found = 1;
            break;
         }
      if (!found) {
         /* Match error */;
         return;
      }
   }
   i = 0;
   j = n;
   time_update(0);
   for (pp=wp->props; pp; pp=pp->next) {	/* okay, rotate */
      l = i;
      do {
         if (names[i] == pp->name) {
            pp->name = names[(i+delta)%n];
            event_PropertyNotify(wp, pp->name, time_stamp(vtime),
							PropertyNewValue);
            j--;
            break;
         }
         i = ( i + 1) % n;
      } while (i != l);
   }
}

/************************************************************************
*									*
*   prop_free								*
*									*
************************************************************************/
void
prop_free
   AL((wp))
   DB window_t *wp
   DE
{
   register prop_t *pp, *np;

   for (np=wp->props; pp=np;) {
      np = np->next;
      if (pp->dp)
         free(pp->dp);
      free(pp);
   }
}

/************************************************************************
*									*
*   prop_list								*
*									*
************************************************************************/
void
prop_list
   AL((cp, windowId))
   DB client_t *cp
   DD rid_t windowId
   DE
{
   register window_t *wp;
   register prop_t *pp;
   register list_t *lstp;

   if (		(wp = (window_t *)hash_data(vmap, windowId)) == 0 ||
		wp->dwb.type != W_WINDOW) {
      proto_Error(cp, BadWindow, windowId, 0, X_ListProperties);
      return;
   }
   lstp = list_new();

   for (pp=wp->props; pp; pp=pp->next) {
      list_put(lstp, pp->name);
   }
   proto_ListPropertiesReply(cp, lstp);
   list_free(lstp);
}

/************************************************************************
*									*
*   prop_print								*
*									*
************************************************************************/
void
prop_print
   AL((wp))
   DB window_t *wp
   DE
{
   register prop_t *pp;
   register u8_t *cp;
   register u16_t *sp;
   register u32_t *lp;

   for (pp=wp->props; pp; pp=pp->next) {
      printf("name(%d) type(%d) format(%d) nbytes(%d) sz(%d)\n",
			pp->name, pp->type, pp->format, pp->nbytes, pp->sz);
      if (pp->nbytes)
         if (pp->type == 31)	/* STRING */
            printf("   ->\"%s\"\n", (char *)pp->dp);
         else if (pp->format == 8) {
            cp = (u8_t *)pp->dp;
            printf("   ->%x %x %x %x %x ...\n",cp[0],cp[1],cp[2],cp[3],cp[4]);
         }
         else if (pp->format == 16) {
            sp = (u16_t *)pp->dp;
            printf("   ->%x %x %x %x %x ...\n",sp[0],sp[1],sp[2],sp[3],sp[4]);
         }
         else if (pp->format == 32) {
            lp = (u32_t *)pp->dp;
            printf("   ->%x %x %x %x %x ...\n",lp[0],lp[1],lp[2],lp[3],lp[4]);
         }
   }
}
