/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   ptc.c								*
*									*
*	Protocol translation code module.  Routines to create, manage	*
*	and apply X protocol translation "programs."			*
*									*
************************************************************************/
#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/X.h>
#include <X11/Xproto.h>
#include <xmc.h>
#include "xmx.h"
#include "df.h"
#include "ptc.h"
#include "incl/ptc.pvt.h"

static ptc_t *ptcfreelist;
static char *blkfreelist;

/************************************************************************
*									*
*  ptc_set								*
*									*
*	Set the type of the current "open" chunk of the given buffer	*
*	to type "request" and make a new ptc struct its meta-data.	*
*									*
************************************************************************/
ptc_t *
ptc_set
   AL((bp, clinum, seqno))
   DB buffer_t *bp
   DD u8_t clinum
   DD u16_t seqno
   DE
{
   register chunk_t *chp = buf_chunk(bp);
   register ptc_t *ptcp;

   if (chp->type != P_NONE) {
      warn("ptc_set: chunk has type %s, discarding...\n",
					debug_proto_type_str(chp->type));
      if (buf_active(bp)) {
         buf_clear(buf_split(bp, 0));
         chp = buf_chunk(bp);
      }
   }
   ptcp = new_ptc();
   ptcp->count = 0;
   ptcp->clinum = clinum;
   ptcp->seqno = seqno;

   chp->type = P_REQUEST;
   chp->dptr = (void *)ptcp;

   return ptcp;
}

void
ptc_unset
   AL((chp))
   DB chunk_t *chp
   DE
{
   if (chp->type == P_REQUEST) {
      chp->type = P_NONE;
      free_ptc((ptc_t *)chp->dptr);
   }
}

void
ptc_done
   AL((chp))
   DB chunk_t *chp
   DE
{
   PTC_DONE(((ptc_t *)(chp->dptr)));
}

/************************************************************************
*									*
*   ptc_apply								*
*									*
*	This is a byte-code interpreter.  It reads and executes ptc	*
*	instructions stored as a chunk's meta-data.  The "program"	*
*	contains instructions and data and operates implicitely on	*
*	the chunk to which it is attached.				*
*									*
*	Ptc programs are idempotent.  They have no side effects and	*
*	may well be executed again and again on the same chunk, even	*
*	repeatedly for the same destination.				*
*									*
************************************************************************/
void
ptc_apply
   AL((chp, sp))
   DB chunk_t *chp
   DD server_t *sp
   DE
{
   register uint_t pc;
   register mask_t base;
   register int shift;
   register char *dp, *blkp;
   register ptc_t *ptcp;

   DEBUG3(D_PTC, "ptc_apply: chunk [0x%x] server [%d] [%s]\n",
							chp, sp->fd, sp->tag);
   ptcp = (ptc_t *)chp->dptr;
   dp = buf_data(chp);
   blkp = ptcp->head;
   /*
   **  cache these because client rids are the most common operation
   */
   base = sp->smap.base;
   shift = sp->smap.shift;

   pc = 0;

   while (1) {
      switch (blkp[pc]) {
         case PTC_DONE_OP: {
            return;
         }
         case PTC_SKIP_OP: {
            register op_skip_t *op = (op_skip_t *)&blkp[pc];

            pc += op->count;
            break;
         }
         case PTC_JUMP_OP: {
            register op_jump_t *op = (op_jump_t *)&blkp[pc];

            blkp = op->blkp;
            pc = 0;
            break;
         }
         case PTC_ATOM_OP: {
            register op_atom_t *op = (op_atom_t *)&blkp[pc];

            *(atom_t *)&dp[op->offset] = am_map(sp->smap.amp, op->atom);
            pc += sizeof(op_atom_t);
            break;
         }
         case PTC_CLIENT_RID_OP: {
            register op_rid_t *op = (op_rid_t *)&blkp[pc];

            *(rid_t *)&dp[op->offset] = (op->rid << shift) | base;
            pc += sizeof(op_rid_t);
            break;
         }
         case PTC_SERVER_RID_OP: {
            register op_rid_t *op = (op_rid_t *)&blkp[pc];

            *(rid_t *)&dp[op->offset] = hash_map(sp->smap.mp, op->rid);
            pc += sizeof(op_rid_t);
            break;
         }
         case PTC_EVENT_MASK_OP: {
            register op_mask_t *op = (op_mask_t *)&blkp[pc];

            if (op->mask & IsShellMask)
               *(mask_t *)&dp[op->offset] = IE_SHELLSET;
            else {
               switch (sp->mode) {
                  case Floor:
                     *(mask_t *)&dp[op->offset] =
			(op->mask & IE_FLOORALLOW) | IE_FLOORSET;
                     break;
                  case Seat:
                     *(mask_t *)&dp[op->offset] =
			(op->mask & IE_SEATALLOW) | SeatMaskUnshift(op->mask);
                     break;
                  case View:
                     *(mask_t *)&dp[op->offset] =
			(op->mask & IE_VIEWALLOW) | IE_VIEWSET;
                     break;
               }
               if (op->mask & IsRootMask)
                  *(mask_t *)&dp[op->offset] |= IE_ROOTSET;
            }
            pc += sizeof(op_mask_t);
            break;
         }
         case PTC_ROOT_PIXEL_OP: {
            register op_pixel_t *op = (op_pixel_t *)&blkp[pc];
            /*
            **  pixels can range outside allocated values,
            **  so the pixel mapping table is protected here
            */
            if (	vscreen.wp->mp->mappixels &&
			op->pixel < vscreen.wp->mp->vp->nentries)
               *(pixel_t *)&dp[op->offset] =
				sp->smap.root.pmap[op->pixel];
            else
               *(pixel_t *)&dp[op->offset] = op->pixel;
            pc += sizeof(op_pixel_t);
            break;
         }
         case PTC_IMAGE_LEN_OP : {
            register op_imagelen_t *op = (op_imagelen_t *)&blkp[pc];
            register int depth;

            depth = op->format == ZPixmap ? op->depth : 1;
            *(u16_t *)&dp[op->offset] = sz_xPutImageReq / 4 +
					image_length(	sp->smap.iordv[depth],
							op->width,
							op->height,
							op->leftPad,
							op->depth);
            pc += sizeof(op_imagelen_t);
            break;
         }
         case PTC_G_EXPOSE_OP: {
            register op_gexpose_t *op = (op_gexpose_t *)&blkp[pc];

               /*
               ** This is not exactly right, but it minimizes the
               ** number of GraphicsExpose and NoExpose events that are
               ** generated.
               **
               ** Basically, we only request the events from the current
               ** query server and ignore all the others.
               */
            *(u32_t *)&dp[op->offset] = (sp == qservp) ? 1 : 0;
            pc += sizeof(op_gexpose_t);
            break;
         }
         case PTC_EXTENSION_OP: {
            register op_extension_t *op = (op_extension_t *)&blkp[pc];
            ext_ptc(op->code, op, sp, chp);
            pc += sizeof(op_extension_t);
            break;
         }
         default:
            quit(-1, "ptc_apply: bad opcode [%d]\n", blkp[pc]);
      }
   }
}

/*
**   new_ptc
**
**	Ptc's always have a block.
*/
static ptc_t *
new_ptc
   VOID
{
   register ptc_t *ptcp;

   if (ptcfreelist) {
      ptcp = ptcfreelist;
      ptcfreelist = ptcp->next;
   }
   else if (MALLOC(ptcp, ptc_t *, sizeof(ptc_t)))
      quit(-1, "new_ptc: fatal\n");
   else {
      ptcp->head = ptcp->tail = ptc_new_block();
   }
   ptcp->pc = 0;
   ptcp->count = 0;

   return ptcp;
}

/*
**   free_ptc
**
**	Free all but head block first.
*/
static void
free_ptc
   AL((ptcp))
   DB ptc_t *ptcp
   DE
{
   register char *blkp, *nblkp;

   blkp = ptcp->head;
   while (blkp != ptcp->tail) {
      nblkp = ((op_jump_t *)&blkp[PTC_LASTPC])->blkp;
      free_block(blkp);
      blkp = nblkp;
   }
   ptcp->head = ptcp->tail;

   ptcp->next = ptcfreelist;
   ptcfreelist = ptcp;
}


/************************************************************************
*									*
*  ptc_new_block							*
*									*
************************************************************************/
char *
ptc_new_block
   VOID
{
   register char *blkp;

   if (blkfreelist) {
      blkp = blkfreelist;
      blkfreelist = *(char **)blkp;
   }
   else if (MALLOC(blkp, char *, PTC_BLOCKSZ))
      quit(-1, "(ptc)new_block: fatal\n");

   return blkp;
}

/*
**  free_block
*/
static void
free_block
   AL((blkp))
   DB char *blkp
   DE
{
   *(char **)blkp = blkfreelist;
   blkfreelist = blkp;
}
