/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*	queue.c								*
*									*
*	Managed queues of buffer chunks.				*
*									*
************************************************************************/
#include <sys/time.h>
#include <sys/types.h>
#include <sys/uio.h>
#ifdef _AIX
#include <sys/select.h>
#endif
#include <netinet/in.h>		/* ip.h needs this */
#include <netinet/in_systm.h>	/* ...and this */
#include <netinet/ip.h>		/* for IP_MAXPACKET */
#include <errno.h>

#include <X11/X.h>
#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/Xproto.h>

#include <xmcp.h>

#include "xmx.h"
#include "df.h"
#include "cx.h"
#include "rx.h"
#include "res.h"
#include "ptc.h"
#include "rs.h"
#include "es.h"
#include "incl/queue.pvt.h"
#include "fd.h"

/*
**  IOV_MAX isn't defined under SunOS5.[34], but 16 is the proper value
*/
#ifndef IOV_MAX
#define IOV_MAX	16
#endif

#ifndef IP_MAXPACKET
#define IP_MAXPACKET	8192
#endif

#ifdef HIGHER
#define QHIGHWATER	(IP_MAXPACKET - (IP_MAXPACKET / 8))
#else
#define QHIGHWATER	(IP_MAXPACKET)
#endif

#define QINC	32

static queue_t *qfree;

/************************************************************************
*									*
*   queue_hold								*
*									*
************************************************************************/
void
queue_hold
   AL((qp))
   DB queue_t *qp
   DE
{
   qp->hold++;
   if (qp->qnel)
      fd_wclear(qp->fd);
}

/************************************************************************
*									*
*   queue_release							*
*									*
************************************************************************/
void
queue_release
   AL((qp))
   DB queue_t *qp
   DE
{
   switch (qp->hold) {
      case 0:
         break;
      case 1:
         qp->hold = 0;
         if (qp->qnel) {
            fd_write(qp->fd);
            if (qpoll(qp))
               queue_flush(qp);
         }
         break;
      default:
         qp->hold--;
         break;
   }
}

/************************************************************************
*									*
*   queue_new								*
*									*
************************************************************************/
queue_t *
queue_new
   AL((dest, dp))
   DB u16_t dest
   DD char *dp
   DE
{
   queue_t *qp;

   if (qfree) {
      qp = qfree;
      qfree = qfree->next;
   }
   else
      if (MALLOC(qp, queue_t *, sizeof(queue_t)))
         return 0;

   switch (qp->dest = dest) {
      case Q_XSERVER:
         qp->dst.sp = (server_t *)dp;
         qp->fd = ((server_t *)dp)->fd;
         break;
      case Q_XCLIENT:
         qp->dst.cp = (client_t *)dp;
         qp->fd = ((client_t *)dp)->fd;
         break;
      case Q_XMCCLIENT:
         qp->dst.xp = (xmc_t *)dp;
         qp->fd = ((xmc_t *)dp)->fd;
         break;
      default:
         qp->dst.np = (char *)0;
         break;
   }
   qp->hold = 0;
   qp->nbytes = 0;
   qp->pp = 0;
   qp->off = 0;
   qp->qnel = 0;
   qp->wnel = 0;
   qp->qsz = 0;
   qp->wsz = 0;
   qp->queue = 0;
   qp->wait = 0;
   qp->next = 0;

   return qp;
}

/************************************************************************
*									*
*   queue_free								*
*									*
************************************************************************/
void
queue_free
   AL((qp))
   DB queue_t *qp
   DE
{
   register int i;

   if (qp->pp)
      pp_free(qp->pp);

   for (i=0; i<(int)qp->qnel; i++)
      buf_clear(qp->queue[i]);

   for (i=0; i<(int)qp->wnel; i++)
      buf_clear(qp->wait[i]);

   qp->next = qfree;
   qfree = qp;
}

/************************************************************************
*									*
*   queue_free_freelist							*
*									*
************************************************************************/
void
queue_free_freelist
   VOID
{
   register queue_t *qp, *last;

   for (qp=qfree; last=qp;) {
      qp = qp->next;
      free(last);
   }
   qfree = 0;
}

/************************************************************************
*									*
*   queue_client_reply							*
*									*
************************************************************************/
void
queue_client_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   if (chp->tpp)	/* if more, push this routine to handle it */
      rx_push(cp->rxqp, major, minor, seqno, queue_client_reply);

   queue_add(cp->qp, chp);
}

/************************************************************************
*									*
*   queue_allserv							*
*									*
************************************************************************/
int
queue_allserv
   AL((chp))
   DB chunk_t *chp
   DE
{
   register int i, n;

   n = 0;
   for (i=0; i<num_serv; i++)
      if (servers[i]->state == S_READY) {
         queue_add(servers[i]->qp, chp);
         n++;
      }
   return n;
}

/************************************************************************
*									*
*   queue_allbutoneserver						*
*									*
************************************************************************/
void
queue_allbutoneserver
   AL((sp, chp))
   DB server_t *sp
   DD chunk_t *chp
   DE
{
   register int i;

   for (i=0; i<num_serv; i++)
      if (servers[i] != sp && servers[i]->state == S_READY)
         queue_add(servers[i]->qp, chp);
}

/************************************************************************
*									*
*   queue_add								*
*									*
*	Add chunk to queue.  If the last chunk added was incomplete,	*
*	and the new chunk does not complete it, put it on the queue's	*
*	waitlist.  If the new chunk completes it, queue it and queue	*
*	as many chunks from the waitlist as possible.			*
*									*
************************************************************************/
void
queue_add
   AL((qp, chp))
   DB queue_t *qp
   DD chunk_t *chp
   DE
{
   register int i, j, skip;
   register chunk_t **chpp;

   chp->refs++;
   if (qp->pp != chp->lpp) {		/* defer chunk to waitlist */
      if (qp->wnel == qp->wsz) {
         if (MALLOC(chpp, chunk_t **, (qp->wsz+QINC) * sizeof(chunk_t *)))
            return;	/* this is catastrophic (chp->refs is wrong) */
         qp->wsz += QINC;
         if (qp->wnel)
            bcopy((char *)qp->wait, (char *)chpp, qp->wnel * sizeof(chunk_t *));
         if (qp->wait)
            free(qp->wait);
         qp->wait = chpp;
      }
      qp->wait[qp->wnel++] = chp;
   }
   else {					/* queue it */
      add_chunk(qp, chp);	/* chp may be freed here */
      if (qp->wnel) {			/* queue from waitlist, if possible */
         /*
         **  Queue chunks in tag-match order primarily, and
         **  queue order secondarily.  Searches the entire waitlist
         **  for tag matches, but otherwise honors the queue order.
         */
         skip = -1;
         for (i=0; i<(int)qp->wnel;) {
            if (qp->wait[i])
               if (qp->pp == qp->wait[i]->lpp) {
                  add_chunk(qp, qp->wait[i]);
                  qp->wait[i] = 0;
                  /*
                  **  If no particular leading tag is needed, and we
                  **  skipped some chunks, go back to them.
                  */
                  if (qp->pp == 0 && skip >= 0) {
                     i = skip;
                     skip = -1;
                     continue;
                  }
               }
               else if (skip < 0)
                  skip = i;
            i++;
         }
         for (i=j=0; i<(int)qp->wnel; i++)	/* slide waitlist left */
            if (qp->wait[i])
               qp->wait[j++] = qp->wait[i];

         qp->wnel = j;
      }
   }
#ifdef QPOLL_EXPENSIVE
   if (qp->nbytes > QHIGHWATER && qpoll(qp))	/* necessary? */
      queue_flush(qp);
#endif
}

/*
**  add_chunk
**
**	Queue a chunk.
*/
static void
add_chunk
   AL((qp, chp))
   DB queue_t *qp
   DD chunk_t *chp
   DE
{
   register int n;
   register chunk_t **chpp;
   register xGenericReply *rp;
   register xEvent *ep;

#ifdef DEBUG
if (chp->valid == 0) {
   warn("add_chunk: unvalid chunk\n");
   abort();
}
#endif
   /*
   **  queue-time chunk processing
   **
   **  The statements below are guaranteed to be executed exactly once
   **  for each chunk queue event, and for each chunk in exactly the order
   **  in which it will be written to the destination.
   **
   **  Hint: this is where sequence number processing should occur.
   */
   switch (chp->type) {
      case P_NONE:
         break;
      case P_REQUEST:
         ptc_done(chp);
         sm_add(qp->dst.sp->smp, ptc_clinum(chp), ptc_seqno(chp),
							ptc_count(chp));
         break;
      case P_REPLY:
         rp = (xGenericReply *)buf_data(chp);
         qp->dst.cp->oseqno = rp->sequenceNumber;
         break;
      case P_ERROR:
         rp = (xGenericReply *)buf_data(chp);
         qp->dst.cp->oseqno = rp->sequenceNumber;
         break;
      case P_EVENT:
         n = es_get(chp);
         for (ep=(xEvent *)buf_data(chp); n--; ep++) {
            /*
            **  save for actual write, in queue_flush, below
            */
            list_put(qp->dst.cp->eslp, qp->dst.cp->seqno);
         }
         qp->dst.cp->oseqno = qp->dst.cp->seqno;
         break;
      case P_IMAGE:
         break;
      case P_CONNECT:
         break;
      case P_EXT:
         break;
   }
   if (qp->qnel == qp->qsz) {
      if (MALLOC(chpp, chunk_t **, (qp->qsz+QINC) * sizeof(chunk_t *)))
         return;	/* this is catastrophic */
      qp->qsz += QINC;
      if (qp->qnel)
         bcopy((char *)qp->queue, (char *)chpp, qp->qnel * sizeof(chunk_t *));
      if (qp->queue)
         free(qp->queue);
      qp->queue = chpp;
   }
   qp->queue[qp->qnel++] = chp;

   if (qp->qnel == 1 && qp->hold == 0)	/* first one, mark queue ready */
      fd_write(qp->fd);

   if (qp->pp)
      pp_free(qp->pp);
   pp_assign(qp->pp, chp->tpp);

   qp->nbytes += buf_chunksize(chp);
}

/************************************************************************
*									*
*   queue_flush_servers							*
*									*
************************************************************************/
void
queue_flush_servers
   VOID
{
   register int i;
   fd_set wfds;
   static struct timeval tv;	/* static ensures it is zero */

   /*
   **	serverfds is a global defined in fd.h
   */
   bcopy((char *)&serverfds, (char *)&wfds, sizeof(fd_set));
   (void) select(MAXFD+1, 0, fdcast(&wfds), 0, &tv);

   for (i=0; i<num_serv; i++)
      if (FD_ISSET(servers[i]->fd, &wfds))
         queue_flush(servers[i]->qp);
}

/************************************************************************
*									*
*   queue_flush								*
*									*
*	Apply per-destination mappings to queued chunks, assemble them	*
*	into an iovec and write them all at once.  If some data is not	*
*	written, leave it queued for next time.				*
*									*
*	Note that chunks may be assigned to more than one queue, so	*
*	we mustn't muck up the data irretrievably.			*
*									*
*	More chunks may be queued than we can fit in an iovec, so	*
*	there is an ugly goto loop around the thing in case we have	*
*	to break up the write into IOV_MAX size pieces.			*
*									*
************************************************************************/
void
queue_flush
   AL((qp))
   DB queue_t *qp
   DE
{
   register int max;
   register int r;
   register int i, j;
   register int nevents;
   register uint_t nwritten, nbytes, left;
   register chunk_t *chp;
   register xEvent *ep;
   struct iovec iov[IOV_MAX];
   uint_t olen[IOV_MAX];

   if (qp->hold)	/* until released, nothing goes out */
      return;

   if (qp->qnel == 0) {
#ifdef BLORT_TODO
      warn("--flushing an empty queue--[%s]\n", debug_queue_dest_str(qp->dest));
#endif
      return;
   }

   startagain:

   nbytes = 0;
   max = MIN(qp->qnel, IOV_MAX);
   nevents = 0;

   DEBUG3(D_NETWORK, "SOCKET WRITE [%d] [%s] off [%d]{\n",
			qp->fd, debug_queue_dest_str(qp->dest), qp->off);
   for (i=0; i<max; i++) {
      chp = qp->queue[i];
#ifdef DEBUG
if (chp->valid == 0) {
   warn("queue_flush: unvalid chunk\n");
   abort();
}
#endif
#ifdef DEBUG
      if (chp->refs == 0) {
         warn("queue_flush: chunk [0x%x] has no refs\n", chp);
         abort();
      }
#endif
      /*
      **  apply per-destination mappings
      */
      switch (chp->type) {
         case P_NONE:
            break;
         case P_REQUEST:
            ptc_apply(chp, qp->dst.sp);
#ifdef DEBUG
            if (debug)
               if (	debug & D_PROTO2 ||
			debug & D_PROTOq && qp->dst.sp == qservp)
                  dprx_request_chunk(chp, qp->dst.sp->smp);
#endif
            break;
         case P_REPLY:
            D_CALL3(D_PROTO4, dprx_reply, (xGenericReply *)buf_data(chp),
					rs_major(chp), rs_minor(chp));
            if (qp->dst.cp->swap)
               swap_reply((xGenericReply *)buf_data(chp),
					rs_major(chp), rs_minor(chp));
            break;
         case P_ERROR:
            D_CALL1(D_PROTO4, dprx_error, (xError *)buf_data(chp));
            if (qp->dst.cp->swap)
               swap_error((xError *)buf_data(chp));
            break;
         case P_EVENT:
            ep = (xEvent *)buf_data(chp);
            if (nevents++ == 0)
               list_reset(qp->dst.cp->eslp);
            for (j=es_get(chp); j; j--) {
               ep->u.u.sequenceNumber = list_next(qp->dst.cp->eslp);
               D_CALL1(D_PROTO4, dprx_event, ep);
               D_CALL2(D_FOCUS, print_event_time, qp->dst.cp, ep);
               if (qp->dst.cp->swap)
                  swap_event(ep);
               ep++;
            }
            if (qp->dst.cp->swap)
               chp->type = P_EVENT_SWAPPED;
            break;
         case P_EVENT_SWAPPED:
            ep = (xEvent *)buf_data(chp);
            if (nevents++ == 0)
               list_reset(qp->dst.cp->eslp);
            for (j=buf_chunksize(chp)/sz_xEvent; j; j--) {
               if (!qp->dst.cp->swap) {
                  swap_event(ep);
                  ep->u.u.sequenceNumber = list_get(qp->dst.cp->eslp);
                  D_CALL1(D_PROTO4, dprx_event, ep);
               }
               else {
                  ep->u.u.sequenceNumber = list_get(qp->dst.cp->eslp);
                  SWAP2(ep->u.u.sequenceNumber);
                  DEBUG0(D_PROTO4, "<swapped event>\n");
               }
               ep++;
            }
            if (!qp->dst.cp->swap)
               chp->type = P_EVENT;
            break;
         case P_IMAGE:
            chp = image_swap(chp, qp->dest == Q_XSERVER ?
				&qp->dst.sp->smap.iordv[0] : &viordv[0]);
#ifdef DEBUG
            if (debug & D_PROTO4) {
               warn("[CHUNK-> [0x%x] off[%d] sz[%d] refs[%d] bp[0x%x]\n",
				chp, chp->off, chp->sz, chp->refs, chp->bp);
               warn("image, size %d\n", buf_chunksize(chp));
               warn("<-CHUNK]\n");
            }
#endif
#ifdef DEBUG
if (chp->valid == 0) {
   warn("queue_flush: P_IMAGE - unvalid chunk\n");
   abort();
}
#endif
            break;
         case P_CONNECT:
         case P_CONNECT_SWAPPED:
            {
               register xConnSetupPrefix *prefix;
               register xConnSetup *setup;
 
               prefix = (xConnSetupPrefix *)buf_data(chp);
 
               if (prefix->success) {
                  setup = (xConnSetup *)(prefix + 1);
                  if (chp->type == P_CONNECT_SWAPPED) {
                     CPSWAP4(qp->dst.cp->base, setup->ridBase);
                     DEBUG0(D_PROTO4, "<successful conn setup - swapped>\n");
                  }
                  else {
                     setup->ridBase = qp->dst.cp->base;
                     D_CALL1(D_PROTO4, dprx_connsetup, buf_data(chp));
                  }
               }
               else {	/* failures are never pre-swapped */
                  D_CALL1(D_PROTO4, dprx_ConnSetupPrefix,
					(xConnSetupPrefix *)buf_data(chp));
                  if (qp->dst.cp->swap)
                     swap_connsetup(buf_data(chp));
               }
            }
            break;
         case P_EXT:
            break;
         default:
            warn("queue_flush: unknown chunk type <%d>\n", chp->type);
            break;
      }
      if (i == 0 && qp->off) {
         iov[i].iov_base = buf_data(chp) + qp->off;
         iov[i].iov_len = olen[i] = buf_chunksize(chp) - qp->off;
         nbytes += buf_chunksize(chp) - qp->off;
      }
      else {
         iov[i].iov_base = buf_data(chp);
         iov[i].iov_len = olen[i] = buf_chunksize(chp);
         nbytes += buf_chunksize(chp);
      }
   }
   DEBUG1(D_NETWORK, "} SOCKET WRITE [%d]\n", qp->fd);

   i = 0;
   left = nbytes;

   for (;;) {
      if ((r = writev(qp->fd, &iov[i], max - i)) < 0) {
         if (errno == EWOULDBLOCK) {	/* blocked, save unwritten data */
            nwritten = nbytes - left;
            /*
            **	clear fully written chunks first
            */
            for (i=0; i<max && nwritten>=olen[i]; i++) {
               nwritten -= olen[i];
               qp->nbytes -= buf_chunksize(qp->queue[i]);
               /*
               **  this is icky, but it must be done here.
               **  client-specific sequence number mapping data
               **  can only be cleared when that client's queue
               **  reference to the data is decremented, whether
               **  or not the chunk is actually freed.  this is
               **  also done below.
               */
               if (	qp->queue[i]->type == P_EVENT ||
			qp->queue[i]->type == P_EVENT_SWAPPED)
                  list_clear(qp->dst.cp->eslp, es_get(qp->queue[i]));
               buf_clear(qp->queue[i]);		/* might free it */
            }
            if (nwritten)	/* this is approximate, but that's okay */
               qp->nbytes = qp->nbytes > nwritten ? qp->nbytes - nwritten : 0;

            if (i == 0)
               qp->off += nwritten;
            else
               qp->off = nwritten;
            /*
            **	slide unwritten chunks left in queue array
            */
            for (j=0; i<(int)qp->qnel; i++)
               qp->queue[j++] = qp->queue[i];

            qp->qnel = j;
         }
         else {			/* lost connection, clean up */
            switch (qp->dest) {
               case Q_NONE:
                  warn("error!\n");	/* TODO */
                  break;
               case Q_XSERVER:
                  xmc_drop(qp->dst.sp);
                  break;
               case Q_XCLIENT:
                  /*
                  ** It's generally not a good idea to call a routine
                  ** that messes with queues while inside a routine 
                  ** that messes with queues.  But it's necessary here
                  ** and the queues we are writing to (the servers) do
                  ** NOT include the queue we are working on in this
                  ** section of this routine (a client), so this is safe.
                  */
                  cliz_kill_client(qp->dst.cp);
                  break;
               case Q_XMCCLIENT:
                  xmc_close(qp->dst.xp);
                  break;
            }
         }
         break;			/** loop exit **/
      }
      else if (r < (int)left) {			/* partial write */
         left -= r;
         while (iov[i].iov_len <= r)
            r -= iov[i++].iov_len;
         if (r) {
            iov[i].iov_len -= r;
            iov[i].iov_base += r;
         }
      }
      else {					/* completely written */
         for (i=0; i<max; i++) {
            qp->nbytes -= buf_chunksize(qp->queue[i]);
            /* see comment above */
            if (qp->queue[i]->type == P_EVENT ||
		qp->queue[i]->type == P_EVENT_SWAPPED)
               list_clear(qp->dst.cp->eslp, es_get(qp->queue[i]));
            buf_clear(qp->queue[i]);		/* might free it */
         }
         for (j=0; i<(int)qp->qnel; i++)	/* slide array left */
            qp->queue[j++] = qp->queue[i];
         qp->qnel = j;
         qp->off = 0;

         if (qp->qnel)
            goto startagain;	/* so sue me */
         else {
            qp->nbytes = 0;
            fd_wclear(qp->fd);		/* queue is now inactive */
         }

         break;			/** loop exit **/
      }
   }
   D_CALL0(D_VERIFY, buf_verify_chunks);
}

/*
**  qpoll
**
**	can we write to the queue's destination?
*/
static int
qpoll
   AL((qp))
   DB queue_t *qp
   DE
{
   fd_set wfds;
   static struct timeval tv;	/* static ensures it is zero */

   FD_ZERO(&wfds);
   FD_SET(qp->fd, &wfds);
   return (select(qp->fd+1, 0, fdcast(&wfds), 0, &tv) > 0);
}

static void
print_event_time
   AL((cp, ep))
   DB client_t *cp
   DD xEvent *ep
   DE
{
   register u32_t time = 0;

   switch (ep->u.u.type) {
      case KeyPress:
      case KeyRelease:
      case ButtonPress:
      case ButtonRelease:
      case MotionNotify:
         time = ep->u.keyButtonPointer.time;
         break;
      case EnterNotify:
      case LeaveNotify:
         time = ep->u.enterLeave.time;
         break;
      case PropertyNotify:
         time = ep->u.property.time;
         break;
      case SelectionClear:
         time = ep->u.selectionClear.time;
         break;
      case SelectionRequest:
         time = ep->u.selectionRequest.time;
         break;
      case SelectionNotify:
         time = ep->u.selectionNotify.time;
         break;
   }
   if (time)
      warn("sending to cp [%d] event [%s] window [0x%08x] time [%u]\n",
	cp->fd, dprx_event_str(ep->u.u.type), dprx_event_window(ep), time);
}
