/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   request.c								*
*									*
************************************************************************/
#include <sys/time.h>
#include <sys/types.h>
#include <sys/uio.h>
#ifdef _AIX
#include <sys/select.h>
#endif
#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/Xproto.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>

#include "xmx.h"
#include "df.h"
#include "cx.h"
#include "rx.h"
#include "res.h"
#include "ptc.h"
#include "ptc_x.h"
#include "kpm.h"
#include "incl/request.pvt.h"
#include "fd.h"

/************************************************************************
*									*
*   request_process							*
*									*
*	Process a chunk of X client requests.				*
*									*
*	First process any unfinished request.				*
*	Next update xmx's resource model.				*
*	Cull unneeded requests.						*
*	Queue it.							*
*									*
************************************************************************/

#define DEVNULL	0
#define ALLSERV	1
#define QSERV	2
/*
**	Okay to process if at least n bytes are present.
*/
#define WANT(n)\
   if (left < (n)) {\
      len = 0; /* exit loop */\
      break;\
   }\
   cp->seqno++;\
   D_CALL2(D_PROTO1, dprx_request, rp, (s32_t)cp->seqno)

#define QUEUE(dest, chp)\
   switch (dest) {\
      case QSERV:\
         queue_add(qservp->qp, (chp));\
         if (rx_repeats(cp->rxqp))\
            rx_clear_repeat(cp->rxqp);\
         break;\
      case ALLSERV:\
         {  register int nn;\
            nn = queue_allserv(chp);\
            if (rx_repeats(cp->rxqp)) {\
               rx_incr(cp->rxqp, nn);\
               rx_clear_repeat(cp->rxqp);\
            }\
         }\
         break;\
   }\
   buf_clear(chp)

#define DISPATCH(pp)\
   if (nbytes) {\
      register chunk_t *_chp;\
      _chp = buf_split(bp, nbytes);\
      pp_assign(_chp->tpp, pp);\
      QUEUE(curdest, _chp);\
      nbytes = 0;\
   }

#define DEST(dest) {\
   if (dest != curdest || nbytes + len > MAXCHUNKSIZE) {\
      DISPATCH(0);\
      curdest = dest;\
   }\
   nbytes += len;\
}

#define INSERT(dest, chp)	{DISPATCH(0) QUEUE(dest, chp); (chp) = 0;}

#define CULL()			DEST(DEVNULL)
#define FORWARD()		DEST(ALLSERV)
#define QUERY()			DEST(QSERV)

void
request_process
   AL((cp, bp))
   DB client_t *cp
   DD buffer_t *bp
   DE
{
   register uint_t nbytes = 0;	/* running total of bytes processed */
   register uint_t left, len;
   register u32_t curdest;
   register window_t *wp;
   register chunk_t *chp;
   chunk_t *chp1;
   register xReq *rp;
   register pp_t *pp;

   if (cp->swap)
      left = request_swap(bp);
   else
      left = buf_active(bp);

   curdest = 0;
   chp1 = 0;
   for (rp=(xReq *)buf_curdata(bp);; rp=(xReq *)((char *)rp+len)) {
      if (cp->pp) {	/* a partially processed request */
         if (chp = pp_process(bp, &cp->pp)) {
            len = buf_chunksize(chp);
            left -= len;
            if (cp->pp == 0)
               fd_unpartial(cp->fd);
            (void)queue_allserv(chp);		/* hardwired default... */
            buf_clear(chp);
         }
         else		/* not enough data */
            break;

         continue;
      }/* else... */

#ifdef DEBUG
if (left > buf_active(bp)) {
warn("request_process: left[%d] > chunk size[%d]\n", left,
	buf_active(bp));
abort();
}
#endif
      if (left < sz_xReq)
         break;		/* that's all, folks */

      len = rp->length * 4;

      switch (rp->reqType) {
         case X_CreateWindow:
            {  register xCreateWindowReq *p = (xCreateWindowReq *)rp;
               register window_t *wp;

               WANT(len);
               if ((wp=window_create(cp, p, &chp1)) == 0) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_CreateWindow(bp, cp->clinum, cp->seqno, p,
							wp->mp->mappixels);
                  if (chp1) {
                     INSERT(ALLSERV, chp1);
                  }
               }
            }
            break;
         case X_ChangeWindowAttributes:
            {  register xChangeWindowAttributesReq *p =
					(xChangeWindowAttributesReq *)rp;
               WANT(len);
               if ((wp=window_change_attributes(cp, p, &chp1)) == 0) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_ChangeWindowAttributes(bp, cp->clinum, cp->seqno, p,
							wp->mp->mappixels);
                  if (chp1) {
                     INSERT(ALLSERV, chp1);
                  }
               }
            }
            break;
         case X_GetWindowAttributes:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               window_get_attributes(cp, p);
               CULL();
            }
            break;
         case X_DestroyWindow:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               if (window_destroy(cp, p)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_DestroyWindow(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_DestroySubwindows:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               if (window_destroy_subwindows(cp, p)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_DestroySubwindows(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_ChangeSaveSet:
            {  register xChangeSaveSetReq *p = (xChangeSaveSetReq *)rp;
               WANT(len);
               window_change_save_set(cp, p);
               CULL();
            }
            break;
         case X_ReparentWindow:
            {  register xReparentWindowReq *p = (xReparentWindowReq *)rp;
               WANT(len);
               if (window_reparent(cp, p)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_ReparentWindow(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_MapWindow:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               if (window_map(cp, p)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_MapWindow(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_MapSubwindows:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               if (window_map_subwindows(cp, p)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_MapSubwindows(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_UnmapWindow:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               if (window_unmap(cp, p)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_UnmapWindow(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_UnmapSubwindows:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               if (window_unmap_subwindows(cp, p)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_UnmapSubwindows(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_ConfigureWindow:
            {  register xConfigureWindowReq *p = (xConfigureWindowReq *)rp;
               WANT(len);
               if (window_configure(cp, p)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_ConfigureWindow(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_CirculateWindow:
            {  register xCirculateWindowReq *p = (xCirculateWindowReq *)rp;
               WANT(len);
               if (window_circulate(cp, p)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_CirculateWindow(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_GetGeometry:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               window_get_geometry(cp, p);
               CULL();
            }
            break;
         case X_QueryTree:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               window_query_tree(cp, p);
               CULL();
            }
            break;
         case X_InternAtom:
            {  register xInternAtomReq *p = (xInternAtomReq *)rp;
               WANT(len);
               atom_intern(cp, p);
               CULL();
            }
            break;
         case X_GetAtomName:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               atom_getname(cp, p);
               CULL();
            }
            break;
         case X_ChangeProperty:
            {  register xChangePropertyReq *p = (xChangePropertyReq *)rp;
               WANT(len);
               prop_change(	cp,
				p->window,
				p->property,
				p->type,
				p->format,
				p->mode,
				p->nUnits,
				(char *)(p+1));
               CULL();
            }
            break;
         case X_DeleteProperty:
            {  register xDeletePropertyReq *p = (xDeletePropertyReq *)rp;
               WANT(len);
               prop_delete(cp, p->window, p->property);
               CULL();
            }
            break;
         case X_GetProperty:
            {  register xGetPropertyReq *p = (xGetPropertyReq *)rp;
               WANT(len);
               prop_get(	cp,
				p->window,
				p->property, 
				p->type, 
				p->longOffset, 
				p->longLength, 
				p->delete);
               CULL();
            }
            break;
         case X_ListProperties:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               prop_list(cp, p->id);
               CULL();
            }
            break;
         case X_SetSelectionOwner:
            {  register xSetSelectionOwnerReq *p = (xSetSelectionOwnerReq *)rp;
               WANT(len);
               sel_set_owner(cp, p);
               CULL();
            }
            break;
         case X_GetSelectionOwner:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               sel_get_owner(cp, p);
               CULL();
            }
            break;
         case X_ConvertSelection:
            {  register xConvertSelectionReq *p = (xConvertSelectionReq *)rp;
               WANT(len);
               sel_convert(cp, p);
               CULL();
            }
            break;
         case X_SendEvent:
            {  register xSendEventReq *p = (xSendEventReq *)rp;
               WANT(len);
               event_send_event(cp, p);
               CULL();
            }
            break;
         case X_GrabPointer:
            {  register xGrabPointerReq *p = (xGrabPointerReq *)rp;
               WANT(len);
               inp_grab_pointer(cp, p);
               CULL();
            }
            break;
         case X_UngrabPointer:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               inp_ungrab_pointer(cp, p);
               CULL();
            }
            break;
         case X_GrabButton:
            {  register xGrabButtonReq *p = (xGrabButtonReq *)rp;
               WANT(len);
               inp_grab_button(cp, p);
               CULL();
            }
            break;
         case X_UngrabButton:
            {  register xUngrabButtonReq *p = (xUngrabButtonReq *)rp;
               WANT(len);
               inp_ungrab_button(cp, p);
               CULL();
            }
            break;
         case X_ChangeActivePointerGrab:
            {  register xChangeActivePointerGrabReq *p =
					(xChangeActivePointerGrabReq *)rp;
               WANT(len);
               inp_change_pointer_grab(cp, p);
               CULL();
            }
            break;
         case X_GrabKeyboard:
            {  register xGrabKeyboardReq *p = (xGrabKeyboardReq *)rp;
               WANT(len);
               inp_grab_keyboard(cp, p);
               CULL();
            }
            break;
         case X_UngrabKeyboard:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               inp_ungrab_keyboard(cp, p);
               CULL();
            }
            break;
         case X_GrabKey:
            {  register xGrabKeyReq *p = (xGrabKeyReq *)rp;
               WANT(len);
               inp_grab_key(cp, p);
               CULL();
            }
            break;
         case X_UngrabKey:
            {  register xUngrabKeyReq *p = (xUngrabKeyReq *)rp;
               WANT(len);
               inp_ungrab_key(cp, p);
               CULL();
            }
            break;
         case X_AllowEvents:
            {  register xAllowEventsReq *p = (xAllowEventsReq *)rp;
               WANT(len);
               inp_allow_events(cp, p);
               CULL();
            }
            break;
         case X_GrabServer:
            {  register xReq *p = (xReq *)rp;
               WANT(len);
               fd_grab(cp->fd);
               DEBUG1(D_GRABS, "request_process: GrabServer fd %d\n", cp->fd);
               CULL();
            }
            break;
         case X_UngrabServer:
            {  register xReq *p = (xReq *)rp;
               WANT(len);
               fd_ungrab();
               DEBUG1(D_GRABS, "request_process: UngrabServer (fd %d)\n",
								cp->fd);
               CULL();
            }
            break;
         case X_QueryPointer:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               inp_query_pointer(cp, p);
               CULL();
            }
            break;
         case X_GetMotionEvents:
            {  register xGetMotionEventsReq *p = (xGetMotionEventsReq *)rp;
               WANT(len);
               inp_get_motion(cp, p);
               CULL();
            }
            break;
         case X_TranslateCoords:
            {  register xTranslateCoordsReq *p = (xTranslateCoordsReq *)rp;
               WANT(len);
               inp_translate_coords(cp, p);
               CULL();
            }
            break;
         case X_WarpPointer:
            {  register xWarpPointerReq *p = (xWarpPointerReq *)rp;
               WANT(len);
               /* might want to do something here... TODO */
               CULL();
            }
            break;
         case X_SetInputFocus:
            {  register xSetInputFocusReq *p = (xSetInputFocusReq *)rp;
               WANT(len);
               focus_set(cp, p);
               CULL();
            }
            break;
         case X_GetInputFocus:
            {  register xReq *p = (xReq *)rp;
               WANT(len);
               focus_get(cp);
               CULL();
            }
            break;
         case X_QueryKeymap:
            {  register xReq *p = (xReq *)rp;
               WANT(len);
               inp_query_keymap(cp, p);
               CULL();
            }
            break;
         case X_OpenFont:
            {  register xOpenFontReq *p = (xOpenFontReq *)rp;
               WANT(len);
               if (font_open(cp, p)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_OpenFont(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_CloseFont:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               if (font_close(cp, p)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_CloseFont(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_QueryFont:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               QUERY();
               ptc_QueryFont(bp, cp->clinum, cp->seqno, p);
               rx_queue(cp->rxqp, X_QueryFont, 0, cp->seqno,
						cliz_scan_fontprops);
            }
            break;
         case X_QueryTextExtents:
            {  register xQueryTextExtentsReq *p = (xQueryTextExtentsReq *)rp;
               WANT(len);
               QUERY();
               ptc_QueryTextExtents(bp, cp->clinum, cp->seqno, p);
               rx_queue(cp->rxqp, X_QueryTextExtents, 0, cp->seqno,
							queue_client_reply);
            }
            break;
         case X_ListFonts:
            {  register xListFontsReq *p = (xListFontsReq *)rp;
               WANT(len);
               QUERY();
               ptc_ListFonts(bp, cp->clinum, cp->seqno, p);
               rx_queue(cp->rxqp, X_ListFonts, 0, cp->seqno,
							queue_client_reply);
            }
            break;
         case X_ListFontsWithInfo:
            {  register xListFontsReq *p = (xListFontsReq *)rp;
               WANT(len);
               QUERY();
               ptc_ListFontsWithInfo(bp, cp->clinum, cp->seqno, p);
               rx_queue(cp->rxqp, X_ListFontsWithInfo, 0, cp->seqno,
							cliz_scan_fontprops);
            }
            break;
         case X_SetFontPath:
            {  register xSetFontPathReq *p = (xSetFontPathReq *)rp;
               WANT(len);
               FORWARD();
               font_set_path(cp, p);
               ptc_SetFontPath(bp, cp->clinum, cp->seqno, p);
            }
            break;
         case X_GetFontPath:
            {  register xReq *p = (xReq *)rp;
               WANT(len);
               QUERY();
               ptc_GetFontPath(bp, cp->clinum, cp->seqno, p);
               rx_queue(cp->rxqp, X_GetFontPath, 0, cp->seqno,
							font_get_path_reply);
            }
            break;
         case X_CreatePixmap:
            {  register xCreatePixmapReq *p = (xCreatePixmapReq *)rp;
               WANT(len);
               if (pixmap_create(cp, p)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_CreatePixmap(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_FreePixmap:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               if (pixmap_free(cp, p)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_FreePixmap(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_CreateGC:
            {  register xCreateGCReq *p = (xCreateGCReq *)rp;
               register gc_t *gcp;

               WANT(len);
               if ((gcp=gc_create(cp, p, &chp1)) == 0) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_CreateGC(bp, cp->clinum, cp->seqno, p,
						gcp->wp->mp->mappixels);
                  if (chp1) {
                     INSERT(ALLSERV, chp1);
                  }
               }
            }
            break;
         case X_ChangeGC:
            {  register xChangeGCReq *p = (xChangeGCReq *)rp;
               register gc_t *gcp;

               WANT(len);
               if ((gcp=gc_change(cp, p)) == 0) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_ChangeGC(bp, cp->clinum, cp->seqno, p,
						gcp->wp->mp->mappixels);
               }
            }
            break;
         case X_CopyGC:
            {  register xCopyGCReq *p = (xCopyGCReq *)rp;
               WANT(len);
               if (gc_copy(cp, p)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_CopyGC(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_SetDashes:
            {  register xSetDashesReq *p = (xSetDashesReq *)rp;
               WANT(len);
               FORWARD();
               ptc_SetDashes(bp, cp->clinum, cp->seqno, p);
            }
            break;
         case X_SetClipRectangles:
            {  register xSetClipRectanglesReq *p = (xSetClipRectanglesReq *)rp;
               WANT(len);
               FORWARD();
               ptc_SetClipRectangles(bp, cp->clinum, cp->seqno, p);
            }
            break;
         case X_FreeGC:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               if (gc_free(cp, p)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_FreeGC(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_ClearArea:
            {  register xClearAreaReq *p = (xClearAreaReq *)rp;
               WANT(len);
               FORWARD();
               ptc_ClearArea(bp, cp->clinum, cp->seqno, p);
            }
            break;
         case X_CopyArea:
            {  register xCopyAreaReq *p = (xCopyAreaReq *)rp;
               WANT(len);
               FORWARD();
               ptc_CopyArea(bp, cp->clinum, cp->seqno, p);
            }
            break;
         case X_CopyPlane:
            {  register xCopyPlaneReq *p = (xCopyPlaneReq *)rp;
               WANT(len);
               FORWARD();
               ptc_CopyPlane(bp, cp->clinum, cp->seqno, p);
            }
            break;
         case X_PolyPoint:
         case X_PolyLine:
            {  register xPolyPointReq *p = (xPolyPointReq *)rp;
#ifdef TODO
               WANT(sz_xPolyPointReq);
               TRUNC_SHORTS();
#else
               WANT(len);
#endif
               FORWARD();
               ptc_PolyPoint(bp, cp->clinum, cp->seqno, p);
            }
            break;
         case X_PolySegment:
         case X_PolyRectangle:
         case X_PolyArc:
            {  register xPolySegmentReq *p = (xPolySegmentReq *)rp;
#ifdef TODO
               WANT(sz_xPolySegmentReq);
               TRUNC_SHORTS();
#else
               WANT(len);
#endif
               FORWARD();
               ptc_PolySegment(bp, cp->clinum, cp->seqno, p);
            }
            break;
         case X_FillPoly:
            {  register xFillPolyReq *p = (xFillPolyReq *)rp;
#ifdef TODO
               WANT(sz_xFillPolyReq);
               TRUNC_SHORTS();
#else
               WANT(len);
#endif
               FORWARD();
               ptc_FillPoly(bp, cp->clinum, cp->seqno, p);
            }
            break;
         case X_PolyFillRectangle:
         case X_PolyFillArc:
            {  register xPolySegmentReq *p = (xPolySegmentReq *)rp;
#ifdef TODO
               WANT(sz_xPolySegmentReq);
               TRUNC_SHORTS();
#else
               WANT(len);
#endif
               FORWARD();
               ptc_PolySegment(bp, cp->clinum, cp->seqno, p);
            }
            break;
         case X_PutImage:
            {  register xPutImageReq *p = (xPutImageReq *)rp;
               register int depth;

               WANT(sz_xPutImageReq);

               depth = p->format == ZPixmap ? p->depth : 1;
               pp = pp_image(len - sz_xPutImageReq, viordv[depth], p->width,
								p->depth);
               pp_assign(cp->pp, pp);

               fd_partial(cp->fd);

               len = sz_xPutImageReq;

               FORWARD();
               ptc_PutImage(bp, cp->clinum, cp->seqno, p);
               DISPATCH(pp);
            }
            break;
         case X_GetImage:
            {  register xGetImageReq *p = (xGetImageReq *)rp;
               register drawable_t *dp;

               WANT(len);
               if ((dp = (drawable_t *)hash_data(vmap, p->drawable)) == 0) {
                  proto_Error(cp, BadDrawable, p->drawable, 0, X_GetImage);
                  CULL();
               }
               else {	/* save width from request for reply */
                  QUERY();
                  ptc_GetImage(bp, cp->clinum, cp->seqno, p);
                  df_put_i((int)p->format);
                  df_put_i((int)p->width);
                  rx_queue(cp->rxqp, X_GetImage, 0, cp->seqno,
							image_client_reply);
               }
            }
            break;
         case X_PolyText8:
         case X_PolyText16:
            {  register xPolyTextReq *p = (xPolyTextReq *)rp;
#ifdef TODO
               WANT(sz_xPolyTextReq);
               TRUNC_SHORTS();
#else
               WANT(len);
#endif
               FORWARD();
               ptc_PolyText8(bp, cp->clinum, cp->seqno, p);
            }
            break;
         case X_ImageText8:
         case X_ImageText16:
            {  register xImageTextReq *p = (xImageTextReq *)rp;
#ifdef TODO
               WANT(sz_xImageTextReq);
               TRUNC_BYTES();
#else
               WANT(len);
#endif
               FORWARD();
               ptc_ImageText8(bp, cp->clinum, cp->seqno, p);
            }
            break;
         case X_CreateColormap:
            {  register xCreateColormapReq *p = (xCreateColormapReq *)rp;
               WANT(len);
               if (color_create_cmap(cp, p, &chp1)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_CreateColormap(bp, cp->clinum, cp->seqno, p);
                  if (chp1) {
                     INSERT(ALLSERV, chp1);
                  }
               }
            }
            break;
         case X_FreeColormap:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               if (color_free_cmap(cp, p, &chp1)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_FreeColormap(bp, cp->clinum, cp->seqno, p);
                  if (chp1) {
                     INSERT(ALLSERV, chp1);
                  }
               }
            }
            break;
         case X_CopyColormapAndFree:
            {  register xCopyColormapAndFreeReq *p =
						(xCopyColormapAndFreeReq *)rp;
               WANT(len);
               color_copy_cmap_and_free(cp, p, &chp1);
               if (chp1) {
                  INSERT(ALLSERV, chp1);
               }
               CULL();
            }
            break;
         case X_InstallColormap:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               color_install_cmap(cp, p, &chp1);
               if (chp1) {
                  INSERT(ALLSERV, chp1);
               }
               CULL();
            }
            break;
         case X_UninstallColormap:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               color_uninstall_cmap(cp, p, &chp1);
               if (chp1) {
                  INSERT(ALLSERV, chp1);
               }
               CULL();
            }
            break;
         case X_ListInstalledColormaps:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               color_list_cmaps(cp, p);
               CULL();
            }
            break;
         case X_AllocColor:
            {  register xAllocColorReq *p = (xAllocColorReq *)rp;
               WANT(len);
               if (color_alloc(cp, p, &chp1)) {
                  if (chp1) {
                     INSERT(ALLSERV, chp1);
                  }
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_AllocColor(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_AllocNamedColor:
            {  register xAllocNamedColorReq *p = (xAllocNamedColorReq *)rp;
               WANT(len);
               if (color_alloc_named(cp, p, &chp1)) {
                  if (chp1) {
                     INSERT(ALLSERV, chp1);
                  }
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_AllocNamedColor(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_AllocColorCells:
            {  register xAllocColorCellsReq *p = (xAllocColorCellsReq *)rp;
               WANT(len);
               if (color_alloc_cells(cp, p)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_AllocColorCells(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_AllocColorPlanes:
            {  register xAllocColorPlanesReq *p = (xAllocColorPlanesReq *)rp;
               WANT(len);
               if (color_alloc_planes(cp, p)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_AllocColorPlanes(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_FreeColors:
            {  register xFreeColorsReq *p = (xFreeColorsReq *)rp;
               register colormap_t *mp;

               WANT(len);
               if (mp = color_free(cp, p)) {
                  FORWARD();
                  ptc_FreeColors(bp, cp->clinum, cp->seqno, p, mp->mappixels);
               }
	       else {
                  CULL();
               }
            }
            break;
         case X_StoreColors:
            {  register xStoreColorsReq *p = (xStoreColorsReq *)rp;
               register colormap_t *mp;

               WANT(len);
               if (mp = color_store(cp, p)) {
                  FORWARD();
                  ptc_StoreColors(bp, cp->clinum, cp->seqno, p, mp->mappixels);
               }
               else {
                  CULL();
               }
            }
            break;
         case X_StoreNamedColor:
            {  register xStoreNamedColorReq *p = (xStoreNamedColorReq *)rp;
               register colormap_t *mp;

               WANT(len);
               if (mp = color_store_named(cp, p)) {
                  FORWARD();
                  ptc_StoreNamedColor(bp, cp->clinum, cp->seqno, p,
							mp->mappixels);
               }
               else {
                  CULL();
               }
            }
            break;
         case X_QueryColors:
            {  register xQueryColorsReq *p = (xQueryColorsReq *)rp;
               WANT(len);
               color_query(cp, p);
               CULL();
            }
            break;
         case X_LookupColor:
            {  register xLookupColorReq *p = (xLookupColorReq *)rp;
               WANT(len);
               color_lookup(cp, p);
               CULL();
            }
            break;
         case X_CreateCursor:
            {  register xCreateCursorReq *p = (xCreateCursorReq *)rp;
               WANT(len);
               if (cursor_create(cp, p, &chp1)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_CreateCursor(bp, cp->clinum, cp->seqno, p);
                  if (chp1) {
                     INSERT(ALLSERV, chp1);
                  }
               }
            }
            break;
         case X_CreateGlyphCursor:
            {  register xCreateGlyphCursorReq *p =
						(xCreateGlyphCursorReq *)rp;
               WANT(len);
               if (cursor_create_glyph(cp, p, &chp1)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_CreateGlyphCursor(bp, cp->clinum, cp->seqno, p);
                  if (chp1) {
                     INSERT(QSERV, chp1);
                  }
               }
            }
            break;
         case X_FreeCursor:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANT(len);
               if (cursor_free(cp, p)) {
                  CULL();
               }
               else {
                  FORWARD();
                  ptc_FreeCursor(bp, cp->clinum, cp->seqno, p);
               }
            }
            break;
         case X_RecolorCursor:
            {  register xRecolorCursorReq *p = (xRecolorCursorReq *)rp;
               WANT(len);
               FORWARD();
               ptc_RecolorCursor(bp, cp->clinum, cp->seqno, p);
            }
            break;
         case X_QueryBestSize:
            {  register xQueryBestSizeReq *p = (xQueryBestSizeReq *)rp;
               WANT(len);
               QUERY();
               ptc_QueryBestSize(bp, cp->clinum, cp->seqno, p);
               rx_queue(cp->rxqp, X_QueryBestSize,0, cp->seqno,
							queue_client_reply);
            }
            break;
         case X_QueryExtension:
            {  register xQueryExtensionReq *p = (xQueryExtensionReq *)rp;
               WANT(len);
               ext_query(cp, p);
               CULL();
            }
            break;
         case X_ListExtensions:
            {  register xReq *p = (xReq *)rp;
               WANT(len);
               ext_list(cp, p);
               CULL();
            }
            break;
         case X_ChangeKeyboardMapping:
            {  register xChangeKeyboardMappingReq *p =
						(xChangeKeyboardMappingReq *)rp;
               WANT(len);
	       kpm_set_keys(cp, p);
               CULL();
            }
            break;
         case X_GetKeyboardMapping:
            {  register xGetKeyboardMappingReq *p =
						(xGetKeyboardMappingReq *)rp;
               WANT(len);
	       kpm_get_keys(cp, p);
               CULL();
            }
            break;
         case X_ChangeKeyboardControl:
            {  register xChangeKeyboardMappingReq *p =
						(xChangeKeyboardMappingReq *)rp;
               WANT(len);
               CULL();
            }
            break;
         case X_GetKeyboardControl:
            {  register xReq *p = (xReq *)rp;
               WANT(len);
               QUERY();
               ptc_GetKeyboardControl(bp, cp->clinum, cp->seqno, p);
               rx_queue(cp->rxqp, X_GetKeyboardControl, 0, cp->seqno,
							queue_client_reply);
            }
            break;
         case X_Bell:
            {  register xBellReq *p = (xBellReq *)rp;
               WANT(len);
               FORWARD();
               ptc_Bell(bp, cp->clinum, cp->seqno, p);
            }
            break;
         case X_ChangePointerControl:
            {  register xChangePointerControlReq *p =
						(xChangePointerControlReq *)rp;
               WANT(len);
               CULL();
            }
            break;
         case X_GetPointerControl:
            {  register xReq *p = (xReq *)rp;
               WANT(len);
               QUERY();
               ptc_GetPointerControl(bp, cp->clinum, cp->seqno, p);
               rx_queue(cp->rxqp, X_GetPointerControl, 0, cp->seqno,
							queue_client_reply);
            }
            break;
         case X_SetScreenSaver:
            {  register xSetScreenSaverReq *p = (xSetScreenSaverReq *)rp;
               WANT(len);
               FORWARD();
               ptc_SetScreenSaver(bp, cp->clinum, cp->seqno, p);
            }
            break;
         case X_GetScreenSaver:
            {  register xReq *p = (xReq *)rp;
               WANT(len);
               QUERY();
               ptc_GetScreenSaver(bp, cp->clinum, cp->seqno, p);
               rx_queue(cp->rxqp, X_GetScreenSaver, 0, cp->seqno,
							queue_client_reply);
            }
            break;
         case X_ChangeHosts:
            {  register xChangeHostsReq *p = (xChangeHostsReq *)rp;
               WANT(len);
               access_change(cp, p);
               CULL();
            }
            break;
         case X_ListHosts:
            {  register xListHostsReq *p = (xListHostsReq *)rp;
               WANT(len);
               access_list(cp);
               CULL();
            }
            break;
         case X_SetAccessControl:
            {  register xChangeModeReq *p = (xChangeModeReq *)rp;
               WANT(len);
               access_set(cp, p->mode);
               CULL();
            }
            break;
         case X_SetCloseDownMode:
            {  register xChangeModeReq *p = (xChangeModeReq *)rp;
               WANT(len);
               cp->closedownmode = p->mode;
               CULL();
            }
            break;
         case X_KillClient:
            {  register xResourceReq *p = (xResourceReq *)rp;
               register resource_t *tp;
               register client_t *kcp = 0;

               WANT(len);
               if (p->id)
                  if ((tp = (resource_t *)hash_data(vmap, p->id)) == 0)
                     proto_Error(cp, BadValue, p->id, 0, X_KillClient);
                  else
                     kcp = tp->client;

               CULL();
               if (kcp)
                  /*
                  **  inserts lots of protocol
                  */
                  cliz_kill_client(kcp);
            }
            break;
         case X_RotateProperties:
            {  register xRotatePropertiesReq *p = (xRotatePropertiesReq *)rp;
               WANT(len);
               prop_rotate(	cp,
				p->window,
				p->nPositions,
				p->nAtoms,
				(atom_t *)(p+1));
               CULL();
            }
            break;
         case X_ForceScreenSaver:
            {  register xChangeModeReq *p = (xChangeModeReq *)rp;
               WANT(len);
               FORWARD();
               ptc_ForceScreenSaver(bp, cp->clinum, cp->seqno, p);
            }
            break;
         case X_SetPointerMapping:
            {  register xSetPointerMappingReq *p = (xSetPointerMappingReq *)rp;
               WANT(len);
	       kpm_set_buttons(cp, p);
               CULL();
            }
            break;
         case X_GetPointerMapping:
            {  register xReq *p = (xReq *)rp;
               WANT(len);
	       kpm_get_buttons(cp, p);
               CULL();
            }
            break;
         case X_SetModifierMapping:
            {  register xSetModifierMappingReq *p =
						(xSetModifierMappingReq *)rp;
               WANT(len);
	       kpm_set_mods(cp, p);
               CULL();
            }
            break;
         case X_GetModifierMapping:
            {  register xReq *p = (xReq *)rp;
               WANT(len);
	       kpm_get_mods(cp, p);
               CULL();
            }
            break;
         case X_NoOperation:
            {  register xReq *p = (xReq *)rp;
               WANT(len);
               CULL();
            }
            break;
         default:	/* this needs fixing (TODO) */
            len = ext_request_process(cp, rp, left);
            break;
      }
      if (len)
         left -= len;
      else
         break;		/* exit loop */
   }
   DISPATCH(0);
}
#undef WANT
#undef TRUNC_BYTES
#undef TRUNC_SHORTS

/************************************************************************
*									*
*   request_swap							*
*									*
*	Swap any new request data in the given buffer.  Returns the	*
*	total number of swapped bytes, including any that were swapped	*
*	previously.							*
*									*
*	Splits the chunk if not all data is swapped (should almost	*
*	never happen).							*
*									*
************************************************************************/

#define WANT(n)\
   if (left < (n))\
      return nbytes;\
   p->length = length

#define WANTALL()\
	WANT(len);\
	left -= len;\
	nbytes += len

#ifdef TODO
#define SWAPSHORTS(sz)\
	if (left < len)\
		t = (left - (sz)) & ~1;\
	else\
		t = (len - (sz)) & ~1;\
	left -= (sz) + t;\
	nbytes += (sz) + t;\
	shp = (u16_t *)(p+1);\
	t /= 2;\
	for (i=0; i<t; i++)\
		SWAP2(shp[i])

#define TRUNCBYTES()\
	t = left < len ? left : len;\
	left -= t;\
	nbytes += t
#else
#define SWAPSHORTS(sz)\
	t = (len - (sz)) / 2;\
	shp = (u16_t *)(p+1);\
	for (i=0; i<t; i++)\
		SWAP2(shp[i])

#define TRUNCBYTES()
#endif


int
request_swap
   AL((bp))
   DB buffer_t *bp
   DE
{
   register int i, t;
   register uint_t nbytes = 0;
   u16_t length;
   register uint_t len, left;
   register xReq *rp;
   register u16_t *shp;
   register chunk_t *chp;

   chp = buf_chunk(bp);
   left = buf_chunksize(chp);
   rp = (xReq *)buf_data(chp);

   for (; left >= sz_xReq; rp=(xReq *)((char *)rp+len)) {
      ((char *)&length)[0] = ((char *)&rp->length)[1];
      ((char *)&length)[1] = ((char *)&rp->length)[0];

      len = length * 4;

      DEBUG1(D_SWAP, "request_swap: %s\n", dprx_reqType_str(rp->reqType));
      switch (rp->reqType) {
         case X_CreateWindow:
            {  register xCreateWindowReq *p = (xCreateWindowReq *)rp;

               WANTALL();
               SWAP4(p->wid);
               SWAP4(p->parent);
               SWAP2(p->x);
               SWAP2(p->y);
               SWAP2(p->width);
               SWAP2(p->height);
               SWAP2(p->borderWidth);
               SWAP2(p->class);
               SWAP4(p->visual);
               SWAP4(p->mask);
               for (i=sz_xCreateWindowReq/4; i<(int)p->length; i++)
                  SWAP4(((u32_t *)p)[i]);
            }
            break;
         case X_ChangeWindowAttributes:
            {  register xChangeWindowAttributesReq *p =
					(xChangeWindowAttributesReq *)rp;
               WANTALL();
               SWAP4(p->window);
               SWAP4(p->valueMask);
               for (i=sz_xChangeWindowAttributesReq/4; i<(int)p->length; i++)
                  SWAP4(((u32_t *)p)[i]);
            }
            break;
         case X_GetWindowAttributes:
         case X_DestroyWindow:
         case X_DestroySubwindows:
            {  register xResourceReq *p = (xResourceReq *)rp;

               WANTALL();
               SWAP4(p->id);
            }
            break;
         case X_ChangeSaveSet:
            {  register xChangeSaveSetReq *p = (xChangeSaveSetReq *)rp;

               WANTALL();
               SWAP4(p->window);
            }
            break;
         case X_ReparentWindow:
            {  register xReparentWindowReq *p = (xReparentWindowReq *)rp;

               WANTALL();
               SWAP4(p->window);
               SWAP4(p->parent);
               SWAP2(p->x);
               SWAP2(p->y);
            }
            break;
         case X_MapWindow:
         case X_MapSubwindows:
         case X_UnmapWindow:
         case X_UnmapSubwindows:
            {  register xResourceReq *p = (xResourceReq *)rp;

               WANTALL();
               SWAP4(p->id);
            }
            break;
         case X_ConfigureWindow:
            {  register xConfigureWindowReq *p = (xConfigureWindowReq *)rp;

               WANTALL();
               SWAP4(p->window);
               SWAP2(p->mask);
               for (i=sz_xConfigureWindowReq/4; i<(int)p->length; i++)
                  SWAP4(((u32_t *)p)[i]);
            }
            break;
         case X_CirculateWindow:
            {  register xCirculateWindowReq *p = (xCirculateWindowReq *)rp;

               WANTALL();
               SWAP4(p->window);
            }
            break;
         case X_GetGeometry:
         case X_QueryTree:
            {  register xResourceReq *p = (xResourceReq *)rp;

               WANTALL();
               SWAP4(p->id);
            }
            break;
         case X_InternAtom:
            {  register xInternAtomReq *p = (xInternAtomReq *)rp;

               WANTALL();
               SWAP2(p->nbytes);
            }
            break;
         case X_GetAtomName:
            {  register xResourceReq *p = (xResourceReq *)rp;

               WANTALL();
               SWAP4(p->id);
            }
            break;
         case X_ChangeProperty:
            {  register xChangePropertyReq *p = (xChangePropertyReq *)rp;

               WANTALL();
               SWAP4(p->window);
               SWAP4(p->property);
               SWAP4(p->type);
               SWAP4(p->nUnits);
               switch (p->format) {
                  case 8:
                     break;
                  case 16:
                     SWAP2L(p+1, p->nUnits);
                     break;
                  case 32:
                     SWAP4L(p+1, p->nUnits);
                     break;
               }
            }
            break;
         case X_DeleteProperty:
            {  register xDeletePropertyReq *p = (xDeletePropertyReq *)rp;

               WANTALL();
               SWAP4(p->window);
               SWAP4(p->property);
            }
            break;
         case X_GetProperty:
            {  register xGetPropertyReq *p = (xGetPropertyReq *)rp;

               WANTALL();
               SWAP4(p->window);
               SWAP4(p->property);
               SWAP4(p->type);
               SWAP4(p->longOffset);
               SWAP4(p->longLength);
            }
            break;
         case X_ListProperties:
            {  register xResourceReq *p = (xResourceReq *)rp;

               WANTALL();
               SWAP4(p->id);
            }
            break;
         case X_SetSelectionOwner:
            {  register xSetSelectionOwnerReq *p = (xSetSelectionOwnerReq *)rp;

               WANTALL();
               SWAP4(p->window);
               SWAP4(p->selection);
               SWAP4(p->time);
            }
            break;
         case X_GetSelectionOwner:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANTALL();
               SWAP4(p->id);
            }
            break;
         case X_ConvertSelection:
            {  register xConvertSelectionReq *p = (xConvertSelectionReq *)rp;
               WANTALL();
               SWAP4(p->requestor);
               SWAP4(p->selection);
               SWAP4(p->target);
               SWAP4(p->property);
               SWAP4(p->time);
            }
            break;
         case X_SendEvent:
            {  register xSendEventReq *p = (xSendEventReq *)rp;
               WANTALL();
               SWAP4(p->destination);
               SWAP4(p->eventMask);
               swap_event(&p->event);
            }
            break;
         case X_GrabPointer:
            {  register xGrabPointerReq *p = (xGrabPointerReq *)rp;
               WANTALL();
               SWAP4(p->grabWindow);
               SWAP2(p->eventMask);
               SWAP4(p->confineTo);
               SWAP4(p->cursor);
               SWAP4(p->time);
            }
            break;
         case X_UngrabPointer:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANTALL();
               SWAP4(p->id);
            }
            break;
         case X_GrabButton:
            {  register xGrabButtonReq *p = (xGrabButtonReq *)rp;
               WANTALL();
               SWAP4(p->grabWindow);
               SWAP2(p->eventMask);
               SWAP4(p->confineTo);
               SWAP4(p->cursor);
            }
            break;
         case X_UngrabButton:
            {  register xUngrabButtonReq *p = (xUngrabButtonReq *)rp;
               WANTALL();
               SWAP4(p->grabWindow);
               SWAP2(p->modifiers);
            }
            break;
         case X_ChangeActivePointerGrab:
            {  register xChangeActivePointerGrabReq *p =
					(xChangeActivePointerGrabReq *)rp;
               WANTALL();
               SWAP4(p->cursor);
               SWAP4(p->time);
               SWAP2(p->eventMask);
            }
            break;
         case X_GrabKeyboard:
            {  register xGrabKeyboardReq *p = (xGrabKeyboardReq *)rp;
               WANTALL();
               SWAP4(p->grabWindow);
               SWAP4(p->time);
            }
            break;
         case X_UngrabKeyboard:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANTALL();
               SWAP4(p->id);
            }
            break;
         case X_GrabKey:
            {  register xGrabKeyReq *p = (xGrabKeyReq *)rp;
               WANTALL();
               SWAP4(p->grabWindow);
               SWAP2(p->modifiers);
            }
            break;
         case X_UngrabKey:
            {  register xUngrabKeyReq *p = (xUngrabKeyReq *)rp;
               WANTALL();
               SWAP4(p->grabWindow);
               SWAP2(p->modifiers);
            }
            break;
         case X_AllowEvents:
            {  register xAllowEventsReq *p = (xAllowEventsReq *)rp;
               WANTALL();
               SWAP4(p->time);
            }
            break;
         case X_GrabServer:
         case X_UngrabServer:
            {  register xReq *p = (xReq *)rp;
               WANTALL();
            }
            break;
         case X_QueryPointer:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANTALL();
               SWAP4(p->id);
            }
            break;
         case X_GetMotionEvents:
            {  register xGetMotionEventsReq *p = (xGetMotionEventsReq *)rp;
               WANTALL();
               SWAP4(p->window);
               SWAP4(p->start);
               SWAP4(p->stop);
            }
            break;
         case X_TranslateCoords:
            {  register xTranslateCoordsReq *p = (xTranslateCoordsReq *)rp;
               WANTALL();
               SWAP4(p->srcWid);
               SWAP4(p->dstWid);
               SWAP2(p->srcX);
               SWAP2(p->srcY);
            }
            break;
         case X_WarpPointer:
            {  register xWarpPointerReq *p = (xWarpPointerReq *)rp;
               WANTALL();
               SWAP4(p->srcWid);
               SWAP4(p->dstWid);
               SWAP2(p->srcX);
               SWAP2(p->srcY);
               SWAP2(p->srcWidth);
               SWAP2(p->srcHeight);
               SWAP2(p->dstX);
               SWAP2(p->dstY);
            }
            break;
         case X_SetInputFocus:
            {  register xSetInputFocusReq *p = (xSetInputFocusReq *)rp;
               WANTALL();
               SWAP4(p->focus);
               SWAP4(p->time);
            }
            break;
         case X_GetInputFocus:
         case X_QueryKeymap:
            {  register xReq *p = (xReq *)rp;
               WANTALL();
            }
            break;
         case X_OpenFont:
            {  register xOpenFontReq *p = (xOpenFontReq *)rp;
               WANTALL();
               SWAP4(p->fid);
               SWAP2(p->nbytes);
            }
            break;
         case X_CloseFont:
         case X_QueryFont:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANTALL();
               SWAP4(p->id);
            }
            break;
         case X_QueryTextExtents:
            {  register xQueryTextExtentsReq *p = (xQueryTextExtentsReq *)rp;
               WANTALL();
               SWAP4(p->fid);
            }
            break;
         case X_ListFonts:
         case X_ListFontsWithInfo:
            {  register xListFontsReq *p = (xListFontsReq *)rp;
               WANTALL();
               SWAP2(p->maxNames);
               SWAP2(p->nbytes);
            }
            break;
         case X_SetFontPath:
            {  register xSetFontPathReq *p = (xSetFontPathReq *)rp;
               WANTALL();
               SWAP2(p->nFonts);
            }
            break;
         case X_GetFontPath:
            {  register xReq *p = (xReq *)rp;
               WANTALL();
            }
            break;
         case X_CreatePixmap:
            {  register xCreatePixmapReq *p = (xCreatePixmapReq *)rp;
               WANTALL();
               SWAP4(p->pid);
               SWAP4(p->drawable);
               SWAP2(p->width);
               SWAP2(p->height);
            }
            break;
         case X_FreePixmap:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANTALL();
               SWAP4(p->id);
            }
            break;
         case X_CreateGC:
            {  register xCreateGCReq *p = (xCreateGCReq *)rp;
               WANTALL();
               SWAP4(p->gc);
               SWAP4(p->drawable);
               SWAP4(p->mask);
               for (i=sz_xCreateGCReq/4; i<(int)p->length; i++)
                  SWAP4(((u32_t *)p)[i]);
            }
            break;
         case X_ChangeGC:
            {  register xChangeGCReq *p = (xChangeGCReq *)rp;
               WANTALL();
               SWAP4(p->gc);
               SWAP4(p->mask);
               for (i=sz_xChangeGCReq/4; i<(int)p->length; i++)
                  SWAP4(((u32_t *)p)[i]);
            }
            break;
         case X_CopyGC:
            {  register xCopyGCReq *p = (xCopyGCReq *)rp;
               WANTALL();
               SWAP4(p->srcGC);
               SWAP4(p->dstGC);
               SWAP4(p->mask);
            }
            break;
         case X_SetDashes:
            {  register xSetDashesReq *p = (xSetDashesReq *)rp;
               WANTALL();
               SWAP4(p->gc);
               SWAP2(p->dashOffset);
               SWAP2(p->nDashes);
            }
            break;
         case X_SetClipRectangles:
            {  register xSetClipRectanglesReq *p = (xSetClipRectanglesReq *)rp;
               WANTALL();
               SWAP4(p->gc);
               SWAP2(p->xOrigin);
               SWAP2(p->yOrigin);
               SWAP2L(p+1, (len-sz_xSetClipRectanglesReq)/2);
            }
            break;
         case X_FreeGC:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANTALL();
               SWAP4(p->id);
            }
            break;
         case X_ClearArea:
            {  register xClearAreaReq *p = (xClearAreaReq *)rp;
               WANTALL();
               SWAP4(p->window);
               SWAP2(p->x);
               SWAP2(p->y);
               SWAP2(p->width);
               SWAP2(p->height);
            }
            break;
         case X_CopyArea:
            {  register xCopyAreaReq *p = (xCopyAreaReq *)rp;
               WANTALL();
               SWAP4(p->srcDrawable);
               SWAP4(p->dstDrawable);
               SWAP4(p->gc);
               SWAP2(p->srcX);
               SWAP2(p->srcY);
               SWAP2(p->dstX);
               SWAP2(p->dstY);
               SWAP2(p->width);
               SWAP2(p->height);
            }
            break;
         case X_CopyPlane:
            {  register xCopyPlaneReq *p = (xCopyPlaneReq *)rp;
               WANTALL();
               SWAP4(p->srcDrawable);
               SWAP4(p->dstDrawable);
               SWAP4(p->gc);
               SWAP2(p->srcX);
               SWAP2(p->srcY);
               SWAP2(p->dstX);
               SWAP2(p->dstY);
               SWAP2(p->width);
               SWAP2(p->height);
               SWAP4(p->bitPlane);
            }
            break;
         case X_PolyPoint:
         case X_PolyLine:
            {  register xPolyPointReq *p = (xPolyPointReq *)rp;
#ifdef TODO
               WANT(sz_xPolyPointReq);
#else
               WANTALL();
#endif
               SWAP4(p->drawable);
               SWAP4(p->gc);
               SWAPSHORTS(sz_xPolyPointReq);
            }
            break;
         case X_PolySegment:
         case X_PolyRectangle:
         case X_PolyArc:
            {  register xPolySegmentReq *p = (xPolySegmentReq *)rp;
#ifdef TODO
               WANT(sz_xPolySegmentReq);
#else
               WANTALL();
#endif
               SWAP4(p->drawable);
               SWAP4(p->gc);
               SWAPSHORTS(sz_xPolySegmentReq);
            }
            break;
         case X_FillPoly:
            {  register xFillPolyReq *p = (xFillPolyReq *)rp;
#ifdef TODO
               WANT(sz_xFillPolyReq);
#else
               WANTALL();
#endif
               SWAP4(p->drawable);
               SWAP4(p->gc);
               SWAPSHORTS(sz_xFillPolyReq);
            }
            break;
         case X_PolyFillRectangle:
         case X_PolyFillArc:
            {  register xPolySegmentReq *p = (xPolySegmentReq *)rp;
#ifdef TODO
               WANT(sz_xPolySegmentReq);
#else
               WANTALL();
#endif
               SWAP4(p->drawable);
               SWAP4(p->gc);
               SWAPSHORTS(sz_xPolySegmentReq);
            }
            break;
         case X_PutImage:
            {  register xPutImageReq *p = (xPutImageReq *)rp;
#ifdef TODO
               WANT(sz_xPutImageReq);
#else
               WANTALL();
#endif
               SWAP4(p->drawable);
               SWAP4(p->gc);
               SWAP2(p->width);
               SWAP2(p->height);
               SWAP2(p->dstX);
               SWAP2(p->dstY);
               TRUNCBYTES();
            }
            break;
         case X_GetImage:
            {  register xGetImageReq *p = (xGetImageReq *)rp;
               WANTALL();
               SWAP4(p->drawable);
               SWAP2(p->x);
               SWAP2(p->y);
               SWAP2(p->width);
               SWAP2(p->height);
               SWAP4(p->planeMask);
            }
            break;
         case X_PolyText8:
         case X_PolyText16:
            {  register xPolyTextReq *p = (xPolyTextReq *)rp;
#ifdef TODO
               WANT(sz_xPolyTextReq);
#else
               WANTALL();
#endif
               SWAP4(p->drawable);
               SWAP4(p->gc);
               SWAP2(p->x);
               SWAP2(p->y);
               TRUNCBYTES();
            }
            break;
         case X_ImageText8:
         case X_ImageText16:
            {  register xImageTextReq *p = (xImageTextReq *)rp;
#ifdef TODO
               WANT(sz_xImageTextReq);
#else
               WANTALL();
#endif
               SWAP4(p->drawable);
               SWAP4(p->gc);
               SWAP2(p->x);
               SWAP2(p->y);
               TRUNCBYTES();
            }
            break;
         case X_CreateColormap:
            {  register xCreateColormapReq *p = (xCreateColormapReq *)rp;
               WANTALL();
               SWAP4(p->mid);
               SWAP4(p->window);
               SWAP4(p->visual);
            }
            break;
         case X_FreeColormap:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANTALL();
               SWAP4(p->id);
            }
            break;
         case X_CopyColormapAndFree:
            {  register xCopyColormapAndFreeReq *p =
						(xCopyColormapAndFreeReq *)rp;
               WANTALL();
               SWAP4(p->mid);
               SWAP4(p->srcCmap);
            }
            break;
         case X_InstallColormap:
         case X_UninstallColormap:
         case X_ListInstalledColormaps:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANTALL();
               SWAP4(p->id);
            }
            break;
         case X_AllocColor:
            {  register xAllocColorReq *p = (xAllocColorReq *)rp;
               WANTALL();
               SWAP4(p->cmap);
               SWAP2(p->red);
               SWAP2(p->green);
               SWAP2(p->blue);
            }
            break;
         case X_AllocNamedColor:
            {  register xAllocNamedColorReq *p = (xAllocNamedColorReq *)rp;
               WANTALL();
               SWAP4(p->cmap);
               SWAP2(p->nbytes);
            }
            break;
         case X_AllocColorCells:
            {  register xAllocColorCellsReq *p = (xAllocColorCellsReq *)rp;
               WANTALL();
               SWAP4(p->cmap);
               SWAP2(p->colors);
               SWAP2(p->planes);
            }
            break;
         case X_AllocColorPlanes:
            {  register xAllocColorPlanesReq *p = (xAllocColorPlanesReq *)rp;
               WANTALL();
               SWAP4(p->cmap);
               SWAP2(p->colors);
               SWAP2(p->red);
               SWAP2(p->green);
               SWAP2(p->blue);
            }
            break;
         case X_FreeColors:
            {  register xFreeColorsReq *p = (xFreeColorsReq *)rp;
               WANTALL();
               SWAP4(p->cmap);
               SWAP2(p->planeMask);
               SWAP4L(p+1, (len-sz_xFreeColorsReq)/4);
            }
            break;
         case X_StoreColors:
            {  register xStoreColorsReq *p = (xStoreColorsReq *)rp;
               WANTALL();
               SWAP4(p->cmap);
               swap_coloritems(	(xColorItem *)(p+1),
				(len-sz_xStoreColorsReq)/sz_xColorItem);
            }
            break;
         case X_StoreNamedColor:
            {  register xStoreNamedColorReq *p = (xStoreNamedColorReq *)rp;
               WANTALL();
               SWAP4(p->cmap);
               SWAP4(p->pixel);
               SWAP2(p->nbytes);
            }
            break;
         case X_QueryColors:
            {  register xQueryColorsReq *p = (xQueryColorsReq *)rp;
               WANTALL();
               SWAP4(p->cmap);
               SWAP4L(p+1, (len-sz_xQueryColorsReq)/4);
            }
            break;
         case X_LookupColor:
            {  register xLookupColorReq *p = (xLookupColorReq *)rp;
               WANTALL();
               SWAP4(p->cmap);
               SWAP2(p->nbytes);
            }
            break;
         case X_CreateCursor:
            {  register xCreateCursorReq *p = (xCreateCursorReq *)rp;
               WANTALL();
               SWAP4(p->cid);
               SWAP4(p->source);
               SWAP4(p->mask);
               SWAP2(p->foreRed);
               SWAP2(p->foreGreen);
               SWAP2(p->foreBlue);
               SWAP2(p->backRed);
               SWAP2(p->backGreen);
               SWAP2(p->backBlue);
               SWAP2(p->x);
               SWAP2(p->y);
            }
            break;
         case X_CreateGlyphCursor:
            {  register xCreateGlyphCursorReq *p = (xCreateGlyphCursorReq *)rp;
               WANTALL();
               SWAP4(p->cid);
               SWAP4(p->source);
               SWAP4(p->mask);
               SWAP2(p->sourceChar);
               SWAP2(p->maskChar);
               SWAP2(p->foreRed);
               SWAP2(p->foreGreen);
               SWAP2(p->foreBlue);
               SWAP2(p->backRed);
               SWAP2(p->backGreen);
               SWAP2(p->backBlue);
            }
            break;
         case X_FreeCursor:
            {  register xResourceReq *p = (xResourceReq *)rp;
               WANTALL();
               SWAP4(p->id);
            }
            break;
         case X_RecolorCursor:
            {  register xRecolorCursorReq *p = (xRecolorCursorReq *)rp;
               WANTALL();
               SWAP4(p->cursor);
               SWAP2(p->foreRed);
               SWAP2(p->foreGreen);
               SWAP2(p->foreBlue);
               SWAP2(p->backRed);
               SWAP2(p->backGreen);
               SWAP2(p->backBlue);
            }
            break;
         case X_QueryBestSize:
            {  register xQueryBestSizeReq *p = (xQueryBestSizeReq *)rp;
               WANTALL();
               SWAP4(p->drawable);
               SWAP2(p->width);
               SWAP2(p->height);
            }
            break;
         case X_QueryExtension:
            {  register xQueryExtensionReq *p = (xQueryExtensionReq *)rp;
               WANTALL();
               SWAP2(p->nbytes);
            }
            break;
         case X_ListExtensions:
            {  register xReq *p = (xReq *)rp;
               WANTALL();
            }
            break;
         case X_ChangeKeyboardMapping:
            {  register xChangeKeyboardMappingReq *p =
						(xChangeKeyboardMappingReq *)rp;
               WANTALL();
               SWAP4(p->firstKeyCode);
               SWAP4L(p+1, p->keyCodes * p->keySymsPerKeyCode);
            }
            break;
         case X_GetKeyboardMapping:
            {  register xGetKeyboardMappingReq *p =
						(xGetKeyboardMappingReq *)rp;
               WANTALL();
            }
            break;
         case X_ChangeKeyboardControl:
            {  register xChangeKeyboardControlReq *p =
						(xChangeKeyboardControlReq *)rp;
               WANTALL();
               SWAP4(p->mask);
            }
            break;
         case X_GetKeyboardControl:
            {  register xReq *p = (xReq *)rp;
               WANTALL();
            }
            break;
         case X_Bell:
            {  register xBellReq *p = (xBellReq *)rp;
               WANTALL();
            }
            break;
         case X_ChangePointerControl:
            {  register xChangePointerControlReq *p =
						(xChangePointerControlReq *)rp;
               WANTALL();
               SWAP2(p->accelNum);
               SWAP2(p->accelDenum);
               SWAP2(p->threshold);
            }
            break;
         case X_GetPointerControl:
            {  register xReq *p = (xReq *)rp;
               WANTALL();
            }
            break;
         case X_SetScreenSaver:
            {  register xSetScreenSaverReq *p = (xSetScreenSaverReq *)rp;
               WANTALL();
               SWAP2(p->timeout);
               SWAP2(p->interval);
            }
            break;
         case X_GetScreenSaver:
            {  register xReq *p = (xReq *)rp;
               WANTALL();
            }
            break;
         case X_ChangeHosts:
            {  register xChangeHostsReq *p = (xChangeHostsReq *)rp;
               WANTALL();
               SWAP2(p->hostLength);
            }
            break;
         case X_ListHosts:
            {  register xListHostsReq *p = (xListHostsReq *)rp;
               WANTALL();
            }
            break;
         case X_SetAccessControl:
         case X_SetCloseDownMode:
            {  register xChangeModeReq *p = (xChangeModeReq *)rp;
               WANTALL();
            }
            break;
         case X_KillClient:
            {  register xResourceReq *p = (xResourceReq *)rp;
               SWAP4(p->id);
               WANTALL();
            }
            break;
         case X_RotateProperties:
            {  register xRotatePropertiesReq *p = (xRotatePropertiesReq *)rp;
               WANTALL();
               SWAP4(p->window);
               SWAP2(p->nAtoms);
               SWAP2(p->nPositions);
               SWAP4L(p+1, p->nAtoms);
            }
            break;
         case X_ForceScreenSaver:
            {  register xChangeModeReq *p = (xChangeModeReq *)rp;
               WANTALL();
            }
            break;
         case X_SetPointerMapping:
            {  register xSetPointerMappingReq *p = (xSetPointerMappingReq *)rp;
               WANTALL();
            }
            break;
         case X_GetPointerMapping:
            {  register xReq *p = (xReq *)rp;
               WANTALL();
            }
            break;
         case X_SetModifierMapping:
            {  register xSetModifierMappingReq *p =
						(xSetModifierMappingReq *)rp;
               WANTALL();
               SWAP4L(p+1, (len-sz_xSetModifierMappingReq)/4);
            }
            break;
         case X_GetModifierMapping:
            {  register xReq *p = (xReq *)rp;
               WANTALL();
            }
            break;
         case X_NoOperation:
            {  register xReq *p = (xReq *)rp;
               WANTALL();
            }
            break;
         default: {
               ext_request_swap(rp, left);
            }
            break;
      }
   }
   return nbytes;
}
#undef WANT
#undef WANTALL
#undef SWAPSHORTS
#undef TRUNCBYTES
