/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   rx.c								*
*									*
*	Routines for managing request/reply contexts.			*
*									*
*	Think of this as a kind of delayed procedure call.  When a	*
*	server-bound protocol request is generated by xmx, a marker	*
*	is placed in a queue noting that a certain procedure should	*
*	be called with certain parameters when the reply is received.	*
*	They're like callbacks, except we know that the order in	*
*	which they are registered is the order in which they are	*
*	called, and they are only called once (per server).		*
*									*
*	In this way, xmx/server dialogs are more easily written with	*
*	less explicit overhead.						*
*									*
************************************************************************/
#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/X.h>
#include <X11/Xproto.h>
#include "xmx.h"
#include "df.h"
#include "cx.h"
#include "rx.h"
#include "incl/rx.pvt.h"

static rx_t *rxfree;		/* free list */

/************************************************************************
*									*
*   rx_new								*
*   rx_free								*
*									*
*	Allocate/free a reply context stack structure.			*
*									*
************************************************************************/
rxq_t *
rx_new
   VOID
{
   register rxq_t *rxqp;

   if (MALLOC(rxqp, rxq_t *, sizeof(rxq_t)))
      return (rxq_t *)0;

   rxqp->repeat = 0;
   rxqp->in = rxqp->out = 0;
   rxqp->head = rxqp->tail = 0;
   rxqp->cstk = 0;

   return rxqp;
}

void
rx_free
   AL((rxqp))
   DB rxq_t *rxqp
   DE
{
   register rx_t *rxp, *last;

   for (rxp=rxqp->out; last=rxp;) {
      warn("rx_free: discarding contexts for pending replies\n");
      rxp = rxp->next;
      if (last->dfp)
         df_free(last->dfp);
      free_rx(last);
   }
   free(rxqp);
}

/************************************************************************
*									*
*   rx_queue								*
*									*
*	Put a context block on the end of the queue.  If a repeat	*
*	block is in progress, adds to it.				*
*									*
************************************************************************/
void
rx_queue
   AL((rxqp, major, minor, seqno, fptr))
   DB rxq_t *rxqp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD reply_proc fptr
   DE
{
   register rx_t *rxp;

   rxp = new_rx();
   rxp->major = major;
   rxp->minor = minor;
   rxp->seqno = seqno;
   rxp->count = 0;
   rxp->fptr = fptr;
   rxp->dfp = df_current();

   rxp->next = 0;
   if (rxqp->in)
      rxqp->in->next = rxp;
   else
      rxqp->out = rxp;
   rxqp->in = rxp;

   if (rxqp->repeat)
      if (rxqp->tail)
         rxqp->tail = rxp;
      else
         rxqp->head = rxqp->tail = rxp;
}

/************************************************************************
*									*
*   rx_push								*
*									*
*	Put a context block at the front of the queue.  If a repeat	*
*	block is in progress, this routine does not affect it.		*
*									*
************************************************************************/
void
rx_push
   AL((rxqp, major, minor, seqno, fptr))
   DB rxq_t *rxqp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD reply_proc fptr
   DE
{
   register rx_t *rxp;

   rxp = new_rx();
   rxp->major = major;
   rxp->minor = minor;
   rxp->seqno = seqno;
   rxp->count = 0;
   rxp->fptr = fptr;
   rxp->dfp = df_current();

   rxp->next = rxqp->out;
   rxqp->out = rxp;
}

/************************************************************************
*									*
*   rx_next_seqno							*
*									*
*	Returns zero if no replies are pending, non-zero if any are,	*
*	and the seqno of the next one is passed back.			*
*									*
************************************************************************/
int
rx_next_seqno
   AL((rxqp, seqnop))
   DB rxq_t *rxqp
   DD u16_t *seqnop
   DE
{
   if (rxqp->out) {
      *seqnop = rxqp->out->seqno;
      return 1;
   }
   return 0;
}

/************************************************************************
*									*
*   rx_begin_repeat							*
*   rx_end_repeat							*
*									*
*	Delimit a "repeat block" of contexts.  This is a marked		*
*	group of contexts that can have their individual reply		*
*	counts incremented as a group.					*
*									*
************************************************************************/
void
rx_begin_repeat
   AL((rxqp))
   DB rxq_t *rxqp
   DE
{
   rxqp->repeat = 1;
   /*
   **  if a previous repeat block remains, "delete" it
   */
   if (rxqp->head)
      rxqp->head = rxqp->tail = 0;
}

void
rx_end_repeat
   AL((rxqp))
   DB rxq_t *rxqp
   DE
{
   rxqp->repeat = 0;
}

void
rx_clear_repeat
   AL((rxqp))
   DB rxq_t *rxqp
   DE
{
   rxqp->head = rxqp->tail = 0;
}

/************************************************************************
*									*
*   rx_incr								*
*									*
*	Increment the reply counts of each context in a "repeat		*
*	block."								*
*									*
************************************************************************/
void
rx_incr
   AL((rxqp, n))
   DB rxq_t *rxqp
   DD int n
   DE
{
   register rx_t *rxp;

   for (rxp=rxqp->head; rxp; rxp=rxp->next) {
      rxp->count += n;
      if (rxp == rxqp->tail)
         break;		/* loop exit */
   }
}

rx_t *
rx_get
   AL((rxqp, seqno))
   DB rxq_t *rxqp
   DD u16_t seqno
   DE
{
   register rx_t *rxp;

   for (rxp=rxqp->out; rxp; rxp=rxp->next)
      if (rxp->seqno == seqno)
         return rxp;
   return 0;
}

/************************************************************************
*									*
*   rx_continue								*
*									*
*	Find a client-associated reply context, and execute the		*
*	continuation function.  Contexts may be reused: if identical	*
*	requests went out to multiple servers, all replies will use	*
*	the same context.						*
*									*
*	When the reply context queue empties, the continuation		*
*	context stack is checked, and any pending functions there	*
*	are then executed, until either the reply context queue		*
*	fills again or both are empty.					*
*									*
************************************************************************/
void
rx_continue
   AL((rxqp, rxp, sp, cp, seqno, chp))
   DB rxq_t *rxqp
   DD rx_t *rxp
   DD server_t *sp
   DD client_t *cp
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   register rx_t *trxp;

   DEBUG1(D_NETWORK, "<reply to [%d]>\n", cp ? cp->fd : 0);
#ifdef DEBUG
   if (debug && chp)
      if (debug & D_PROTO3 || debug & D_PROTOq && sp == qservp)
         if (chp->lpp)
            warn("<partial data, %d bytes>\n", buf_chunksize(chp));
         else {
            dprx_chunk_beg(chp);
            dprx_reply((xGenericReply *)buf_data(chp), rxp->major, rxp->minor);
            dprx_chunk_end();
         }
#endif

   if (rxp->fptr && chp) {
      if (rxp->dfp)
         df_reset(rxp->dfp);
      (*(rxp->fptr))(rxp->dfp, sp, cp, rxp->major, rxp->minor, seqno, chp);
   }
   switch (rxp->count) {
      case 0:
      case 1:
         /*
         **  nuke it
         */
         if (rxqp->out == rxp) {	/* this is the most common case */
            rxqp->out = rxp->next;
            if (rxqp->in == rxp)
               rxqp->in = rxqp->out;
         }
         else {
            for (trxp=rxqp->out; trxp && trxp->next != rxp; trxp=trxp->next);
            if (trxp) {
               trxp->next = rxp->next;
               if (rxp->next == 0)
                  rxqp->in = trxp;
            }
            else
               warn("rx_continue: context not found in list!\n");
         }
         if (rxp->dfp)
            df_free(rxp->dfp);
         free_rx(rxp);
         /*
         **  if reply context queue is now empty and
         **  continue context stack is not, pop the latter
         */
         while (rxqp->out == 0 && rxqp->cstk)
            cx_continue(rxqp);
         break;
      default:			/* not done yet */
         rxp->count--;
         break;
   }
}

/*
**  new_rx	- allocate a context
*/
static rx_t *
new_rx
   VOID
{
   register rx_t *rxp;

   if (rxfree) {
      rxp = rxfree;
      rxfree = rxfree->next;
   }
   else if (MALLOC(rxp, rx_t *, sizeof(rx_t)))
      return 0;

   return rxp;
}

/*
**  free_rx	- put context on free list
*/
void
free_rx
   AL((rxp))
   DB rx_t *rxp
   DE
{
   rxp->next = rxfree;
   rxfree = rxp;
}

/************************************************************************
*									*
*   rx_free_freelist							*
*									*
************************************************************************/
void
rx_free_freelist
   VOID
{
   register rx_t *rxp, *last;

   for (rxp=rxfree; last=rxp;) {
      rxp = rxp->next;
      free(last);
   }
   rxfree = 0;
}

/************************************************************************
*									*
*   rx_print								*
*									*
************************************************************************/
void
rx_print
   AL((rxqp))
   DB rxq_t *rxqp
   DE
{
   register int inblock;
   register rx_t *rxp;

   warn("reply context queue [0x%x] repeat [%d]\n", rxqp, rxqp->repeat);
   
   inblock = 0;
   for (rxp=rxqp->out; rxp; rxp=rxp->next) {
      if (rxp == rxqp->head)
         inblock = 1;
      warn("%s%s seqno %d count %d fptr 0x%x dfp 0x%x\n",
				inblock ? "  *" : "   ",
				dprx_reqType_str(rxp->major),
				rxp->seqno, rxp->count, rxp->fptr, rxp->dfp);
      if (rxp == rxqp->tail)
         inblock = 0;
   }
}
