/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   sm.c								*
*									*
*	Sequence number mapping.  Seq maps are tables of seq map	*
*	entries.  Entries map server sequence numbers to clients and	*
*	client sequence numbers.					*
*									*
*	Maps are linked lists of arrays of map entries.  Each entry	*
*	provides a mapping for from 1 to SM_MAXCOUNT consecutive	*
*	requests from a single client.  The arrays (or "blocks") are	*
*	maintained as a FIFO, with new entries added at one end, while	*
*	old blocks are discarded as they are deemed unneeded.		*
*									*
*	Gaps in sequences are explicitly allowed, since we may want	*
*	to selectively keep track for efficiency's sake.  Also, adds	*
*	are called a lot more often than gets, so those routines	*
*	should do as little as possible.				*
*									*
************************************************************************/
#include "xmx.h"
#include "df.h"
#include "cx.h"
#include "rx.h"
#include "sm.h"
#include "incl/sm.pvt.h"

static sm_blk_t *smblkfree;

/************************************************************************
*									*
*   sm_alloc								*
*									*
*	Allocate a new sequence number map.				*
*									*
************************************************************************/
sm_t *
sm_alloc
   VOID
{
   sm_t *smp;

   if (MALLOC(smp, sm_t *, sizeof(sm_t)))
      return (sm_t *)0;

   smp->hix = smp->tix = 0;
   smp->sseqno = 1;		/* the first seqno is 1 */
   smp->rseqno = 0;			/* this is a garbage value */
   smp->bseqno = 1;
   smp->blkcnt = 1;
   /*
   ** there's always a block
   */
   smp->tail = smp->head = new_block();

   smp->tail->next = 0;
   /*
   **  the first client will probably be xmx, and the first seqno
   **  will be one, so we prime it this way.
   */
   smp->tail->e[0].clinum = 0;
   smp->tail->e[0].count = 0;
   smp->tail->e[0].cseqno = 1;

   return smp;
}

void
sm_free
   AL((smp))
   DB sm_t *smp
   DE
{
   register sm_blk_t *smbp, *last;

   for (smbp=smp->head; last=smbp;) {
      smbp = smbp->next;
      free_block(last);
   }
   free(smp);
}

/************************************************************************
*									*
*   sm_next_seqno							*
*									*
************************************************************************/
u16_t
sm_next_seqno
   AL((smp))
   DB sm_t *smp
   DE
{
   return smp->sseqno;
}

/************************************************************************
*									*
*   sm_add								*
*									*
*	Add or extend an entry in the given map.  Creates blocks as	*
*	needed, and removes old ones if there are many and they are	*
*	not needed.							*
*									*
************************************************************************/
void
sm_add
   AL((smp, clinum, cseq, n))
   DB sm_t *smp
   DD u8_t clinum
   DD u16_t cseq
   DD uint_t n		/* how many requests? */
   DE
{
   register uint_t tix, t;
   u16_t seqno;
   register sm_blk_t *sbp, *last;

   DEBUG4(D_SM, "sm_add: [%d] seqno %d->%d [%d]\n",
				smp->tix, cseq, smp->sseqno, n);
   tix = smp->tix;
   sbp = smp->tail;

   smp->sseqno += (u16_t)n;		/* sseqno is never used... TODO */
   /*
   **	see if we can increment the current entry
   */
   if (	clinum == sbp->e[tix].clinum &&
	cseq - sbp->e[tix].count == sbp->e[tix].cseqno) {
      if (n + (uint_t)sbp->e[tix].count > SM_MAXCOUNT) {
         t = SM_MAXCOUNT - sbp->e[tix].count;
         n -= t;
         cseq += t;
         sbp->e[tix].count = SM_MAXCOUNT;
      }
      else {
         sbp->e[tix].count += (u8_t)n;
         return;			/* RETURN */
      }
   }
   /*
   **	create new entries as needed
   */
   for (tix++;;tix++) {
      if (tix == SM_BLKSZ) {
         sbp = new_block();
         sbp->next = 0;
         smp->tail->next = sbp;
         smp->tail = sbp;
         tix = 0;
         smp->blkcnt++;
         /*
         **  if we're accumulating blocks, remove the oldest ones.
         **  adjusts the current block pointer.  this only happens
         **  if a client is spewing requests that generate no
         **  events, errors or replies.
         */
         if (smp->blkcnt > 3) {		/* 3 is pretty arbitrary */
            if (rx_next_seqno(rx_q(client_ptr(clinum)), &seqno)) {
               /*
               **  advance the map to the oldest pending reply
               */
               if (advance(smp, seqno)) {
                  DEBUG0(D_SM, "sm_add: dropping unneeded blocks...\n");
                  DEBUG3(D_SM, "\t...[%d] not found (bseqno[%d], last[%d])\n",
					seqno, smp->bseqno, smp->sseqno-1);
               }
            }
            else {
               /*
               **  just drop all but the most recent three
               */
               while (smp->blkcnt > 3) {
                  for (; smp->hix < SM_BLKSZ; smp->hix++)
                     smp->bseqno += smp->head->e[smp->hix].count;
                  smp->hix = 0;
                  last = smp->head;
                  smp->head = smp->head->next;
                  free_block(last);
                  smp->blkcnt--;
               }
            }
         }
      }
      sbp->e[tix].clinum = clinum;
      sbp->e[tix].cseqno = cseq;
      if (n > SM_MAXCOUNT) {
         n -= SM_MAXCOUNT;
         cseq += SM_MAXCOUNT;
         sbp->e[tix].count = SM_MAXCOUNT;
      }
      else {
         sbp->e[tix].count = (u8_t)n;
         break;				/* LOOP EXIT */
      }
   }
   smp->tix = tix;
}

/************************************************************************
*									*
*   sm_mapping								*
*									*
*	Returns the client->server sequence number mapping that		*
*	resulted from a previous add.  This is a brute force algorithm	*
*	but it is used only for debugging.				*
*									*
************************************************************************/
u16_t
sm_mapping
   AL((smp, clinum, cseq))
   DB sm_t *smp
   DD u8_t clinum
   DD u16_t cseq
   DE
{
   register uint_t max, hix;
   register u16_t bseqno;
   register sm_blk_t *sbp;

   hix = smp->hix;
   bseqno = smp->bseqno;
   sbp = smp->head;

   max = sbp->next == 0 ? smp->tix : SM_BLKSZ - 1;
   for (;;) {
      if (clinum == sbp->e[hix].clinum) {
         if (	SEQNO_GE(cseq, sbp->e[hix].cseqno) &&
		SEQNO_LT(cseq, sbp->e[hix].cseqno + sbp->e[hix].count)) {

            if (cseq < sbp->e[hix].cseqno)
               bseqno += cseq + 65536 - sbp->e[hix].cseqno;
            else
               bseqno += cseq - sbp->e[hix].cseqno;

            return bseqno;		/* success - loop exit */
         }
      }
      bseqno += sbp->e[hix].count;
      if (++hix > max) {
         hix = 0;
         if ((sbp = sbp->next) == 0)
            break;			/* not found - loop exit */
         max = sbp->next == 0 ? smp->tix : SM_BLKSZ - 1;
      }
   }
   return 0;	/* not a good error return */
}

/************************************************************************
*									*
*   sm_get_last								*
*									*
*	Return the same information as the last successful sm_get.	*
*	If no successful sm_get's have taken place, returns garbage.	*
*	If many sm_add's took place since the last sm_get, the response	*
*	can be garbage.  But this is used for partial packets, so	*
*	neither of those will happen.					*
*									*
************************************************************************/
void
sm_get_last
   AL((smp, clinump, cliseqp))
   DB sm_t *smp
   DD u8_t *clinump
   DD u16_t *cliseqp
   DE
{
   *clinump = smp->head->e[smp->hix].clinum;
   *cliseqp = smp->head->e[smp->hix].cseqno + (smp->rseqno - smp->bseqno);
}

/************************************************************************
*									*
*   sm_get								*
*									*
*	Returns the client number and client sequence numbers		*
*	corresponding to the server sequence number given.  Returns	*
*	non-zero if not found.						*
*									*
*	The same seqno may be retrieved repeatedly until a newer seqno	*
*	is retrieved.  After that, there is no guarantee that older	*
*	seqnos will be found.						*
*									*
*	Works correctly when sequence numbers roll over.		*
*									*
************************************************************************/
int
sm_get
   AL((smp, seqno, clinump, cliseqp))
   DB sm_t *smp
   DD u16_t seqno
   DD u8_t *clinump
   DD u16_t *cliseqp
   DE
{
   smp->rseqno = seqno;

   if (advance(smp, seqno)) {
      DEBUG3(D_SM, "sm_get: [%d] not found (bseqno[%d], last[%d])\n",
					seqno, smp->bseqno, smp->sseqno-1);
      return -1;
   }
   *clinump = smp->head->e[smp->hix].clinum;
   *cliseqp = smp->head->e[smp->hix].cseqno + (seqno - smp->bseqno);

   return 0;
}

/*
**  advance
**
**	search for the given seqno in the map, freeing map blocks
**	as we advance beyond them.  returns zero if successful and
**	leaves the map head at the entry containing the seqno.
*/
static int
advance
   AL((smp, seqno))
   DB sm_t *smp
   DD u16_t seqno
   DE
{
   register uint_t max, hix = smp->hix;
   register u16_t bseqno = smp->bseqno;
   register sm_blk_t *sbp, *last;

   if (SEQNO_GE(seqno, bseqno)) {
      sbp = smp->head;

      max = sbp->next == 0 ? smp->tix : SM_BLKSZ - 1;

      while (SEQNO_GE(seqno, bseqno + sbp->e[hix].count)) {
         bseqno += sbp->e[hix].count;
         if (++hix > max) {
            if (sbp->next == 0)
               return -1;

            last = sbp;
            sbp = sbp->next;
            free_block(last);
#ifdef DEBUG
	    if (smp->blkcnt == 0)
               quit(-1, "sm.c/advance: smp->blkcnt is zero - fatal.\n");
#endif
            smp->blkcnt--;
            hix = 0;
            max = sbp->next == 0 ? smp->tix : SM_BLKSZ - 1;
         }
      }
      smp->head = sbp;
   }
   else
#ifdef DEBUG_ABORT
   {
      warn("sm.c/advance: seqno [%d] precedes oldest [%d] in map\n", seqno, bseqno);
      sm_print(smp);
      abort();
   }
#else
      return err(-1, "sm.c/advance: seqno [%d] precedes oldest [%d] in map\n",
						seqno, bseqno);
#endif

   smp->hix = hix;
   smp->bseqno = bseqno;

   return 0;
}

/************************************************************************
*									*
*   sm_print								*
*									*
************************************************************************/
void
sm_print
   AL((smp))
   DB sm_t *smp
   DE
{
   register uint_t ix, max;
   register u16_t sseqno;
   register sm_blk_t *sbp;

   warn("smp [0x%x] sseqno [%d] rseqno [%d] blkcnt [%d] hix [%d] tix [%d]\n",
	smp, smp->sseqno, smp->rseqno, smp->blkcnt, smp->hix, smp->tix);

   sseqno = 0;
   ix = smp->hix;
   sseqno = smp->bseqno;

   for (sbp=smp->head; sbp; sbp=sbp->next) {
      warn("[");	/* indicates start of block */

      max = sbp == smp->tail ? smp->tix : SM_BLKSZ - 1;

      for (; ix<=max; ix++) {
         warn("\t[%d] clinum [%d] sseqno [%d] cseqno [%d] count [%d]\n",
					ix, sbp->e[ix].clinum, sseqno,
					sbp->e[ix].cseqno, sbp->e[ix].count);
         sseqno += sbp->e[ix].count;
      }
      warn("]");
      ix = 0;
   }
   warn("\n");
}

static sm_blk_t *
new_block
   VOID
{
   register sm_blk_t *sbp;
 
   if (smblkfree) {
      sbp = smblkfree;
      smblkfree = smblkfree->next;
   }
   else if (MALLOC(sbp, sm_blk_t *, sizeof(sm_blk_t)))
      return 0;
 
   return sbp;
}

static void
free_block
   AL((sbp))
   DB sm_blk_t *sbp
   DE
{
   sbp->next = smblkfree;
   smblkfree = sbp;
}
