/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   socket.c								*
*									*
*	All socket setup and io system calls are done here.		*
*									*
************************************************************************/
#include <sys/param.h>
#include <sys/types.h>
#include <sys/socket.h>
#ifdef UNIXCONN
#include <sys/un.h>
#endif
#ifdef TCPCONN
#include <netinet/in.h>
#include <netinet/tcp.h>
#endif
#include <netdb.h>
#include <fcntl.h>
#include <errno.h>
#ifdef _AIX
#include <sys/select.h>
#undef FNDELAY
#endif
#ifndef FNDELAY
#include <sys/ioctl.h>
#endif
#ifdef SVR4
#include <sys/filio.h>
#endif

#include <xmcp.h>

#include "xmx.h"
#include "incl/socket.pvt.h"

#define X_UNIX_DIR	"/tmp/.X11-unix"
#define X_UNIX_PATH	"/tmp/.X11-unix/X"
#define X_TCP_PORT	6000

static char xpath[MAXPATHLEN];
static char xmcpath[MAXPATHLEN];

/************************************************************************
*									*
*   socket_as_Xserver							*
*									*
*	Create X server style sockets, bind to well known addresses	*
*	and listen on them.						*
*									*
************************************************************************/
int
socket_as_Xserver
   AL((display))
   DB int display
   DE
{
   register int rv;
   mode_t umsk;

   umsk = umask(0);
   if (!mkdir(X_UNIX_DIR, 0777))
      chmod(X_UNIX_DIR, 0777);

   (void)sprintf(xpath, "%s%d", X_UNIX_PATH, display);
   rv = create_port(XPORT, X_TCP_PORT+display, xpath);

   (void)umask(umsk);
   return rv;
}

/************************************************************************
*									*
*   socket_as_xmc							*
*									*
*	Create X multiplexor control server sockets.			*
*									*
************************************************************************/
int
socket_as_xmc
   AL((display))
   DB int display
   DE
{
   register int rv;
   mode_t umsk;

   umsk = umask(0);
   if (!mkdir(XMC_UNIX_DIR, 0777))
      chmod(XMC_UNIX_DIR, 0777);

   (void)sprintf(xmcpath, "%s%d", XMC_UNIX_PATH, display);
   rv = create_port(XMCPORT, XMC_TCP_PORT-display, xmcpath);

   (void)umask(umsk);
   return rv;
}

/************************************************************************
*									*
*   socket_cleanup							*
*									*
*	Cleanup upon exit.						*
*									*
************************************************************************/
void
socket_cleanup
   VOID
{
#ifdef UNIXCONN
   if (*xpath)
      (void)unlink(xpath);
   if (*xmcpath)
      (void)unlink(xmcpath);
#endif
}

static int
create_port
   AL((type, portno, path))
   DB u16_t type
   DD u16_t portno
   DD char *path
   DE
{
   register int len, i, s;
   struct linger linger;
   port_t *portp;
#ifdef UNIXCONN
   static struct sockaddr_un usock;
#endif
#ifdef TCPCONN
   static struct sockaddr_in isock;
#endif

#ifdef UNIXCONN
					/* create UNIX domain socket */
   usock.sun_family = AF_UNIX;
   (void)strcpy(usock.sun_path, path);
   len = strlen(usock.sun_path);
#ifdef _AIX
   usock.sun_len = ++len;
#endif
   (void)unlink(usock.sun_path);
   if ((s = socket(AF_UNIX, SOCK_STREAM, 0)) < 0)
      return perr(-1, usock.sun_path);

   len += sizeof(usock.sun_family);
#ifdef _AIX
   len += sizeof(usock.sun_len);
#endif
   if (bind(s, (struct sockaddr *)&usock, len))
      return perr(-1, usock.sun_path);

   if (listen(s, 5))
      return perr(-1, usock.sun_path);

   if (MALLOC(portp, port_t *, sizeof(port_t)))
      return -1;
   portp->family = FamilyLocal;
   main_open_conn(s, type, (void *)portp);
#endif

					/* create TCP socket */
#ifdef TCPCONN
   isock.sin_family = AF_INET;
   isock.sin_port = htons(portno);
   isock.sin_addr.s_addr = htonl(INADDR_ANY);
   if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0)
      return perr(-1, "tcp socket");

   for (i=20; i && bind(s, (struct sockaddr *)&isock, sizeof(isock)); i--) {
      pwarn("create_port: tcp bind (port %d)", portno);
      sleep(10);
   }
   if (i == 0)
      return perr(-1, "tcp socket");

   linger.l_onoff = 0;
   linger.l_linger = 0;
   if (setsockopt(s, SOL_SOCKET, SO_LINGER, (char *)&linger, sizeof(linger)))
      return perr(-1, "tcp socket");

   if (listen(s, 5))
      return perr(-1, "tcp socket");

   if (MALLOC(portp, port_t *, sizeof(port_t)))
      return -1;
   portp->family = FamilyInternet;
   main_open_conn(s, type, (void *)portp);
#endif

   return 0;
}

/************************************************************************
*									*
*   socket_as_xdm							*
*									*
*	Create X Display Manager style sockets.				*
*									*
************************************************************************/
int
socket_as_xdm
   AL((fdsp))
   DB int **fdsp
   DE
{
   int nfd = 0;

   return nfd;
}

/************************************************************************
*									*
*   socket_accept							*
*									*
*	Accept a pending connection.  Returns the new socket file	*
*	descriptor or -1 if an error occurred.				*
*									*
************************************************************************/
int
socket_accept
   AL((s))
   DB int s
   DE
{
   int i;
   int ns;
   int addrlen;
   union {
      struct sockaddr sa;
#ifdef UNIXCONN
      struct sockaddr_un s_un;
#endif
#ifdef TCPCONN
      struct sockaddr_in s_in;
#endif
   }addr;
#ifdef FIONBIO
   int flag = 1;
#endif

   addrlen = sizeof(addr);
   if ((ns = accept(s, &addr.sa, &addrlen)) < 0)
      return perr(-1, "accept");

#ifdef TCPCONN
   if (addr.sa.sa_family == AF_INET) {
      i = 1;
      if (setsockopt(ns, IPPROTO_TCP, TCP_NODELAY, (char *)&i, sizeof(i)))
         return perr(-1, "socket");
   }
#endif
#ifdef O_NONBLOCK
   (void)fcntl(ns, F_SETFL, O_NONBLOCK);
#else
#ifdef FNDELAY
   (void)fcntl(ns, F_SETFL, FNDELAY);
#else
#ifdef FIONBIO
   (void)ioctl(ns, FIONBIO, &flag);
#else
   cannot make socket non blocking	/* break compile */
#endif
#endif
#endif

   return ns;
}

/************************************************************************
*									*
*   socket_addr								*
*									*
************************************************************************/
int
socket_addr
   AL((s, family, ap))
   DB int s
   DD u16_t family
   DD hostaddr_t *ap
   DE
{
   int addrlen;
   static union {
      struct sockaddr sa;
#ifdef UNIXCONN
      struct sockaddr_un s_un;
#endif
#ifdef TCPCONN
      struct sockaddr_in s_in;
#endif
   }addr;

   if (family == FamilyLocal) {
      ap->family = FamilyLocal;
      ap->length = 0;
   }
   else {
      addrlen = sizeof(addr);
      if (getpeername(s, &addr.sa, &addrlen))
         return perr(-1, "getpeername");
      switch (addr.sa.sa_family) {
#ifdef UNIXCONN
         case AF_UNIX:
            ap->length = strlen(addr.s_un.sun_path);
            ap->address = addr.s_un.sun_path;
            break;
#endif
         case AF_INET:
            ap->length = sizeof(addr.s_in.sin_addr);
            ap->address = (char *)&addr.s_in.sin_addr;
            break;
         default:
            return err(-1, "socket_addr: address family [%d] not supported\n",
							addr.sa.sa_family);
      }
      if ((ap->family = util_fam_utox(addr.sa.sa_family)) == FamilyWild)
         return -1;
   }
   return 0;
}

/************************************************************************
*									*
*   socket_as_client							*
*									*
*	Connect to an X server.  Returns the socket file descriptor or	*
*	-1 if an error occurred.					*
*									*
************************************************************************/
int
socket_as_client
   AL((family, length, address, display, fdp))
   DB u16_t family
   DD u16_t length
   DD char *address
   DD u16_t display
   DD int *fdp
   DE
{
   register int s, len;
   struct sockaddr *sock;
#ifdef UNIXCONN
   static struct sockaddr_un usock;
#endif
#ifdef TCPCONN
   static struct sockaddr_in isock;
#endif
#ifndef FNDELAY
   int flag = 1;
#endif

   switch (family) {
      case FamilyInternet:
         isock.sin_family = AF_INET;
         isock.sin_port = htons((u_short)(X_TCP_PORT + display));
         if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0)
            return perr(-1, "tcp socket");
         bcopy((char *)address, (char *)&isock.sin_addr, length);
         sock = (struct sockaddr *)&isock;
         len = sizeof(isock);
         break;
      case FamilyDECnet:
         errno = EAFNOSUPPORT;		/* fake system error */
         return err(-1, "DECnet protocol not supported\n");
      case FamilyChaos:
         errno = EAFNOSUPPORT;		/* fake system error */
         return err(-1, "Chaos protocol not supported\n");
      case FamilyLocal:
#ifdef UNIXCONN
         usock.sun_family = AF_UNIX;
         sprintf(usock.sun_path, "%s%d", X_UNIX_PATH, display);
         if ((s = socket(AF_UNIX, SOCK_STREAM, 0)) < 0)
            return perr(-1, usock.sun_path);
         sock = (struct sockaddr *)&usock;
         len = sizeof(usock);
#endif
         break;
   }
#ifdef O_NONBLOCK
   (void)fcntl(s, F_SETFL, O_NONBLOCK);
#else
#ifdef FNDELAY
   (void)fcntl(s, F_SETFL, FNDELAY);
#else
#ifdef FIONBIO
   (void)ioctl(s, FIONBIO, &flag);
#else
   cannot make socket non blocking	/* break compile */
#endif
#endif
#endif

   *fdp = s;
   if (connect(s, sock, len) < 0) {	/* this is a non-blocking connect */
      if (errno != EINPROGRESS) {
         pwarn("connect");
         (void)close(s);
      }
      return -1;
   }
   return 0;
}
