/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   time.c								*
*									*
************************************************************************/
#include <sys/time.h>
#include <sys/types.h>
#ifdef _AIX
#include <sys/select.h>
#endif
#include "xmx.h"
#include "fd.h"
#include "incl/time.pvt.h"

void
time_update
   AL((stamp))
   DB timestamp_t stamp
   DE
{
   register u32_t ms;
   struct timeval tv;

   if (stamp)
      ms = stamp;
   else {
      gettimeofday(&tv, 0);
      ms = tv.tv_sec * 1000 + tv.tv_usec / 1000;
   }
   if (ms < vtime.ms)
      if (vtime.ms - ms > (u32_t)1<<31)
         vtime.mo++;
      else
         return;	/* time can't go backward */

   vtime.ms = ms ? ms : 1;	/* no zero timestamps */
}

void
time_convert
   AL((stamp, tp))
   DB timestamp_t stamp
   DD etime_t *tp
   DE
{
   if (stamp == 0) {
      tp->mo = vtime.mo;
      tp->ms = vtime.ms;
   }
   else {
      tp->mo = vtime.mo;
      tp->ms = stamp;
      if (stamp < vtime.ms) {
         if (vtime.ms - stamp > (u32_t)1<<31)
            tp->mo++;
      }
      else if (stamp - vtime.ms > (u32_t)1<<31)
         tp->mo--;
   }
}

/************************************************************************
*									*
*   Connection timeout routines.					*
*									*
************************************************************************/
typedef struct _to_t {
   int			fd;	/* connection */
   u16_t		to;	/* timeout in seconds */
   long			time;	/* time of connection */
   struct _to_t *	next;
}to_t;

static to_t *timeouts;
static fd_set tofds;

/************************************************************************
*									*
*   time_start								*
*									*
*	Associate a connection time with a file descriptor.		*
*									*
************************************************************************/
void
time_start
   AL((fd, to))
   DB int fd
   DD u16_t to
   DE
{
   register to_t *tp;
   struct timeval tv;

   if (MALLOC(tp, to_t *, sizeof(to_t)))
      quit(-1, "time_start: fatal memory allocation error\n");

   gettimeofday(&tv, 0);
   tp->time = tv.tv_sec;

   tp->fd = fd;
   tp->to = to;
   tp->next = timeouts;
   timeouts = tp;

   FD_SET(fd, &tofds);
   ntimeouts++;
}

/************************************************************************
*									*
*   time_out								*
*									*
*	Iterate through timed-out connections.  Returns one expired	*
*	connection each time it is invoked, or -1 if there are none.	*
*									*
************************************************************************/
int
time_out
   VOID
{
   int fd;
   to_t *np;
   static to_t *lp;
   static struct timeval tv;

   if (lp == 0) {
      gettimeofday(&tv, 0);
      np = timeouts;
   }
   else
      np = lp->next;

   while (np) {
      if (tv.tv_sec - np->time >= (long)np->to) {
         fd = np->fd;
         if (lp)
            lp->next = np->next;
         else
            timeouts = np->next;
         free(np);
         FD_CLR(fd, &tofds);
         ntimeouts--;
         return fd;
      }
      else {
         lp = np;
         np = np->next;
      }
   }
   lp = 0;
   return -1;
}

/************************************************************************
*									*
*   time_end								*
*									*
*	Disassociate a file descriptor from a connection time.		*
*									*
************************************************************************/
void
time_end
   AL((fd))
   DB int fd
   DE
{
   to_t *tp, *lp;

   if (FD_ISSET(fd, &tofds)) {
      lp = 0;
      for (tp=timeouts; tp && tp->fd != fd; lp=tp,tp=tp->next);

      if (tp) {
         if (lp)
            lp->next = tp->next;
         else
            timeouts = tp->next;

         free(tp);
         FD_CLR(fd, &tofds);
         ntimeouts--;
      }
      else
         warn("time_end: sanity - no such fd [%d]\n", fd);
   }
}

/************************************************************************
*									*
*   time_str								*
*									*
*	Return a static buffer containing the string representation of	*
*	an etime_t struct.  Up to 6 values returned by this routine	*
*	can be used at once, before they begin to be overwritten.	*
*	For debugging only.						*
*									*
************************************************************************/
char *
time_str
   AL((tp))
   DB etime_t *tp
   DE
{
   register char *cp;
   static int cur;		/* current buffer */
   static char buf[6][24];	/* six 24 character buffers */

   if (tp == 0)
      return "now";

   cp = &buf[cur++][0];
   cur %= 6;
   sprintf(cp, "%u/%u", tp->mo, tp->ms);  /* cannot exceed 24 in length */
   return cp;
}
