/* MAIN PART OF DISASSEMBLER */

#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "inst-dis.h"
#include "z80-asm.h"
#include "disasm.h"
#include "regs.h"
#include "console.h"
#include "asm.h"
#include "file.h"

_uchar A,B,C,D,E,F,H,L,I,R;
_uchar A_,B_,C_,D_,E_,F_,H_,L_;
_uchar IM,EI;
_ushort PC,SP,IX,IY,MEMP;
char string[256];
char load_name[256];
char save_name[256];


int turbo;
int run;
int current_pc;
int follow;
int row;     /* row for printing next instruction */
int mode;    /* mode for decode functions: printing (M_PRINT), executing (M_EXEC) */
_uchar memory[65536];
_uchar tmp_memory[64];
_uchar *memory_ptr;


void take_line(char *line)
{
 sprintf(line," %s",string);
}


/* pressing the R key switches run flag to 0 */
void warning(char* message)
{
 static char txt[512];
 unsigned char k;
/* c_bell(); */
 c_goto(0,23);
 c_setcolor(9);
 sprintf(txt,"Warning: %s",message);
 c_print(txt);
 c_setcolor(7);
 k=c_getkey();
 if (k=='r'||k=='R') run=0;
 c_clear(0,22,39,22);
 c_clear(0,23,79,23);
}


/* pressing the R key switches run flag to 0 */
void error(int n,char *line,char* message)
{
 static char txt[512];
 unsigned char k;
 n=n;
 c_bell();
 c_goto(0,22);
 c_print(line);
 c_goto(0,23);
 c_setcolor(9);
 sprintf(txt,"Error: %s",message);
 c_print(txt);
 c_setcolor(7);
 k=c_getkey();
 if (k=='r'||k=='R') run=0;
 c_clear(0,22,39,22);
 c_clear(0,23,79,23);
}


char
__printable(char c)
{
 return (c>=32&&c<127)?c:'.';
}


/* prints register to str */
void
flag_str (char *str,_ushort arg)
{
 static const char *f[N_FLAGS+1]={"","nz","nc","c","pe","po","m","p","z"};

 sprintf(str,"%s",f[arg%(N_FLAGS+1)]);  /* ochrana pameti */
 return;
}


void
reg_str (char *str,_ushort arg,_uchar type)
{
 static const char *r[N_REGISTERS+1]={"","a","b","c","d","e","h","l","i","r","af","bc","de","hl","sp","ix","iy","af'"};
 switch (type)
 {
  case A_REG:
  sprintf(str,"%s",r[arg%(N_REGISTERS+1)]);  /* ochrana pameti */
  return;

  case A_PODLE_REG:
  sprintf(str,"(%s)",r[arg%(N_REGISTERS+1)]);  /* ochrana pameti */
  return;

  case A_PODLE_IX_PLUS:
  sprintf(str,"(ix+%u)",arg);
  return;

  case A_PODLE_IY_PLUS:
  sprintf(str,"(iy+%u)",arg);
  return;
 }
}


/* converts ix instrucions to iy version */
void
toiy(unsigned char *t1,unsigned char *a1,unsigned char *t2,unsigned char *a2)
{
 switch (*t1)
 {
  case A_PODLE_IX_PLUS:
  *t1=A_PODLE_IY_PLUS;
  break;

  case A_REG:
  case A_PODLE_REG:
  if (*a1==R_IX)*a1=R_IY;
  break;
 }
 
 switch (*t2)
 {
  case A_PODLE_IX_PLUS:
  *t2=A_PODLE_IY_PLUS;
  break;

  case A_REG:
  case A_PODLE_REG:
  if (*a2==R_IX)*a2=R_IY;
  break;
 }
}


/* returns pointer to an 8-bit register number n (n=R_A,R_B,...) */
_uchar * 
reg_ptr(_ushort n)
{
 _uchar *r[9]={&A,&B,&C,&D,&E,&H,&L,&I,&R};
 return r[(n-1)%9];   /* ochrana pameti */
}


/* reads and calls instruction */
/* exec: 0=don't exec instruction, only adjust PC                     *
 *       1=execute instruction                                        *
 * store_pc: pointer to value stored to PC before calling instruction *
 *           or null (PC untouched)                                   */
void
decode(_ushort *store_pc,int exec)
{
 static struct instruction_type current_instr;
 int a,x1=0,x2=0;
 _ushort a1=0,a2=0;
 _ushort old_pc=PC;
 int iy=0;
 _uchar b=0;
 struct instruction_type *p;
 
 switch (memory_ptr[PC])
 {
  case 0xcb:    /* 0xcb, opcode */
  p=instr_cb;
  PC++;
  goto normal;

  case 0xfd:
  iy=1;
  case 0xdd:
  p=instr_dd;
  PC++;
  if (memory_ptr[PC]!=0xcb)goto normal;   /* 0xdd (0xfd), opcode */
  p=instr_dd_cb;   /* 0xdd (0xfd), 0xcb, byte, opcode    table for IX and IY instructions is the same */
  x1=2;x2=2;
  PC++;
  b=memory_ptr[PC];
  PC++;
  p+=memory_ptr[PC];
  PC++;
  if (!(p->func))
  {
   current_instr.func=f_defb;
   current_instr.arg1=memory_ptr[old_pc];
   current_instr.type1=0;
   current_instr.arg2=0;
   current_instr.type2=0;
   PC=old_pc+1;
  }
  else
  {
   current_instr.func=p->func;
   current_instr.arg1=((p->arg1)==1)?b:p->arg1;
   current_instr.type1=p->type1;
   current_instr.arg2=((p->arg2)==1)?b:p->arg2;
   current_instr.type2=p->type2;
   if (iy)
    toiy(&current_instr.type1,
         &current_instr.arg1,
	 &current_instr.type2,
	 &current_instr.arg2);
  }
  break;

  case 0xed:  /* 0xed, opcode    table for IX and IY instructions is the same */
  p=instr_ed;
  PC++;
  goto normal;

  default:  /* opcode */
  p=instr_normal;
  normal:
  p+=memory_ptr[PC];
  PC++;
  if (!(p->func))
  {
   current_instr.func=f_defb;
   current_instr.arg1=memory_ptr[old_pc];
   current_instr.type1=0;
   current_instr.arg2=0;
   current_instr.type2=0;
   PC=old_pc+1;
  }
  else
  {
   current_instr.func=p->func;
   current_instr.arg1=p->arg1;
   current_instr.type1=p->type1;
   current_instr.arg2=p->arg2;
   current_instr.type2=p->type2;
   if (iy)
    toiy(&current_instr.type1,
         &current_instr.arg1,
	 &current_instr.type2,
	 &current_instr.arg2);
  }
  break;
 }
 
 if (!x1&&(current_instr.type1==A_NUM||current_instr.type1==A_PODLE_NUM||current_instr.type1==A_PODLE_IY_PLUS||current_instr.type1==A_PODLE_IX_PLUS))
 {
  x1=1;
  for (a=0;a<current_instr.arg1;a++,PC++)
   a1+=(memory_ptr[PC])<<(a<<3);
 }
 if (!x2&&(current_instr.type2==A_NUM||current_instr.type2==A_PODLE_NUM||current_instr.type2==A_PODLE_IY_PLUS||current_instr.type2==A_PODLE_IX_PLUS))
 {
  x2=1;
  for (a=0;a<current_instr.arg2;a++,PC++)
   a2+=(memory_ptr[PC])<<(a<<3);
 }
 
 if (!current_instr.func){error(0,"","Undocumented instruction, haha.");return;}
 if (store_pc)PC=*store_pc;
 if (exec)
 current_instr.func(
  x1==1?a1:current_instr.arg1,
  current_instr.type1,
  x2==1?a2:current_instr.arg2,
  current_instr.type2
 );
}


/* prints instruction */
void
print(char *str)
{
 char txt[256];

 
 sprintf(txt," %04x  ",current_pc);
 c_goto(0,row);
 c_setcolor(row==12?15:11);
 c_print(txt);
 c_setcolor(row==12?15:7);
 c_print(str);
}


/* returns 1 if flag is set, 0 otherwise; on F_EMPTY returns always 1 */
_uchar
is_flag(_uchar f)
{
 int s=5,xor=0;

 switch(f)
 {
  case F_EMPTY:
  return 1;
  
  case F_NC:
  xor=1;
  case F_C:
  s=0;
  break;
  
  case F_NN:
  xor=1;
  case F_N:
  s=1;
  break;
  
  case F_PO:
  xor=1;
  case F_PE:
  s=2;
  break;
  
  case F_NH:
  xor=1;
  case F_H:
  s=4;
  break;
  
  case F_NZ:
  xor=1;
  case F_Z:
  s=6;
  break;
  
  case F_P:
  xor=1;
  case F_M:
  s=7;
  break;
 }
 return ((F&(1<<s))>>s)^xor;
}


void
set_flag(_uchar f)
{
 int a=0,s=5;

 switch(f)
 {
  case F_C:
  a=1;
  case F_NC:
  s=0;
  break;
  
  case F_N:
  a=1;
  case F_NN:
  s=1;
  break;
  
  case F_PE:
  a=1;
  case F_PO:
  s=2;
  break;
  
  case F_H:
  a=1;
  case F_NH:
  s=4;
  break;
  
  case F_Z:
  a=1;
  case F_NZ:
  s=6;
  break;
  
  case F_M:
  a=1;
  case F_P:
  s=7;
  break;
 }
 F&=255-(1<<s);		/* reset apropriate bit in flag register */
 F|=a<<s;		/* set the bit to value a */
}


void 
__print_rr(int x,int y,char *t,int R)
{
 char txt[256];
 c_goto(x,y);
 sprintf(txt,"%s=",t);
 c_setcolor(10);
 c_print(txt);
 sprintf(txt,"%05d 0x%04x",R,R);
 c_setcolor(7);
 c_print(txt);
}


void
tobin(char *s,_uchar x)
{
 int a;
 for (a=0;a<8;a++)
  s[7-a]=((x&(1<<a))>>a)?'1':'0';
 s[8]=0;
}


void 
__print_r(int x,int y,char *t,_uchar R)
{
 char txt[256];
 char b[256];
 tobin(b,R);
 c_goto(x,y);
 sprintf(txt,"%s=",t);
 c_setcolor(10);
 c_print(txt);
 sprintf(txt,"%03d (%s)",R,b);
 c_setcolor(7);
 c_print(txt);
}


void
print_regs(void)
{
 __print_rr(46,2,"AF",(A<<8)+F);
 __print_rr(46,3,"BC",(B<<8)+C);
 __print_rr(46,4,"DE",(D<<8)+E);
 __print_rr(46,5,"HL",(H<<8)+L);

 __print_rr(63,2,"AF'",(A_<<8)+F_);
 __print_rr(63,3,"BC'",(B_<<8)+C_);
 __print_rr(63,4,"DE'",(D_<<8)+E_);
 __print_rr(63,5,"HL'",(H_<<8)+L_);
 
 __print_rr(56,7,"IX",IX);
 __print_rr(56,8,"IY",IY);
              
 __print_rr(56,9,"SP",SP);
 __print_rr(56,10,"PC",PC);

 __print_r(46,12,"A",A);
 __print_r(46,13,"F",F);
 __print_r(46,14,"B",B);
 __print_r(46,15,"C",C);
 __print_r(46,16,"D",D);
 __print_r(46,17,"E",E);
 __print_r(46,18,"H",H);
 __print_r(46,19,"L",L);
 
 __print_r(63,12,"A'",A_);
 __print_r(63,13,"F'",F_);
 __print_r(63,14,"B'",B_);
 __print_r(63,15,"C'",C_);
 __print_r(63,16,"D'",D_);
 __print_r(63,17,"E'",E_);
 __print_r(63,18,"H'",H_);
 __print_r(63,19,"L'",L_);
 
 __print_r(56,21,"I",I);
 __print_r(56,22,"R",R);
 
}


void
print_flags(void)
{
 char txt[256];

 c_goto(46,0);
 c_setcolor(is_flag(F_M)?15:7);
 c_print(is_flag(F_P)?"P ":"M ");
 c_setcolor(is_flag(F_Z)?15:7);
 c_print(is_flag(F_Z)?" Z ":"NZ ");
 c_setcolor(is_flag(F_H)?15:7);
 c_print(is_flag(F_H)?" H ":"NH ");
 c_setcolor(is_flag(F_PE)?15:7);
 c_print(is_flag(F_PE)?"PE ":"PO ");
 c_setcolor(is_flag(F_N)?15:7);
 c_print(is_flag(F_N)?" N ":"NN ");
 c_setcolor(is_flag(F_C)?15:7);
 c_print(is_flag(F_C)?" C":"NC");

 sprintf(txt,"IM=%d",IM);
 c_goto(70,0);
 c_setcolor(15);
 c_print(txt);
 
 c_goto(66,0);
 c_setcolor(EI?15:7);
 c_print(EI?"EI":"DI");
 c_setcolor(7);
 
}


void
print_instr(void)
{
 _ushort old_pc=PC;
 
 c_clear(7,12,22,22);
 mode=M_PRINT;
 for (row=12;row<22;row++)
 {
  current_pc=PC;
  decode(0,1);
 }
 c_goto(7,11);
 c_setcolor(10);
 c_print("PROGRAM");
 c_goto(0,12);
 c_setcolor(15);
 c_print(">");
 c_setcolor(7);
 PC=old_pc;
}


void
print_stack(void)
{
 char txt[256];
 _ushort b,c,d;
 int r,a;

 c_goto(30,11);
 c_setcolor(10);
 c_print("STACK");
 c_setcolor(7);
 for (r=12,a=SP+12;a>=SP-6;a-=2,r++)
 {
  c=(a+65536)&65535;
  d=(a+1+65536)&65535;
  b=memory[c]+(memory[d]<<8);
  sprintf(txt,"%04x  ",c);
  c_goto(24,r);
  c_setcolor(r==18?15:11);
  c_print(txt);
  c_setcolor(r==18?15:7);
  sprintf(txt,"%05d %04x ",b,b);
  c_print(txt);
  sprintf(txt,"%c%c",__printable(memory[d]),__printable(memory[c]));
  c_setcolor(12);
  c_print(txt);
  c_setcolor(7);
 }
 c_goto(23,18);
 c_setcolor(15);
 c_print(">");
 c_setcolor(7);
}


#define M memory
void
print_mem(void)
{
 char txt[256];
 _ushort a;
 int r,c;
 _ushort memp;

 if (follow)memp=PC;
 else memp=MEMP;

 c_goto(14,0);
 c_setcolor(10);
 c_print("MEMORY");
 for (r=0,c=memp;c<memp+80;c+=8,r++)
 {
  a=(c+65536)&65535;
  sprintf(txt,"%04x  ",a);
  c_goto(0,1+r);
  c_setcolor(11);
  c_print(txt);
  c_setcolor(7);
  sprintf(txt,"%02x %02x %02x %02x  %02x %02x %02x %02x ",
  M[a],M[(_ushort)(a+1)],M[(_ushort)(a+2)],M[(_ushort)(a+3)],M[(_ushort)(a+4)],M[(_ushort)(a+5)],M[(_ushort)(a+6)],M[(_ushort)(a+7)]);
  c_print(txt);
  sprintf(txt,"%c%c%c%c%c%c%c%c",
  __printable(M[a]),
  __printable(M[(_ushort)(a+1)]),
  __printable(M[(_ushort)(a+2)]),
  __printable(M[(_ushort)(a+3)]),
  __printable(M[(_ushort)(a+4)]),
  __printable(M[(_ushort)(a+5)]),
  __printable(M[(_ushort)(a+6)]),
  __printable(M[(_ushort)(a+7)]));
  c_setcolor(12);
  c_print(txt);
  c_setcolor(7);
 } 
}
#undef M


void
print_status(void)
{
 c_goto(0,23);
 c_setcolor(13);
 c_print(follow?"FOLLOW ":"       ");
 c_goto(7,23);
 c_print(run?"RUN ":"    ");
 c_goto(11,23);
 c_print(turbo?"TURBO ":"      ");
 c_setcolor(7);
}


/* draw screen */
void
print_panel(void)
{
 print_regs();
 print_flags();
 print_instr();
 print_stack();
 print_mem();
 print_status();
}


void
show_keys(void)
{
 c_cls();
 #include "k_layout.h"
 c_goto(0,23);
 c_print("PRESS ANY KEY TO RETURN.");
 c_getkey();
 c_cls();
}

/* initialization of variables */
void
reset(void)
{
 PC=0;
 MEMP=0;
 A=0;B=0;C=0;D=0;E=0;F=0;H=0;L=0;I=0;R=0;
 A_=0;B_=0;C_=0;D_=0;E_=0;F_=0;H_=0;L_=0;
 SP=0;IX=0;IY=0;
 EI=0;IM=0;
 run=0;
 bzero(memory,65536);
}


void
ask_flag(void)
{
 unsigned char c;
 c_clear(0,23,79,23);
 c_goto(0,23);
 c_print("Toggle flag: Zero, Carry, Parity, Sign, H, N?");
 c=c_getkey();
 switch(c)
 {
  case 'z':
  case 'Z':
  set_flag(is_flag(F_Z)?F_NZ:F_Z);
  break;

  case 'c':
  case 'C':
  set_flag(is_flag(F_C)?F_NC:F_C);
  break;

  case 'p':
  case 'P':
  set_flag(is_flag(F_PE)?F_PO:F_PE);
  break;

  case 's':
  case 'S':
  set_flag(is_flag(F_M)?F_P:F_M);
  break;

  case 'h':
  case 'H':
  set_flag(is_flag(F_H)?F_NH:F_H);
  break;

  case 'n':
  case 'N':
  set_flag(is_flag(F_N)?F_NN:F_N);
  break;

  default:
  c_bell();
  break;
 }
 c_clear(0,23,79,23);
}


/* reads string from input (stored in pointer), writes prompt message */
/* string cannot be longer than max_len */
/* return value: 0=ok, 1=escape pressed */
int
ask_str(char *pointer,char *message,int max_len)
{
 unsigned char c;
 int a=strlen(pointer),l=strlen(message);
 
 c_clear(0,23,79,23);
 c_cursor(C_NORMAL);
 c_goto(0,23);
 c_setcolor(15);
 c_print(message);
 c_setcolor(7);
 c_print(pointer);
 while(1)
 {
  c=c_getkey();
  if (c==K_ESCAPE)
  {
   pointer[0]=0;
   c_clear(0,23,79,23);
   c_cursor(C_HIDE);
   return 1;
  }
  if (c==K_ENTER)
  {
   c_clear(0,23,79,23);
   c_cursor(C_HIDE);
   return 0;
  }
  if (c==K_BACKSPACE&&a)
  {
   a--;
   pointer[a]=0;
   c_clear(l+strlen(pointer),23,79,23);
   c_goto(l+strlen(pointer),23);
   continue;
  }
  if (c>=32&&a<max_len)
  {
   pointer[a]=c;   /* do charu se dava unsigned char */
   a++;
   pointer[a]=0;
   c_goto(l,23);
   c_print(pointer);
   c_clear(l+strlen(pointer),23,79,23);
   c_goto(l+strlen(pointer),23);
  }
 }
}


/* reads a non negative integer from input */
/* on escape or error returns -1 */
int
ask(char *str,int init_val)
{
 int val;
 static char txt[256];
 char *p;
 
 if (init_val)sprintf(txt,"%d",init_val);
 else txt[0]=0;
 if (ask_str(txt,str,16))return -1;
 val=strtol(txt,&p,0);
 if (*p)return -1;
 return val;
}


/*-------------------------------MAIN-----------------------------------------*/
int
main(int argc,char **argv)
{
 unsigned char c=0;
 int b,a;
 int start;
 FILE *stream;
 int x;

 if (argc>1&&(!strcmp(argv[1],"-h")||!strcmp(argv[1],"--help")))
 {
  printf("Z80 disassembler\n(c)1999-2000 Brainsoft\n");
  printf("Usage: z80-dis [<filename> ...]\n");
  return 0;
 }

 reset();

 for (b=1;b<argc;b++)
 {
  stream=fopen(argv[b],"r");
  if (!stream){fprintf(stderr,"Error: Can't read file \"%s\".\n",argv[b]);return(1);}
  if (read_header(stream,&start,&x)){fprintf(stderr,"Error: \"%s\" is not a Z80 ASM file.\n",argv[b]);return(1);}
  fread(memory+start,1,x+start<65536?x:65536-start,stream);
  fclose(stream);
 }
 
 c_init();
 c_cursor(C_HIDE);
 follow=1;
 disable_defx=0;
 memory_ptr=memory;
 print_panel();
 while (1)
 {
  c=0;
  if (run)
  {
   mode=M_EXEC;
   decode(0,1);
   if (!turbo)print_panel();
   else print_regs();
  }
  else
  {
   usleep(1000);
  }
  if (c_kbhit())c=c_getkey();
  switch(c)
  {
   case '?':
   case 'h':
   case 'H':
   show_keys();
   print_panel();
   break;
   
   case 'q': /* quit */
   case 'Q':
   c_shutdown();
   return 0;

   case 'x':
   case 'X':
   disable_defx=1;
   address=0;
   memory_ptr=tmp_memory;
   string[0]=0;
   ask_str(string,"Execute instruction: ",40);
   compile();
   address=PC;
   PC=0;
   mode=M_EXEC;
   decode(&address,1);
   memory_ptr=memory;
   print_panel();
   disable_defx=0;
   break;
   
   case ' ':  /* move to next instruction */
   mode=M_EXEC;
   decode(0,0);
   print_panel();
   break;

   case 'e': /* exec one instruction */
   case 'E':
   if (run)break;
   mode=M_EXEC;
   decode(0,1);
   print_panel();
   break;

   case 'f': /* toggle flag */
   case 'F':
   ask_flag();
   print_panel();
   break;
   
   case 's': /* change SP */
   x=ask("SP=",SP);
   if (x<0||x>65535){c_bell();print_status();break;}
   run=0;
   SP=x;
   print_panel();
   break;
   
   case 't': /* turbo */
   case 'T':
   turbo^=1;
   print_panel();
   break;

   case 'r': /* run */
   case 'R':
   run^=1;
   print_panel();
   break;

   case 'i': /* change IM */
   case 'I':
   IM++;
   IM%=3;
   print_panel();
   break;

   case K_ENTER: /* enter instruction */
   string[0]=0;
   ask_str(string,"Put instruction: ",40);
   address=PC;
   compile();
   print_panel();
   break;
   
   case 'd': /* toggle EI/DI */
   case 'D':
   EI^=1;
   print_panel();
   break;

   case '*': /* reset */
   reset();
   print_panel();
   break;

   case 'p': /* ask about address */
   case 'P':
   x=ask("PC=",PC);
   if (x<0||x>65535){c_bell();print_status();break;}
   run=0;
   PC=x;
   print_panel();
   break;

   case 'a':  /* defs */
   case 'A':
   string[0]='d';
   string[1]='e';
   string[2]='f';
   string[3]='s';
   string[4]=' ';
   string[5]='"';
   string[6]=0;
   if (ask_str(string+6,"defs ",64))break;
   string[strlen(string)]='"';
   address=MEMP;
   compile();
   print_panel();
   break;
   
   case 'w':  /* defw */
   case 'W':
   string[0]='d';
   string[1]='e';
   string[2]='f';
   string[3]='w';
   string[4]=' ';
   string[5]=0;
   if (ask_str(string+5,"defw ",64))break;
   address=MEMP;
   compile();
   print_panel();
   break;
   
   case 'b':  /* defb */
   case 'B':
   string[0]='d';
   string[1]='e';
   string[2]='f';
   string[3]='b';
   string[4]=' ';
   string[5]=0;
   if (ask_str(string+5,"defb ",64))break;
   address=MEMP;
   compile();
   print_panel();
   break;
   
   case K_TAB:
   follow^=1;
   print_panel();
   break;
   
   case 'm': /* ask about start of memory dump */
   case 'M':
   follow=0;
   x=ask("Enter address: ",MEMP);
   if (x<0||x>65535){c_bell();break;}
   MEMP=x;
   print_panel();
   break;

   case 'L':   /* load */
   if (ask_str(load_name,"Load file: ",40)){c_bell();print_status();break;}
   stream=fopen(load_name,"r");
   if (!stream){error(0,string,"Can't read file.");print_status();break;}
   if (read_header(stream,&a,&x)){error(0,string,"Not a Z80 ASM file.");print_status();fclose(stream);break;}
   fread(memory+a,1,x+a<65536?x:65536-a,stream);
   fclose(stream);
   print_panel();
   break;

   case 'S':   /* save */
   if (ask_str(save_name,"Save as: ",40)){c_bell();break;}
   x=ask("Length: ",0);
   if (x<=0||PC+x>65536)break;
   stream=fopen(save_name,"w");
   if (!stream){error(0,string,"Can't write to file.");print_status();break;}
   write_header(stream,PC);
   fwrite(memory+PC,1,x,stream);
   fclose(stream);
   print_panel();
   break;
  }
 }
}
